/*
** Implementation for the CPyFactory class
*/

#include "stdafx.h"

#include <Import.h>		/* for PyImport_ImportModule() */

#include "PythonCOM.h"
#include "PyFactory.h"
#include "PythonCOMServer.h"

extern void PyCom_LogF(const char *fmt, ...);
#define LogF PyCom_LogF

CPyFactory::CPyFactory(REFCLSID guidClassID) :
	m_guidClassID(guidClassID),
	m_cRef(1)
{
	PyCom_DLLAddRef();
}
CPyFactory::~CPyFactory()
{
	PyCom_DLLReleaseRef();
}

STDMETHODIMP CPyFactory::QueryInterface(REFIID iid, void **ppv)
{
	*ppv = NULL;

	if ( IsEqualIID(iid, IID_IUnknown) ||
		 IsEqualIID(iid, IID_IClassFactory) )
	{
		*ppv = this;
		AddRef();
		return S_OK;
	}

	return E_NOINTERFACE;
}

STDMETHODIMP_(ULONG) CPyFactory::AddRef(void)
{
	return InterlockedIncrement(&m_cRef);
}

STDMETHODIMP_(ULONG) CPyFactory::Release(void)
{
	LONG cRef = InterlockedDecrement(&m_cRef);
	if ( cRef == 0 )
		delete this;
	return cRef;
}

STDMETHODIMP CPyFactory::CreateInstance(
	IUnknown *punkOuter,
	REFIID riid,
	void **ppv
	)
{
//	LogF("in CPyFactory::CreateInstance");

	if ( ppv == NULL )
		return CLASS_E_NOAGGREGATION;
	*ppv = NULL;

	if ( punkOuter != NULL )
		return E_INVALIDARG;

	PyObject *pNewInstance;
	HRESULT hr = CreateNewPythonInstance(m_guidClassID, riid, &pNewInstance);
	if ( FAILED(hr) )
	{
		LogF("CPyFactory::CreateInstance failed to create instance. (%lx)", hr);
		return hr;
	}

/**********
   NEW CODE - CreateInstance now returns an object already all wrapped
   up (giving more flexibility to the Python programmer.
**********/
	// Note we only pass IID_IUnknown, as the Python programmer may be pulling
	// tricks.  We assume they have done the right thing. We return their
	// gateway as it is (without checking is really is of type IID*
	if (!PyCom_InterfaceFromPyObject(pNewInstance, IID_NULL, ppv, FALSE)) {
		LogF("CPyFactory::CreateInstance failed to get gateway to returned object");
		hr = E_FAIL;
	}
	PyTS_DECREF(pNewInstance); // Dont need it any more.

/******** 
    OLD CODE

	// Note we dont need to QI, as we make a gateway of the specific type
	// first time around.  QI _will_ be called on the resulting object by
	// the caller, but QI will return "this" - ie, we prevent the creation
	// of a temporary PyGUnknown object.
	hr = PyCom_MakeRegisteredGatewayObject(riid, pNewInstance, ppv);
	if ( SUCCEEDED(hr) )
	{
		// pNewInstance has a new Python ref in the new object.
		PyTS_DECREF(pNewInstance);
//		pPyCOM->m_pPyObject = pNewInstance;
//		hr = pPyCOM->QueryInterface(riid, ppv);
	}
	else
	{
		LogF("failed to create gateway. (%lx)", hr);
	}
//	pPyCOM->Release();
*/
	return hr;
}

STDMETHODIMP CPyFactory::LockServer(BOOL fLock)
{
	if ( fLock )
		PyCom_DLLAddRef();
	else
		PyCom_DLLReleaseRef();

	return S_OK;
}

STDMETHODIMP CPyFactory::CreateNewPythonInstance(REFCLSID rclsid, REFCLSID rReqiid, PyObject **ppNewInstance)
{
	extern BOOL LoadGatewayModule(PyObject **);
	PyObject *pPyModule;

	if ( ppNewInstance == NULL )
		return E_INVALIDARG;

	if ( !LoadGatewayModule(&pPyModule) )
		return E_FAIL;

	PyCom_EnterPython();
	{
		PyObject *obiid = PyCom_PyIIDObjectFromIID(rclsid);
		PyObject *obReqiid = PyCom_PyIIDObjectFromIID(rReqiid);
		if ( !obiid || !obReqiid)
		{
			Py_XDECREF(obiid);
			Py_XDECREF(obReqiid);
			PyErr_Clear(); // nothing Python can do!
			PyCom_LeavePython();
			return E_OUTOFMEMORY;
		}

		// If we ignore Factory reference counts, there is a possibility
		// that the DLL global ref count will transition 1->0->1 during the
		// creation process.  To prevent this, we add an artificial lock
		// and remove it when done.
		PyCom_DLLAddRef();
		*ppNewInstance = PyObject_CallMethod(pPyModule, "CreateInstance",
											 "OO", obiid, obReqiid);
		Py_DECREF(obiid);
		Py_DECREF(obReqiid);
		PyCom_DLLReleaseRef();


//		if ( !*ppNewInstance )PyRun_SimpleString("import traceback;traceback.print_exc()");
		PyErr_Clear();	/* ### what to do with exceptions? ... */
	}
	PyCom_LeavePython();

	if ( !*ppNewInstance )
	{
		LogF("ERROR: server.policy could not create an instance.");
		return E_FAIL;
	}

	return S_OK;
}

/*
** Load our C <-> Python gateway module if needed
*/
static PyObject *g_pPyModule = NULL;
BOOL LoadGatewayModule(PyObject **ppModule)
{
	/* fast path */
	if ( g_pPyModule != NULL )
	{
		*ppModule = g_pPyModule;
		return TRUE;
	}

	/* we want just one thread performing this */
	DLLAcquireGlobalLock();
	{
		/* still NULL? */
		if ( g_pPyModule == NULL )
		{
			AllocThreadState();
			PyObject *module;

			module = PyImport_ImportModule("ni");
			if ( module )
			{
				PyObject *importer = PyObject_CallMethod(module, "PackageImporter", NULL);
				Py_DECREF(module);

				if ( importer )
				{
					module = PyObject_CallMethod(importer,
												 "import_module",
												 "s",
												 "win32com.server.policy");
					Py_DECREF(importer);

					if ( module )
					{
						PyObject *module2 = PyObject_GetAttrString(module, "server");
						Py_DECREF(module);

						if ( module2 )
						{
							g_pPyModule = PyObject_GetAttrString(module2, "policy");
							Py_DECREF(module2);
						}
					}
				}
			}

			FreeThreadState();
		}
	}
	DLLReleaseGlobalLock();

	if ( !g_pPyModule )
	{
		LogF("PythonCOM Server - The policy module could not be loaded.");
//		PyRun_SimpleString("import traceback;traceback.print_exc()");
		/* ### propagate the exception? */
		PyErr_Clear();
		return FALSE;
	}

	*ppModule = g_pPyModule;
	return TRUE;
}
void FreeGatewayModule(void)
{
	if ( g_pPyModule != NULL )
	{
		Py_DECREF(g_pPyModule);
		g_pPyModule = NULL;
	}
}
