// PyComHelpers.cpp
//
// Most of the PyCom_ helpers.

// @doc
#include "stdafx.h"
#include "PythonCOM.h"
#include "PythonCOMServer.h"
#include "PyWinObjects.h" // Until this is converted to the new API

extern PyObject *g_obPyCom_MapIIDToType;
extern PyObject *g_obPyCom_MapServerIIDToGateway;

// String conversions
// Convert a Python string object to a BSTR - allow embedded NULLs, etc.
BSTR PyCom_BstrFromPyString(PyObject *stringObject)
{
	int size=PyString_Size(stringObject);
	const char *buf = PyString_AsString(stringObject);
	if (buf==NULL) return NULL;
	int wideSize = size*2;
	LPWSTR wstr = (LPWSTR)alloca(wideSize);
	size = MultiByteToWideChar(CP_ACP, 0, buf, size, wstr, wideSize);
	return SysAllocStringLen(wstr, size);
}

// Convert a Python object to a BSTR - allow embedded NULLs, None, etc.
BOOL PyCom_BstrFromPyObject(PyObject *stringObject, BSTR *pResult, BOOL bNoneOK /*= FALSE*/)
{
	if (PyString_Check(stringObject))
		*pResult = PyCom_BstrFromPyString(stringObject);
	else if (PyUnicode_Check(stringObject))
		*pResult = SysAllocString(((PyUnicode *)stringObject)->m_bstrValue);
	else if (stringObject == Py_None) {
		if (bNoneOK) {
			*pResult = NULL;
			return TRUE;
		} else {
			OleSetTypeError("None is not a valid string in this context");
			return FALSE;
		}
	}
	if (!pResult) {
		OleSetMemoryError("Allocating BSTR");
		return FALSE;
	}
	return TRUE;
}

// MakeBstrToObj - convert a BSTR into a Python string.
//
// ONLY USE THIS FOR TRUE BSTR's - Use the fn below for OLECHAR *'s.
// NOTE - does not use standard macros, so NULLs get through!
PyObject *MakeBstrToObj(const BSTR bstr)
{
	if (bstr==NULL) {
		Py_INCREF(Py_None);
		return Py_None;
	}

	int numChars = SysStringLen(bstr);
	int numBytes = numChars * 2;
	LPSTR str = (LPSTR)alloca(numBytes);
   	numChars = WideCharToMultiByte(CP_ACP, 0, bstr, numChars, str, numBytes, NULL, NULL);

	return PyString_FromStringAndSize(str, numChars);
}

// Size info is available (eg, a fn returns a string and also fills in a size variable)
PyObject *MakeOLECHARToObj(const OLECHAR * str, int numChars)
{
	if (str==NULL) {
		Py_INCREF(Py_None);
		return Py_None;
	}
	int numBytes = numChars * 2;
	LPSTR ascstr = (LPSTR)alloca(numBytes);
   	numChars = WideCharToMultiByte(CP_ACP, 0, str, numChars, ascstr, numBytes, NULL, NULL);
	return PyString_FromStringAndSize(ascstr, numChars);
}

// No size info avail.
PyObject *MakeOLECHARToObj(const OLECHAR * str)
{
	USES_CONVERSION;
	if (str==NULL) {
		Py_INCREF(Py_None);
		return Py_None;
	}
	return PyString_FromString(OLE2T(str));
}

// Interface conversions
PyObject *PyCom_PyObjectFromIUnknown(IUnknown *punk, REFIID riid, BOOL bAddRef /* = FALSE */)
{
	// Quick exit.
	if (punk==NULL) {
		Py_INCREF(Py_None);
		return Py_None;
	}

	// Look up the map, and create the object.
	PyObject *obiid = PyCom_PyIIDObjectFromIID(riid);
	if (!obiid) return NULL;

	PyObject *createType = PyDict_GetItem(g_obPyCom_MapIIDToType, obiid);
	PyTS_DECREF(obiid);
	if (createType==NULL) {
		PyErr_Clear();
		return OleSetTypeError("There is no interface object registered that supports this IID");
	}
	if (!PyType_Check(createType)) {
		return OleSetTypeError("The Python IID map is invalid - the value is not a Python type object");
	}
	// Should check it is actually a type of mine?
	PyComTypeObject *myCreateType = (PyComTypeObject *)createType;
	if (myCreateType->ctor==NULL) {
		return OleSetTypeError("The type does not declare a PyCom constructor");
	}
	PyIUnknown *ret = (*myCreateType->ctor)(punk);
#ifdef _DEBUG_LIFETIMES
	PyCom_LogF("Object %s created at 0x%0xld, IUnknown at 0x%0xld",
		 myCreateType->tp_name, ret, ret->m_obj);
#endif
	if (ret && bAddRef && punk) punk->AddRef();
	return ret;
}


BOOL PyCom_InterfaceFromPyInstanceOrObject(PyObject *ob, REFIID iid, LPVOID *ppv, BOOL bNoneOK /* = TRUE */)
{
	if (ob && PyInstance_Check(ob)) {
		// Get the _oleobj_ attribute
		ob = PyObject_GetAttrString(ob, "_oleobj_");
		if (ob==NULL) return FALSE;
	} else {
		Py_XINCREF(ob);
	}
	BOOL rc = PyCom_InterfaceFromPyObject(ob, iid, ppv, bNoneOK );
	Py_XDECREF(ob);
	return rc;
}

BOOL PyCom_InterfaceFromPyObject(PyObject *ob, REFIID iid, LPVOID *ppv, BOOL bNoneOK /* = TRUE */)
{
	if ( ob == NULL )
	{
		// don't overwrite an error message
		if ( !PyErr_Occurred() )
			OleSetTypeError("the Python object is invalid");
		return FALSE;
	}
	if ( ob == Py_None )
	{
		if ( bNoneOK )
		{
			*ppv = NULL;
			return TRUE;
		}
		else
		{
			OleSetTypeError("None is not a invalid interface object in this context");
			return FALSE;
		}
	}

	if ( !PyIBase::is_object(ob, &PyIUnknown::type) )
	{
		PyErr_SetString(PyExc_ValueError, "argument is not a COM object");
		return FALSE;
	}
	IUnknown *punk = PyIUnknown::GetI(ob);
	if ( !punk )
		return FALSE;	/* exception was set by GetI() */
	/* note: we don't explicitly hold a reference to punk */

	HRESULT hr = punk->QueryInterface(iid, ppv);
	if ( FAILED(hr) )
	{
		OleSetOleError(hr);
		return FALSE;
	}
	/* note: the QI added a ref for the return value */

	return TRUE;
}

BOOL PyCom_IUnknownFromPyObject(PyObject *obUnk, IUnknown **ppunk, BOOL bNoneOK /* = TRUE */)
{
	if ( !PyCom_InterfaceFromPyObject(obUnk, IID_IUnknown, (LPVOID*)ppunk, bNoneOK) )
		return FALSE;

	// remove an extra addref
	// ### we shouldn't be doing this... the return value should have the ref
	if ( *ppunk != NULL )
		(*ppunk)->Release();

	return TRUE;
}

HRESULT PyCom_MakeRegisteredGatewayObject(REFIID iid, PyObject *instance, void **ppv)
{
	if ( g_obPyCom_MapServerIIDToGateway == NULL )
		return E_NOINTERFACE;

	HRESULT hr = E_FAIL;

	PyCom_EnterPython();
	{
		PyObject *keyObject = PyCom_PyIIDObjectFromIID(iid);
		if ( keyObject )
		{
			PyObject *valueObject = PyDict_GetItem(g_obPyCom_MapServerIIDToGateway,keyObject);
			Py_DECREF(keyObject);
			if ( valueObject )
			{
				pfnPyGatewayConstructor ctor = (pfnPyGatewayConstructor)PyInt_AsLong(valueObject);
				// ctor takes reference count to instance.
				hr = (*ctor)(instance, ppv, iid);
			}
			else
			{
				hr = E_NOINTERFACE;
			}
		}
	}
	PyCom_LeavePython();

	return hr;
}

/* Converts a STATSTG structure to a Python tuple

  NOTE - DOES NOT free the string - this is the callers responsibility
  (see the STATSTG doco for details)
*/
// @object STATSTG|A tuple representing a STATSTG structure
PyObject *PyCom_PyObjectFromSTATSTG(STATSTG *pStat)
{
	USES_CONVERSION;
	if (pStat==NULL) {
		Py_INCREF(Py_None);
		return Py_None;
	}
 	PyObject *obSize = NULL;
	PyObject *obmtime = NULL;
	PyObject *obctime = NULL;
	PyObject *obatime = NULL;
	PyObject *obCLSID = NULL;
	obSize = PyObjectFromULARGE_INTEGER(pStat->cbSize);
	obmtime = new PyTime(pStat->mtime);
	obctime = new PyTime(pStat->ctime);
	obatime = new PyTime(pStat->atime);
	obCLSID = PyCom_PyIIDObjectFromIID(pStat->clsid);
	// Seems to be a bug somewhere.
//	PyObject *obName = new PyUnicode(pStat->pwcsName+1);
	char *szName = pStat->pwcsName==NULL ? NULL : OLE2T(pStat->pwcsName);
	PyObject *result = Py_BuildValue("ziOOOOiiOii", 
		           szName, // @pyparm string|name||The name of the storage object
				   pStat->type, // @pyparm int|type||Indicates the type of storage object. This is one of the values from the storagecon.STGTY_* values.
				   obSize, // @pyparm <o ULARGE_INTEGER>|size||Specifies the size in bytes of the stream or byte array.
				   obmtime, // @pyparm <o PyTime>|modificationTime||Indicates the last modification time for this storage, stream, or byte array.
				   obctime,	 // @pyparm <o PyTime>|creationTime||Indicates the creation time for this storage, stream, or byte array.
				   obatime,	 // @pyparm <o PyTime>|accessTime||Indicates the last access time for this storage, stream or byte array.
				   pStat->grfMode, // @pyparm int|mode||Indicates the access mode specified when the object was opened. This member is only valid in calls to Stat methods.
				   pStat->grfLocksSupported,// @pyparm int|locksSupported||Indicates the types of region locking supported by the stream or byte array. See the storagecon.LOCKTYPES_* constants for the values available. This member is not used for storage objects.
				   obCLSID, // @pyparm <o PyIID>|clsid||Indicates the class identifier for the storage object; set to CLSID_NULL for new storage objects. This member is not used for streams or byte arrays.
				   pStat->grfStateBits, // @pyparm int|stateBits||Indicates the current state bits of the storage object, that is, the value most recently set by the <om PyIStorage.SetStateBits> method. This member is not valid for streams or byte arrays.
				   pStat->reserved);  // @pyparm int|storageFormat||Indicates the format of the storage object. This is one of the values from the STGFMT_* constants (yeah right - these constants dont yet exist!).
	Py_XDECREF(obSize);
	Py_XDECREF(obmtime);
	Py_XDECREF(obctime);
	Py_XDECREF(obatime);
	Py_XDECREF(obCLSID);
	return result;
}

BOOL PyCom_PyObjectAsSTATSTG(PyObject *ob, STATSTG *pStat, DWORD flags /* = 0 */)
{
	char *szName;
	PyObject *obSize;
	PyObject *obmtime, *obctime, *obatime;
	PyObject *obCLSID;
	if (!PyArg_ParseTuple(ob, "ziOOOOiiOii",
		                &szName,
						&pStat->type,
						&obSize,
						&obmtime,
						&obctime,
						&obatime,
						&pStat->grfMode,
						&pStat->grfLocksSupported,
						&obCLSID,
						&pStat->grfStateBits,
						&pStat->reserved))
		return NULL;
	pStat->pwcsName = NULL; // XXX - need to fix this
	// When fixed, should honour the STATFLAG_NONAME
	if (!PyObjectToULARGE_INTEGER(obSize, &pStat->cbSize))
		return FALSE;
	if (!PyTime_Check(obmtime) || !PyTime_Check(obctime) || !PyTime_Check(obatime)) {
		OleSetTypeError("The time entries in a STATSTG tuple must be PyTime objects");
		return FALSE;
	}
	if (!((PyTime *)obmtime)->GetTime(&pStat->mtime))
		return FALSE;
	if (!((PyTime *)obctime)->GetTime(&pStat->ctime))
		return FALSE;
	if (!((PyTime *)obatime)->GetTime(&pStat->atime))
		return FALSE;
	if (!PyCom_CLSIDFromPyObject(obCLSID, &pStat->clsid))
		return FALSE;
	return TRUE;
}
