"""Python.Dictionary COM Server.

This module implements a simple COM server that acts much like a Python
dictionary or as a standard string-keyed VB Collection.  The keys of
the dictionary are strings and are case-insensitive.

It uses a highly customized policy to fine-tune the behavior exposed to
the COM client.

The object exposes the following properties:

    int Count                       (readonly)
    VARIANT Item(BSTR key)          (propget for Item)
    Item(BSTR key, VARIANT value)   (propput for Item)

    Note that 'Item' is the default property, so the following forms of
    VB code are acceptable:

        set ob = CreateObject("Python.Dictionary")
        ob("hello") = "there"
        ob.Item("hi") = ob("HELLO")

All keys are defined, returning VT_NULL (None) if a value has not been
stored.  To delete a key, simply assign VT_NULL to the key.

The object responds to the _NewEnum method by returning an enumerator over
the dictionary's keys. This allows for the following type of VB code:

    for each name in ob
        debug.print name, ob(name)
    next
"""

import string
import ni
import win32com.pythoncom
from win32com.server import util, policy, exception
import winerror
import types
import pywintypes

from win32com.pythoncom import DISPATCH_METHOD, DISPATCH_PROPERTYGET
from winerror import S_OK

UnicodeType = pywintypes.UnicodeType
StringType = types.StringType


class DictionaryPolicy(policy.BasicWrapPolicy):
  ### BasicWrapPolicy looks for this
  _com_interfaces_ = [ ]

  ### BasicWrapPolicy looks for this
  _name_to_dispid_ = {
    'item' : win32com.pythoncom.DISPID_VALUE,
    'count' : 1,
    }

  def _CreateInstance_(self, clsid, reqIID):
    self._wrap_({ })
    return win32com.pythoncom.WrapObject(self, reqIID)

  def _wrap_(self, ob):
    self._obj_ = ob	# ob should be a dictionary

  def _invoke_(self, dispid, lcid, wFlags, args):
    if dispid == 0:	# item
      l = len(args)
      if l < 1:
        return winerror.DISP_E_BADPARAMCOUNT

      key = args[0]
      if type(key) == UnicodeType:
        pass
      elif type(key) == StringType:
        key = pywintypes.Unicode(key)
      else:
        ### the nArgErr thing should be 0-based, not reversed... sigh
        return winerror.DISP_E_TYPEMISMATCH, len(args) - 1

      key = key.lower()

      if wFlags & (DISPATCH_METHOD | DISPATCH_PROPERTYGET):
        if l > 1:
          return winerror.DISP_E_BADPARAMCOUNT
        try:
          return S_OK, 0, self._obj_[key]
        except KeyError:
          return S_OK, 0, None	# unknown keys return None (VT_NULL)

      if l > 2:
        return winerror.DISP_E_BADPARAMCOUNT
      if args[1] is None:
        # delete a key when None is assigned to it
        try:
          del self._obj_[key]
        except KeyError:
          pass
      else:
        self._obj_[key] = args[1]
      return S_OK

    if dispid == 1:	# count
      if not wFlags & DISPATCH_PROPERTYGET:
        return winerror.DISP_E_MEMBERNOTFOUND	# not found
      if len(args) != 0:
        return winerror.DISP_E_BADPARAMCOUNT
      return S_OK, 0, len(self._obj_)

    if dispid == win32com.pythoncom.DISPID_NEWENUM:
      return S_OK, 0, util.NewEnum(self._obj_.keys())

    return winerror.DISP_E_MEMBERNOTFOUND

  def _getidsofnames_(self, names, lcid):
    ### this is a copy of MappedWrapPolicy._getidsofnames_ ...

    # Note: these names will always be StringType
    name = string.lower(names[0])
    try:
      return (self._name_to_dispid_[name],)
    except KeyError:
      raise exception.Exception(scode=winerror.DISP_E_MEMBERNOTFOUND,
                                desc="Member not found")


if __name__ == '__main__':
  from win32com.server.register import RegisterServer
  RegisterServer('{39b61048-c755-11d0-86fa-00c04fc2e03e}',
                 className='Python Dictionary',
                 progID='Python.Dictionary',
                 verProgID='Python.Dictionary.1',
                 policy='win32com.servers.dictionary.DictionaryPolicy'
                 )
  print 'Registered: Python.Dictionary'
