"""A utility for browsing COM objects.

 Usage:

  Command Prompt

    Use the command *"python.exe catbrowse.py"*.  This will display
    display a fairly small, modal dialog.

  Pythonwin

    Use the "Run Script" menu item, and this will create the browser in an
    MDI window.  This window can be fully resized.

 Details

   This module allows browsing of registered Type Libraries, COM categories, 
   and running COM objects.  The display is similar to the Pythonwin object
   browser, and displays the objects in a hierarchical window.

   Note that this module requires the win32ui (ie, Pythonwin) diestribution to
   work.

"""
import win32con
import win32api, win32ui
import string
import ni, win32com.pythoncom
pythoncom=win32com.pythoncom
from win32com.client import util
import browser

class HLIRoot(browser.HLIPythonObject):
	def __init__(self, title):
		self.name = title
	def GetSubList(self):
		return [HLIHeadingCategory(), HLI_IEnumMoniker(win32com.pythoncom.GetRunningObjectTable().EnumRunning(), "Running Objects"), HLIHeadingRegisterdTypeLibs()]

class HLICOM(browser.HLIPythonObject):
	def GetText(self):
		return self.name
	def CalculateIsExpandable(self):
		return 1

class HLICLSID(HLICOM):
	def __init__(self, myobject, name=None ):
		if type(myobject)==type(''):
			myobject = win32com.pythoncom.MakeIID(myobject)
		if name is None:
			try:
				name = win32com.pythoncom.ProgIDFromCLSID(myobject)
			except win32com.pythoncom.com_error: 
				name = str(myobject)
			name = "IID: " + name
		HLICOM.__init__(self, myobject, name)
	def CalculateIsExpandable(self):
		return 0
	def GetSubList(self):
		return []

class HLI_Interface(HLICOM):
	pass

class HLI_Enum(HLI_Interface):
	def GetBitmapColumn(self):
		return 0 # Always a folder.
	def CalculateIsExpandable(self):
		if self.myobject is not None:
			rc = len(self.myobject.Next(1))>0
			self.myobject.Reset()
		else:
			rc = 0
		return rc
	pass

class HLI_IEnumMoniker(HLI_Enum):
	def GetSubList(self):
		ctx = win32com.pythoncom.CreateBindCtx()
		ret = []
		for mon in util.Enumerator(self.myobject):
			ret.append(HLI_IMoniker(mon, mon.GetDisplayName(ctx, None)))
		return ret

class HLI_IMoniker(HLI_Interface):
	def GetSubList(self):
		ret = []
		ret.append(browser.MakeHLI(self.myobject.Hash(), "Hash Value"))
		subenum = self.myobject.Enum(1)
		ret.append(HLI_IEnumMoniker(subenum, "Sub Monikers"))
		return ret

class HLIHeadingCategory(HLICOM):
	"A tree heading for registered categories"
	def GetText(self):
		return "Registered Categories"
	def GetSubList(self):
		catinf=win32com.pythoncom.CoCreateInstance(pythoncom.CLSID_StdComponentCategoriesMgr,None,win32com.pythoncom.CLSCTX_INPROC,pythoncom.IID_ICatInformation)
		enum=util.Enumerator(catinf.EnumCategories())
		ret = []
		for catid, lcid, desc in enum:
			ret.append(HLICategory((catid, lcid, desc)))
		return ret

class HLICategory(HLICOM):
	"An actual Registered Category"
	def GetText(self):
		desc =  self.myobject[2]
		if not desc: desc = "(unnamed category)"
		return desc
	def GetSubList(self):
		win32ui.DoWaitCursor(1)
		catid, lcid, desc = self.myobject
		catinf=win32com.pythoncom.CoCreateInstance(pythoncom.CLSID_StdComponentCategoriesMgr,None,win32com.pythoncom.CLSCTX_INPROC,pythoncom.IID_ICatInformation)
		ret = []
		for clsid in util.Enumerator(catinf.EnumClassesOfCategories((catid,),())):
			ret.append(HLICLSID(clsid))
		win32ui.DoWaitCursor(0)

		return ret

class HLIHelpFile(HLICOM):
	def CalculateIsExpandable(self):
		return 0
	def GetText(self):
		import os
		fname, ctx = self.myobject
		base = os.path.split(fname)[1]
		return "Help reference in %s" %( base)

	def TakeDefaultAction(self):
		fname, ctx = self.myobject
		if ctx:
			cmd = win32con.HELP_CONTEXT
		else:
			cmd = win32con.HELP_FINDER
		win32api.WinHelp(win32ui.GetMainFrame().GetSafeHwnd(), fname, cmd, ctx)
	def GetBitmapColumn(self):
		return 6

class HLIRegisteredTypeLibrary(HLICOM):
	def GetSubList(self):
		import os
		clsidstr, versionStr = self.myobject
		collected = []
		helpPath = ""
		key = win32api.RegOpenKey(win32con.HKEY_CLASSES_ROOT, "TypeLib\\%s\\%s" % (clsidstr, versionStr))
		win32ui.DoWaitCursor(1)
		try:
			num = 0
			while 1:
				try:
					subKey = win32api.RegEnumKey(key, num)
				except win32api.error:
					break
				value = win32api.RegQueryValue(key, subKey)
				if subKey=="HELPDIR":
					helpPath = value
				elif subKey=="Flags":
					flags = value
				else:
					try:
						lcid = string.atof(subKey)
						lcidkey = win32api.RegOpenKey(key, subKey)
						# Enumerate the platforms
						lcidnum = 0
						while 1:
							try:
								platform = win32api.RegEnumKey(lcidkey, lcidnum)
							except win32api.error:
								break
							fname = win32api.RegQueryValue(lcidkey, platform)
							collected.append(lcid, platform, fname)	
							lcidnum = lcidnum + 1
						win32api.RegCloseKey(lcidkey)
					except ValueError:
						pass
				num = num + 1
		finally:
			win32ui.DoWaitCursor(0)
			win32api.RegCloseKey(key)
		# Now, loop over my collected objects, adding a TypeLib and a HelpFile
		ret = []
#		if helpPath: ret.append(browser.MakeHLI(helpPath, "Help Path"))
		ret.append(HLICLSID(clsidstr))
		for lcid, platform, fname in collected:
			extraDescs = []
			if platform!="win32":
				extraDescs.append(platform)
			if lcid:
				extraDescs.append("locale=%s"%lcid)
			extraDesc = ""
			if extraDescs: extraDesc = " (%s)" % string.join(extraDescs, ", ")
			ret.append(HLITypeLib(fname, "Type Library" + extraDesc))
		return ret

class HLITypeLibEntry(HLICOM):
	def GetText(self):
		tlb, index = self.myobject
		name, doc, ctx, helpFile = tlb.GetDocumentation(index)
		try:
			typedesc = HLITypeKinds[tlb.GetTypeInfoType(index)][1]
		except KeyError:
			typedesc = "Unknown!"
		return name + " - " + typedesc
	def GetSubList(self):
		tlb, index = self.myobject
		name, doc, ctx, helpFile = tlb.GetDocumentation(index)
		ret = []
		if doc: ret.append(browser.HLIDocString(doc, "Doc"))
		if helpFile: ret.append(HLIHelpFile(	(helpFile, ctx) ))
		return ret

class HLITypeLibMethod(HLITypeLibEntry):
	def __init__(self, ob, name = None):
		self.entry_type = "Method"
		HLITypeLibEntry.__init__(self, ob, name)
	def GetSubList(self):
		ret = HLITypeLibEntry.GetSubList(self)
		tlb, index = self.myobject
		typeinfo = tlb.GetTypeInfo(index)
		attr = typeinfo.GetTypeAttr()
		for i in range(attr[7]):
			ret.append(HLITypeLibProperty((typeinfo, i)))
		for i in range(attr[6]):
			ret.append(HLITypeLibFunction((typeinfo, i)))
		return ret

class HLITypeLibEnum(HLITypeLibEntry):
	def __init__(self, myitem):
		typelib, index = myitem
		typeinfo = typelib.GetTypeInfo(index)
		self.id = typeinfo.GetVarDesc(index)[0]
		name = typeinfo.GetNames(self.id)[0]
		HLITypeLibEntry.__init__(self, myitem, name)
	def GetText(self):
		return self.name + " - Enum/Module"
	def GetSubList(self):
		ret = []
		typelib, index = self.myobject
		typeinfo = typelib.GetTypeInfo(index)
		attr = typeinfo.GetTypeAttr()
		for j in range(attr[7]):
			vdesc = typeinfo.GetVarDesc(j)
			name = typeinfo.GetNames(vdesc[0])[0]
			ret.append(browser.MakeHLI(vdesc[1], name))
		return ret

class HLITypeLibProperty(HLICOM):
	def __init__(self, myitem):
		typeinfo, index = myitem
		self.id = typeinfo.GetVarDesc(index)[0]
		name = typeinfo.GetNames(self.id)[0]
		HLICOM.__init__(self, myitem, name)
	def GetText(self):
		return self.name + " - Property"
	def GetSubList(self):
		ret = []
		typeinfo, index = self.myobject
		names = typeinfo.GetNames(self.id)
		if len(names)>1:
			ret.append(browser.MakeHLI(names[1:], "Named Params"))
		vd = typeinfo.GetVarDesc(index)
		ret.append(browser.MakeHLI(self.id, "Dispatch ID"))
		ret.append(browser.MakeHLI(vd[1], "Value"))
		ret.append(browser.MakeHLI(vd[2], "Elem Desc"))
		ret.append(browser.MakeHLI(vd[3], "Var Flags"))
		ret.append(browser.MakeHLI(vd[4], "Var Kind"))
		return ret

class HLITypeLibFunction(HLICOM):
	funckinds = {win32com.pythoncom.FUNC_VIRTUAL : "Virtual",
	             win32com.pythoncom.FUNC_PUREVIRTUAL : "Pure Virtual",
	             win32com.pythoncom.FUNC_STATIC : "Static",
	             win32com.pythoncom.FUNC_DISPATCH : "Dispatch",
		}
	invokekinds = {win32com.pythoncom.INVOKE_FUNC: "Function",
	             win32com.pythoncom.INVOKE_PROPERTYGET : "Property Get",
	             win32com.pythoncom.INVOKE_PROPERTYPUT : "Property Put",
	             win32com.pythoncom.INVOKE_PROPERTYPUTREF : "Property Put by reference",
		}
	funcflags = [(win32com.pythoncom.FUNCFLAG_FRESTRICTED, "Restricted"),
                   (win32com.pythoncom.FUNCFLAG_FSOURCE, "Source"),
                   (win32com.pythoncom.FUNCFLAG_FBINDABLE, "Bindable"),
                   (win32com.pythoncom.FUNCFLAG_FREQUESTEDIT, "Request Edit"),
                   (win32com.pythoncom.FUNCFLAG_FDISPLAYBIND, "Display Bind"),
                   (win32com.pythoncom.FUNCFLAG_FDEFAULTBIND, "Default Bind"),
                   (win32com.pythoncom.FUNCFLAG_FHIDDEN, "Hidden"),
                   (win32com.pythoncom.FUNCFLAG_FUSESGETLASTERROR, "Uses GetLastError"),
                   ]

	vartypes = {win32com.pythoncom.VT_EMPTY: "Empty",
                win32com.pythoncom.VT_NULL: "NULL",
                win32com.pythoncom.VT_I2: "Integer 2",
                win32com.pythoncom.VT_I4: "Integer 4",
                win32com.pythoncom.VT_R4: "Real 4",
                win32com.pythoncom.VT_R8: "Real 8",
                win32com.pythoncom.VT_CY: "CY",
                win32com.pythoncom.VT_DATE: "Date",
                win32com.pythoncom.VT_BSTR: "String",
                win32com.pythoncom.VT_DISPATCH: "IDispatch",
                win32com.pythoncom.VT_ERROR: "Error",
                win32com.pythoncom.VT_BOOL: "BOOL",
                win32com.pythoncom.VT_VARIANT: "Variant",
                win32com.pythoncom.VT_UNKNOWN: "IUnknown",
                win32com.pythoncom.VT_DECIMAL: "Decimal",
                win32com.pythoncom.VT_I1: "Integer 1",
                win32com.pythoncom.VT_UI1: "Unsigned integer 1",
                win32com.pythoncom.VT_UI2: "Unsigned integer 2",
                win32com.pythoncom.VT_UI4: "Unsigned integer 4",
                win32com.pythoncom.VT_I8: "Integer 8",
                win32com.pythoncom.VT_UI8: "Unsigned integer 8",
                win32com.pythoncom.VT_INT: "Integer",
                win32com.pythoncom.VT_UINT: "Unsigned integer",
                win32com.pythoncom.VT_VOID: "Void",
                win32com.pythoncom.VT_HRESULT: "HRESULT",
                win32com.pythoncom.VT_PTR: "Pointer",
                win32com.pythoncom.VT_SAFEARRAY: "SafeArray",
                win32com.pythoncom.VT_CARRAY: "C Array",
                win32com.pythoncom.VT_USERDEFINED: "User Defined",
                win32com.pythoncom.VT_LPSTR: "Pointer to string",
                win32com.pythoncom.VT_LPWSTR: "Pointer to Wide String",
                win32com.pythoncom.VT_FILETIME: "File time",
                win32com.pythoncom.VT_BLOB: "Blob",
                win32com.pythoncom.VT_STREAM: "IStream",
                win32com.pythoncom.VT_STORAGE: "IStorage",
                win32com.pythoncom.VT_STORED_OBJECT: "Stored object",
                win32com.pythoncom.VT_STREAMED_OBJECT: "Streamed object",
                win32com.pythoncom.VT_BLOB_OBJECT: "Blob object",
                win32com.pythoncom.VT_CF: "CF",
                win32com.pythoncom.VT_CLSID: "CLSID",
        }

	type_flags = [ (win32com.pythoncom.VT_VECTOR, "Vector"),
                   (win32com.pythoncom.VT_ARRAY, "Array"),
                   (win32com.pythoncom.VT_BYREF, "ByRef"),
                   (win32com.pythoncom.VT_RESERVED, "Reserved"),
        ]               
		
	def __init__(self, myitem):
		typeinfo, index = myitem
		self.id = typeinfo.GetFuncDesc(index)[0]
		name = typeinfo.GetNames(self.id)[0]
		HLICOM.__init__(self, myitem, name)
	def GetText(self):
		return self.name + " - Function"
	def MakeReturnTypeName(self, typ):
		justtyp = typ & win32com.pythoncom.VT_TYPEMASK
		try:
			typname = self.vartypes[justtyp]
		except KeyError:
			typname = "?Bad type?"
		for (flag, desc) in self.type_flags:
			if flag & typ:
				typname = "%s(%s)" % (desc, typname)
		return typname			
	def MakeReturnType(self, returnTypeDesc):
		if type(returnTypeDesc)==type(()):
			first = returnTypeDesc[0]
			result = self.MakeReturnType(first)
			if first != win32com.pythoncom.VT_USERDEFINED:
				result = result + " " + self.MakeReturnType(returnTypeDesc[1])
			return result
		else:
			return self.MakeReturnTypeName(returnTypeDesc)
			
	def GetSubList(self):
		ret = []
		typeinfo, index = self.myobject
		names = typeinfo.GetNames(self.id)
		ret.append(browser.MakeHLI(self.id, "Dispatch ID"))
		if len(names)>1:
			ret.append(browser.MakeHLI(string.join(names[1:], ", "), "Named Params"))
		fd = typeinfo.GetFuncDesc(index)
		if fd[1]:
			ret.append(browser.MakeHLI(fd[1], "Possible result values"))
		if fd[8]:
			typ, flags = fd[8]
			val = self.MakeReturnType(typ)
			if flags:
				val = "%s (Flags=%d)" % (val, flags)
			ret.append(browser.MakeHLI(val, "Return Type"))
				
		for argDesc in fd[2]:
			typ, flags = argDesc
			val = self.MakeReturnType(typ)
			if flags:
				val = "%s (Flags=%d)" % (val, flags)
			ret.append(browser.MakeHLI(val, "Argument"))
		
		try:
			fkind = self.funckinds[fd[3]]
		except KeyError:
			fkind = "Unknown"
		ret.append(browser.MakeHLI(fkind, "Function Kind"))
		try:
			ikind = self.invokekinds[fd[4]]
		except KeyError:
			ikind = "Unknown"
		ret.append(browser.MakeHLI(ikind, "Invoke Kind"))
		# 5 = call conv
		# 5 = offset vtbl
		ret.append(browser.MakeHLI(fd[6], "Number Optional Params"))
		flagDescs = []
		for flag, desc in self.funcflags:
			if flag & fd[9]:
				flagDescs.append(desc)
		if flagDescs:
			ret.append(browser.MakeHLI(string.join(flagDescs, ", "), "Function Flags"))
		return ret
	
HLITypeKinds = {
		pythoncom.TKIND_ENUM : (HLITypeLibEnum, 'Enumeration'),
		pythoncom.TKIND_RECORD : (HLITypeLibEntry, 'Record'),
		pythoncom.TKIND_MODULE : (HLITypeLibEnum, 'Module'),
		pythoncom.TKIND_INTERFACE : (HLITypeLibMethod, 'Interface'),
		pythoncom.TKIND_DISPATCH : (HLITypeLibMethod, 'Dispatch'),
		pythoncom.TKIND_COCLASS : (HLITypeLibEntry, 'CoClass'),
		pythoncom.TKIND_ALIAS : (HLITypeLibEntry, 'Alias'),
		pythoncom.TKIND_UNION : (HLITypeLibEntry, 'Union')
	}

class HLITypeLib(HLICOM):
	def GetSubList(self):
		ret = []
		ret.append(browser.MakeHLI(self.myobject, "Filename"))
		try:
			tlb = win32com.pythoncom.LoadTypeLib(self.myobject)
		except win32com.pythoncom.com_error:
			return [browser.MakeHLI("%s can not be loaded" % self.myobject)]
			
		for i in range(tlb.GetTypeInfoCount()):
			try:
				ret.append(HLITypeKinds[tlb.GetTypeInfoType(i)][0]( (tlb, i) ) )
			except pythoncom.com_error:
				ret.append(browser.MakeHLI("The type info can not be loaded!"))
		return ret
		
class HLIHeadingRegisterdTypeLibs(HLICOM):
	"A tree heading for registered type libraries"
	def GetText(self):
		return "Registered Type Libraries"
	def GetSubList(self):
		# Explicit lookup in the registry.
		ret = []
		key = win32api.RegOpenKey(win32con.HKEY_CLASSES_ROOT, "TypeLib")
		win32ui.DoWaitCursor(1)
		try:
			num = 0
			while 1:
				try:
					keyName = win32api.RegEnumKey(key, num)
				except win32api.error:
					break
				# Enumerate all version info
				subKey = win32api.RegOpenKey(key, keyName)
				try:
					subNum = 0
					bestVersion = 0.0
					while 1:
						try:
							versionStr = win32api.RegEnumKey(subKey, subNum)
						except win32api.error:
							break
						try:
							versionFlt = string.atof(versionStr)
						except ValueError:
							versionFlt = 0 # ????
						if versionFlt > bestVersion:
							bestVersion = versionFlt
							name = win32api.RegQueryValue(subKey, versionStr)
						subNum = subNum + 1
				finally:
					win32api.RegCloseKey(subKey)
				ret.append(HLIRegisteredTypeLibrary((keyName, versionStr), name))
				num = num + 1
		finally:
			win32api.RegCloseKey(key)
			win32ui.DoWaitCursor(0)
		return ret

if __name__=='__main__':
	import hierlist, sys
	root = HLIRoot("COM Browser")
	if sys.modules.has_key("app"):
		# do it in a window
		browser.MakeTemplate()
		browser.template.OpenObject(root)
	else:
#		list=hierlist.HierListWithItems( root, win32ui.IDB_BROWSER_HIER )
#		dlg=hierlist.HierDialog("COM Browser",list)
		dlg = browser.dynamic_browser(root)
		dlg.DoModal()

