"""Contains knowledge to build a COM object definition.

This module is used by both the @dynamic@ and @makepy@ modules to build
all knowledge of a COM object.

This module contains classes which contain the actual knowledge of the object.
This include paramater and return type information, the COM dispid and CLSID, etc.

Other modules may use this information to generate .py files, use the information
dynamically, or possibly even generate .html documentation for objects.
"""

#
# NOTES: DispatchItem and MapEntry used by dynamic.py.
#        the rest is used by makepy.py
#
#        OleItem, DispatchItem, MapEntry, BuildCallList() is used by makepy

import sys
import string
import ni
import types

#from win32com import pythoncom
import pythoncom

error = "PythonCOM.Client.Build error"
DropIndirection="DropIndirection"
	
class MapEntry:
	"Simple holder for named attibutes - items in a map."

	def __init__(self, desc_or_id, names=None, doc=None, resultCLSID=pythoncom.IID_NULL, resultDoc = None, hidden=0):
		if type(desc_or_id)==type(0):
			self.dispid = desc_or_id
			self.desc = None
		else:
			self.dispid = desc_or_id[0]
			self.desc = desc_or_id
		self.names = names
		self.doc = doc
		self.resultCLSID = resultCLSID
		self.resultDocumentation = resultDoc
		self.wasProperty = 0 # Have I been transfored into a function so I can pass args?
		self.hidden = hidden
	def GetResultCLSID(self):
		rc = self.resultCLSID
		if rc == pythoncom.IID_NULL: return None
		return rc
	# Return a string, suitable for output - either "'{...}'" or "None"
	def GetResultCLSIDStr(self):
		rc = self.GetResultCLSID()
		if rc is None: return "None"
		return repr(str(rc)) # Convert the IID object to a string, then to a string in a string.

	def GetResultName(self):
		if self.resultDocumentation is None:
			return None
		return self.resultDocumentation[0]

class OleItem:
  typename = "OLEITEM"

  def __init__(self, doc=None):
    self.doc = doc
    self.bWritten = 0
    self.clsid = None

class DispatchItem(OleItem):
	typename = "DISPATCH"

	def __init__(self, typeinfo=None, attr=None, doc=None, bForUser=1):
		OleItem.__init__(self,doc)
		self.propMap = {}
		self.propMapGet = {}
		self.propMapPut = {}
		self.mapFuncs = {}
		self.defaultDispatchName = None
		self.hidden = 0

                if typeinfo:
                  self.Build(typeinfo, attr, bForUser)

	def Build(self, typeinfo, attr, bForUser = 1):
		self.clsid = attr[0]
		if typeinfo is None: return
		# Loop over all methods
		for j in xrange(attr[6]):
			fdesc = list(typeinfo.GetFuncDesc(j))
			id = fdesc[0]
			funcflags = fdesc[9]

                        # skip [restricted] methods, unless it is the
                        # enumerator (which, being part of the "system",
                        # we know about and can use)
                        if funcflags & pythoncom.FUNCFLAG_FRESTRICTED and \
                           id != pythoncom.DISPID_NEWENUM:
                          continue

                        # skip any methods that can't be reached via DISPATCH
                        if fdesc[3] != pythoncom.FUNC_DISPATCH:
                          continue

			try:
                          names = typeinfo.GetNames(id)
			except pythoncom.ole_error:
                          names = None

			doc = None
			try:
                          if bForUser:
                            doc = typeinfo.GetDocumentation(id)
			except pythoncom.ole_error:
                          pass

			if id==0 and names and names[0]:
                          self.defaultDispatchName = names[0]

			invkind = fdesc[4]

			# Translate any Alias or Enums in result.
			fdesc[8], resultCLSID, resultDoc = _ResolveUserDefined(fdesc[8], typeinfo)
			# Translate any Alias or Enums in argument list.
			argList = []			
			for argDesc in fdesc[2]:
                          argList.append(_ResolveUserDefined(argDesc, typeinfo)[0])
			fdesc[2] = tuple(argList)
	
			hidden = (funcflags & pythoncom.FUNCFLAG_FHIDDEN) != 0
#			if hidden and names[0] != "Item": # XXX - special hack case.
#				continue

			if id == pythoncom.DISPID_NEWENUM:
                          map = self.mapFuncs
                        elif invkind == pythoncom.INVOKE_PROPERTYGET:
                          map = self.propMapGet
			elif invkind == pythoncom.INVOKE_PROPERTYPUT:
                          map = self.propMapPut
			elif invkind == pythoncom.INVOKE_FUNC:
                          map = self.mapFuncs
			else:
                          map = None
			if not map is None: 
#				if map.has_key(names[0]):
#					sys.stderr.write("Warning - overwriting existing method/attribute %s\n" % names[0])
				map[names[0]] = MapEntry(tuple(fdesc), names, doc, resultCLSID, resultDoc, hidden)

		self.propMap = {}
		# Loop over all variables (ie, properties)
		for j in xrange(attr[7]):
			fdesc = list(typeinfo.GetVarDesc(j))

                        ### need pythoncom.VARFLAG_FRESTRICTED ...
                        ### then check it

			if fdesc[4] == pythoncom.VAR_DISPATCH:
				id = fdesc[0]
				names = typeinfo.GetNames(id)
				# Translate any Alias or Enums in result.
				fdesc[2], resultCLSID, resultDoc = _ResolveUserDefined(fdesc[2], typeinfo)
				doc = None
				try:
					if bForUser: doc = typeinfo.GetDocumentation(id)
				except pythoncom.ole_error:
					pass

                                # handle the enumerator specially
                                if id == pythoncom.DISPID_NEWENUM:
                                  map = self.mapFuncs
                                  ### hack together a minimal FUNCDESC
                                  fdesc = (fdesc[0], None, None, None, pythoncom.INVOKE_PROPERTYGET, )
                                else:
                                  map = self.propMap

				map[names[0]] = MapEntry(tuple(fdesc), names, doc, resultCLSID, resultDoc)
		
		# Now post-process the maps.  For any "Get" or "Set" properties
		# that have arguments, we must turn them into methods.  If a method
		# of the same name already exists, change the name.
		for key, item in self.propMapGet.items():
			ins, outs, opts = self.CountInOutOptArgs(item.desc[2])
			if ins > 0: # if a Get property takes _any_ in args:
				if item.desc[6]==ins or ins==opts:
					newKey = "Get" + key
					deleteExisting = 0 # This one is still OK
				else:
					deleteExisting = 1 # No good to us
					if self.mapFuncs.has_key(key):
						newKey = "Get" + key
					else:
						newKey = key
				item.wasProperty = 1
				self.mapFuncs[newKey] = item
				if deleteExisting:
					del self.propMapGet[key]
					
		for key, item in self.propMapPut.items():
			ins, outs, opts = self.CountInOutOptArgs(item.desc[2])
			if ins>1: # if a Put property takes more than 1 arg:
				if opts+1==ins or ins==item.desc[6]+1:
					newKey = "Set" + key
					deleteExisting = 0 # This one is still OK
				else:
					deleteExisting = 1 # No good to us
					if self.mapFuncs.has_key(key) or self.propMapGet.has_key(key):
						newKey = "Set" + key
					else:
						newKey = key
				item.wasProperty = 1
				self.mapFuncs[newKey] = item
				if deleteExisting:
					del self.propMapPut[key]

	def CountInOutOptArgs(self, argTuple):
		"Return tuple counting in/outs/OPTS.  Sum of result may not be len(argTuple), as some args may be in/out."
		ins = out = opts = 0
		for argCheck in argTuple:
			inOut = argCheck[1]
			if inOut==0:
				ins = ins + 1
				out = out + 1
			else:
				if inOut & pythoncom.PARAMFLAG_FIN:
					ins = ins + 1
				if inOut & pythoncom.PARAMFLAG_FOPT:
					opts = opts + 1
				if inOut & pythoncom.PARAMFLAG_FOUT:
					out = out + 1
		return ins, out, opts

	def MakeFuncMethod(self, entry, name, bMakeClass = 1):
		# If we have a type description, and its a dispatch, and not varargs...
		if not entry.desc is None and entry.desc[3]==pythoncom.FUNC_DISPATCH and entry.desc[6]!=-1:
			return self.MakeDispatchFuncMethod(entry, name, bMakeClass)
		else:
			return self.MakeInterfaceFuncMethod(entry, name, bMakeClass)
	def MakeDispatchFuncMethod(self, entry, name, bMakeClass = 1):
		fdesc = entry.desc
		doc = entry.doc
		names = entry.names
		ret = []
		if bMakeClass:
			linePrefix = "\t"
			defNamedOptArg = "defaultNamedOptArg"
			defNamedNotOptArg = "defaultNamedNotOptArg"
			defUnnamedArg = "defaultUnnamedArg"
		else:
			linePrefix = ""
			defNamedOptArg = "pythoncom.Missing"
			defNamedNotOptArg = "pythoncom.Missing"
			defUnnamedArg = "pythoncom.Missing"
		id = fdesc[0]
		s = linePrefix + 'def ' + name + '(self' + BuildCallList(fdesc, names, defNamedOptArg, defNamedNotOptArg, defUnnamedArg) + '):'
		ret.append(s)
		if doc and doc[1]:
			ret.append(linePrefix + '\t"' + doc[1] + '"')

#		print "fdesc is ", fdesc

		resclsid = entry.GetResultCLSID()
		if resclsid:
			resclsid = "'%s'" % resclsid
		else:
			resclsid = 'None'
		s = '%s\treturn self._ApplyTypes_(0x%x, %s, %s, %s, %s, %s%s)' % (linePrefix, id, fdesc[4], fdesc[8], fdesc[2], `name`, resclsid, _BuildArgList(fdesc, names))

		ret.append(s)
		ret.append("")
		return ret

	# This makes an interface method, when the only type info available
	# is for a TKIND_INTERFACE, rather than TKIND_DISPATCH.
	def MakeInterfaceFuncMethod(self, entry, name, bMakeClass = 1):
		fdesc = entry.desc
		names = entry.names
		doc = entry.doc
		ret = []
		argPrefix = "self"
		if bMakeClass:
			linePrefix = "\t"
			defArg = "defaultArg"
		else:
			linePrefix = ""
			defArg = "pythoncom.Missing"
		ret.append(linePrefix + 'def ' + name + '(' + argPrefix + ', *args):')
		if doc and doc[1]: ret.append(linePrefix + '\t"' + doc[1] + '"')
		if fdesc:
			invoketype = fdesc[4]
		else:
			invoketype = pythoncom.DISPATCH_METHOD
		s = linePrefix + '\treturn self._get_good_object_(apply(self._oleobj_.Invoke,('
		ret.append(s + hex(entry.dispid) + ",0,%d,1)+args),'%s')" % (invoketype, names[0]))
		ret.append("")
		return ret


def _DoResolveType(typeinfo, hrefResult):
  resultTypeInfo = typeinfo.GetRefTypeInfo(hrefResult)
  resultAttr = resultTypeInfo.GetTypeAttr()
  typeKind = resultAttr[5]
  if typeKind == pythoncom.TKIND_ALIAS:
    ai = resultAttr[14]
    if type(ai)==type(0):
      return ai, resultTypeInfo
    if type(ai)==type(()):
      return _DoResolveType(resultTypeInfo, ai[1])
    return None

  elif typeKind == pythoncom.TKIND_ENUM or \
       typeKind == pythoncom.TKIND_MODULE:
    # For now, assume Long
    return pythoncom.VT_I4, None

  elif typeKind == pythoncom.TKIND_DISPATCH or \
       typeKind == pythoncom.TKIND_INTERFACE:
    return pythoncom.VT_DISPATCH, resultTypeInfo

  elif typeKind == pythoncom.TKIND_COCLASS:
    # Should get default?
    return pythoncom.VT_DISPATCH, resultTypeInfo

def _DoResolveTypeTuple(typeTuple, typeinfo):
  t = typeTuple
  if type(t) != types.TupleType:
    return t, None, None
  if type(t[0]) == types.IntType and type(t[1]) == types.IntType:
    if t[0]==pythoncom.VT_USERDEFINED:
      # Have href.
      rc = _DoResolveType(typeinfo, t[1])
      if rc is None: return t, None, None
      retType, retTypeInfo = rc
      if retTypeInfo:
              retCLSID = retTypeInfo.GetTypeAttr()[0]
              retDocumentation = retTypeInfo.GetDocumentation(-1)
      else:
              retCLSID = retDocumentation = None
      if retType==pythoncom.VT_DISPATCH and retCLSID:
              raise DropIndirection, (retType, retCLSID, retDocumentation)
      return retType, retCLSID, retDocumentation
  if type(t[1])==types.TupleType:	
    try:
      resolved, ret2, ret3 = _DoResolveTypeTuple(t[1],typeinfo)
      return (t[0],resolved), ret2, ret3
    except DropIndirection, (resolved, ret2, ret3):
      return (resolved), ret2, ret3

  return t, None, None
		
def _ResolveUserDefined(typeTuple, typeinfo):
  "Resolve VT_USERDEFINED (often aliases or typed IDispatches)"
  realTypeTuple, inOut = typeTuple
  try:
    resolved, ret2, ret3 = _DoResolveTypeTuple(realTypeTuple, typeinfo)
  except DropIndirection, det:
    resolved, ret2, ret3 = det
  return (resolved, inOut), ret2, ret3

def _BuildArgList(fdesc, names):
  "Builds list of args to the underlying Invoke method."
  str = ''

  # Word has TypeInfo for Insert() method, but says "no args"
  numArgs = max(fdesc[6], len(fdesc[2]))

  for arg in xrange(numArgs):
    try:
      argName = names[arg+1]
      if argName[:2]=='__':
        argName = argName[1:] # These may cause strife
    except IndexError:
      argName = "arg%d" % (arg+1)
    str = str + ', ' + argName
  return str

def BuildCallList(fdesc, names, defNamedOptArg, defNamedNotOptArg, defUnnamedArg):
  "Builds a Python declaration for a method."
  # Names[0] is the func name - param names are from 1.
  numArgs = len(fdesc[2])
  numOptArgs = fdesc[6]
  str = ''
  if numOptArgs==-1:	# Special value that says "var args after here"
    numArgs = numArgs - 1
  else:
    firstOptArg = numArgs - numOptArgs
  for arg in xrange(numArgs):
    try:
      argName = names[arg+1] 
      # Is a named argument
      if arg >= firstOptArg:
        defArgVal = defNamedOptArg
      else:
        defArgVal = defNamedNotOptArg
    except IndexError:
      # Unnamed arg - always allow default values.
      argName = "arg%d" % (arg+1)
      defArgVal = defUnnamedArg

    try:
      inOut = fdesc[2][arg][1]
    except IndexError:
      # something strange - assume is in param.
      inOut = pythoncom.PARAMFLAG_FIN

    if not inOut==pythoncom.PARAMFLAG_NONE and not (inOut & pythoncom.PARAMFLAG_FIN):
      # If it is not an inout param, override default to "None".
      # This allows us to "hide" out params.
      defArgVal = "None"

    if argName[:2]=='__':
      argName = argName[1:] # These may cause strife
    str = str + ", " + argName
    if defArgVal:
      str = str + "=" + defArgVal
  if numOptArgs==-1:
    str = str + ", *" + names[-1]

  return str


if __name__=='__main__':
  print "Use 'makepy.py' to generate Python code - this module is just a helper"
