#include "windows.h"
#include "Python.h"
#include <atlbase.h>
#include "pywintypes.h"
#include "PythonCOM.h"
#include "mapix.h"
#include "PyMAPIUtil.h"

PyObject *PyMAPIObject_FromTypedUnknown( ULONG typ, IUnknown *pUnk, BOOL bAddRef)
{
	const IID *pIID;
	switch (typ) {
		case MAPI_FOLDER:
			pIID = &IID_IMAPIFolder;
			break;
		case MAPI_SESSION:
			pIID = &IID_IMAPISession;
			break;
		case MAPI_MESSAGE:
			pIID = &IID_IMessage;
			break;
		case MAPI_ATTACH:
			pIID = &IID_IAttachment;
			break;
		case MAPI_STORE:
		case MAPI_ADDRBOOK:
		case MAPI_ABCONT:
		case MAPI_MAILUSER:
		case MAPI_DISTLIST:
		case MAPI_PROFSECT:
		case MAPI_STATUS:
		case MAPI_FORMINFO:
			pIID = &IID_IUnknown;
			break;
		default:
			pIID = &IID_IUnknown;
			break;
	}
    return PyCom_PyObjectFromIUnknown(pUnk, *pIID, bAddRef );
}

BOOL PyMAPIObject_AsSPropValue(PyObject *Valob, SPropValue *pv, void *pAllocMoreLinkBlock)
{
	PyObject *ob;
	if (!PyArg_ParseTuple(Valob, "iO:SPropValue item", &pv->ulPropTag, &ob )) {
		PyErr_Clear();
		PyErr_SetString(PyExc_TypeError, "An SPropValue item must be a tuple of (integer, object)");
		return NULL;
	}
	BOOL ok = TRUE;
	PyErr_Clear();
	switch (PROP_TYPE(pv->ulPropTag)) {
		case PT_I2:	//		case PT_SHORT:
			pv->Value.i = (int)PyInt_AsLong(ob);
			break;
		case PT_I4: //		case PT_LONG:
			pv->Value.l = PyInt_AsLong(ob);
			break;
		case PT_R4: //		case PT_FLOAT:
			pv->Value.flt = (float)PyFloat_AsDouble(ob);
			break;
		case PT_R8: //		case PT_DOUBLE:
			pv->Value.dbl = PyFloat_AsDouble(ob);
			break;
		case PT_BOOLEAN:
			pv->Value.b = (BOOL)PyInt_AsLong(ob);
			break;
/*
		case PT_CURRENCY:
			val = PyFloat_FromDouble(pv->Value.cur); ??
			break;
		case PT_APPTIME :
			at 
			double 
*/
		case PT_SYSTIME:
			ok = PyWinObject_AsFILETIME(ob, &pv->Value.ft);
			break;
		case PT_STRING8:
			// Maybe copy into new MAPI memory block?
			pv->Value.lpszA = PyString_AsString(ob);
			break;
		case PT_UNICODE:
			{ // Bit of a hack - need to copy into MAPI block.
			BSTR wstr = NULL;
			ok = PyWinObject_AsBstr(ob, &wstr, FALSE);
			if (ok) {
				DWORD bufSize = sizeof(WCHAR) * (SysStringLen(wstr)+1);
				HRESULT hr = MAPIAllocateMore(bufSize, pAllocMoreLinkBlock, (void **)&pv->Value.lpszW);
				if (S_OK!=hr) {
					OleSetOleError(hr);
					ok = FALSE;
				} else {
					memcpy(pv->Value.lpszW, wstr, bufSize-2);
					// Null terminate
					memcpy(((char *)pv->Value.lpszW)+(bufSize-2),"\0\0", 2);
				}
			}
			SysFreeString(wstr);
			break;
			}

		case PT_BINARY:
			pv->Value.bin.lpb = (unsigned char *)PyString_AsString(ob);
			pv->Value.bin.cb = PyString_Size(ob);
			break;

		case PT_CLSID:
			{
			HRESULT hr = MAPIAllocateMore(sizeof(CLSID), pAllocMoreLinkBlock, (void **)&pv->Value.lpguid);
			if (S_OK != hr) {
				OleSetOleError(hr);
				ok = FALSE;
			} else
				ok = PyCom_CLSIDFromPyObject(ob, pv->Value.lpguid);
			break;
			}
		case PT_I8:
//		case PT_LONGLONG:
			ok = PyObjectToLARGE_INTEGER(ob, &pv->Value.li);
			break;
		case PT_ERROR:
			pv->Value.err = (SCODE)PyInt_AsLong(ob);
			break;
/*

			li 
			LARGE_INTEGER 

		case PT_MV_I2:
			MVi 
			SShortArray 
		case PT_MV_LONG:
			MVI 
			SLongArray 
		case PT_MV_R4:
			MVflt 
			SRealArray 
		case PT_MV_DOUBLE :
			MVdbl 
			SDoubleArray 
		case PT_MV_CURRENCY:
			MVcur 
			SCurrencyArray 
		case PT_MV_APPTIME:
			MVat 
			SAppTimeArray 
		case PT_MV_SYSTIME:
			MVft 
			SDateTimeArray 
		case PT_MV_BINARY:
			MVbin 
			SBinaryArray 
		case PT_MV_STRING8:
			MVszA 
			SLPSTRArray 
		case PT_MV_UNICODE:
			MVszW 
			SWStringArray 
		case PT_MV_CLSID:
			MVguid 
			SGuidArray 
		case PT_MV_I8:
			MVli 
			SLargeIntegerArray 
		case PT_NULL:
		case PT_OBJECT:
			x 
			LONG
*/
		default:
			printf("*** Unsupported MAPI property type %d", PROP_TYPE(pv->ulPropTag));
			PyErr_SetString(PyExc_TypeError, "Unsupported property type");
			ok = FALSE;
		}
	ok = (ok && !PyErr_Occurred());
	return ok;
}

PyObject *PyMAPIObject_FromSPropValue(SPropValue *pv)
{
	PyObject *val;
	switch (PROP_TYPE(pv->ulPropTag)) {
		case PT_I2:	//		case PT_SHORT:
			val = PyInt_FromLong(pv->Value.i);
			break;
		case PT_I4:	//		case PT_LONG:
			val = PyInt_FromLong(pv->Value.l);
			break;
		case PT_R4:	//		case PT_FLOAT:
			val = PyFloat_FromDouble(pv->Value.flt);
			break;
		case PT_R8:	//		case PT_DOUBLE:
			val = PyFloat_FromDouble(pv->Value.dbl);
			break;
		case PT_BOOLEAN:
			val = PyInt_FromLong(pv->Value.b);
			break;
/*
		case PT_CURRENCY:
			val = PyFloat_FromDouble(pv->Value.cur); ??
			break;
		case PT_APPTIME :
			at 
			double 
*/
		case PT_SYSTIME:
			printf("***** Moving FILETIME\n");
			val = PyWinObject_FromFILETIME(pv->Value.ft);
			break;
		case PT_STRING8:
			val = PyString_FromString(pv->Value.lpszA);
			break;
		case PT_UNICODE:
			val = PyWinObject_FromWCHAR(pv->Value.lpszW);
			break;
		case PT_BINARY:
			val = PyString_FromStringAndSize((char *)pv->Value.bin.lpb, pv->Value.bin.cb);
			break;

		case PT_CLSID:
			val = PyCom_PyIIDObjectFromIID(*pv->Value.lpguid);
			break;
		case PT_I8:
//		case PT_LONGLONG:
			val = PyObjectFromLARGE_INTEGER(pv->Value.li);
			break;
		case PT_ERROR:
			val = PyInt_FromLong(pv->Value.err);
			break;
/*

			li 
			LARGE_INTEGER 

		case PT_MV_I2:
			MVi 
			SShortArray 
		case PT_MV_LONG:
			MVI 
			SLongArray 
		case PT_MV_R4:
			MVflt 
			SRealArray 
		case PT_MV_DOUBLE :
			MVdbl 
			SDoubleArray 
		case PT_MV_CURRENCY:
			MVcur 
			SCurrencyArray 
		case PT_MV_APPTIME:
			MVat 
			SAppTimeArray 
		case PT_MV_SYSTIME:
			MVft 
			SDateTimeArray 
		case PT_MV_BINARY:
			MVbin 
			SBinaryArray 
		case PT_MV_STRING8:
			MVszA 
			SLPSTRArray 
		case PT_MV_UNICODE:
			MVszW 
			SWStringArray 
		case PT_MV_CLSID:
			MVguid 
			SGuidArray 
		case PT_MV_I8:
			MVli 
			SLargeIntegerArray 
		case PT_NULL:
		case PT_OBJECT:
			x 
			LONG
*/
		default:
			printf("*** Unsupported MAPI property type %d", PROP_TYPE(pv->ulPropTag));
			PyErr_SetString(PyExc_TypeError, "Unsupported property type");
			return NULL;
		}

	PyObject *rc = PyTuple_New(2);
	if (rc==NULL) {
		Py_DECREF(val);
		PyErr_SetString(PyExc_MemoryError, "Tuple(2) for PROP result");
		return NULL;
	}
	PyTuple_SET_ITEM(rc, 0, PyInt_FromLong(pv->ulPropTag));
	PyTuple_SET_ITEM(rc, 1, val);
	return rc;
}

BOOL PyMAPIObject_AsSRowSet(PyObject *obSeq, SRowSet **ppResult, BOOL bNoneOK)
{
	if (ppResult==NULL || obSeq==NULL)
		return FALSE;
	if (obSeq==NULL) return FALSE;
	*ppResult==NULL;
	if (obSeq==Py_None) {
		if (bNoneOK)
			return TRUE;
		PyErr_SetString(PyExc_TypeError, "None is not a valid SRowSet/ADRLIST in this context");
		return FALSE;
	}
	PyObject *rowObject = NULL;
	PyObject *propObject = NULL;
	BOOL rc = FALSE;
	HRESULT hr;
	ULONG i;
	DWORD allocSize;

	int seqLen = PySequence_Length(obSeq);

	if (seqLen==-1) {
		PyErr_SetString(PyExc_TypeError, "ADRLIST items must be a sequence");
		goto done;
	}

	allocSize = sizeof(SRowSet) + (sizeof(SRow) * seqLen);
	if (S_OK != (hr=MAPIAllocateBuffer(allocSize, (void **)ppResult))) {
		OleSetOleError(hr);
		goto done;
	}
	ZeroMemory(*ppResult, allocSize); // so cleanup works correctly.
	(*ppResult)->cRows = seqLen;

	for (i=0;i<(ULONG)seqLen;i++) {
		rowObject = PySequence_GetItem(obSeq, i);
		if (rowObject==NULL)
			goto done;
		// myob is expected to represent an SRow structure.  This is really an array
		// of property values.
		SRow *pRow = (*ppResult)->aRow+i;
		pRow->cValues = PySequence_Length(rowObject);
		if (pRow->cValues==-1)
			goto done;

		if (pRow->cValues==0)
			pRow->lpProps = NULL;
		else {
			allocSize = sizeof(SPropValue) * pRow->cValues;
			hr = MAPIAllocateBuffer(allocSize, (void **)&pRow->lpProps);
			if (FAILED(hr)) {
				OleSetOleError(hr);
				goto done;
			}
			for (ULONG j=0;j<pRow->cValues;j++) {
				propObject = PySequence_GetItem(rowObject, j);
				if (propObject==NULL)
					goto done;
				if (!PyMAPIObject_AsSPropValue(propObject, pRow->lpProps+j, *ppResult))
					goto done;
				Py_DECREF(propObject);
				propObject = NULL; // important for cleanup
			}
		}
		rowObject = NULL; // important for cleanup
	}
	rc = TRUE;
done:
	if (!rc && (*ppResult)) {
		PyMAPIObject_FreeSRowSet(*ppResult);
	}
	Py_XDECREF(propObject);
	Py_XDECREF(rowObject);
	return rc;
}

void PyMAPIObject_FreeSRowSet(SRowSet *pResult)
{
	if (pResult) {
		for (ULONG i=0;i<pResult->cRows;i++)
			MAPIFreeBuffer( pResult->aRow[i].lpProps );
		MAPIFreeBuffer(pResult);
	}
}

PyObject *PyMAPIObject_FromSRow(SRow *pr)
{
	PyObject *result = PyTuple_New(pr->cValues);
	if (result==NULL) {
		PyErr_SetString(PyExc_MemoryError, "Allocating SRow result");
		return NULL;
	}
	for (ULONG i=0;i<pr->cValues;i++) {
		PyObject *obNew = PyMAPIObject_FromSPropValue(pr->lpProps+i);
		if (obNew==NULL) {
			Py_DECREF(result);
			return NULL;
		}
		PyTuple_SET_ITEM(result, i, obNew);
	}
	return result;
}


PyObject *PyMAPIObject_FromSRowSet(SRowSet *prs)
{
	PyObject *result = PyTuple_New(prs->cRows);
	if (result==NULL) {
		PyErr_SetString(PyExc_MemoryError, "Allocating SRowSet result");
		return NULL;
	}
	for (ULONG i=0;i<prs->cRows;i++) {
		PyObject *obNew = PyMAPIObject_FromSRow(prs->aRow+i);
		if (obNew==NULL) {
			Py_DECREF(result);
			return NULL;
		}
		PyTuple_SET_ITEM(result, i, obNew);
	}
	return result;
}

BOOL PyMAPIObject_AsSPropTagArray(PyObject *obta, SPropTagArray **ppta)
{
	if (obta==Py_None) {
		*ppta = NULL;
		return TRUE;
	}
	BOOL bSeq = TRUE;
	int seqLen;
	if (PySequence_Check(obta)) {
		seqLen = PySequence_Length(obta);
	} else if (PyInt_Check(obta)) {
		seqLen = 1;
		bSeq = FALSE;
	} else {
		PyErr_SetString(PyExc_TypeError, "SPropTagArray must be a sequence of integers");
		return FALSE;
	}

	DWORD cBytes = (seqLen * sizeof(ULONG)) + sizeof(ULONG);
	SPropTagArray *pta;
	HRESULT hr = MAPIAllocateBuffer(cBytes, (void **)&pta);
	if (FAILED(hr)) {
		OleSetOleError(hr);
		return FALSE;
	}
	pta->cValues = seqLen;
	if (bSeq) {
		for (ULONG i=0;i<(ULONG)seqLen;i++) {
			PyObject *obItem = PySequence_GetItem(obta, i);
			if (obItem==NULL) {
				MAPIFreeBuffer(pta);
				return FALSE;
			}
			pta->aulPropTag[i] = PyInt_AsLong(obItem);
			if (PyErr_Occurred()) {
				Py_DECREF(obItem);
				MAPIFreeBuffer(pta);
				return FALSE;
			}
			Py_DECREF(obItem);
		}
	} else {
		// Simple int.
		pta->aulPropTag[0] = PyInt_AsLong(obta);
	}
	*ppta = pta;
	return TRUE;
}

void PyMAPIObject_FreeSPropTagArray(SPropTagArray *pta)
{
	if (pta)
		MAPIFreeBuffer(pta);
}

PyObject *PyMAPIObject_FromSPropTagArray(SPropTagArray *pta)
{
	PyObject *ret = PyTuple_New(pta->cValues);
	for (ULONG i=0;i<pta->cValues;i++) {
		PyTuple_SET_ITEM(ret, i, PyInt_FromLong(pta->aulPropTag[i]));
	}
	return ret;
}

BOOL PyMAPIObject_AsSBinaryArray(PyObject *ob, SBinaryArray *pba)
{
	BOOL bSeq = TRUE;
	int seqLen;
	if (PyString_Check(ob)) {
		seqLen = 1;
		bSeq = FALSE;
	} else if (PySequence_Check(ob)) {
		seqLen = PySequence_Length(ob);
	} else {
		PyErr_SetString(PyExc_TypeError, "SBinaryArray must be a sequence of strings");
		return FALSE;
	}
	DWORD cBytes = (seqLen * sizeof(SBinary));
	SBinary *pBin;
	HRESULT hr = MAPIAllocateBuffer(cBytes, (void **)&pBin);
	pba->lpbin = pBin;
	if (FAILED(hr)) {
		OleSetOleError(hr);
		return FALSE;
	}
	pba->cValues = seqLen;
	if (bSeq) {
		for (ULONG i=0;i<(ULONG)seqLen;i++) {
			PyObject *obItem = PySequence_GetItem(ob, i);
			if (obItem==NULL) {
				MAPIFreeBuffer(pba);
				return FALSE;
			}
			if (!PyString_Check(obItem)) {
				PyErr_SetString(PyExc_TypeError, "SBinary must be a string");
				Py_DECREF(obItem);
				MAPIFreeBuffer(pba);
				return FALSE;
			}
			pBin[i].cb = PyString_Size(obItem);
			pBin[i].lpb = (LPBYTE )PyString_AsString(obItem);
			Py_DECREF(obItem);
		}
	} else {
		if (!PyString_Check(ob)) {
			PyErr_SetString(PyExc_TypeError, "SBinary must be a string");
			MAPIFreeBuffer(pba);
			return FALSE;
		}
		// Simple string
		pBin[0].cb = PyString_Size(ob);
		pBin[0].lpb = (LPBYTE)PyString_AsString(ob);
	}
	return TRUE;
}

void PyMAPIObject_FreeSBinaryArray(SBinaryArray *pv)
{
	if (pv->lpbin)
		MAPIFreeBuffer(pv->lpbin);
}

BOOL PyMAPIObject_AsMAPINAMEIDArray(PyObject *ob, MAPINAMEID ***pppNameId, ULONG *pNumIds, BOOL bNoneOK /*= FALSE*/ )
{
	if (bNoneOK && ob==Py_None) {
		*pppNameId = NULL;
		*pNumIds = 0;
		return TRUE;
	}
	PyErr_Clear();
	ULONG len = (ULONG)PySequence_Length(ob);
	if (PyErr_Occurred()) {
		PyErr_Clear();
		PyErr_SetString(PyExc_TypeError, "MAPINAMEID array list be a sequence of tuples");
		return FALSE;
	}
	MAPINAMEID **ppNew = NULL;
	MAPINAMEID *prgIds = NULL;
	IID *pIIDs = NULL;
	HRESULT hr = MAPIAllocateBuffer(len * sizeof(MAPINAMEID *), (void **)&ppNew);
	if (SUCCEEDED(hr)) hr = MAPIAllocateMore(len * sizeof(MAPINAMEID), ppNew, (void **)&prgIds);
	if (SUCCEEDED(hr)) hr = MAPIAllocateMore(len * sizeof(IID), ppNew, (void **)&pIIDs);
	if (FAILED(hr)) {
		MAPIFreeBuffer(ppNew);
		OleSetOleError(hr);
		return FALSE;
	}
	for (ULONG i=0;i<len;i++) {
		ppNew[i] = prgIds+i;
		MAPINAMEID *pNew = prgIds+i;
		PyObject *obIID, *obPropId;
		PyObject *pMe = PySequence_GetItem(ob, i);
		if (pMe==NULL) {
			goto loop_error;
		}
		if (!PyArg_ParseTuple(pMe, "OO", &obIID, &obPropId )) {
			PyErr_Clear();
			PyErr_SetString(PyExc_TypeError, "MAPINAMEIDArray must be a sequence of (iid, string/int) tuples");
			goto loop_error;
		}

		pNew->lpguid = pIIDs+i;
		BSTR bstrVal;
		if (!PyWinObject_AsIID(obIID, pIIDs+i))
			goto loop_error;
		if (PyInt_Check(obPropId)) {
			pNew->ulKind = MNID_ID;
			pNew->Kind.lID = PyInt_AsLong(obPropId);
		} else if (PyWinObject_AsBstr(obPropId, &bstrVal)) {
			// Make a copy of the string
			pNew->ulKind = MNID_STRING;
			DWORD strLen = SysStringLen(bstrVal);
			hr = MAPIAllocateMore(sizeof(WCHAR) * (strLen+1), ppNew, (void **)&pNew->Kind.lpwstrName);
			if (FAILED(hr)) {
				PyWinObject_FreeBstr(bstrVal);
				OleSetOleError(hr);
				goto loop_error;
			}
			wcsncpy(pNew->Kind.lpwstrName, bstrVal, strLen+1);
			PyWinObject_FreeBstr(bstrVal);
		} else {
			PyErr_SetString(PyExc_TypeError, "The type of the new property ID is unknown - must be string/unicode or int");
			goto loop_error;
		}
		Py_DECREF(pMe);
		continue;
loop_error:
			Py_XDECREF(pMe);
			MAPIFreeBuffer(ppNew);
			return NULL;
	}
	*pppNameId = ppNew;
	*pNumIds = len;
	return TRUE;
}

void PyMAPIObject_FreeMAPINAMEIDArray(MAPINAMEID **pv)
{
	MAPIFreeBuffer(pv);
}

