"""Python ActiveX Scripting Implementation

This module implements the Python ActiveX Scripting client.

To register the implementation, simply "run" this Python program - ie
either double-click on it, or run "python.exe pyscript.py" from the
command line.
"""

import framework
import winerror
import win32com
import win32api
from win32com.axscript import axscript
import win32com.server.util, win32com.server.policy, win32com.server.register
import sys
import string
import traceback
import scriptdispatch
import regsub
import win32com.client.dynamic

from framework import RaiseAssert, trace, Exception, SCRIPTTEXT_FORCEEXECUTION, SCRIPTTEXT_ISEXPRESSION

PyScript_CLSID = "{DF630910-1C1D-11d0-AE36-8C0F5E000000}"

debugging_attr = 0

def debug_attr_print(*args):
	if debugging_attr:
		apply(trace, args)

def ExpandTabs(text):
	return regsub.gsub('\t','    ', text)

def AddCR(text):
	return regsub.gsub('\n','\r\n',text)
#	return string.join(string.split(text,'\n'),'\r\n')

def RemoveCR(text):
# No longer just "RemoveCR" - should be renamed to
# FixNewlines, or something.  Idea is to fix arbitary newlines into
# something Python can compile...
	return regsub.gsub('\(\r\n\)\|\r\|\(\n\r\)','\n',text)
	
class AXScriptCodeBlock(framework.AXScriptCodeBlock):
	def GetDisplayName(self):
		return "PyScript - " + framework.AXScriptCodeBlock.GetDisplayName(self)

#
# Restricted execution model.
#
import rexec
class AXRExec(rexec.RExec):
	ok_builtin_modules = rexec.RExec.ok_builtin_modules + ('win32trace',)

	def __init__(self, pretendMain, hooks = None, verbose = 0):
		self.pretendMain = pretendMain
		rexec.RExec.__init__(self, hooks, verbose)
#		mods = list(self.ok_dynamic_modules)
#		mods.append("win32trace")
#		mods = tuple(mods)
#		self.ok_dynamic_modules = mods
	def make_main(self):
		if not self.modules.has_key('__main__'):
			self.modules['__main__'] = self.pretendMain
			self.pretendMain.__builtins__ = self.modules['__builtin__']
			m = self.add_module('__main__')

# Classes that looks and behaves like RExec, but isnt really!
import IHooks
class AXNotRHooks(IHooks.Hooks):
	pass

class AXNotRExec:
	def __init__(self, pretendMain, hooks = None, verbose = 0):
		self.pretendMain = pretendMain
		self.hooks = hooks or AXNotRHooks(verbose)
		self.modules = {'__main__': self.pretendMain}

	def add_module(self, mname):
		if self.modules.has_key(mname):
			return self.modules[mname]
		self.modules[mname] = m = self.hooks.new_module(mname)
#		m.__builtins__ = self.modules['__builtin__']
		return m

# There is only ever _one_ ax object - it exists in the global namespace
# for all script items.
# It performs a search from all global/visible objects
# down.
# This means that if 2 sub-objects of the same name are used
# then only one is ever reachable using the ax shortcut.
class AXScriptAttribute:
	"An attribute in a scripts namespace."
	def __init__(self, engine):
		self.__dict__['_scriptEngine_'] = engine
	def __getattr__(self, attr):
		if attr[1]=="_" and attr[:-1]=="_":
			raise AttributeError, attr
		rc = self._FindAttribute_(string.lower(attr))
		if rc is None:
			raise AttributeError, attr
		return rc
	def _Reset_(self):
		self.__dict__['_scriptEngine_'] = None

	def _DoFindAttribute_(self, obj, attr):
		try:
			return obj.subItems[attr].attributeObject
		except KeyError:
			pass
		# Check out the sub-items
		for item in obj.subItems.values():
			try:
				return self._DoFindAttribute_(item, attr)
			except AttributeError:
				pass
		raise AttributeError, attr

	def _FindAttribute_(self, attr):
		for item in self._scriptEngine_.subItems.values():
			try:
				return self._DoFindAttribute_(item, attr)
			except AttributeError:
				pass
		# All else fails, see if it is a global
		# (mainly b/w compat)
		return getattr(self._scriptEngine_.globalNameSpaceModule, attr)
#		raise AttributeError, attr

class NamedScriptAttribute:
	"An explicitely named object in an objects namespace"
	# Each named object holds a reference to one of these.
	# Whenever a sub-item appears in a namespace, it is really one of these
	# objects.  Has a circular reference back to the item itself, which is
	# closed via _Reset_()
	def __init__(self, scriptItem):
		self.__dict__['_scriptItem_'] = scriptItem
	def __repr__(self):
		return "<NamedItemAttribute" + repr(self._scriptItem_) + ">"
	def __getattr__(self, attr):
		# If a known subitem, return it.
		attr = string.lower(attr)
		try:
			return self._scriptItem_.subItems[attr].attributeObject
		except KeyError:
			# Otherwise see if the dispatch can give it to us
			if self._scriptItem_.dispatchContainer:
				return getattr(self._scriptItem_.dispatchContainer,attr)
		raise AttributeError, attr
	def __setattr__(self, attr, value):
		# XXX - todo - if a known item, then should call its default
		# dispatch method.
		attr = string.lower(attr)
		if self._scriptItem_.dispatchContainer:
			try:
				return setattr(self._scriptItem_.dispatchContainer,attr, value)
			except AttributeError:
				pass
		raise AttributeError, attr
	def _Reset_(self):
		self.__dict__['_scriptItem_'] = None

	
class ScriptItem(framework.ScriptItem):
	def __init__(self, parentItem, name, dispatch, flags):
		framework.ScriptItem.__init__(self, parentItem, name, dispatch, flags)
		self.scriptlets = {}
		self.attributeObject = None
	def Reset(self):
		framework.ScriptItem.Reset(self)
		self.dispatchContainer = None
		self.scriptlets = {}
		if self.attributeObject:
			self.attributeObject._Reset_()
		self.attributeObject = None

	def Register(self):
		framework.ScriptItem.Register(self)
		self.attributeObject = NamedScriptAttribute(self)
		if self.dispatch:
			self.dispatchContainer = win32com.client.dynamic.Dispatch(self.dispatch, userName = self.name)
#			self.dispatchContainer = win32com.client.dynamic.DumbDispatch(self.dispatch, userName = self.name)
			
#	def Connect(self):
#		framework.ScriptItem.Connect(self)
#	def Disconnect(self):
#		framework.ScriptItem.Disconnect(self)
		
class PyScript(framework.COMScript):
	# Setup the auto-registration stuff...
	_reg_verprogid_ = "Python.AXScript.2"
	_reg_progid_ = "Python"
#	_reg_policy_spec_ = default
	_reg_catids_ = [axscript.CATID_ActiveScript,axscript.CATID_ActiveScriptParse]
	_reg_desc_ = "Python ActiveX Scripting Engine"
	_reg_clsid_ = PyScript_CLSID
	_reg_class_spec_ = "win32com.axscript.client.pyscript.PyScript"

	def InitNew(self):
		framework.COMScript.InitNew(self)
		import imp
		self.scriptDispatch = None
		self.globalNameSpaceModule = imp.new_module("__ax_main__")
		self.globalNameSpaceModule.__dict__['ax'] = AXScriptAttribute(self)
		self.rexec_env = None # will be created first time around.
		
		self.pendingCodeBlocks = []
		self.codeBlockCounter = 0

	def Stop(self):
		# Dont execute anything else!
		self.pendingCodeBlocks = []
		return framework.COMScript.Stop(self)


	def _GetNextCodeBlockNumber(self):
		self.codeBlockCounter = self.codeBlockCounter + 1
		return self.codeBlockCounter
		
	def RegisterNamedItem(self, item):
		if self.rexec_env is None:
			if self.safetyOptions:
				# Use RExec.
				self.rexec_env = AXRExec(self.globalNameSpaceModule)
			else:
				# DONT use RExec.
				self.rexec_env = AXNotRExec(self.globalNameSpaceModule)

		wasReg = item.isRegistered
		framework.COMScript.RegisterNamedItem(self, item)
		if not wasReg:
			# Insert into our namespace.
			# Add every item by name
			if item.IsVisible():
				self.globalNameSpaceModule.__dict__[item.name] = item.attributeObject
			if item.IsGlobal():
				# Global items means sub-items are also added...
				for subitem in item.subItems.values():
					self.globalNameSpaceModule.__dict__[subitem.name] = subitem.attributeObject
				# Also add all methods
				for name, entry in item.dispatchContainer._olerepr_.mapFuncs.items():
					if not entry.hidden:
						self.globalNameSpaceModule.__dict__[name] = getattr(item.dispatchContainer,name)

	def DoExecutePendingScripts(self):
		try:
			globs = self.globalNameSpaceModule.__dict__
			for codeBlock in self.pendingCodeBlocks:
				codeObject = self.CompileInScriptedSection(codeBlock.codeText, codeBlock.GetFileName(), "exec")
				if codeObject:
					self.ExecInScriptedSection(codeObject, globs)
		finally:
			self.pendingCodeBlocks = []
	
	def DoRun(self):
		pass
		
	def Close(self):
		self.ResetNamespace()
		self.globalNameSpaceModule = None
		self.pendingCodeBlocks = []
		self.scriptDispatch = None
		framework.COMScript.Close(self)

	def GetScriptDispatch(self, name):
#		trace("GetScriptDispatch with", name)
		if self.scriptDispatch is None:
		
			# Base classes should override.
#			if name:
#				raise Exception(scode=winerror.E_NOTIMPL)
#					RaiseAssert(scode=winerror.E_NOTIMPL, desc = "Not expecting a name!")
			self.scriptDispatch = self._wrap(scriptdispatch.ScriptDispatch(self.globalNameSpaceModule))
		return self.scriptDispatch

	def MakeEventMethodName(self, subItemName, eventName):
		return string.upper(subItemName[0])+subItemName[1:] + "_" + string.upper(eventName[0])+eventName[1:]

	def DoAddScriptlet(self, defaultName, code, itemName, subItemName, eventName, delimiter,sourceContextCookie, startLineNumber):
		# Just store the code away - compile when called.  (JIT :-)
		item = self.GetNamedItem(itemName)
		if itemName==subItemName: # Explicit handlers - eg <SCRIPT LANGUAGE="Python" for="TestForm" Event="onSubmit">
			subItem = item
		else:
			subItem = item.GetCreateSubItem(item, subItemName, None, None)
		funcName = self.MakeEventMethodName(subItemName, eventName)
		
		codeBlock = AXScriptCodeBlock("Script Event %s" %funcName, code, sourceContextCookie, startLineNumber)
		self._AddScriptCodeBlock(codeBlock)
		subItem.scriptlets[funcName] = codeBlock

	def DoProcessScriptItemEvent(self, item, event, lcid, wFlags, args):
#		trace("ScriptItemEvent", self, item, event, event.name, lcid, wFlags, args)
		funcName = self.MakeEventMethodName(item.name, event.name)
		codeBlock = function = None
		try:
			function = item.scriptlets[funcName]
			if type(function)==type(self): # ie, is a CodeBlock instance
				codeBlock = function
				function = None
		except KeyError:
			pass
		if codeBlock:
			realCode = "def %s():\n" % funcName
			for line in string.split(RemoveCR(codeBlock.codeText),"\n"):
				realCode = realCode + '\t' + line + '\n'
			realCode = realCode + '\n'
			codeObject = self.CompileInScriptedSection(realCode, codeBlock.GetFileName(), "exec")
			if not codeObject:
				return
			dict = {}
			self.ExecInScriptedSection(codeObject, self.globalNameSpaceModule.__dict__, dict)
			function = dict[funcName]
			# cache back in scriptlets as a function.
			item.scriptlets[funcName] = function
		if function is None:
			# still no function - see if in the global namespace.
			try:
				function = self.globalNameSpaceModule.__dict__[funcName]
			except KeyError:
				# Not there _exactly_ - do case ins search.
				funcNameLook = string.lower(funcName)
				for attr in self.globalNameSpaceModule.__dict__.keys():
					if funcNameLook==string.lower(attr):
						function = self.globalNameSpaceModule.__dict__[attr]
						# cache back in scriptlets, to avoid this overhead next time
						item.scriptlets[funcName] = function

		if function is None:
			raise Exception(scode=winerror.DISP_E_MEMBERNOTFOUND)
		return self.ApplyInScriptedSection(function, args)

	def DoParseScriptText(self, code, sourceContextCookie, startLineNumber, bWantResult, flags):
		code = RemoveCR(code) + "\n"
		if flags & SCRIPTTEXT_ISEXPRESSION:
			name = "Script Expression"
			exec_type = "eval"
		else:
			name = "Script Block"
			exec_type = "exec"
		num = self._GetNextCodeBlockNumber()
		if num==1: num=""
		name = "%s %s" % (name, num)
		codeBlock = AXScriptCodeBlock(name, code, sourceContextCookie, startLineNumber)
		self._AddScriptCodeBlock(codeBlock)
		globs = self.globalNameSpaceModule.__dict__
		if bWantResult: # always immediate.
			codeObject = self.CompileInScriptedSection(codeBlock.codeText, codeBlock.GetFileName(), exec_type)
			if codeObject:
				if flags & SCRIPTTEXT_ISEXPRESSION:
					return self.EvalInScriptedSection(codeObject, globs)
				else:
					return self.ExecInScriptedSection(codeObject, globs)
				
			# else compile failed, but user chose to keep running...
		else:
			if flags & SCRIPTTEXT_FORCEEXECUTION:
				codeObject = self.CompileInScriptedSection(codeBlock.codeText, codeBlock.GetFileName(), exec_type)
				if codeObject:
					self.ExecInScriptedSection(codeObject, globs)
			else:
				self.pendingCodeBlocks.append(codeBlock)

	def GetNamedItemClass(self):
		return ScriptItem

	def ResetNamespace(self):
		# actually clean out the dictionary!
		try:
			self.globalNameSpaceModule.ax._Reset_()
		except AttributeError:
			pass # ???
		for key, item in self.globalNameSpaceModule.__dict__.items():
			del self.globalNameSpaceModule.__dict__[key]
		self.rexec_env = None
		
	def _wrap_debug(self, obj):
		return win32com.server.util.wrap(obj, usePolicy=win32com.server.policy.DynamicPolicy, useDispatcher=win32com.server.policy.DispatcherWin32trace)
	def _wrap_nodebug(self, obj):
		return win32com.server.util.wrap(obj, usePolicy=win32com.server.policy.DynamicPolicy)

	bDebugging = win32com.server.register.GetRegisteredServerOption(_reg_clsid_, "Debugging")=='1'
	if bDebugging:
		_wrap = _wrap_debug
	else:
		_wrap = _wrap_nodebug

def Register():
	import sys
	win32com.server.register.UseCommandLine(PyScript)
	if '--unregister' in sys.argv:
		# Unregister our special stuff.
		win32com.server.register.recurse_delete_key(".pys")
		win32com.server.register.recurse_delete_key("pysFile")
		print "Unregistration of Scripting Engine complete."
	else:
		# If we are registering, do our extra stuff.
		win32com.server.register._set_subkeys(PyScript._reg_progid_ + "\\OLEScript", {}) # Just a CreateKey
		# Basic Registration for wsh.
		win32com.server.register._set_string(".pys", "pysFile")
		win32com.server.register._set_string("pysFile\\ScriptEngine", PyScript._reg_progid_)
		print "Registration of Scripting Engine complete."
	
	
if __name__=='__main__':
	Register()

