"""AXScript Client Framework

  This module provides a core framework for an ActiveX Scripting client.
  Derived classes actually implement the AX Client itself, including the
  scoping rules, etc.

  There are classes defined for the engine itself, and for ScriptItems
"""
import sys, string
import ni
from win32com.axscript import axscript
import win32com.server.util

import win32com.client.connect # Need simple connection point support

import winerror
import pythoncom
import types

SCRIPTTEXT_FORCEEXECUTION = 0x80000000
SCRIPTTEXT_ISEXPRESSION   = 0x00000020


from win32com.server.exception import Exception
import error # ax.client.error

def trace(*args):
	"""A function used instead of "print" for debugging output.
	"""
	for arg in args:
		print arg,
	print

def RaiseAssert(scode, desc):
	"""A debugging function that raises an exception considered an "Assertion".
	"""
	print "**************** ASSERTION FAILED *******************"
	print desc
	raise Exception(scode, desc)

class AXScriptCodeBlock:
	"""An object which represents a chunk of code in an AX Script
	"""
	def __init__(self, name, codeText, sourceContextCookie, startLineNumber):
		# If codeTextDisplay is none, it assumes the codeText is displayed.
		# This is handy if the actual code has been modified by the engine.
		self.name = name
		self.codeText = codeText
		self.codeObject = None
		self.sourceContextCookie = sourceContextCookie
		self.startLineNumber = startLineNumber
	def GetFileName(self):
		# Gets the "file name" for Python - uses <...> so Python doesnt think
		# it is a real file.
		return "<%s>" % self.name
	def GetDisplayName(self):
		return self.name
	def GetLineNo(self, no):
		pos = -1
		for i in range(no-1):
			pos = string.find(self.codeText, '\n', pos+1)
			if pos==-1: pos=len(self.codeText)
		epos = string.find(self.codeText, '\n', pos+1)
		if epos==-1:
			epos=len(self.codeText)
		return string.strip(self.codeText[pos+1:epos])

class Event:
	"""A single event for a ActiveX named object.
	"""
	def __init__(self):
		self.name = "<None>"
	def __repr__(self):
		return "<%s at %d: %s>" % (self.__class__.__name__, id(self), self.name)
	def Reset(self):
		pass
	def Build(self, typeinfo, funcdesc):
		self.dispid = funcdesc[0]
		self.name = typeinfo.GetNames(self.dispid)[0]
#		print "Event.Build() - Event Name is ", self.name

class EventSink:
	"""A set of events against an item.  Note this is a COM client for connection points.
	"""
	_public_methods_ = []
	def __init__(self, myItem, coDispatch):
		self.events = {}
		self.connection = None
		self.coDispatch = coDispatch
		self.myScriptItem = myItem
		self.myInvokeMethod = myItem.GetEngine().ProcessScriptItemEvent
		self.iid = None
	def Reset(self):
		self.iid = None
		self.myScriptItem = None
		self.myInvokeMethod = None
		self.coDispatch = None
		for event in self.events.values():
			event.Reset()
		self.events = {}
		self.Disconnect()
	# COM Connection point methods.
	def _query_interface_(self, iid):
		if iid==self.iid:
			return win32com.server.util.wrap(self)
	def _invoke_(self, dispid, lcid, wFlags, args):
		try:
			event = self.events[dispid]
		except:
			raise Exception(scode=winerror.DISP_E_MEMBERNOTFOUND)
#		print "Invoke for ", event, "on", self.myScriptItem
		return self.myInvokeMethod(self.myScriptItem, event, lcid, wFlags, args)

	def GetSourceTypeInfo(self, typeinfo):
		"""Gets the typeinfo for the Source Events for the passed typeinfo"""
		attr = typeinfo.GetTypeAttr()
		cFuncs = attr[6]
		typeKind = attr[5]
		if typeKind not in [pythoncom.TKIND_COCLASS, pythoncom.TKIND_INTERFACE]:
			RaiseAssert(winerror.E_UNEXPECTED, "The typeKind of the object is unexpected")
		cImplType = attr[8]
		for i in xrange(cImplType):
			# Look for the [source, default] interface on the coclass
			# that isn't marked as restricted.
			flags = typeinfo.GetImplTypeFlags(i)
			flagsNeeded = pythoncom.IMPLTYPEFLAG_FDEFAULT | pythoncom.IMPLTYPEFLAG_FSOURCE
			if (flags & ( flagsNeeded | pythoncom.IMPLTYPEFLAG_FRESTRICTED))==(flagsNeeded):
				# Get the handle to the implemented interface.
				href = typeinfo.GetRefTypeOfImplType(i)
				return typeinfo.GetRefTypeInfo(href)

	def BuildEvents(self):
		# See if it is an extender object.
		try:
			mainTypeInfo = self.coDispatch.QueryInterface(axscript.IID_IProvideMultipleClassInfo)
			isMulti = 1
			numTypeInfos = mainTypeInfo.GetMultiTypeInfoCount()
		except win32com.pythoncom.com_error:
			isMulti = 0
			numTypeInfos = 1
			try:
				mainTypeInfo = self.coDispatch.QueryInterface(pythoncom.IID_IProvideClassInfo)
			except pythoncom.com_error:
				numTypeInfos = 0
		# Create an event handler for the item.
		for item in xrange(numTypeInfos):
			if isMulti:
				typeinfo, flags = mainTypeInfo.GetInfoOfIndex(item, axscript.MULTICLASSINFO_GETTYPEINFO)
			else:
				typeinfo = mainTypeInfo.GetClassInfo()
			sourceType = self.GetSourceTypeInfo(typeinfo)
			cFuncs = 0
			if sourceType:
				attr = sourceType.GetTypeAttr()
				self.iid = attr[0]
				cFuncs = attr[6]
				for i in xrange(cFuncs):
					funcdesc = sourceType.GetFuncDesc(i)
					event = Event()
					event.Build(sourceType, funcdesc)
					self.events[event.dispid] = event

	def Connect(self):
		if self.connection is not None or self.iid is None: return
#		trace("Connect for sink item", self.myScriptItem.name, "with IID",str(self.iid))
		self.connection = win32com.client.connect.SimpleConnection(self.coDispatch, self, self.iid)
	def Disconnect(self):
		if self.connection:
			self.connection.Disconnect()
			self.connection = None

class ScriptItem:
	"""An item (or subitem) that is exposed to the ActiveX script
	"""
	def __init__(self, parentItem, name, dispatch, flags):
		self.parentItem = parentItem
		self.dispatch = dispatch
		self.name = name
		self.flags = flags
		self.eventSink = None
		self.subItems = {}
		self.createdConnections = 0
		self.isRegistered = 0
#		trace("Creating ScriptItem", name, "of parent", parentItem,"with dispatch", dispatch)

	def __repr__(self):
		flagsDesc=""
		if self.flags is not None and self.flags & axscript.SCRIPTITEM_GLOBALMEMBERS:
			flagsDesc = "/Global"
		return "<%s at %d: %s%s>" % (self.__class__.__name__, id(self), self.name,flagsDesc)

	def _dump_(self, level):
		flagsDesc = ""
		if self.flags is not None and self.flags & axscript.SCRIPTITEM_GLOBALMEMBERS:
			flagsDesc = "GLOBAL!"
		print " " * level, self.name, flagsDesc, self
		for subItem in self.subItems.values():
			subItem._dump_(level+1)

	def Reset(self):
		self.Disconnect()
		self.dispatch = None
		self.parentItem = None
		if self.eventSink:
			self.eventSink.Reset()
			self.eventSink = None
		for subItem in self.subItems.values():
			subItem.Reset()
		self.createdConnections = 0
		self.isRegistered = 0

	def Register(self):
		if self.isRegistered: return
		# Get the type info to use to build this item.
#		if not self.dispatch:
#			id = self.parentItem.dispatch.GetIDsOfNames(self.name)
#			print "DispID of me is", id
#			result = self.parentItem.dispatch.Invoke(id, 0, pythoncom.DISPATCH_PROPERTYGET,1)
#			if type(result)==pythoncom.TypeIIDs[pythoncom.IID_IDispatch]:
#				self.dispatch = result
#			else:
#				print "*** No dispatch"
#				return
#			print "**** Made dispatch"
		self.isRegistered = 1
		# Register the sub-items.				
		for item in self.subItems.values():
			if not item.isRegistered:
				item.Register()

	def IsGlobal(self):
		return self.flags & axscript.SCRIPTITEM_GLOBALMEMBERS

	def IsVisible(self):
		return self.flags & axscript.SCRIPTITEM_ISVISIBLE

	def GetEngine(self):
		item = self
		while item.parentItem.__class__==self.__class__:
			item = item.parentItem
		return item.parentItem

	def _GetFullItemName(self):
		ret = self.name
		if self.parentItem:
			try:
				ret = self.parentItem._GetFullItemName() + "." + ret
			except AttributeError:
				pass
		return ret
	
	def GetSubItemClass(self):
		return self.__class__

	def GetSubItem(self, name):
		return self.subItems[string.lower(name)]

	def GetCreateSubItem(self, parentItem, name, dispatch, flags):
		keyName = string.lower(name)
		try:
			rc = self.subItems[keyName]
			# No changes allowed to existing flags.
			if not rc.flags is None and not flags is None and rc.flags != flags:
				raise Exception(scode=winerror.E_INVALIDARG)
			# Existing item must not have a dispatch.
			if not rc.dispatch is None and not dispatch is None:
				raise Exception(scode=winerror.E_INVALIDARG)
			rc.flags = flags # Setup the real flags.
			rc.dispatch = dispatch
		except KeyError:
			rc = self.subItems[keyName] = self.GetSubItemClass()(parentItem, name, dispatch, flags)
		return rc
#		if self.dispatch is None: 
#			RaiseAssert(winerror.E_UNEXPECTED, "??")

	def CreateConnections(self):
		# Create (but do not connect to) the connection points.
		if self.createdConnections: return
		self.createdConnections = 1
		# Nothing to do unless this is an event source
		# This flags means self, _and_ children, are connectable.
		if self.flags & axscript.SCRIPTITEM_ISSOURCE:
			self.BuildEvents()
			self.FindBuildSubItemEvents()

	def Connect(self):
		# Connect to the already created connection points.
		if self.eventSink:
			self.eventSink.Connect()
		for subItem in self.subItems.values():
			subItem.Connect()
			
	def Disconnect(self):
		# Disconnect from the connection points.
		if self.eventSink:
			self.eventSink.Disconnect()
		for subItem in self.subItems.values():
			subItem.Disconnect()


	def BuildEvents(self):
		if self.eventSink is not None or self.dispatch is None:
			RaiseAssert(winerror.E_UNEXPECTED, "Item already has built events, or no dispatch available?")
		
#		trace("BuildEvents for named item", self._GetFullItemName())
		self.eventSink = EventSink(self, self.dispatch)
		self.eventSink.BuildEvents()

	def FindBuildSubItemEvents(self):
		# Called during connection to event source.  Seeks out and connects to
		# all children.  As per the AX spec, this is not recursive
		# (ie, children sub-items are not seeked)
		try:
			multiTypeInfo = self.dispatch.QueryInterface(axscript.IID_IProvideMultipleClassInfo)
			numTypeInfos = multiTypeInfo.GetMultiTypeInfoCount()
		except win32com.pythoncom.com_error:
			return
		for item in xrange(numTypeInfos):
			typeinfo, flags = multiTypeInfo.GetInfoOfIndex(item, axscript.MULTICLASSINFO_GETTYPEINFO)
			defaultType = self.GetDefaultSourceTypeInfo(typeinfo)
			index = 0
			while 1:
				try:
					fdesc = defaultType.GetFuncDesc(index)
				except pythoncom.com_error:
					break # No more funcs
				index = index + 1
				dispid = fdesc[0]
				funckind = fdesc[3]
				invkind = fdesc[4]
				elemdesc = fdesc[8]
				funcflags = fdesc[9]
				try:
					isSubObject = not (funcflags & pythoncom.FUNCFLAG_FRESTRICTED) and \
						funckind == pythoncom.FUNC_DISPATCH and \
						invkind  == pythoncom.INVOKE_PROPERTYGET and \
						elemdesc[0][0] == pythoncom.VT_PTR and \
						elemdesc[0][1][0] == pythoncom.VT_USERDEFINED
				except:
					isSubObject = 0
				if isSubObject:
						try:
							# We found a sub-object.
							names = typeinfo.GetNames(dispid);
							result = self.dispatch.Invoke(dispid, 0x0, pythoncom.DISPATCH_PROPERTYGET, 1)
							if type(result)==pythoncom.TypeIIDs[pythoncom.IID_IDispatch]:
								name = names[0]
								subObj = self.GetCreateSubItem(self, name, result, axscript.SCRIPTITEM_ISVISIBLE)
								subObj.BuildEvents()
								subObj.Register()

						except pythoncom.com_error:
							pass

	def GetDefaultSourceTypeInfo(self, typeinfo):
		"""Gets the typeinfo for the Default Dispatch for the passed typeinfo"""
		attr = typeinfo.GetTypeAttr()
		cFuncs = attr[6]
		typeKind = attr[5]
		if typeKind not in [pythoncom.TKIND_COCLASS, pythoncom.TKIND_INTERFACE]:
			RaiseAssert(winerror.E_UNEXPECTED, "The typeKind of the object is unexpected")
		cImplType = attr[8]
		for i in xrange(cImplType):
			# Look for the [source, default] interface on the coclass
			# that isn't marked as restricted.
			flags = typeinfo.GetImplTypeFlags(i)
			if (flags & ( pythoncom.IMPLTYPEFLAG_FDEFAULT | pythoncom.IMPLTYPEFLAG_FSOURCE | pythoncom.IMPLTYPEFLAG_FRESTRICTED))==pythoncom.IMPLTYPEFLAG_FDEFAULT:
				# Get the handle to the implemented interface.
				href = typeinfo.GetRefTypeOfImplType(i)
				defTypeInfo = typeinfo.GetRefTypeInfo(href)
				attr = defTypeInfo.GetTypeAttr()
				typeKind = attr[5]
				typeFlags = attr[11]
				if typeKind == pythoncom.TKIND_INTERFACE and typeFlags & pythoncom.TYPEFLAG_FDUAL:
					# Get corresponding Disp interface
					# -1 is a special value which does this for us.
					href = typeinfo.GetRefTypeOfImplType(-1)
					return defTypeInfo.GetRefTypeInfo(href)
				else:
					return defTypeInfo


IActiveScriptMethods = [
   "SetScriptSite", "GetScriptSite", "SetScriptState", "GetScriptState",
   "Close", "AddNamedItem", "AddTypeLib", "GetScriptDispatch",
   "GetCurrentScriptThreadID", "GetScriptThreadID", "GetScriptThreadState",
   "InterruptScriptThread", "Clone" ]
IActiveScriptParseMethods = [
   "InitNew", "AddScriptlet", "ParseScriptText" ]
IObjectSafetyMethods = [
   "GetInterfaceSafetyOptions", "SetInterfaceSafetyOptions"]

# ActiveScriptParseProcedure is a new interface with IIS4/IE4.
IActiveScriptParseProcedureMethods = ['ParseProcedureText']
class COMScript:
	"""An ActiveX Scripting engine base class.

	This class implements the required COM interfaces for ActiveX scripting.
	"""
	_public_methods_ = IActiveScriptMethods + IActiveScriptParseMethods + IObjectSafetyMethods + IActiveScriptParseProcedureMethods
	_com_interfaces_ = [axscript.IID_IActiveScript, axscript.IID_IActiveScriptParse, axscript.IID_IObjectSafety] # axscript.IID_IActiveScriptParseProcedure]
	
	def __init__(self):
#		import sys
#		trace("AXScriptEngine object created\n")
		self.baseThreadId = -1
		self.debugManager = None
		self.threadState = axscript.SCRIPTTHREADSTATE_NOTINSCRIPT
		self.scriptState = axscript.SCRIPTSTATE_UNINITIALIZED
		self.scriptSite = None
		self.safetyOptions = 0
		self.lcid = 0
		self.subItems = {}
		self.persistLoaded = 0
		self.scriptCodeBlocks = {}

	def _query_interface_(self, iid):
		if self.debugManager:
			return self.debugManager._query_interface_for_debugger_(iid)
		trace("ScriptEngine QI - unknown IID", iid)
		return 0
	#
	# IActiveScriptParse
	def InitNew(self):
		if self.persistLoaded:
			raise Exception(scode=winerror.E_UNEXPECTED)

		self.persistLoaded = 1
		if self.scriptSite is not None:
			self.ChangeScriptState(axscript.SCRIPTSTATE_INITIALIZED)
			
	def AddScriptlet(self, defaultName, code, itemName, subItemName, eventName, delimiter, sourceContextCookie, startLineNumber):
#		trace ("AddScriptlet", defaultName, code, itemName, subItemName, eventName, delimiter, sourceContextCookie, startLineNumber)
		self.DoAddScriptlet(defaultName, str(code), itemName, subItemName, eventName, delimiter,sourceContextCookie, startLineNumber)

	def ParseScriptText(self, code, itemName, context, delimiter, sourceContextCookie, startLineNumber, flags, bWantResult):
#		trace ("ParseScriptText", code[:20],"...", itemName, context, delimiter, sourceContextCookie, startLineNumber, flags, bWantResult)
		if bWantResult or self.scriptState == axscript.SCRIPTSTATE_STARTED \
				or self.scriptState == axscript.SCRIPTSTATE_CONNECTED \
				or self.scriptState == axscript.SCRIPTSTATE_DISCONNECTED :
			flags = flags | SCRIPTTEXT_FORCEEXECUTION
		else:
			flags = flags & (~SCRIPTTEXT_FORCEEXECUTION)

		if flags & SCRIPTTEXT_FORCEEXECUTION:
			# About to execute the code.
			self.RegisterNewNamedItems()
		return self.DoParseScriptText(str(code), sourceContextCookie, startLineNumber, bWantResult, flags)
						
	#
	# IActiveScriptParseProcedure
	def ParseProcedureText( self, code, formalParams, procName, itemName, unkContext, delimiter, contextCookie, startingLineNumber, flags):
		trace("ParseProcedureText", code, formalParams, procName, itemName, unkContext, delimiter, contextCookie, startingLineNumber, flags)
		return None
	#
	# IActiveScript
	def SetScriptSite(self, site):
		if self.scriptSite: raise Exception(scode=winerror.E_UNEXPECTED)
		self.scriptSite = site
		import traceback
		try:
			import debug
			self.debugManager = debug.DebugManager(self)
		except (ImportError, pythoncom.com_error), details:
			# COM errors will occur if the debugger interface has never been
			# seen on the target system
			traceback.print_exc()
			trace("*** Debugger Manager could not initialize - %s" % (`details`))
			self.debugManager = None

		try:
			self.lcid = site.GetLCID() 
		except pythoncom.com_error:
			self.lcid = 0 # todo - GetUserDefaultLCID
		self.ResetNamedItems()
		if self.persistLoaded:
			self.ChangeScriptState(axscript.SCRIPTSTATE_INITIALIZED)

	def GetScriptSite(self, iid):
		if self.scriptSite is None: raise Exception(scode=winerror.S_FALSE)
		return self.scriptSite.QueryInterface(iid)

	def SetScriptState(self, state):
#		trace("SetScriptState with %d - currentstate = %d" % (state,self.scriptState))
		if state == self.scriptState: return
		if self.scriptState==axscript.SCRIPTSTATE_CLOSED:
			raise Exception(scode=winerror.E_INVALIDARG)
		if state==axscript.SCRIPTSTATE_INITIALIZED:
			if self.scriptState in [axscript.SCRIPTSTATE_CONNECTED, axscript.SCRIPTSTATE_DISCONNECTED]:
				self.Stop()
			if self.scriptState in [axscript.SCRIPTSTATE_CONNECTED, axscript.SCRIPTSTATE_DISCONNECTED,axscript.SCRIPTSTATE_STARTED]:
				self.Reset()
			if self.scriptState in [axscript.SCRIPTSTATE_CONNECTED, axscript.SCRIPTSTATE_DISCONNECTED,axscript.SCRIPTSTATE_STARTED,axscript.SCRIPTSTATE_UNINITIALIZED]:
				self.ChangeScriptState(axscript.SCRIPTSTATE_INITIALIZED)
		elif state==axscript.SCRIPTSTATE_STARTED:
			if self.scriptState in [axscript.SCRIPTSTATE_CONNECTED, axscript.SCRIPTSTATE_DISCONNECTED]:
				self.Stop()
				self.Reset()
#			if self.scriptState in [axscript.SCRIPTSTATE_CONNECTED, axscript.SCRIPTSTATE_DISCONNECTED, axscript.SCRIPTSTATE_UNINITIALIZED,axscript.SCRIPTSTATE_INITIALIZED]:
			self.ExecutePendingScripts()
			self.ChangeScriptState(axscript.SCRIPTSTATE_STARTED)
		elif state==axscript.SCRIPTSTATE_CONNECTED:
			if self.scriptState in [axscript.SCRIPTSTATE_UNINITIALIZED,axscript.SCRIPTSTATE_INITIALIZED]:
				self.SetScriptState(axscript.SCRIPTSTATE_INITIALIZED) # transition through started
			if self.scriptState in [axscript.SCRIPTSTATE_STARTED, axscript.SCRIPTSTATE_UNINITIALIZED,axscript.SCRIPTSTATE_INITIALIZED]:
				self.ExecutePendingScripts()
				self.Run()
				self.ChangeScriptState(state)
#			if self.scriptState in [axscript.SCRIPTSTATE_DISCONNECTED, axscript.SCRIPTSTATE_STARTED, axscript.SCRIPTSTATE_UNINITIALIZED,axscript.SCRIPTSTATE_INITIALIZED]:
#				self.Reconnect()
		elif state==axscript.SCRIPTSTATE_DISCONNECTED:
			pass #??
		elif state==axscript.SCRIPTSTATE_CLOSED:
			self.Close()
		else:
			raise Exception(scode=winerror.E_INVALIDARG)

	def GetScriptState(self):
		return self.scriptState

	def Close(self):
#		trace("Close")
		if self.scriptState in [axscript.SCRIPTSTATE_CONNECTED, axscript.SCRIPTSTATE_DISCONNECTED]:
			self.Stop()
		if self.scriptState in [axscript.SCRIPTSTATE_CONNECTED, axscript.SCRIPTSTATE_DISCONNECTED, axscript.SCRIPTSTATE_INITIALIZED, axscript.SCRIPTSTATE_STARTED]:
			pass # engine.close??
		if self.scriptState in [axscript.SCRIPTSTATE_UNINITIALIZED, axscript.SCRIPTSTATE_CONNECTED, axscript.SCRIPTSTATE_DISCONNECTED, axscript.SCRIPTSTATE_INITIALIZED, axscript.SCRIPTSTATE_STARTED]:
			self.ChangeScriptState(axscript.SCRIPTSTATE_CLOSED)
			for item in self.subItems.values():
				item.Reset()
			self.subItems = {}
			self.baseThreadId = -1
		if self.debugManager:
			self.debugManager.Close()
			self.debugManager = None
		self.scriptSite = None
		self.scriptCodeBlocks = {}

	def AddNamedItem(self, name, flags):
		if self.scriptSite is None: raise Exception(scode=winerror.E_INVALIDARG)
		name = str(name) # Convert from Unicode.
		unknown = self.scriptSite.GetItemInfo(name, axscript.SCRIPTINFO_IUNKNOWN)[0]
		try:
			dispatch = unknown.QueryInterface(pythoncom.IID_IDispatch)
		except pythoncom.com_error:
			RaiseAssert(winerror.E_NOINTERFACE, "Object has no dispatch interface available.")
		newItem = self.subItems[name] = self.GetNamedItemClass()(self, name, dispatch, flags)
		if newItem.IsGlobal():
			newItem.CreateConnections()

	def GetScriptDispatch(self, name):
		# Base classes should override.
		raise Exception(scode=winerror.E_NOTIMPL)

	def GetCurrentScriptThreadID(self):
		return self.baseThreadId

	def GetScriptThreadID(self, win32ThreadId):
		if self.baseThreadId == -1:
			raise Exception(scode=winerror.E_UNEXPECTED)
		if self.baseThreadId != win32ThreadId:
			raise Exception(scode=winerror.E_INVALIDARG)
		return self.baseThreadId

	def GetScriptThreadState(self, scriptThreadId):
		if self.baseThreadId == -1:
			raise Exception(scode=winerror.E_UNEXPECTED)
		if scriptThreadId != self.baseThreadId:
			raise Exception(scode=winerror.E_INVALIDARG)
		return self.threadState

	def AddTypeLib(self, uuid, major, minor, flags):
		raise Exception("Not Implemented")	

	def InterruptScriptThread(self, state, flags):
		raise Exception("Not Implemented")

	def Clone(self):
		raise Exception("Not Implemented")	
	#
	# IObjectSafety
	def SetInterfaceSafetyOptions(self, iid, optionsMask, enabledOptions):
#		trace ("SetInterfaceSafetyOptions", iid, optionsMask, enabledOptions)
		if optionsMask & enabledOptions == 0:
			return

		if (optionsMask & enabledOptions & \
			~(axscript.INTERFACESAFE_FOR_UNTRUSTED_DATA | axscript.INTERFACESAFE_FOR_UNTRUSTED_CALLER)):
			# request for options we don't understand
			RaiseAssert(scode=winerror.E_FAIL, desc="Unknown safety options")

		if iid in [pythoncom.IID_IPersist, pythoncom.IID_IPersistStream, pythoncom.IID_IPersistStreamInit,
		            axscript.IID_IActiveScript, axscript.IID_IActiveScriptParse]:
			self.safetyOptions = 0
			if optionsMask & enabledOptions & axscript.INTERFACESAFE_FOR_UNTRUSTED_CALLER:
				self.safetyOptions = self.safetyOptions | axscript.INTERFACESAFE_FOR_UNTRUSTED_CALLER
			if optionsMask & enabledOptions & axscript.INTERFACESAFE_FOR_UNTRUSTED_DATA:
				self.safetyOptions = self.safetyOptions | axscript.INTERFACESAFE_FOR_UNTRUSTED_DATA
		else:
			raise Exception(scode=winerror.E_NOINTERFACE)
		
	def GetInterfaceSafetyOptions(self, iid):
		if iid in [pythoncom.IID_IPersist, pythoncom.IID_IPersistStream, pythoncom.IID_IPersistStreamInit,
		            axscript.IID_IActiveScript, axscript.IID_IActiveScriptParse]:
			return self.safetyOptions
		else:
			raise Exception(scode=winerror.E_NOINTERFACE)
	#
	# Other helpers.
	def ExecutePendingScripts(self):
		self.RegisterNewNamedItems()
		self.DoExecutePendingScripts()

	def ProcessScriptItemEvent(self, item, event, lcid, wFlags, args):
#		trace("ProcessScriptItemEvent", item, event, lcid, wFlags, args)
		self.RegisterNewNamedItems()
		return self.DoProcessScriptItemEvent(item, event, lcid, wFlags, args)

	def _DumpNamedItems_(self):
		for item in self.subItems.values():
			item._dump_(0)

	def ResetNamedItems(self):
		for name, item in self.subItems.items():
			item.Reset()
			if item.flags & axscript.SCRIPTITEM_ISPERSISTENT == 0:
				# remove
				del self.subItems[name]
			else:
				# keep and re-process.
				item.Register()
				
	def GetCurrentSafetyOptions(self):
		return self.safetyOptions
	def ProcessNewNamedItemsConnections(self):
		# Process all sub-items.
		for item in self.subItems.values():
			if not item.createdConnections: # Fast-track!
				item.CreateConnections()

	def RegisterNewNamedItems(self):
		# Register all sub-items.
		for item in self.subItems.values():
			if not item.isRegistered: # Fast-track!
				self.RegisterNamedItem(item)

	def RegisterNamedItem(self, item):
		item.Register()

	def CheckConnectedOrDisconnected(self):
		if self.scriptState in [axscript.SCRIPTSTATE_CONNECTED, axscript.SCRIPTSTATE_DISCONNECTED]:
			return
		RaiseAssert(winerror.E_UNEXPECTED, "Not connected or disconnected - %d" % self.scriptState)
	def Run(self):
#		trace("AXScript running...")
		if self.scriptState != axscript.SCRIPTSTATE_INITIALIZED and self.scriptState != axscript.SCRIPTSTATE_STARTED:
			raise Exception(scode=winerror.E_UNEXPECTED)
		self.ProcessNewNamedItemsConnections()
		self.RegisterNewNamedItems()
		self.ConnectEventHandlers()
#		self._DumpNamedItems_()
		self.ExecutePendingScripts()

		self.DoRun()

	def Stop(self):
		self.CheckConnectedOrDisconnected()
		self.Disconnect()
		self.ChangeScriptState(axscript.SCRIPTSTATE_INITIALIZED)
	def Disconnect(self):
		self.CheckConnectedOrDisconnected()
		self.DisconnectEventHandlers()
		self.ChangeScriptState(axscript.SCRIPTSTATE_DISCONNECTED)
	def ConnectEventHandlers(self):
#		trace ("Connecting to event handlers")
		for item in self.subItems.values():
			item.Connect()
		self.ChangeScriptState(axscript.SCRIPTSTATE_CONNECTED);
	def DisconnectEventHandlers(self):
#		trace ("Disconnecting from event handlers")
		for item in self.subItems.values():
			item.Disconnect()
	def Reset(self):
		self.ResetNamedItems()
	def ChangeScriptState(self, state):
		self.DisableInterrupts()
		try:
			self.scriptState = state
			if self.scriptSite: self.scriptSite.OnStateChange(state)
		finally:
			self.EnableInterrupts()

	def ApplyInScriptedSection(self, fn, args):
		self.BeginScriptedSection()
		try:
			try:
				return apply(fn, args)
			finally:
				self.EndScriptedSection()
		except:
			self.HandleException()
			
	def CompileInScriptedSection(self, code, name, type):
		self.BeginScriptedSection()
		try:
			try:
				return compile(code, name, type)
			finally:
				self.EndScriptedSection()
		except:
			self.HandleException()
	
	def ExecInScriptedSection(self, codeObject, globals, locals = None):
		if locals is None: locals = globals
		self.BeginScriptedSection()
		try:
			try:
				exec codeObject in globals, locals
			finally:
				self.EndScriptedSection()
		except:
			self.HandleException()

	def EvalInScriptedSection(self, codeObject, globals, locals = None):
		if locals is None: locals = globals
		self.BeginScriptedSection()
		try:
			try:
				return eval(codeObject, globals, locals)
			finally:
				self.EndScriptedSection()
		except:
			self.HandleException()

	def HandleException(self):
		if type(sys.exc_type)==types.ClassType: # re-raise.
			raise sys.exc_type, sys.exc_value
		exception = error.AXScriptException()
		exception.BuildFromException(self)
#		exception.SetContext(sourceContextCookie, startLineNumber)
		result_exception = error.ProcessAXScriptException(self.scriptSite, exception)
		if result_exception is not None:
			self.scriptSite.OnScriptTerminate(None, result_exception)
			self.SetScriptState(axscript.SCRIPTSTATE_INITIALIZED)
		
	def BeginScriptedSection(self):
		if self.scriptSite is None:
			raise Exception(E_UNEXPECTED)
		self.scriptSite.OnEnterScript()
		if self.debugManager: self.debugManager.OnEnterScript()
	def EndScriptedSection(self):
		if self.scriptSite is None:
			raise Exception(E_UNEXPECTED)
		if self.debugManager: self.debugManager.OnLeaveScript()
		self.scriptSite.OnLeaveScript()
	
	def DisableInterrupts(self):
		pass
	def EnableInterrupts(self):
		pass
	def GetNamedItem(self, name):
		try:
			return self.subItems[name]
		except KeyError:
			raise Exception(scode=winerror.E_INVALIDARG)

	def GetNamedItemClass(self):
		return ScriptItem

	def _AddScriptCodeBlock(self, codeBlock):
		self.scriptCodeBlocks[codeBlock.GetFileName()] = codeBlock
		if self.debugManager:
			self.debugManager.AddScriptBlock(codeBlock)

if __name__=='__main__':
	print "This is a framework class - please use pyscript.py etc"

def dumptypeinfo(typeinfo):
		return
		attr = typeinfo.GetTypeAttr()
		# Loop over all methods
		print "Methods"
		for j in xrange(attr[6]):
			fdesc = list(typeinfo.GetFuncDesc(j))
			id = fdesc[0]
			try:
				names = typeinfo.GetNames(id)
			except pythoncom.ole_error:
				names = None
			doc = typeinfo.GetDocumentation(id)

			print " ", names, "has attr", fdesc

		# Loop over all variables (ie, properties)
		print "Variables"
		for j in xrange(attr[7]):
			fdesc = list(typeinfo.GetVarDesc(j))
			names = typeinfo.GetNames(id)
			print " ", names, "has attr", fdesc

