#ifndef __PYTHONCOM_H__
#define __PYTHONCOM_H__

// #define _DEBUG_LIFETIMES // Trace COM object lifetimes.

#ifdef BUILD_PYTHONCOM
#define PYCOM_EXPORT __declspec(dllexport)
#else
#define PYCOM_EXPORT __declspec(dllimport)
#endif

#include <PyWinTypes.h> // Standard Win32 Types

// "Thread Safe" Python *REF macros.
// (not thread-safe yet, obviously!! - but should be used everywhere)
#define PyTS_DECREF Py_DECREF
#define PyTS_INCREF Py_INCREF
#define PyTS_XDECREF Py_XDECREF
#define PyTS_XINCREF Py_XINCREF

class PyIUnknown;
// To make life interesting/complicated, I use C++ classes for
// all Python objects.  The main advantage is that I can derive
// a PyIDispatch object from a PyIUnknown, etc.  This provides a 
// clean C++ interface, and "automatically" provides all base
// Python methods to "derived" Python types.
//
// Main disadvantage is that any extension DLLs will need to include
// these headers, and link with this .lib
// 
// Base class for (most of) the type objects.

class PYCOM_EXPORT PyComTypeObject : public PyTypeObject {
public:
	PyComTypeObject( const char *name, PyComTypeObject *pBaseType, int typeSize, struct PyMethodDef* methodList, PyIUnknown* (* thector)(IUnknown *)  );
	~PyComTypeObject();
public:
	PyComTypeObject *baseType;
	struct PyMethodDef* methods;
	PyIUnknown * (* ctor)(IUnknown *);
};

// Very very base class - not COM specific - Should exist in the
// Python core somewhere, IMO.
class PYCOM_EXPORT PyIBase : 
		public PyObject 
{
public:
	// virtuals for Python support
	virtual PyObject *getattr(char *name);
	virtual int setattr(char *name, PyObject *v);
	virtual PyObject *repr();

	static struct PyMethodDef PyIBase::empty_methods[];
protected:
	PyIBase();
	virtual ~PyIBase();

public:
	static BOOL is_object( const PyObject *, PyComTypeObject *which);
	BOOL is_object(PyComTypeObject *which);
	static void dealloc(PyObject *ob);
	static PyObject *repr(PyObject *ob);
	static PyObject *getattr(PyObject *self, char *name);
	static int setattr(PyObject *op, char *name, PyObject *v);
};

/* Special Type objects */
extern PYCOM_EXPORT PyTypeObject PyOleEmptyType;     // equivalent to VT_EMPTY
extern PYCOM_EXPORT PyTypeObject PyOleMissingType;     // special Python handling.


// Useful utility functions
// Register a client type extension.
PYCOM_EXPORT int PyCom_RegisterClientType(PyTypeObject *typeOb, const GUID *guid);

// ALL of these set an appropriate Python error on bad return.

// Given a Python object who thinks it is a PyIUnknown (or base), get
// the IUnknown.
PYCOM_EXPORT BOOL PyCom_IUnknownFromPyObject(PyObject *obUnk, IUnknown **ppunk, BOOL bNoneOK = TRUE);

// Given a Python object that is a registered COM type, return a given
// interface pointer on its underlying object, with a new reference added.
PYCOM_EXPORT BOOL PyCom_InterfaceFromPyObject(
	PyObject *ob,
	REFIID iid,
	LPVOID *ppv,
	BOOL bNoneOK=TRUE
	);

// As above, but allows instance with "_oleobj_" attribute.
PYCOM_EXPORT BOOL PyCom_InterfaceFromPyInstanceOrObject(
	PyObject *ob,
	REFIID iid,
	LPVOID *ppv,
	BOOL bNoneOK=TRUE
	);

// Given an IUnknown and an Interface ID, create and return an object
// of the appropriate type. eg IID_Unknown->PyIUnknown,
// IID_IDispatch->PyIDispatch, etc.
// Uses a map that external extension DLLs can populate with their IID/type.
// Under the prinical of least surprise, this will return Py_None is punk is NULL.
//  Otherwise, a valid PyI*, but with NULL m_obj (and therefore totally useless)
//  object is created.
// BOOL bAddRef indicates if a COM reference count should be added to the IUnknown.
//  This depends purely on the context in which it is called.  If the IUnknown is obtained
//  from a function that creates a new ref (eg, CoCreateInstance()) then you should use
//  FALSE.  If you receive the pointer as (eg) a param to a gateway function, then
//  you normally need to pass TRUE, as this is truly a new reference.
//  *** ALWAYS take the time to get this right. ***
PYCOM_EXPORT PyObject *PyCom_PyObjectFromIUnknown(IUnknown *punk, REFIID riid, BOOL bAddRef = FALSE);

// IID <-> PyObject conversion

// Given an object repring a CLSID (either PyIID or string), fill the CLSID.
#define PyCom_CLSIDFromPyObject PyWinObject_AsIID

// return a native PyIID object representing an IID
#define PyCom_PyIIDObjectFromIID PyWinObject_FromIID
// return a string object representing an IID
#define PyCom_PyStringObjectFromIID PyWinStringObject_FromIID

// VARIANT <-> PyObject conversion utilities.
PYCOM_EXPORT BOOL PyCom_VariantFromPyObject(PyObject *obj, VARIANT *var);
PYCOM_EXPORT PyObject *PyCom_PyObjectFromVariant(const VARIANT *var);

inline BOOL PyCom_MakePyObjectToVariant(PyObject *obj, VARIANT *var)
{
	return PyCom_VariantFromPyObject(obj, var);
}
inline PyObject *PyCom_MakeVariantToPyObject(const VARIANT *var)
{
	return PyCom_PyObjectFromVariant(var);
}

// Other conversion helpers...
PYCOM_EXPORT PyObject *PyCom_PyObjectFromSTATSTG(STATSTG *pStat);
PYCOM_EXPORT BOOL PyCom_PyObjectAsSTATSTG(PyObject *ob, STATSTG *pStat, DWORD flags = 0);

#define PyObjectToLARGE_INTEGER PyWinObject_AsLARGE_INTEGER
#define PyObjectToULARGE_INTEGER PyWinObject_AsULARGE_INTEGER
#define PyObjectFromLARGE_INTEGER PyWinObject_FromLARGE_INTEGER
#define PyObjectFromULARGE_INTEGER PyWinObject_FromULARGE_INTEGER

/* Functions for Initializing COM, and also letting the core know about it!
*/
PYCOM_EXPORT HRESULT PyCom_CoInitializeEx(LPVOID reserved, DWORD dwInit);
PYCOM_EXPORT HRESULT PyCom_CoInitialize(LPVOID reserved);
PYCOM_EXPORT void PyCom_CoUninitialize();

///////////////////////////////////////////////////////////////////
// Error related functions
PYCOM_EXPORT void GetScodeString(SCODE sc, char *buf, int bufSize);
PYCOM_EXPORT LPCSTR GetScodeRangeString(SCODE sc);
PYCOM_EXPORT LPCSTR GetSeverityString(SCODE sc);
PYCOM_EXPORT LPCSTR GetFacilityString(SCODE sc);
PYCOM_EXPORT PyObject *OleSetExtendedOleError(HRESULT hr, IUnknown *pUnk, REFIID iid);
PYCOM_EXPORT PyObject* OleSetOleError(SCODE sc, EXCEPINFO *einfo = NULL, UINT nArgErr = -1);
PYCOM_EXPORT PyObject* OleSetError(char *msg);
PYCOM_EXPORT PyObject* OleSetTypeError(char *msg);
PYCOM_EXPORT PyObject* OleSetMemoryError(char *doingWhat);

///////////////////////////////////////////////////////////////////
//
// External C++ helpers - these helpers are for other DLLs which
// may need similar functionality, but dont want to duplicate all

// This helper is for an application that has an IDispatch, and COM arguments
// and wants to call a Python function.  It is assumed the caller can map the IDispatch
// to a Python object, so the Python handler is passed.
// Args:
//   handler : A Python callable object.
//   dispparms : the COM arguments.
//   pVarResult : The variant for the return value of the Python call.
//   pexcepinfo : Exception info the helper may fill out.
//   puArgErr : Argument error the helper may fill out on exception
//   addnArgs : Any additional arguments to the Python function.  May be NULL.
// If addnArgs is NULL, then it is assumed the Python call should be native -
// ie, the COM args are packed as normal Python args to the call.
// If addnArgs is NOT NULL, it is assumed the Python function itself is
// a helper.  This Python function will be called with 2 arguments - both
// tuples - first one is the COM args, second is the addn args.
PYCOM_EXPORT BOOL PyCom_MakeOlePythonCall(PyObject *handler, DISPPARAMS FAR* params, VARIANT FAR* pVarResult,
	EXCEPINFO FAR* pexcepinfo, UINT FAR* puArgErr, PyObject *addnlArgs);

// If a Python exception occurrs call this to do get an HRESULT and to
// optionally fill in an EXCEPINFO structure. If no Python exception has
// occurred, then S_OK is returned (and the EXCEPINFO is zeroed).
// Otherwise, the function will look at the Python exception to try to
// find an HRESULT and to try to fill in the EXCEPINFO.
// In debug builds, this will call "traceback.print_exc()"
PYCOM_EXPORT HRESULT PyCom_HandlePythonFailureToCOM(EXCEPINFO *pExcepInfo = NULL);

// Create a Python object holding the exception information.  The exception
// information is *not* freed by this function.  Python exceptions are
// raised and NULL is returned if an error occurs.
PYCOM_EXPORT PyObject *PyCom_PyObjectFromExcepInfo(const EXCEPINFO *pexcepInfo);

// Fill in an EXCEPINFO structure from a Python instance or tuple object.
PYCOM_EXPORT BOOL PyCom_ExcepInfoFromPyObject(PyObject *obExcepInfo, EXCEPINFO *pexcepInfo);

// Used by sophisticated gateways to SetErrorInfo() full error information.
PYCOM_EXPORT HRESULT PyCom_SetFromExcepInfo(const EXCEPINFO *pexcepinfo, REFIID riid = IID_NULL);

// Used in gateways to SetErrorInfo() with a simple HRESULT, then return it.
PYCOM_EXPORT HRESULT PyCom_SetFromSimple(HRESULT hr, REFIID riid = IID_NULL);

// Used in gateways to SetErrorInfo() the current Python exception
PYCOM_EXPORT HRESULT PyCom_SetFromPyException(REFIID riid = IID_NULL);

/////////////////////////////////////////////////////////////////////////////
// class PyOleEmpty
class PYCOM_EXPORT PyOleEmpty : public PyObject
{
public:
	PyOleEmpty();
};

class PYCOM_EXPORT PyOleMissing : public PyObject
{
public:
	PyOleMissing();
};

// We need to dynamically create C++ Python objects
// These helpers allow each type object to create it.
#define MAKE_PYCOM_CTOR(classname) static PyIUnknown * classname::PyObConstruct(IUnknown *pInitObj) {return new classname(pInitObj);}
#define MAKE_PYCOM_CTOR_ERRORINFO(classname, iid) \
         static PyIUnknown * classname::PyObConstruct(IUnknown *pInitObj) {return new classname(pInitObj);} \
		 static PyObject *SetPythonCOMError(PyObject *self, HRESULT hr) {return OleSetExtendedOleError(hr, GetI(self), iid);}
#define GET_PYCOM_CTOR(classname) classname::PyObConstruct

/////////////////////////////////////////////////////////////////////////////
// class PyIUnknown
class PYCOM_EXPORT PyIUnknown : public PyIBase
{
public:
	MAKE_PYCOM_CTOR(PyIUnknown);
	virtual PyObject *repr();

	static IUnknown *GetI(PyObject *self);
	IUnknown *m_obj;
	static char *szErrMsgObjectReleased;
	static void SafeRelease(PyIUnknown *ob);
	static PyComTypeObject type;

	// The Python methods
	static PyObject *QueryInterface(PyObject *self, PyObject *args);
	static PyObject *SafeRelease(PyObject *self, PyObject *args);

protected:
	PyIUnknown(IUnknown *punk);
	~PyIUnknown();
};


/////////////////////////////////////////////////////////////////////////////
// class PyIDispatch

class PYCOM_EXPORT PyIDispatch : public PyIUnknown
{
public:
	MAKE_PYCOM_CTOR(PyIDispatch);
	static IDispatch *GetI(PyObject *self);
	static PyComTypeObject type;

	// The Python methods
	static PyObject *Invoke(PyObject *self, PyObject *args);
	static PyObject *InvokeTypes(PyObject *self, PyObject *args);
	static PyObject *GetIDsOfNames(PyObject *self, PyObject *args);
	static PyObject *GetTypeInfo(PyObject *self, PyObject *args);
	static PyObject *GetTypeInfoCount(PyObject *self, PyObject *args);
protected:
	PyIDispatch(IUnknown *pdisp);
	~PyIDispatch();
};

/////////////////////////////////////////////////////////////////////////////
// class PyIProvideTypeInfo

class PYCOM_EXPORT PyIProvideClassInfo : public PyIUnknown
{
public:
	MAKE_PYCOM_CTOR(PyIProvideClassInfo);
	static IProvideClassInfo *GetI(PyObject *self);
	static PyComTypeObject type;

	// The Python methods
	static PyObject *GetClassInfo(PyObject *self, PyObject *args);
protected:
	PyIProvideClassInfo(IUnknown *pdisp);
	~PyIProvideClassInfo();
};

class PYCOM_EXPORT PyIProvideClassInfo2 : public PyIProvideClassInfo
{
public:
	MAKE_PYCOM_CTOR(PyIProvideClassInfo2);
	static IProvideClassInfo2 *GetI(PyObject *self);
	static PyComTypeObject type;

	// The Python methods
	static PyObject *GetGUID(PyObject *self, PyObject *args);
protected:
	PyIProvideClassInfo2(IUnknown *pdisp);
	~PyIProvideClassInfo2();
};

/////////////////////////////////////////////////////////////////////////////
// class PyITypeInfo
class PYCOM_EXPORT PyITypeInfo : public PyIUnknown
{
public:
	MAKE_PYCOM_CTOR(PyITypeInfo);
	static PyComTypeObject type;
	static ITypeInfo *GetI(PyObject *self);
	
	PyObject *GetContainingTypeLib();
	PyObject *GetDocumentation(MEMBERID);
	PyObject *GetRefTypeInfo(HREFTYPE href);
	PyObject *GetRefTypeOfImplType(int index);
	PyObject *GetFuncDesc(int pos);
	PyObject *GetIDsOfNames(OLECHAR FAR* FAR*, int);
	PyObject *GetNames(MEMBERID);
	PyObject *GetTypeAttr();
	PyObject *GetVarDesc(int pos);
	PyObject *GetImplTypeFlags(int index);

protected:
	PyITypeInfo(IUnknown *);
	~PyITypeInfo();
};


/////////////////////////////////////////////////////////////////////////////
// class CPyTypeLib

class PYCOM_EXPORT PyITypeLib : public PyIUnknown
{
public:
	MAKE_PYCOM_CTOR(PyITypeLib);
	static PyComTypeObject type;
	static ITypeLib *GetI(PyObject *self);

	PyObject *GetLibAttr();
	PyObject *GetDocumentation(int pos);
	PyObject *GetTypeInfo(int pos);
	PyObject *GetTypeInfoCount();
	PyObject *GetTypeInfoType(int pos);

protected:
	PyITypeLib(IUnknown *);
	~PyITypeLib();
};

/////////////////////////////////////////////////////////////////////////////
// class PyIConnectionPoint

class PYCOM_EXPORT PyIConnectionPoint : public PyIUnknown
{
public:
	MAKE_PYCOM_CTOR(PyIConnectionPoint);
	static PyComTypeObject type;
	static IConnectionPoint *GetI(PyObject *self);

	static PyObject *GetConnectionInterface(PyObject *self, PyObject *args);
	static PyObject *GetConnectionPointContainer(PyObject *self, PyObject *args);
	static PyObject *Advise(PyObject *self, PyObject *args);
	static PyObject *Unadvise(PyObject *self, PyObject *args);
	static PyObject *EnumConnections(PyObject *self, PyObject *args);

protected:
	PyIConnectionPoint(IUnknown *);
	~PyIConnectionPoint();
};

class PYCOM_EXPORT PyIConnectionPointContainer : public PyIUnknown
{
public:
	MAKE_PYCOM_CTOR(PyIConnectionPointContainer);
	static PyComTypeObject type;
	static IConnectionPointContainer *GetI(PyObject *self);

	static PyObject *EnumConnectionPoints(PyObject *self, PyObject *args);
	static PyObject *FindConnectionPoint(PyObject *self, PyObject *args);

protected:
	PyIConnectionPointContainer(IUnknown *);
	~PyIConnectionPointContainer();
};


/////////////////////////////////////////////////////////////////////////////
// class PythonOleArgHelper
//
// A PythonOleArgHelper is used primarily to help out Python helpers
// which need to convert from a Python object when the specific OLE 
// type is known - eg, when a TypeInfo is available.
//
class PYCOM_EXPORT PythonOleArgHelper
{
public:
	PythonOleArgHelper();
	~PythonOleArgHelper();
	BOOL ParseTypeInformation(PyObject *reqdObjectTuple);

	// Using this call with reqdObject != NULL will check the existing 
	// VT_ of the variant.  If not VT_EMPTY, then the result will be coerced to
	// that type.  This contrasts with PyCom_PyObjectToVariant which just
	// uses the Python type to determine the variant type.
	BOOL MakeObjToVariant(PyObject *obj, VARIANT *var, PyObject *reqdObjectTuple = NULL);
	PyObject *MakeVariantToObj(VARIANT *var);

	VARTYPE m_reqdType;
	BOOL m_bParsedTypeInfo;
	BOOL m_bByRef;
	union {
		void *m_pValueHolder;
		short m_sBuf;
		long m_lBuf;
		VARIANT_BOOL m_boolBuf;
		double m_dBuf;
		float m_fBuf;
		IDispatch *m_dispBuf;
		IUnknown *m_unkBuf;
		SAFEARRAY *m_arrayBuf;
	};
};


/////////////////////////////////////////////////////////////////////////////
// global functions and variables
PYCOM_EXPORT BOOL MakePythonArgumentTuples(PyObject **pArgs, PythonOleArgHelper **ppHelpers, 
                     PyObject **pNamedArgs, PythonOleArgHelper **ppNamedHelpers,
                     DISPPARAMS FAR* params);


// MakeBstr - convert a const char * into a BSTR.
inline BSTR MakeBstr(const char *s)
{
	USES_CONVERSION;
	return SysAllocString(T2COLE(s));	// okay if s == NULL
}

// Convert a Python string object to a BSTR - allow embedded NULLs, etc.
PYCOM_EXPORT BSTR PyCom_BstrFromPyString(PyObject *stringObject);

// Convert a Python object to a BSTR - allow embedded NULLs, None, etc.
PYCOM_EXPORT BOOL PyCom_BstrFromPyObject(PyObject *stringObject, BSTR *pResult, BOOL bNoneOK = FALSE);

// MakeBstrToObj - convert a BSTR into a Python string.
//
// ONLY USE THIS FOR TRUE BSTR's - Use the fn below for OLECHAR *'s.
// NOTE - does not use standard macros, so NULLs get through!
PYCOM_EXPORT PyObject *MakeBstrToObj(const BSTR bstr);

// Size info is available (eg, a fn returns a string and also fills in a size variable)
PYCOM_EXPORT PyObject *MakeOLECHARToObj(const OLECHAR * str, int numChars);

// No size info avail.
PYCOM_EXPORT PyObject *MakeOLECHARToObj(const OLECHAR * str);

#endif // __PYTHONCOM_H__
