// PythonCOM.cpp : Implementation of DLL Exports.

#include "stdafx.h"

#include <windows.h>

#include <Python.h>
#include <PythonRun.h>	/* for Py_Initialize() */
#include <Import.h>		/* for PyImport_GetModuleDict() */

#include "PythonCOM.h"
#include "PythonCOMServer.h"
#include "PyFactory.h"

// #define LOG_USE_EVENT_LOG

void LogEvent(LPCSTR pszMessageText)
{
    HANDLE  hEventSource;

	hEventSource = RegisterEventSource(NULL, "PythonCOM");
    if (NULL == hEventSource)
		return;

    BOOL nRetVal = ReportEvent(
        hEventSource,
        EVENTLOG_INFORMATION_TYPE,
        0,				/* category */
        0x40000001ul,	/* event id */
        NULL,			/* security identifier */
        1,				/* string count */
        0,				/* length of binary data */
        &pszMessageText,
        NULL			/* binary data */
    );

	DeregisterEventSource(hEventSource);
}

void LogMessage(LPCSTR pszMessageText)
{
#ifndef LOG_USE_EVENT_LOG
	OutputDebugString(pszMessageText);
	OutputDebugString("\n");
#else
	LogEvent(pszMessageText);
#endif
}

void VLogF(const char *fmt, va_list argptr)
{
	char buff[512];

	wvsprintf(buff, fmt, argptr);

	LogMessage(buff);
}

PYCOM_EXPORT void PyCom_LogF(const char *fmt, ...)
{
	va_list marker;

	va_start(marker, fmt);
	VLogF(fmt, marker);
}
#define LogF PyCom_LogF

/*
** This value counts the number of references to objects that contain code
** within this DLL.  The DLL cannot be unloaded until this reaches 0.
**
** Additional locks (such as via the LockServer method on a CPyFactory) can
** add a "pseudo-reference" to ensure the DLL is not tossed.
*/
static LONG g_cLockCount = 0;

/*
** A waitable event that the COM framework will signal when the lock count hits
** Zero.  This is considered a "private feature", and is used by a EXE that can
** host Python COM objects.  Therefore, this functionality is exported from the DLL, 
but not declared in any headers
*/
static HANDLE hLockCountEvent = 0;
PYCOM_EXPORT void PyCom_SetLockCountWaitEvent( HANDLE hEvent )
{
	hLockCountEvent = hEvent;
}

void PyCom_DLLAddRef(void)
{
	InterlockedIncrement(&g_cLockCount);
}
void PyCom_DLLReleaseRef(void)
{
	InterlockedDecrement(&g_cLockCount);
	// Not thread safe, but Im not sure we _can_ - g_cLockCount
	// could always transition 1->0->1 at some stage, screwing this
	// up.  Oh well...
	if (hLockCountEvent && g_cLockCount==0)
		SetEvent(hLockCountEvent);
}

/////////////////////////////////////////////////////////////////////////////
// DLL Entry Point

static DWORD g_dwCoInitThread = 0;
static BOOL g_bCoInitThreadHasInit = FALSE;
static BOOL g_bPyInited = FALSE;
static BOOL g_bRegisteredInfo = FALSE;

/* declare this outside of DllMain which has "C" scoping */
extern void FreeGatewayModule(void);
extern int PyCom_RegisterCoreSupport(void);
extern int PyCom_UnregisterCoreSupport(void);

extern "C"
BOOL WINAPI DllMain(HINSTANCE hInstance, DWORD dwReason, LPVOID lpReserved)
{
	if ( dwReason == DLL_PROCESS_ATTACH )
	{
//		LogEvent("Loaded pythoncom.dll");

		/*
		** NOTE: we assume that our globals are not shared among processes,
		**       so for all intents and purposes, we can only hit this body
		**       of code "once" (from the standpoint of what our variables
		**       tell us).
		*/

		/*
		** One of two situations is now occurring:
		**
		**   1) Python is loading this DLL as part of its standard import
		**      mechanism.  Python has been initialized already and will
		**      eventually call initwin32com().
		**
		**      We want to call OleInitialize() to set up the OLE environment
		**      because we don't know if anything else has yet.  Note that
		**      OleInitialize() can be called many times, as long as it is
		**      balanced by an OleUninitialize().
		**
		**   2) The OLE system is loading this DLL to serve out a particular
		**      object.  Python may or may not be initialized (it will be
		**      initialized if the client of the object is in this process
		**      and the Python interpreter is running in that process).
		**      initwin32com() will not immediately be called, but it is
		**      possible that it will be called later (if this process has
		**      Python in it and it wants the win32com module).
		**
		**      If Python has not been initialized, then we want to get it
		**      initialized.  We should count the number of times that
		**      initialize is called and then clean up the interpreter on
		**      the last detach (note that if Python is already initialized
		**      on entry, then we never want to clean up).
		**
		**      (actually, we don't have to count because we assume that we
		**      won't be called multiple times per process)
		*/

		/* Note we no longer initialize OLE here - this needed to
		** get a bit smarter - see the PyCom_OleInitialize() etc parms
		*/

		/*
		** determine if Python needs to be initialized or not
		**
		** ### this is a hack, but Python does not export its "inited" flag
		**
		** ### THIS SHOULD NEVER HAPPEN.
		** ### when the python core DLL loads, it initializes itself.
		*/
		PyObject *d = PyImport_GetModuleDict();
		if ( d == NULL )
		{
			Py_Initialize();
			g_bPyInited = TRUE;
		}

		AllocThreadState();
		{
			if ( PyCom_RegisterCoreSupport() == -1 )
			{
#ifdef _DEBUG
				LogMessage("could not register information!");
#endif
				FreeThreadState();
				return FALSE;
			}
		}
		FreeThreadState();
		g_bRegisteredInfo = TRUE;

		/*
		** we don't need to be notified about threads
		*/
		DisableThreadLibraryCalls(hInstance);
	}
	else if ( dwReason == DLL_PROCESS_DETACH )
	{
//		LogEvent("Terminated pythoncom.dll");

		AllocThreadState();
		{
			/* free the gateway module if loaded (see PythonCOMObj.cpp) */
			FreeGatewayModule();

			if ( g_bRegisteredInfo )
			{
				(void)PyCom_UnregisterCoreSupport();
				g_bRegisteredInfo = FALSE;
			}

			/*
			** ### THIS SHOULD NEVER HAPPEN.
			** ### when the python core DLL loads, it initializes itself,
			** ### meaning that we wouldn't have init'd it.
			*/
			if ( g_bPyInited )
			{
				Py_Cleanup();
				PyImport_Cleanup();
				g_bPyInited = FALSE;
			}
		}
		FreeThreadState();	// ### I know this can be called after Cleanup

		PyCom_CoUninitialize();
	}

	return TRUE;    // ok
}

HRESULT PyCom_CoInitializeEx(LPVOID reserved, DWORD dwInit)
{
	CEnterLeavePython celp;
	if (g_bCoInitThreadHasInit && g_dwCoInitThread == GetCurrentThreadId())
		return S_OK;
	// Do a LoadLibrary, as the Ex version may not always exist
	// on Win95.
	HMODULE hMod = GetModuleHandle("ole32.dll");
	if (hMod==0) return E_HANDLE;
	FARPROC fp = GetProcAddress(hMod, "CoInitializeEx");
	if (fp==NULL) return E_NOTIMPL;

	HRESULT (*mypfn)(void *pvReserved, DWORD dwCoInit);
	mypfn = (HRESULT (*)(void *pvReserved, DWORD dwCoInit))fp;

	HRESULT hr = (*mypfn)(reserved, dwInit);

	if ( (hr != RPC_E_CHANGED_MODE) && FAILED(hr) )
	{
#ifdef _DEBUG
		LogF("OLE initialization failed! (0x%08lx)", hr);
#endif
		return hr;
	}
	// If we have never been initialized before, then consider this
	// thread our "main initializer" thread.
	if (g_dwCoInitThread==0 && hr == S_OK) {
		g_dwCoInitThread = GetCurrentThreadId();
		g_bCoInitThreadHasInit = TRUE;
	}
	return hr;
}

HRESULT PyCom_CoInitialize(LPVOID reserved)
{
	CEnterLeavePython celp;
	// If our "main" thread has ever called this before, just
	// ignore it.  If it is another thread, then that thread
	// must manage itself.
	if (g_bCoInitThreadHasInit && g_dwCoInitThread == GetCurrentThreadId())
		return S_OK;
	HRESULT hr = CoInitialize(reserved);
	if ( (hr != RPC_E_CHANGED_MODE) && FAILED(hr) )
	{
#ifdef _DEBUG
		LogF("OLE initialization failed! (0x%08lx)", hr);
#endif
		return hr;
	}
	// If we have never been initialized before, then consider this
	// thread our "main initializer" thread.
	if (g_dwCoInitThread==0 && hr == S_OK) {
		g_dwCoInitThread = GetCurrentThreadId();
		g_bCoInitThreadHasInit = TRUE;
	}
	return hr;
}

void PyCom_CoUninitialize()
{
	CEnterLeavePython celp;
	if (g_dwCoInitThread == GetCurrentThreadId()) {
		// being asked to terminate on our "main" thread
		// Check our flag, but always consider it success.
		if (g_bCoInitThreadHasInit) {
			CoUninitialize();
			g_bCoInitThreadHasInit = FALSE;
		}
	} else {
		// Not our thread - assume caller knows what they are doing
		CoUninitialize();
	}
}

/////////////////////////////////////////////////////////////////////////////
// Used to determine whether the DLL can be unloaded by OLE

STDAPI DllCanUnloadNow(void)
{
	return g_cLockCount ? S_FALSE : S_OK;
}

/////////////////////////////////////////////////////////////////////////////
// Returns a class factory to create an object of the requested type

STDAPI DllGetClassObject(REFCLSID rclsid, REFIID riid, LPVOID* ppv)
{
	//LogMessage("in DllGetClassObject\n");

	if ( ppv == NULL )
		return E_INVALIDARG;
	if ( !IsEqualIID(riid, IID_IUnknown) &&
		 !IsEqualIID(riid, IID_IClassFactory) )
		return E_INVALIDARG;

	// ### validate that we support rclsid?

	/* Put the factory right into *ppv; we know it supports <riid> */
	*ppv = (LPVOID*) new CPyFactory(rclsid);
	if ( *ppv == NULL )
		return E_OUTOFMEMORY;

	return S_OK;
}
