// PyIDispatch

// @doc
#include "stdafx.h"
#include "PythonCOM.h"

static BOOL HandledDispatchFailure(HRESULT hr, EXCEPINFO *einfo, UINT nArgErr, UINT cArgs)
{
	if ( hr == DISP_E_EXCEPTION )
	{
		if ( einfo->scode != DISP_E_TYPEMISMATCH &&
			 einfo->scode != DISP_E_PARAMNOTFOUND )
			nArgErr = -1;
		else
			nArgErr = cArgs - nArgErr;	/* convert to usable index */
		OleSetOleError(hr, einfo, nArgErr);
		return TRUE;
	}

	if ( FAILED(hr) )
	{
		if ( hr != DISP_E_TYPEMISMATCH && hr != DISP_E_PARAMNOTFOUND )
			nArgErr = -1;
		else
			nArgErr = cArgs - nArgErr;	/* convert to usable index */
		OleSetOleError(hr, NULL, nArgErr);
		return TRUE;
	}
	return FALSE;
}

PyIDispatch::PyIDispatch(IUnknown *pDisp) :
	PyIUnknown(pDisp)
{
	ob_type = &type;
}

PyIDispatch::~PyIDispatch()
{
}

/*static*/ IDispatch *PyIDispatch::GetI(PyObject *self)
{
	return (IDispatch *)PyIUnknown::GetI(self);
}

// @pymethod (tuple of ints)|PyIDispatch|GetIDsOfNames|Get the DISPID for the passed names.
PyObject *PyIDispatch::GetIDsOfNames(PyObject *self, PyObject *args)
{
	UINT i;

	// @pyparm string|name||A name to query for
	// @pyparmalt1 [string, ...]|[name, ...]||A list of string names to query
	int argc = PyObject_Length(args);
	if ( argc == -1 )
		return NULL;
	if ( argc < 1 )
		return OleSetTypeError("At least one argument must be supplied");

	LCID lcid = LOCALE_SYSTEM_DEFAULT;
	UINT offset = 0;
	if ( argc > 1 )
	{
		PyObject *ob = PySequence_GetItem(args, 0);
		if ( !ob )
			return NULL;
		if ( PyInt_Check(ob) )
		{
			lcid = PyInt_AS_LONG((PyIntObject *)ob);
			offset = 1;
		}
		Py_DECREF(ob);
	}

	UINT cNames = argc - offset;
	OLECHAR FAR* FAR* rgszNames = new LPOLESTR[cNames];

	USES_CONVERSION;
	for ( i = 0 ; i < cNames; ++i )
	{
		PyObject *ob = PySequence_GetItem(args, i + offset);
		if ( !ob )
		{
			delete [] rgszNames;
			return NULL;
		}
		if ( !PyString_Check(ob) )
		{
			Py_DECREF(ob);
			OleSetTypeError("Names must be strings");
			delete [] rgszNames;
			return NULL;
		}
		char *s = PyString_AS_STRING((PyStringObject *)ob);
		rgszNames[i] = A2OLE(s);
		Py_DECREF(ob);
	}

	DISPID FAR* rgdispid = new DISPID[cNames];
	IDispatch *pMyDispatch = GetI(self);
	if (pMyDispatch==NULL) return NULL;
	HRESULT hr = pMyDispatch->GetIDsOfNames(IID_NULL, rgszNames, cNames, lcid, rgdispid);

	delete [] rgszNames;

	if ( FAILED(hr) )
		return OleSetOleError(hr);

	PyObject *result;

	/* if we have just one name, then return a single DISPID (int) */
	if ( cNames == 1 )
	{
		result = PyInt_FromLong(rgdispid[0]);
	}
	else
	{
		result = PyTuple_New(cNames);
		if ( result )
		{
			for ( i = 0; i < cNames; ++i )
			{
				PyObject *ob = PyInt_FromLong(rgdispid[i]);
				if ( !ob )
				{
					delete [] rgdispid;
					return NULL;
				}
				PyTuple_SET_ITEM(result, i, ob);
			}
		}
	}

	delete [] rgdispid;
	return result;
}

// @pymethod int|PyIDispatch|Invoke|Invokes a DISPID, using the passed arguments.
PyObject * PyIDispatch::Invoke(PyObject *self, PyObject *args)
{
	/* Invoke(dispid, lcid, wflags, bResultWanted, arg1, arg2...) */
	PyErr_Clear();
	int argc = PyObject_Length(args);
	if ( argc == -1 )
		return NULL;
	if ( argc < 4 )
		return OleSetTypeError("not enough arguments (at least 4 needed)");

	DISPID dispid = PyInt_AsLong(PyTuple_GET_ITEM(args, 0));
	LCID lcid = PyInt_AsLong(PyTuple_GET_ITEM(args, 1));
	UINT wFlags = PyInt_AsLong(PyTuple_GET_ITEM(args, 2));
	BOOL bResultWanted = (BOOL)PyInt_AsLong(PyTuple_GET_ITEM(args, 3));
	if ( PyErr_Occurred() )
		return NULL;

	IDispatch *pMyDispatch = GetI(self);
	if ( pMyDispatch==NULL )
		return NULL;

	DISPID dispidNamed = DISPID_PROPERTYPUT;
	DISPPARAMS dispparams = { NULL, NULL, argc - 4, 0 };
	if ( dispparams.cArgs )
	{
		dispparams.rgvarg = new VARIANTARG[dispparams.cArgs];
		for ( UINT i = 0; i < dispparams.cArgs; ++i )
		{
			VariantInit(&dispparams.rgvarg[i]);
			// args in reverse order.
			if ( !PyCom_VariantFromPyObject(PyTuple_GET_ITEM(args, argc-i-1), &dispparams.rgvarg[i]) )
			{
				if ( !PyErr_Occurred() )
					OleSetTypeError("Bad argument");
				while ( i-- > 0 )
					VariantClear(&dispparams.rgvarg[i]);
				return NULL;
			}
		}

		/* puts and putrefs need a named argument */
		if ( wFlags & (DISPATCH_PROPERTYPUT | DISPATCH_PROPERTYPUTREF) )
		{
			dispparams.rgdispidNamedArgs = &dispidNamed;
			dispparams.cNamedArgs = 1;
		}
	}

	VARIANT varResult;
	VARIANT *pVarResultUse;
	if ( bResultWanted ) {
		VariantInit(&varResult);
		pVarResultUse = &varResult;
	} else
		pVarResultUse = NULL;

	// initialize EXCEPINFO struct
	EXCEPINFO excepInfo;
	memset(&excepInfo, 0, sizeof excepInfo);

	UINT nArgErr = (UINT)-1;  // initialize to invalid arg
	HRESULT hr = pMyDispatch->Invoke(dispid, IID_NULL, lcid, wFlags, &dispparams, pVarResultUse, &excepInfo, &nArgErr);
	if ( dispparams.rgvarg )
	{
		for ( UINT i = dispparams.cArgs; i--; )
			VariantClear(&dispparams.rgvarg[i]);
		delete [] dispparams.rgvarg;
	}

	if ( HandledDispatchFailure(hr, &excepInfo, nArgErr, dispparams.cArgs) )
	{
		if ( pVarResultUse )
			VariantClear(pVarResultUse);
		return NULL;
	}

	PyObject *result;
	if (pVarResultUse) {
		result = PyCom_PyObjectFromVariant(pVarResultUse);
		VariantClear(pVarResultUse);
	} else {
		result = Py_None;
		PyTS_INCREF(result);
	}
	return result;
}

// @pymethod int|PyIDispatch|InvokeTypes|Invokes a DISPID, using the passed arguments and type descriptions.
PyObject * PyIDispatch::InvokeTypes(PyObject *self, PyObject *args)
{
	/* InvokeType(dispid, lcid, wflags, ELEMDESC resultType, ELEMDESC[] argTypes, arg1, arg2...) */
	PyErr_Clear();
	int argc = PyObject_Length(args);
	if ( argc == -1 )
		return NULL;
	if ( argc < 5 )
		return OleSetTypeError("not enough arguments (at least 5 needed)");

	DISPID dispid = PyInt_AsLong(PyTuple_GET_ITEM(args, 0));
	LCID lcid = PyInt_AsLong(PyTuple_GET_ITEM(args, 1));
	UINT wFlags = PyInt_AsLong(PyTuple_GET_ITEM(args, 2));
	PyObject *resultElemDesc = PyTuple_GET_ITEM(args, 3);
	PyObject *argsElemDescArray = PyTuple_GET_ITEM(args, 4);
	if ( PyErr_Occurred() )
		return NULL;
	int numArgs;
	int argTypesLen = PyObject_Length(argsElemDescArray);
	if (!PyTuple_Check(argsElemDescArray) || argTypesLen<argc-5)
		return OleSetTypeError("The array of argument types must be a tuple whose size is <= to the number of arguments.");
	// See how many _real_ entries - count until end or
	// first param marked as Missing.
	for (numArgs = 0;numArgs<argc-5; numArgs++) {
		if (PyTuple_GET_ITEM(args, numArgs+5)->ob_type==&PyOleMissingType) {
			break;
		}
	}

	// these will all be cleared before returning
	PythonOleArgHelper *ArgHelpers = NULL;
	PyObject *result = NULL;
	DISPID dispidNamed = DISPID_PROPERTYPUT;
	DISPPARAMS dispparams = { NULL, NULL, 0, 0 };
	PythonOleArgHelper resultArgHelper;

	// This gets confusing.  If we have typeinfo for a byref arg, but the
	// arg is not specified by the user, then we _do_ present the arg to
	// COM.  If the arg does not exist, and it is not byref, then we do
	// not present it _at_all - ie, the arg count does _not_ include it.
	// So - first we must loop over the arg types, using this info to 
	// decide how big the arg array is!

	// If we have type info for an arg but not specified by the user, we will still process
	// the arg fully.
	// Note numArgs can not be > argTypesLen (as checked above)
	UINT numArgArray = 0;
	UINT i;

	if (argTypesLen>0) {
		ArgHelpers = new PythonOleArgHelper[argTypesLen]; // new may! except.
		if (ArgHelpers==NULL) return OleSetMemoryError("Allocating ArgHelpers array");
		for ( i = 0; i < (UINT)argTypesLen; i++ ) {
			if (!ArgHelpers[i].ParseTypeInformation(PyTuple_GET_ITEM(argsElemDescArray,i)))
				goto error;
			if (i<(UINT)numArgs || ArgHelpers[i].m_bByRef)
				numArgArray = i+1;
		}
	}

	dispparams.cArgs = numArgArray;
	if ( dispparams.cArgs ) {
		dispparams.rgvarg = new VARIANTARG[dispparams.cArgs];
		if (dispparams.rgvarg==NULL) {
			OleSetMemoryError("Allocating dispparams.rgvarg array");
			goto error;
		}

		for ( i = dispparams.cArgs; i--; )
			VariantInit(&dispparams.rgvarg[i]);

		for ( i = 0; i < dispparams.cArgs; ++i ) {
			// args in reverse order.
			// arg-helpers in normal order.
			UINT offset = dispparams.cArgs - i - 1;
			// See if the user actually specified this arg.
			PyObject *arg = i>=(UINT)numArgs ? Py_None : PyTuple_GET_ITEM(args, i+5);
			if ( !ArgHelpers[i].MakeObjToVariant(arg, &dispparams.rgvarg[offset], PyTuple_GET_ITEM(argsElemDescArray,i)) )
				goto error;
		}
	}

	/* puts and putrefs need a named argument */
	if ( wFlags & (DISPATCH_PROPERTYPUT | DISPATCH_PROPERTYPUTREF) )
	{
		dispparams.rgdispidNamedArgs = &dispidNamed;
		dispparams.cNamedArgs = 1;
	}

	if (!resultArgHelper.ParseTypeInformation(resultElemDesc)) {
		OleSetError("The return type information could not be parsed");
		goto error;
	}

	BOOL bResultWanted;
	bResultWanted = (resultArgHelper.m_reqdType != VT_VOID && resultArgHelper.m_reqdType != VT_EMPTY);

	VARIANT varResult;
	VARIANT *pVarResultUse;
	if ( bResultWanted ) {
		VariantInit(&varResult);
		pVarResultUse = &varResult;
	} else
		pVarResultUse = NULL;

	// initialize EXCEPINFO struct
	EXCEPINFO excepInfo;
	memset(&excepInfo, 0, sizeof excepInfo);

	HRESULT hr;
	UINT nArgErr;
	IDispatch *pMyDispatch;

	pMyDispatch = GetI(self);
	if (pMyDispatch==NULL) goto error;
	nArgErr = (UINT)-1;  // initialize to invalid arg
	hr = pMyDispatch->Invoke(dispid, IID_NULL, lcid, wFlags, &dispparams, pVarResultUse, &excepInfo, &nArgErr);

	if ( !HandledDispatchFailure(hr, &excepInfo, nArgErr, dispparams.cArgs) )
	{
		// Now get fancy with the args.  Any args specified as BYREF get returned
		// to Python.
		int retSize = 0;
		if (pVarResultUse)
			retSize++;
		for (UINT arg=0;arg<numArgArray;arg++)
			if (ArgHelpers[arg].m_bByRef)
				retSize++;
		if (retSize==0) {  // result is None.
			result = Py_None;
			PyTS_INCREF(result);
		} else if (retSize==1) { // result is a simple object.
			if (pVarResultUse) { // only retval is actual result.
				result = resultArgHelper.MakeVariantToObj(pVarResultUse);
			} else { // only result in one of the params - seek it.
				for (UINT arg=0;arg<numArgArray;arg++) {
					if (ArgHelpers[arg].m_bByRef) {
						result = ArgHelpers[arg].MakeVariantToObj(dispparams.rgvarg+(numArgArray-arg-1));
						break;
					}
				}
			}
		} else { // result is a tuple.
			result = PyTuple_New(retSize);
			int tupleItem = 0;
			if (pVarResultUse) {
				PyTuple_SetItem(result, tupleItem++, resultArgHelper.MakeVariantToObj(pVarResultUse));
			}
			// Loop over all the args, reverse order, setting the byrefs.
			for (int arg=numArgArray-1;arg>=0;arg--)
				if (ArgHelpers[numArgArray-arg-1].m_bByRef)
					PyTuple_SetItem(result, tupleItem++, ArgHelpers[numArgArray-arg-1].MakeVariantToObj(dispparams.rgvarg+(arg)));
		}
	}
	if (pVarResultUse) VariantClear(pVarResultUse); // wipe the result.

  error:
	if ( dispparams.rgvarg )
	{
		for ( i = dispparams.cArgs; i--; )
			VariantClear(&dispparams.rgvarg[i]);
		delete [] dispparams.rgvarg;
	}
	delete [] ArgHelpers;
	return result;
}

// @pymethod <o PyITypeInfo>|PyIDispatch|GetTypeInfo|Get type information for the object.
PyObject *PyIDispatch::GetTypeInfo(PyObject *self, PyObject *args)
{
	LCID locale = LOCALE_USER_DEFAULT;
	// @pyparm int|locale|LOCALE_USER_DEFAULT|The locale to use.
	if (!PyArg_ParseTuple(args, "|i:GetTypeInfo", &locale))
		return NULL;

	IDispatch *pMyDispatch = GetI(self);
	if (pMyDispatch==NULL) return NULL;
	ITypeInfo *pti = NULL;
	SCODE sc = pMyDispatch->GetTypeInfo(0, locale, &pti);
	if (sc!=S_OK) // S_OK is only acceptable result.
		return OleSetOleError(sc);
	return PyCom_PyObjectFromIUnknown(pti, IID_ITypeInfo);
}

// @pymethod int|PyIDispatch|GetTypeInfoCount|Retrieves the number of interfaces the object provides.
PyObject *PyIDispatch::GetTypeInfoCount(PyObject *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args,":GetTypeInfoCount"))
		return NULL;
	unsigned int ret;

	IDispatch *pMyDispatch = GetI(self);
	if (pMyDispatch==NULL) return NULL;
	HRESULT hr = pMyDispatch->GetTypeInfoCount(&ret);
	if ( FAILED(hr) )
		return OleSetOleError(hr);
	return Py_BuildValue("i", ret);
}

// @object PyIDispatch|A OLE automation client object.
static struct PyMethodDef PyIDispatch_methods[] =
{
	{"Invoke",         PyIDispatch::Invoke,  1}, // @pymeth Invoke|Invokes a DISPID, using the passed arguments.
	{"InvokeTypes",    PyIDispatch::InvokeTypes,  1}, // @pymeth InvokeTypes|Invokes a DISPID, using the passed arguments and type descriptions.
	{"GetIDsOfNames",  PyIDispatch::GetIDsOfNames,  1}, // @pymeth GetIDsOfNames|Get the DISPID for the passed names.
	{"GetTypeInfo",    PyIDispatch::GetTypeInfo,  1}, // @pymeth GetTypeInfo|Get type information for the object.
	{"GetTypeInfoCount",PyIDispatch::GetTypeInfoCount,  1}, // @pymeth GetTypeInfoCount|Retrieves the number of interfaces the object provides.
	{NULL,  NULL}        
};

PyComTypeObject PyIDispatch::type("PyIDispatch",
                 &PyIUnknown::type,
                 sizeof(PyIDispatch),
                 PyIDispatch_methods,
				 GET_PYCOM_CTOR(PyIDispatch));

