"""Utilities for registering objects.

This module contains utility functions to register Python objects as
valid COM Servers.  The RegisterServer function provides all information
necessary to allow the COM framework to respond to a request for a COM object,
construct the necessary Python object, and dispatch COM events.

"""
import sys
import win32api
import win32con
import ni
import win32com.pythoncom
import winerror

CATID_PythonCOMServer = "{B3EF80D0-68E2-11D0-A689-00C04FD658FF}"

def _set_subkeys(keyName, valueDict, base=win32con.HKEY_CLASSES_ROOT):
  hkey = win32api.RegCreateKey(base, keyName)
  try:
    for key, value in valueDict.items():
      win32api.RegSetValueEx(hkey, key, None, win32con.REG_SZ, value)
  finally:
    win32api.RegCloseKey(hkey)
			
def _set_string(path, value, base=win32con.HKEY_CLASSES_ROOT):
  "Set a string value in the registry."

  win32api.RegSetValue(base,
                       path,
                       win32con.REG_SZ,
                       value)

def _get_string(path, base=win32con.HKEY_CLASSES_ROOT):
  "Get a string value from the registry."

  try:
    return win32api.RegQueryValue(base, path)
  except win32api.error:
    return None

def _remove_key(path, base=win32con.HKEY_CLASSES_ROOT):
  "Remove a string from the registry."

  try:
    win32api.RegDeleteKey(base, path)
  except win32api.error, (code, fn, msg):
    if code != winerror.ERROR_FILE_NOT_FOUND:
      raise win32api.error, (code, fn, msg)

def recurse_delete_key(path, base=win32con.HKEY_CLASSES_ROOT):
  """Recursively delete registry keys.

  This is needed since you can't blast a key when subkeys exist.
  """
  try:
    h = win32api.RegOpenKey(base, path)
  except win32api.error, (code, fn, msg):
    if code != winerror.ERROR_FILE_NOT_FOUND:
      raise win32api.error, (code, fn, msg)
  else:
    # parent key found and opened successfully. do some work, making sure
    # to always close the thing (error or no).
    try:
      # remove all of the subkeys
      while 1:
        try:
          subkeyname = win32api.RegEnumKey(h, 0)
        except win32api.error, (code, fn, msg):
          if code != winerror.ERROR_NO_MORE_ITEMS:
            raise win32api.error, (code, fn, msg)
          break
        recurse_delete_key(path + '\\' + subkeyname, base)

      # remove the parent key
      _remove_key(path, base)
    finally:
      win32api.RegCloseKey(h)

def _cat_registrar():
  return win32com.pythoncom.CoCreateInstance(
    win32com.pythoncom.CLSID_StdComponentCategoriesMgr,
    None,
    win32com.pythoncom.CLSCTX_INPROC_SERVER,
    win32com.pythoncom.IID_ICatRegister
    )

def RegisterServer(clsid, 
                   ### would be nice to move the className to after the clsid
                   ### but backwards-compat prevents us...
                   pythonInstString=None, className=None,
                   progID=None, verProgID=None,
                   defIcon=None,
                   threadingModel="both",
                   policy=None,
                   catids=[], other={},
                   addPyComCat=1,
                   dispatcher = None):
  """Registers a Python object as a COM Server.  This enters almost all necessary
     information in the system registry, allowing COM to use the object.

     clsid -- The (unique) CLSID of the server.
     pythonInstString -- A string holding the instance name that will be created
                   whenever COM requests a new object.
     className -- ??
     progID -- The user name of this object (eg, Word.Document)
     verProgId -- The user name of this version's implementation (eg Word.6.Document)
     defIcon -- The default icon for the object.
     threadingModel -- The threading model this object supports.
     policy -- The policy to use when creating this object.
     catids -- A list of category ID's this object belongs in.
     other -- A dictionary of extra items to be registered.
     addPyComCat -- A flag indicating if the object should be added to the list
              of Python servers installed on the machine.
     dispatcher -- The dispatcher to use when creating this object.
  """


  ### backwards-compat check
  if not className:
    raise TypeError, 'className required'

  # get the module loaded
  try:
    dllName = win32api.GetModuleFileName(win32api.GetModuleHandle("pythoncom.dll"))
  except win32api.error:
    raise ImportError, "Could not locate the PythonCOM extension"

  keyNameRoot = "CLSID\\%s" % str(clsid)
  _set_string(keyNameRoot, className)

  _set_subkeys(keyNameRoot + "\\InprocServer32",
               { None : dllName,
                 "ThreadingModel" : threadingModel,
                 })

  if pythonInstString:
    _set_string(keyNameRoot + '\\PythonCOM', pythonInstString)
  else:
    _remove_key(keyNameRoot + '\\PythonCOM')
  if policy:
    _set_string(keyNameRoot + '\\PythonCOMPolicy', policy)
  else:
    _remove_key(keyNameRoot + '\\PythonCOMPolicy')

  if dispatcher:
    _set_string(keyNameRoot + '\\PythonCOMDispatcher', dispatcher)
  else:
    _remove_key(keyNameRoot + '\\PythonCOMDispatcher')

  if defIcon:
    _set_string(keyNameRoot + '\\DefaultIcon', defIcon)

  if addPyComCat:
    catids = catids + [ CATID_PythonCOMServer ]

  # Set up the implemented categories
  if catids:
    regCat = _cat_registrar()
    regCat.RegisterClassImplCategories(clsid, catids)

  # set up any other reg values they might have
  if other:
    for key, value in other.items():
      _set_string(keyNameRoot + '\\' + key, value)

  if progID:
    # set the progID as the most specific that was given to us
    if verProgID:
      _set_string(keyNameRoot + '\\ProgID', verProgID)
    else:
      _set_string(keyNameRoot + '\\ProgID', progID)

    # Set up the root entries - version independent.
    _set_string(progID, className)
    _set_string(progID + '\\CLSID', str(clsid))

    # Set up the root entries - version dependent.
    if verProgID:
      # point from independent to the current version
      _set_string(progID + '\\CurVer', verProgID)

      # point to the version-independent one
      _set_string(keyNameRoot + '\\VersionIndependentProgID', progID)

      # set up the versioned progID
      _set_string(verProgID, className)
      _set_string(verProgID + '\\CLSID', str(clsid))


def UnregisterServer(clsid, progID=None, verProgID=None):
  """Unregisters a Python COM server."""

  # remove the main CLSID registration
  recurse_delete_key("CLSID\\%s" % str(clsid))

  # remove the versioned ProgID registration
  if verProgID:
    recurse_delete_key(verProgID)

  ### it might be nice at some point to "roll back" the independent ProgID
  ### to an earlier version if one exists, and just blowing away the
  ### specified version of the ProgID (and its corresponding CLSID)
  ### another time, though...

  # blow away the independent ProgID. we can't leave it since we just
  # torched the class.
  ### could potentially check the CLSID... ?
  if progID:
    recurse_delete_key(progID)

  ### NOTE: ATL simply blows away the above three keys without the
  ### potential checks that I describe.  Assuming that defines the
  ### "standard" then we have no additional changes necessary.

def GetRegisteredServerOption(clsid, optionName):
  """Given a CLSID for a server and option name, return the option value
  """
  keyNameRoot = "CLSID\\%s\\%s" % (str(clsid), str(optionName))
  return _get_string(keyNameRoot)


def _get(ob, attr, default=None):
  try:
    return getattr(ob, attr)
  except AttributeError:
    return default

def RegisterClasses(*classes, **flags):
  quiet = flags.has_key('quiet') and flags['quiet']
  debugging = flags.has_key('debug') and flags['debug']
  for cls in classes:
    clsid = cls._reg_clsid_
    desc = cls._reg_desc_
    spec = _get(cls, '_reg_class_spec_')
    progID = _get(cls, '_reg_progid_')
    verProgID = _get(cls, '_reg_verprogid_')
    defIcon = _get(cls, '_reg_icon_')
    threadingModel = _get(cls, '_reg_threading_', 'both')
    catids = _get(cls, '_reg_catids_', [])
    options = _get(cls, '_reg_options_', {})
    policySpec = _get(cls, '_reg_policy_spec_')
    addPyComCat = not _get(cls, '_reg_disable_pycomcat_', 0)
    if debugging:
      # If the class has a debugging dispatcher specified, use it, otherwise
      # use our default dispatcher.
      dispatcherSpec = _get(cls, '_reg_debug_dispatcher_spec_')
      if dispatcherSpec is None:
        dispatcherSpec = "DispatcherWin32trace"
      # And remember the debugging flag as servers may wish to use it at runtime.
      debuggingDesc = "(for debugging)"
      options['Debugging'] = "1"
    else:
      dispatcherSpec = _get(cls, '_reg_dispatcher_spec_')
      debuggingDesc = ""
      options['Debugging'] = "0"

    RegisterServer(clsid, spec, desc, progID, verProgID, defIcon,
                   threadingModel, policySpec, catids, options,
                   addPyComCat, dispatcherSpec)
    if not quiet:
      print 'Registered:', progID or spec, debuggingDesc


def UnregisterClasses(*classes, **flags):
  quiet = flags.has_key('quiet') and flags['quiet']
  for cls in classes:
    clsid = cls._reg_clsid_
    progID = _get(cls, '_reg_progid_')
    verProgID = _get(cls, '_reg_verprogid_')

    UnregisterServer(clsid, progID, verProgID)
    if not quiet:
      print 'Unregistered:', progID or str(clsid)


def UseCommandLine(*classes, **flags):
  unregister = '--unregister' in sys.argv
  flags['quiet'] = '--quiet' in sys.argv
  flags['debug'] = '--debug' in sys.argv
  if unregister:
    apply(UnregisterClasses, classes, flags)
  else:
    apply(RegisterClasses, classes, flags)


def RegisterPyComCategory():
  """ Register the Python COM Server component category.
  """
  regCat = _cat_registrar()
  regCat.RegisterCategories( [ (CATID_PythonCOMServer,
                                0x0409,
                                "Python COM Server") ] )

try:
  win32api.RegQueryValue(win32con.HKEY_CLASSES_ROOT,
                         'Component Categories\\%s' % CATID_PythonCOMServer)
except win32api.error:
  RegisterPyComCategory()
