"""Support for dynamic COM client support.

Introduction
 Dynamic COM client support is the ability to use a COM server without
 prior knowledge of the server.  This can be used to talk to almost all
 COM servers, including much of MS Office.
 
 In general, you should not use this module directly - see below.
 
Example
 >>> import win32com.client
 >>> xl = win32com.client.Dispatch("Excel.Application")
 # The line above invokes the functionality of this class.
 # xl is now an object we can use to talk to Excel.
 >>> xl.Visible = 1 # The Excel window becomes visible.

"""
import traceback
import string
import new

import pythoncom
#import win32com.pythoncom
#pythoncom=win32com.pythoncom
import build

from types import StringType, IntType, TupleType, ListType
from pywintypes import UnicodeType

debugging=0			# General debugging
debugging_attr=0	# Debugging dynamic attribute lookups.

LCID = 0x0

def debug_print(*args):
	if debugging:
		for arg in args:
			print arg,
		print

def debug_attr_print(*args):
	if debugging_attr:
		for arg in args:
			print arg,
		print

# get the dispatch type in use.
dispatchType = pythoncom.TypeIIDs[pythoncom.IID_IDispatch]

mapKnownObjects = {}

def _GetGoodDispatch(IDispatch):
  if type(IDispatch) in [StringType,pythoncom.PyIIDType,UnicodeType]:
    userName = IDispatch
    try:
      IDispatch = pythoncom.connect(IDispatch)
    except pythoncom.ole_error:
      IDispatch = pythoncom.new(IDispatch)
  return IDispatch

def Dispatch(IDispatch, userName = None, createClass = None, typeinfo = None, UnicodeToString=1):
  if userName is None: userName = "<unknown>"
  IDispatch = _GetGoodDispatch(IDispatch)
  if createClass is None:
    createClass = CDispatch
  try:
    if typeinfo is None:
      typeinfo = IDispatch.GetTypeInfo()
    attr = typeinfo.GetTypeAttr()
    # If the type info is a special DUAL interface, magically turn it into
    # a DISPATCH typeinfo.
    if attr[5] == pythoncom.TKIND_INTERFACE and attr[11] & pythoncom.TYPEFLAG_FDUAL:
      # Get corresponding Disp interface;
      # -1 is a special value which does this for us.
      href = typeinfo.GetRefTypeOfImplType(-1);
      typeinfo = typeinfo.GetRefTypeInfo(href)
      attr = typeinfo.GetTypeAttr()
    try:
      olerepr = mapKnownObjects[attr[0]]
    except KeyError:
      olerepr = build.DispatchItem(typeinfo, attr, None, 0)
      mapKnownObjects[attr[0]] = olerepr
  except pythoncom.ole_error:
    olerepr = build.DispatchItem()
  return createClass(IDispatch, olerepr, userName,UnicodeToString)

def DumbDispatch(IDispatch, userName = None, createClass = None,UnicodeToString=1):
  "Dispatch without attempt at using type information"
  # XXX - Not "truly" dumb - ie, an object returned may not be dumb.
  # to fix, needs its own subclass of CDispatch, with _wrap_dispatch method.
  IDispatch = _GetGoodDispatch(IDispatch)
  if createClass is None:
    createClass = CDispatch
  olerepr = build.DispatchItem()
  return createClass(IDispatch, olerepr, userName,UnicodeToString=UnicodeToString)

class CDispatch:
    def __init__(self, IDispatch, olerepr, userName =  None, UnicodeToString=1):
          if userName is None: userName = "<unknown>"
          self.__dict__['_oleobj_'] = IDispatch
          self.__dict__['_username_'] = userName
          self.__dict__['_olerepr_'] = olerepr
          self.__dict__['_mapCachedItems_'] = {}
          self.__dict__['_builtMethods_'] = {}
          self.__dict__['_enum_'] = None
          self.__dict__['_unicode_to_string_'] = UnicodeToString

# del must not release - other object may be sharing!
#	def __del__(self):
#		try:
#			self._Release_()
#		except:
#			traceback.print_exc()
#			print "Error in CDispatch __del__"

    def __call__(self, *args):
          "Provide 'default dispatch' COM functionality - allow instance to be called"
          if self._olerepr_.defaultDispatchName:
            invkind, dispid = self._find_dispatch_type_(self._olerepr_.defaultDispatchName)
            if invkind:
              allArgs = (pythoncom.DISPID_VALUE,LCID,invkind,1) + args
              return self._get_good_object_(apply(self._oleobj_.Invoke,allArgs),self._olerepr_.defaultDispatchName,None)
          raise TypeError, "This dispatch object does not define a default method"
          
    def __nonzero__(self):
    	return 1 # ie "if object:" should always be "true" - without this, __len__ is tried.
    	# _Possibly_ want to defer to __len__ if available, but Im not sure this is
    	# desirable???

    def __len__(self):
          invkind, dispid = self._find_dispatch_type_("Count")
          if invkind:
            return self._oleobj_.Invoke(dispid, LCID, invkind, 1)
          raise TypeError, "This dispatch object does not define a Count method"

    def __getitem__(self, index):
          if self._enum_ is None:
            invkind, dispid = self._find_dispatch_type_("_NewEnum")
            if not invkind:
              raise TypeError, "This object does not support enumeration"
            import util
            self._enum_ = util.WrapEnum(self._oleobj_.InvokeTypes(pythoncom.DISPID_NEWENUM,LCID,invkind,(13, 10),()),None)
          return self._get_good_object_(self._enum_.__getitem__(index))

    def _find_dispatch_type_(self, methodName):
          if self._olerepr_.mapFuncs.has_key(methodName):
            item = self._olerepr_.mapFuncs[methodName]
            return item.desc[4], item.dispid

          if self._olerepr_.propMapGet.has_key(methodName):
            item = self._olerepr_.propMapGet[methodName]
            return item.desc[4], item.dispid

          try:
            dispid = self._oleobj_.GetIDsOfNames(methodName)
          except:	### what error?
            return None, None
          return pythoncom.DISPATCH_METHOD | pythoncom.DISPATCH_PROPERTYGET, dispid

    def __repr__(self):
          return "<COMObject %s>" % (self._username_)

    def _ApplyTypes_(self, dispid, wFlags, retType, argTypes, user, resultCLSID, *args):
          result = apply(self._oleobj_.InvokeTypes,
                         (dispid, LCID, wFlags, retType, argTypes) + args)
          return self._get_good_object_(result, user, resultCLSID)

    def _wrap_dispatch_(self, ob, userName = None, returnCLSID = None, UnicodeToString = 1):
		# Given a dispatch object, wrap it in a class
		return Dispatch(ob, userName, UnicodeToString=UnicodeToString)

    def _get_good_single_object_(self,ob,userName = None, ReturnCLSID=None):
        if dispatchType==type(ob):
          # make a new instance of (probably this) class.
          return self._wrap_dispatch_(ob, userName, ReturnCLSID)
        elif self._unicode_to_string_ and UnicodeType==type(ob):  
          return str(ob)
        else:
          return ob
		
    def _get_good_object_(self,ob,userName = None, ReturnCLSID=None):
          """Given an object (usually the retval from a method), make it a good object to return.
             Basically checks if it is a COM object, and wraps it up.
             Also handles the fact that a retval may be a tuple of retvals"""
          if ob is None: # Quick exit!
            return None
          elif type(ob)==TupleType:
            return tuple(map(lambda o, s=self, oun=userName, rc=ReturnCLSID: s._get_good_single_object_(o, oun, rc),  ob))
          else:
            return self._get_good_single_object_(ob)
		
    def _make_method_(self, name):
          "Make a method object - Assumes in olerepr funcmap"
          methodCodeList = self._olerepr_.MakeFuncMethod(self._olerepr_.mapFuncs[name], name,0)
          methodCode = string.join(methodCodeList,"\n")
          try:
#	    print "Method code for %s is:\n" % self._username_, methodCode
#	    self._print_details_()
            codeObject = compile(methodCode, "<COMObject %s>" % self._username_,"exec")
            # Exec the code object
            tempNameSpace = {}
            exec codeObject in globals(), tempNameSpace # self.__dict__, self.__dict__
            # Save the function in map.
            fn = self._builtMethods_[name] = tempNameSpace[name]
            import new
            newMeth = new.instancemethod(fn, self, self.__class__)
            return newMeth
          except:
            debug_print("Error building OLE definition for code ", methodCode)
            traceback.print_exc()
          return None
		
    def _Release_(self):
          """Cleanup object - like a close - to force cleanup when you dont 
           want to rely on Python's reference counting."""
          for childCont in self._mapCachedItems_.values():
            childCont._Release_()
          self._mapCachedItems_ = {}
          if self._oleobj_:
            self._oleobj_.Release()
            self.__dict__['_oleobj_'] = None
          if self._olerepr_:
            self.__dict__['_olerepr_'] = None
          self._enum_ = None

    def _proc_(self, name, *args):
          """Call the named method as a procedure, rather than function.
             Mainly used by Word.Basic, which whinges about such things."""
          try:
            item = self._olerepr_.mapFuncs[name]
            dispId = item.dispid
            return self._get_good_object_(apply( self._oleobj_.Invoke, (dispId, LCID, item.desc[4], 0 ) + (args) ))
          except KeyError:
            raise AttributeError, name
		
    def _print_details_(self):
		"Debug routine - dumps what it knows about an object."
		print "AxDispatch container",self._username_
		try:
                  print "Methods:"
                  for method in self._olerepr_.mapFuncs.keys():
                    print "\t", method
                  print "Props:"
                  for prop, entry in self._olerepr_.propMap.items():
                    print "\t%s = 0x%x - %s" % (prop, entry.dispid, `entry`)
                  print "Get Props:"
                  for prop, entry in self._olerepr_.propMapGet.items():
                    print "\t%s = 0x%x - %s" % (prop, entry.dispid, `entry`)
                  print "Put Props:"
                  for prop, entry in self._olerepr_.propMapPut.items():
                    print "\t%s = 0x%x - %s" % (prop, entry.dispid, `entry`)
		except:
                  traceback.print_exc()

    def __getattr__(self, attr):
		if attr[0]=='_' and attr[-1]=='_': # Fast-track.
                  raise AttributeError, attr
		# If a known method, create new instance and return.
		try:
                  return new.instancemethod(self._builtMethods_[attr], self, self.__class__)
		except KeyError:
                  pass
		# XXX - Note that we current are case sensitive in the method.
		#debug_attr_print("GetAttr called for %s on DispatchContainer %s" % (attr,self._username_))
		# First check if it is in the method map.  Note that an actual method
		# must not yet exist, (otherwise we would not be here).  This
		# means we create the actual method object - which also means
		# this code will never be asked for that method name again.
		if self._olerepr_.mapFuncs.has_key(attr):
                  return self._make_method_(attr)
		
		# Delegate to property maps/cached items
		retEntry = None
		if self._olerepr_ and self._oleobj_:
                  # first check general property map, then specific "put" map.
                  if self._olerepr_.propMap.has_key(attr):
                    retEntry = self._olerepr_.propMap[attr]
                  if retEntry is None and self._olerepr_.propMapGet.has_key(attr):
                    retEntry = self._olerepr_.propMapGet[attr]
                  # Not found so far - See what COM says.
                  if retEntry is None:
                    try:
                      debug_attr_print("Calling GetIDsOfNames for property %s in Dispatch container %s" % (attr, self._username_))
                      dispid = self._oleobj_.GetIDsOfNames(0,attr)
                      retEntry = build.MapEntry(dispid, (attr,))
                    except pythoncom.ole_error:
                      pass # No prop by that name - retEntry remains None.
		if not retEntry is None: # see if in my cache
                  try:
                    ret = self._mapCachedItems_[retEntry.dispid]
                    debug_attr_print ("Cached items has attribute!", ret)
                    return ret
                  except (KeyError, AttributeError):
                    debug_attr_print("Attribute %s not in cache" % attr)

		# If we are still here, and have a retEntry, get the OLE item
		if not retEntry is None:
                  debug_attr_print("Getting property Id 0x%x from OLE object" % retEntry.dispid)
                  try:
                    ret = self._oleobj_.Invoke(retEntry.dispid,0,pythoncom.DISPATCH_PROPERTYGET,1)
                  except pythoncom.com_error, details:
                    import winerror
                    if details[0] in [winerror.DISP_E_MEMBERNOTFOUND, winerror.DISP_E_BADPARAMCOUNT, winerror.DISP_E_PARAMNOTOPTIONAL]:
                      # May be a method.
                      self._olerepr_.mapFuncs[attr] = retEntry
                      return self._make_method_(attr)
                    raise pythoncom.com_error, details
                  self._olerepr_.propMap[attr] = retEntry
                  debug_attr_print("OLE returned ", ret)
                  return self._get_good_object_(ret)

		# no where else to look.
		raise AttributeError, "%s.%s" % (self._username_, attr)

    def __setattr__(self, attr, value):
		if self.__dict__.has_key(attr): # Fast-track - if already in our dict, just make the assignment.
                  # XXX - should maybe check method map - if someone assigns to a method,
                  # it could mean something special (not sure what, tho!)
                  self.__dict__[attr] = value
                  return
		# Allow property assignment.
		debug_attr_print("SetAttr called for %s.%s=%s on DispatchContainer" % (self._username_, attr, `value`))
		if self._olerepr_:
			# Check the "general" property map.
			if self._olerepr_.propMap.has_key(attr):
                          self._oleobj_.Invoke(self._olerepr_.propMap[attr].dispid, 0, pythoncom.DISPATCH_PROPERTYPUT, 0, value)
                          return
			# Check the specific "put" map.
			if self._olerepr_.propMapPut.has_key(attr):
                          self._oleobj_.Invoke(self._olerepr_.propMapPut[attr].dispid, 0, pythoncom.DISPATCH_PROPERTYPUT, 0, value)
                          return

                # Try the OLE Object				
		if self._oleobj_:
                  try:
                    debug_attr_print("Calling GetIDsOfNames for property %s in Dispatch container %s" % (attr, self._username_))
                    dispid = self._oleobj_.GetIDsOfNames(0,attr)
                    entry = build.MapEntry(dispid)
                    self._olerepr_.propMap[attr] = entry
                    self._oleobj_.Invoke(entry.dispid, 0, pythoncom.DISPATCH_PROPERTYPUT, 0, value)
                    debug_attr_print("__setattr__ property %s (id=0x%x) in Dispatch container %s" % (attr, entry.dispid, self._username_))
                    return
                  except pythoncom.ole_error:
                    pass # No prop by that name.
		raise AttributeError, "Property '%s.%s' can not be set." % (self._username_, attr)
