/***********************************************************

win32net.cpp -- module for interface into Network API

NOTE: The Network API for NT uses UNICODE.  Therefore, you
can not simply pass python strings to the API functioms - some
conversion is required.

To make my life easier, I include MFC, and use the
A2W/W2A style macros.  No MFC code should be linked in
(afaik!)

Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc

******************************************************************/

#include "windows.h"
#include "lmaccess.h"
#include "lmapibuf.h"
#include "lmerr.h"
#include "lmmsg.h"

#include "Python.h"
#include "atlbase.h"
#include "atlconv.cpp"

static PyObject *module_error;

/* error helper */
PyObject *ReturnError(char *msg, char *fnName = NULL, int errCode = 0)
{
	PyObject *v = Py_BuildValue("(izs)", 0, fnName, msg);
	if (v != NULL) {
		PyErr_SetObject(module_error, v);
		Py_DECREF(v);
	}
	return NULL;
}
/* error helper - GetLastError() is provided, but this is for exceptions */
PyObject *ReturnNetError(char *fnName, long err = 0)
{
	const int bufSize = 512;
	char buf[bufSize];
	DWORD errorCode = err == 0 ? GetLastError() : err;
	BOOL bHaveMessage = FALSE;
	if (errorCode) {
		HMODULE hLibrary;
		if (hLibrary = LoadLibrary(MESSAGE_FILENAME)) {
			bHaveMessage = (0 != FormatMessage( FORMAT_MESSAGE_FROM_HMODULE,
				(LPVOID)hLibrary, errorCode, 0, buf, bufSize, NULL));
			FreeLibrary(hLibrary);
		}
		if (!bHaveMessage) {
			bHaveMessage = (0 != FormatMessage( FORMAT_MESSAGE_FROM_SYSTEM,
				NULL, errorCode, 0, buf, bufSize, NULL));
		}
	}
	if (!bHaveMessage)
		strcpy(buf,"No error message is available");
	/* strip trailing cr/lf */
	int end = strlen(buf)-1;
	if (end>1 && (buf[end-1]=='\n' || buf[end-1]=='\r'))
		buf[end-1] = '\0';
	else
		if (end>0 && (buf[end]=='\n' || buf[end]=='\r'))
			buf[end]='\0';
	PyObject *v = Py_BuildValue("(iss)", errorCode, fnName, buf);
	if (v != NULL) {
		PyErr_SetObject(module_error, v);
		Py_DECREF(v);
	}
	return NULL;
}
/*
BOOL PyUserInfoFromDict(user_info_2*pBuf)
{
	char *prop = NULL;
	if (v = PyDict_GetItemString (font_props, (prop="name"))) {
		if (
		if (bOK=(PyString_Check (v))
			_tcscpy(pBuf->usri2_auth_flags, A2W(PyString_AsString(v);
	}
	if (bOK && (v = PyDict_GetItemString (font_props, (prop="auth_flags")))) {
		if (bOK=(PyInt_Check (v))
			pBuf->usri2_auth_flags = PyInt_AsLong(v);
	}



	switch (level) {
	if (strcmp(name, "auth_flags"))
case 1:

}
*/

// @pymethod |win32net|NetMessageBufferSend|sends a string to a registered message alias.
static PyObject *
PyNetMessageBufferSend( PyObject *self, PyObject *args)
{
	USES_CONVERSION;
	DWORD rc;
	char *serverName, *msgName, *fromName, *message;
	int msgLen;
	if (!PyArg_ParseTuple(args, "zszs#:NetMessageBufferSend", 
	          &serverName,  // @pyparm string|domain||Specifies the name of the remote server on which the function is to execute. None or empty string the local computer.
	          &msgName, // @pyparm string|userName||Specifies the message name to which the message buffer should be sent.
	          &fromName, // @pyparm string|fromName||The user the message is to come from, or None for the current user.
	          &message, &msgLen)) // @pyparm string|message||The message text
		return NULL;
	if ((rc=NetMessageBufferSend( A2W(serverName), A2W(msgName), A2W(fromName), (BYTE *)A2W(message), msgLen * sizeof(WCHAR))))
		return ReturnNetError("NetMessageBufferSend",rc);	// @pyseeapi NetMessageBufferSend

	Py_INCREF(Py_None);
	return Py_None;
}

// @pymethod |win32net|NetUserChangePassword|Changes a users password on the specified domain.
static PyObject *
PyNetUserChangePassword( PyObject *self, PyObject *args)
{
	USES_CONVERSION;
	DWORD rc;
	char *domain, *userName, *oldPassword, *newPassword;
	if (!PyArg_ParseTuple(args, "zszs:NetUserChangePassword", 
	          &domain,  // @pyparm string|domain||Specifies the domain for which to change the password.  If None, the users logon domain is used.
	          &userName, // @pyparm string|userName||Specifies the username for which to change the password.
	          &oldPassword, // @pyparm string|oldPassword||The users old password, or None.
	          &newPassword)) // @pyparm string|newPassword||The users new password.
	                // One value has a special meaning: If dwDuration is  - 1, the function 
	                // operates asynchronously and produces sound until called again. 
		return NULL;
	if ((rc=NetUserChangePassword( A2W(domain), A2W(userName), A2W(oldPassword), A2W(newPassword))))
		return ReturnNetError("NetUserChangePassword",rc);	// @pyseeapi NetUserChangePassword

	Py_INCREF(Py_None);
	return Py_None;
	// @comm A server or domain can be configured to require that a
	// user log on to change the password on a user account.
	// If that is the case, you need administrator or account operator access
	// to change the password for another user acount.
	// If logging on is not required, you can change the password for
	// any user account, so long as you know the current password.
}

// @pymethod |win32net|NetUserGetGroups|Returns a list of groups,attributes for all groups for the user.
static PyObject *
PyNetUserGetGroups( PyObject *self, PyObject *args)
{
	USES_CONVERSION;
	DWORD rc;
	char *domain, *userName;
	if (!PyArg_ParseTuple(args, "zs:NetUserGetGroups", 
	          &domain,  // @pyparm string|domain||Specifies the domain for which to change the password.  If None, the users logon domain is used.
	          &userName)) // @pyparm string|userName||Specifies the username for which to change the password.
		return NULL;

		DWORD read;
	DWORD total;
	GROUP_USERS_INFO_1 *buf = NULL;
	// I can not make enumeration work correctly (ie, can not make a
	// "loop while rc==ERROR_MORE_DATA" work - it _always_ returns the
	// first groups (ie, the loop never terminates).
	// Therefore, I loop until I get success, doubling the buffer
	// each time.
	DWORD bufSize = 8192;
	DWORD newSize;
	WCHAR *wDomain = A2W(domain);
	WCHAR *wUserName = A2W(userName);
	while (1) {
		rc=NetUserGetGroups( wDomain, wUserName, 1, (unsigned char **)&buf, bufSize, &read, &total );
		if (rc==ERROR_SUCCESS) break;
		if (buf) {
			NetApiBufferFree(buf);
			buf = NULL;
		}
		newSize = bufSize * 2; // this may overflow if many many many groups - just fail.
		if (newSize<bufSize || (rc!=ERROR_MORE_DATA && rc!=NERR_BufTooSmall))
			return ReturnNetError("NetUserGetGroups",rc);	// @pyseeapi NetUserGetGroups
		bufSize = newSize;
	}
	PyObject *retList = PyList_New(0);
	for (DWORD i=0;i<read;i++) {
		PyObject *newOb = Py_BuildValue("si", W2A(buf[i].grui1_name), buf[i].grui1_attributes);
		PyList_Append(retList, newOb);
		Py_DECREF(newOb);
	}
	NetApiBufferFree(buf);
	return retList;
}

// @pymethod |win32net|NetUserGetLocalGroups|Returns a list of groups,attributes for all groups for the user.
static PyObject *
PyNetUserGetLocalGroups( PyObject *self, PyObject *args)
{
	USES_CONVERSION;
	DWORD rc;
	DWORD flags = LG_INCLUDE_INDIRECT;
	char *domain, *userName;
	if (!PyArg_ParseTuple(args, "zs|i:NetUserGetLocalGroups", 
	          &domain,  // @pyparm string|domain||Specifies the domain for which to change the password.  If None, the users logon domain is used.
	          &userName, // @pyparm string|userName||Specifies the username for which to change the password.
			  &flags))  // @pyparm int|flags|LG_INCLUDE_INDIRECT|Flags for the operation.
		return NULL;
	
	DWORD read;
	DWORD total;
	GROUP_USERS_INFO_0 *buf = NULL;
	// I can not make enumeration work correctly (ie, can not make a
	// "loop while rc==ERROR_MORE_DATA" work - it _always_ returns the
	// first groups (ie, the loop never terminates).
	// Therefore, I loop until I get success, doubling the buffer
	// each time.
	DWORD bufSize = 8192;
	DWORD newSize;
	WCHAR *wDomain = A2W(domain);
	WCHAR *wUserName = A2W(userName);
	while (1) {
		rc=NetUserGetLocalGroups( wDomain, wUserName, 0, flags, (unsigned char **)&buf, bufSize, &read, &total );
		if (rc==ERROR_SUCCESS) break;
		if (buf) {
			NetApiBufferFree(buf);
			buf = NULL;
		}
		newSize = bufSize * 2; // this may overflow if many many many groups - just fail.
		if (newSize<bufSize || (rc!=ERROR_MORE_DATA && rc!=NERR_BufTooSmall))
			return ReturnNetError("NetUserGetLocalGroups",rc);	// @pyseeapi NetUserGetLocalGroups
		bufSize = newSize;
	}
	PyObject *retList = PyList_New(0);
	for (DWORD i=0;i<read;i++) {
		PyObject *newOb = Py_BuildValue("s", W2A(buf[i].grui0_name));
		PyList_Append(retList, newOb);
		Py_DECREF(newOb);
	}
	NetApiBufferFree(buf);
	return retList;
}

/* List of functions exported by this module */
// @module win32net|A module encapsulating the Windows Network API.
static struct PyMethodDef win32net_functions[] = {
	{"NetMessageBufferSend",         (PyCFunction)PyNetMessageBufferSend, METH_VARARGS}, // @pymeth NetMessageBufferSend|sends a string to a registered message alias.
	{"NetUserChangePassword",        (PyCFunction)PyNetUserChangePassword,  METH_VARARGS }, // @pymeth NetUserChangePassword|Changes a users password on the specified domain.
	{"NetUserGetGroups",             (PyCFunction)PyNetUserGetGroups,  METH_VARARGS }, // @pymeth NetUserGetGroups|Determines all global groups for the specified user.
	{"NetUserGetLocalGroups",        (PyCFunction)PyNetUserGetLocalGroups,  METH_VARARGS }, // @pymeth NetUserGetLocalGroups|Determines all local groups for the specified user.

	{NULL,			NULL}
};


extern "C" __declspec(dllexport) void
initwin32net(void)
{
  PyObject *dict, *module;
  module = Py_InitModule("win32net", win32net_functions);
  dict = PyModule_GetDict(module);
  module_error = PyString_FromString("win32net error");
  PyDict_SetItemString(dict, "error", module_error);
}
