/* File : win32file.i */
// @doc

%module win32file // An interface to the win32 File API's

%{
//#define UNICODE
%}

%include "typemaps.i"
%include "pywin32.i"

#define FILE_GENERIC_READ FILE_GENERIC_READ
#define FILE_GENERIC_WRITE FILE_GENERIC_WRITE
#define FILE_ALL_ACCESS FILE_ALL_ACCESS

#define GENERIC_READ GENERIC_READ 
// Specifies read access to the object. Data can be read from the file and the file pointer can be moved. Combine with GENERIC_WRITE for read-write access. 
#define GENERIC_WRITE GENERIC_WRITE 
// Specifies write access to the object. Data can be written to the file and the file pointer can be moved. Combine with GENERIC_READ for read-write access. 
#define GENERIC_EXECUTE GENERIC_EXECUTE 
// Specifies execute access.

#define FILE_SHARE_DELETE  FILE_SHARE_DELETE 
// Windows NT only: Subsequent open operations on the object will succeed only if delete access is requested. 
#define FILE_SHARE_READ FILE_SHARE_READ 
// Subsequent open operations on the object will succeed only if read access is requested. 
#define FILE_SHARE_WRITE FILE_SHARE_WRITE 
// Subsequent open operations on the object will succeed only if write access is requested. 
 
#define CREATE_NEW CREATE_NEW 
// Creates a new file. The function fails if the specified file already exists.
#define CREATE_ALWAYS CREATE_ALWAYS 
// Creates a new file. The function overwrites the file if it exists.
#define OPEN_EXISTING OPEN_EXISTING 
// Opens the file. The function fails if the file does not exist.
#define OPEN_ALWAYS OPEN_ALWAYS 
// Opens the file, if it exists. If the file does not exist, the function creates the file as if dwCreationDistribution were CREATE_NEW.
#define TRUNCATE_EXISTING TRUNCATE_EXISTING 
// Opens the file. Once opened, the file is truncated so that its size is zero bytes. The calling process must open the file with at least GENERIC_WRITE access. The function fails if the file does not exist.
 
#define FILE_ATTRIBUTE_ARCHIVE FILE_ATTRIBUTE_ARCHIVE 
// The file should be archived. Applications use this attribute to mark files for backup or removal.
#define FILE_ATTRIBUTE_COMPRESSED FILE_ATTRIBUTE_COMPRESSED 
// The file or directory is compressed. For a file, this means that all of the data in the file is compressed. For a directory, this means that compression is the default for newly created files and subdirectories.
#define FILE_ATTRIBUTE_HIDDEN FILE_ATTRIBUTE_HIDDEN 
// The file is hidden. It is not to be included in an ordinary directory listing.
#define FILE_ATTRIBUTE_NORMAL FILE_ATTRIBUTE_NORMAL 
// The file has no other attributes set. This attribute is valid only if used alone.
#define FILE_ATTRIBUTE_OFFLINE FILE_ATTRIBUTE_OFFLINE 
// The data of the file is not immediately available. Indicates that the file data has been physically moved to offline storage.
#define FILE_ATTRIBUTE_READONLY FILE_ATTRIBUTE_READONLY 
// The file is read only. Applications can read the file but cannot write to it or delete it.
#define FILE_ATTRIBUTE_SYSTEM FILE_ATTRIBUTE_SYSTEM 
// The file is part of or is used exclusively by the operating system.
#define FILE_ATTRIBUTE_TEMPORARY FILE_ATTRIBUTE_TEMPORARY 
// The file is being used for temporary storage. File systems attempt to keep all of the data in memory for quicker access rather than flushing the data back to mass storage. A temporary file should be deleted by the application as soon as it is no longer needed.
 
#define FILE_FLAG_WRITE_THROUGH FILE_FLAG_WRITE_THROUGH 
// Instructs the system to write through any intermediate cache and go directly to disk. Windows can still cache write operations, but cannot lazily flush them.
#define FILE_FLAG_OVERLAPPED FILE_FLAG_OVERLAPPED 
// Instructs the system to initialize the object, so that operations that take a significant amount of time to process return ERROR_IO_PENDING. When the operation is finished, the specified event is set to the signaled state.
	// When you specify FILE_FLAG_OVERLAPPED, the ReadFile and WriteFile functions must specify an OVERLAPPED structure. That is, when FILE_FLAG_OVERLAPPED is specified, an application must perform overlapped reading and writing.
	// When FILE_FLAG_OVERLAPPED is specified, the system does not maintain the file pointer. The file position must be passed as part of the lpOverlapped parameter (pointing to an OVERLAPPED structure) to the ReadFile and WriteFile functions.
	// This flag also enables more than one operation to be performed simultaneously with the handle (a simultaneous read and write operation, for example).
#define FILE_FLAG_NO_BUFFERING FILE_FLAG_NO_BUFFERING 
// Instructs the system to open the file with no intermediate buffering or caching. 
	// When combined with FILE_FLAG_OVERLAPPED, the flag gives maximum asynchronous performance, 
	// because the I/O does not rely on the synchronous operations of the memory 
	// manager. However, some I/O operations will take longer, because data is 
	// not being held in the cache. An application must meet certain requirements 
	// when working with files opened with FILE_FLAG_NO_BUFFERING:
	// <nl>	File access must begin at byte offsets within the file that are integer multiples of the volume's sector size.
	// <nl>	File access must be for numbers of bytes that are integer multiples of the volume's sector size. 
	// For example, if the sector size is 512 bytes, an application can request reads and writes of 512, 1024, or 2048 bytes, but not of 335, 981, or 7171 bytes.
	// <nl>	Buffer addresses for read and write operations must be aligned on addresses in memory that are integer multiples of the volume's sector size. 
	// One way to align buffers on integer multiples of the volume sector size is to use VirtualAlloc to allocate the 
	// buffers. It allocates memory that is aligned on addresses that are integer multiples of the operating system's memory page size. Because both memory page 
	// and volume sector sizes are powers of 2, this memory is also aligned on addresses that are integer multiples of a volume's sector size. An application can 
	// determine a volume's sector size by calling the GetDiskFreeSpace function. 
#define FILE_FLAG_RANDOM_ACCESS FILE_FLAG_RANDOM_ACCESS 
// Indicates that the file is accessed randomly. The system can use this as a hint to optimize file caching.
#define FILE_FLAG_SEQUENTIAL_SCAN FILE_FLAG_SEQUENTIAL_SCAN 
// Indicates that the file is to be accessed sequentially from beginning to end. The system can use this as a hint to optimize file caching. 
	// If an application moves the file pointer for random access, optimum caching may not occur; however, correct operation is still guaranteed.
	// Specifying this flag can increase performance for applications that read large files using sequential access. 
	// Performance gains can be even more noticeable for applications that read large files mostly sequentially, but occasionally skip over small ranges of bytes.
#define FILE_FLAG_DELETE_ON_CLOSE FILE_FLAG_DELETE_ON_CLOSE 
// Indicates that the operating system is to delete the file immediately after all of its handles have been closed, 
	// not just the handle for which you specified FILE_FLAG_DELETE_ON_CLOSE. Subsequent open requests for the file will fail, unless FILE_SHARE_DELETE is used. 
#define FILE_FLAG_BACKUP_SEMANTICS FILE_FLAG_BACKUP_SEMANTICS 
// Windows NT only: Indicates that the file is being opened or created for a backup or restore operation. 
	// The operating system ensures that the calling process overrides file security checks, provided it has the necessary permission to do so. The relevant permissions are SE_BACKUP_NAME and SE_RESTORE_NAME.
	// You can also set this flag to obtain a handle to a directory. A directory handle can be passed to some Win32 functions in place of a file handle.
#define FILE_FLAG_POSIX_SEMANTICS FILE_FLAG_POSIX_SEMANTICS 
// Indicates that the file is to be accessed according to POSIX rules. 
	// This includes allowing multiple files with names, differing only in case, for file systems that support such naming. 
	// Use care when using this option because files created with this flag may not be accessible by applications written for MS-DOS or Windows.
 
#define SECURITY_ANONYMOUS SECURITY_ANONYMOUS 
// Specifies to impersonate the client at the Anonymous impersonation level.
#define SECURITY_IDENTIFICATION SECURITY_IDENTIFICATION 
// Specifies to impersonate the client at the Identification impersonation level.
#define SECURITY_IMPERSONATION SECURITY_IMPERSONATION 
// Specifies to impersonate the client at the Impersonation impersonation level.
#define SECURITY_DELEGATION SECURITY_DELEGATION 
// Specifies to impersonate the client at the Delegation impersonation level.
#define SECURITY_CONTEXT_TRACKING SECURITY_CONTEXT_TRACKING 
// Specifies that the security tracking mode is dynamic. If this flag is not specified, Security Tracking Mode is static.
#define SECURITY_EFFECTIVE_ONLY SECURITY_EFFECTIVE_ONLY 
// Specifies that only the enabled aspects 	

// @pyswig int|AreFileApisANSI|Determines whether a set of Win32 file functions is using the ANSI or OEM character set code page. This function is useful for 8-bit console input and output operations.
BOOL AreFileApisANSI(void);

// BOOLAPI CancelIO(HANDLE handle);

// @pyswig |CopyFile|Copies a file
BOOLAPI CopyFile(
    TCHAR *from, // @pyparm <o PyUnicode>|from||The name of the file to copy from
    TCHAR *to, // @pyparm <o PyUnicode>|to||The name of the file to copy to
    BOOL bFailIfExists); // @pyparm int|bFailIfExists||Indicates if the operation should fail if the file exists.

// CopyFileEx

// @pyswig |CreateDirectory|Creates a directory
BOOLAPI CreateDirectory(
    TCHAR *name, // @pyparm <o PyUnicode>|name||The name of the directory to create
    SECURITY_ATTRIBUTES *pSA); // @pyparm <o PySECURITY_ATTRIBUTES>|sa||The security attributes, or None

// @pyswig |CreateDirectoryEx|Creates a directory
BOOLAPI CreateDirectoryEx(
    TCHAR *templateName, // @pyparm <o PyUnicode>|templateName||Specifies the path of the directory to use as a template when creating the new directory. 
    TCHAR *newDirectory, // @pyparm <o PyUnicode>|newDirectory||Specifies the name of the new directory
    SECURITY_ATTRIBUTES *pSA); // @pyparm <o PySECURITY_ATTRIBUTES>|sa||The security attributes, or None

// @pyswig <o PyHANDLE>|CreateFile|Creates or opens the a file or other object and returns a handle that can be used to access the object.
// @comm The following objects can be opened:<nl>files<nl>pipes<nl>mailslots<nl>communications resources<nl>disk devices (Windows NT only)<nl>consoles<nl>directories (open only)
HANDLE CreateFile(
    TCHAR *lpFileName,	// @pyparm <o PyUnicode>|templateName||The name of the file
    DWORD dwDesiredAccess,	// @pyparm int|desiredAccess||access (read-write) mode
    DWORD dwShareMode,	// @pyparm int|shareMode||share mode
    SECURITY_ATTRIBUTES *lpSecurityAttributes,	// @pyparm <o PySECURITY_ATTRIBUTES>|attributes||The security attributes, or None
    DWORD dwCreationDistribution,	// @pyparm int|creationDistribution||how to create
    DWORD dwFlagsAndAttributes,	// @pyparm int|flagsAndAttributes||file attributes
    HANDLE hTemplateFile // @pyparm <o PyHANDLE>|hTemplateFile||Specifies a handle with GENERIC_READ access to a template file. The template file supplies file attributes and extended attributes for the file being created.   Under Win95, this must be 0, else an exception will be raised.
);

//CreateIoCompletionPort

// @pyswig |DefineDosDevice|Lets an application define, redefine, or delete MS-DOS device names. 
BOOLAPI DefineDosDevice(
    DWORD dwFlags,	// @pyparm int|flags||flags specifying aspects of device definition  
    TCHAR *lpDeviceName,	// @pyparm <o PyUnicode>|deviceName||MS-DOS device name string  
    TCHAR *lpTargetPath	// @pyparm <o PyUnicode>|targetPath||MS-DOS or path string for 32-bit Windows.
);

// @pyswig |DeleteFile|Deletes a file.
BOOLAPI DeleteFile(TCHAR *fileName);
// @pyparm <o PyUnicode>|fileName||The filename to delete

//FileIOCompletionRoutine	

// @pyswig |FindClose|Closes a handle opened with <om win32file.FindOpen>
BOOLAPI FindClose(INT_HANDLE hFindFile);	// @pyparm int|hFindFile||file search handle
 
// @pyswig |FindCloseChangeNotification|Closes a handle.
BOOLAPI FindCloseChangeNotification(
    INT_HANDLE hChangeHandle 	// @pyparm int|hChangeHandle||handle to change notification to close
);

// @pyswig int|FindFirstChangeNotification|Creates a change notification handle and sets up initial change notification filter conditions. A wait on a notification handle succeeds when a change matching the filter conditions occurs in the specified directory or subtree. 
INT_HANDLE FindFirstChangeNotification(
    TCHAR *lpPathName,	// @pyparm <o PyUnicode>|pathName||Name of directory to watch  
    BOOL bWatchSubtree,	// @pyparm int|bWatchSubtree||flag for monitoring directory or directory tree  
    DWORD dwNotifyFilter 	// @pyparm int|notifyFilter||filter conditions to watch for 
);

//FindFirstFile	
//FindFirstFileEx	
// FindNextFile	

// @pyswig int|FindNextChangeNotification|Requests that the operating system signal a change notification handle the next time it detects an appropriate change,
BOOLAPI FindNextChangeNotification(
    INT_HANDLE hChangeHandle 	//  @pyparm int|hChangeHandle||handle to change notification to signal  
);

// @pyswig |FlushFileBuffers|Clears the buffers for the specified file and causes all buffered data to be written to the file. 
BOOLAPI FlushFileBuffers(
   HANDLE hFile 	// @pyparm <o PyHANDLE>|hFile||open handle to file whose buffers are to be flushed 
);

// @pyswig int|GetBinaryType|Determines whether a file is executable, and if so, what type of executable file it is. That last property determines which subsystem an executable file runs under.
BOOLAPI GetBinaryType(
    TCHAR *lpApplicationName,	// @pyparm <o PyUnicode>|appName||Fully qualified path of file to test
    unsigned long *OUTPUT	// DWORD
   );
#define SCS_32BIT_BINARY SCS_32BIT_BINARY // A Win32-based application
#define SCS_DOS_BINARY SCS_DOS_BINARY // An MS-DOS - based application
#define SCS_OS216_BINARY SCS_OS216_BINARY // A 16-bit OS/2-based application
#define SCS_PIF_BINARY SCS_PIF_BINARY // A PIF file that executes an MS-DOS - based application
#define SCS_POSIX_BINARY SCS_POSIX_BINARY // A POSIX - based application
#define SCS_WOW_BINARY SCS_WOW_BINARY // A 16-bit Windows-based application


//GetCurrentDirectory

// @pyswig (int, int, int, int)|GetDiskFreeSpace|Determines the free space on a device.
BOOLAPI GetDiskFreeSpace(
    TCHAR *lpRootPathName,	// @pyparm <o PyUnicode>|rootPathName||address of root path
    unsigned long *OUTPUT, // LPDWORD 
    unsigned long *OUTPUT, // LPDWORD 
    unsigned long *OUTPUT, // LPDWORD 
    unsigned long *OUTPUT // LPDWORD 
// @rdesc The result is a tuple of integers representing (sectors per cluster, bytes per sector, number of free clusters, total number of clusters)
);

// GetDiskFreeSpaceEx	

// @pyswig int|GetDriveType|Determines whether a disk drive is a removable, fixed, CD-ROM, RAM disk, or network drive. 
long GetDriveType(
    TCHAR *rootPathName
// @rdesc The result is one of the DRIVE_* constants.
);

#define DRIVE_UNKNOWN DRIVE_UNKNOWN // The drive type cannot be determined.
#define DRIVE_NO_ROOT_DIR DRIVE_NO_ROOT_DIR // The root directory does not exist.
#define DRIVE_REMOVABLE DRIVE_REMOVABLE // The disk can be removed from the drive.
#define DRIVE_FIXED DRIVE_FIXED // The disk cannot be removed from the drive.
#define DRIVE_REMOTE DRIVE_REMOTE // The drive is a remote (network) drive.
#define DRIVE_CDROM DRIVE_CDROM // The drive is a CD-ROM drive.
#define DRIVE_RAMDISK DRIVE_RAMDISK // The drive is a RAM disk.


// @pyswig int|GetFileAttributes|Determines a files attributes.
DWORD GetFileAttributes(
    TCHAR *fileName); // @pyparm <o PyUnicode>|fileName||Name of the file to retrieve attributes for.

//GetFileAttributesEx	
//GetFileInformationByHandle	

%{
PyObject *MyGetCompressedFileSize(TCHAR *fname)
{
	DWORD dwSizeLow, dwSizeHigh;
    Py_BEGIN_ALLOW_THREADS
	dwSizeLow = GetCompressedFileSize(fname, &dwSizeHigh);
    Py_END_ALLOW_THREADS
	// If we failed ... 
	if (dwSizeLow == 0xFFFFFFFF && 
	    GetLastError() != NO_ERROR )
		return PyWin_SetAPIError("GetFileSize");
	return PyLong_FromTwoInts(dwSizeHigh, dwSizeLow);
}
PyObject *MyGetFileSize(HANDLE hFile)
{
	DWORD dwSizeLow, dwSizeHigh;
    Py_BEGIN_ALLOW_THREADS
	dwSizeLow = GetFileSize (hFile, &dwSizeHigh);
    Py_END_ALLOW_THREADS
	// If we failed ... 
	if (dwSizeLow == 0xFFFFFFFF && 
	    GetLastError() != NO_ERROR )
		return PyWin_SetAPIError("GetFileSize");
	return PyLong_FromTwoInts(dwSizeHigh, dwSizeLow);
}

// @pyswig int|ReadFile|Reads a string from a file
PyObject *MyReadFile(PyObject *self, PyObject *args)
{
	OVERLAPPED *pOverlapped;
	PyObject *obhFile;
	HANDLE hFile;
	DWORD bufSize;
	PyObject *obOverlapped = NULL;

	if (!PyArg_ParseTuple(args, "Ol|O:ReadFile", 
		&obhFile, // @pyparm int|hFile||Handle to the file
		&bufSize, // @pyparm int|bufSize||Size of the buffer to create for the read.
		&obOverlapped))	// @pyparm <o PyOVERLAPPED>|ol|None|An overlapped structure
		return NULL;
	if (obOverlapped==NULL)
		pOverlapped = NULL;
	else {
		if (!PyWinObject_AsOVERLAPPED(obOverlapped, &pOverlapped))
			return NULL;
	}
	if (!PyWinObject_AsHANDLE(obhFile, &hFile))
		return NULL;

	void *buf = malloc(bufSize);
	DWORD numRead;
	BOOL ok;
    Py_BEGIN_ALLOW_THREADS
	ok = ReadFile(hFile, buf, bufSize, &numRead, pOverlapped);
    Py_END_ALLOW_THREADS
	if (!ok) {
		free(buf);
		return PyWin_SetAPIError("ReadFile");
	}
	PyObject *result = PyString_FromStringAndSize((char *)buf, numRead);
	free(buf);
	return result;
}

// @pyswig int|WriteFile|Writes a string to a file
// @rdesc The result is the number of bytes written.
PyObject *MyWriteFile(PyObject *self, PyObject *args)
{
	OVERLAPPED *pOverlapped;
	PyObject *obhFile;
	HANDLE hFile;
	char *writeData;
	DWORD writeSize;
	PyObject *obOverlapped = NULL;

	if (!PyArg_ParseTuple(args, "Os#|O:Write", 
		&obhFile, // @pyparm <o PyHANDLE>|hFile||Handle to the file
		&writeData, &writeSize, // @pyparm string|data||The data to write.
		&obOverlapped))	// @pyparm <o PyOVERLAPPED>|ol|None|An overlapped structure
		return NULL;
	if (obOverlapped==NULL)
		pOverlapped = NULL;
	else {
		if (!PyWinObject_AsOVERLAPPED(obOverlapped, &pOverlapped))
			return NULL;
	}
	if (!PyWinObject_AsHANDLE(obhFile, &hFile))
		return NULL;
	DWORD numWritten;
	BOOL ok;
    Py_BEGIN_ALLOW_THREADS
	ok = WriteFile(hFile, writeData, writeSize, &numWritten, pOverlapped);
    Py_END_ALLOW_THREADS
	if (!ok) {
		return PyWin_SetAPIError("WriteFile");
	}
	return PyInt_FromLong(numWritten);
}

// @pyswig |CloseHandle|Closes an open handle.
static PyObject *MyCloseHandle(PyObject *self, PyObject *args)
{
	PyObject *obHandle;
	if (!PyArg_ParseTuple(args, "O:CloseHandle",
			&obHandle)) // @pyparm <o PyHANDLE>/int|handle||A previously opened handle.
		return NULL;
	if (!PyWinObject_CloseHANDLE(obHandle))
		return NULL
	Py_INCREF(Py_None);
	return Py_None;
}

%}

// @pyswig <o PyLARGE_INTEGER>|GetCompressedFileSize|Determines the compressed size of a file.
BOOLAPI GetCompressedFileSize(
    TCHAR *fileName,	// @pyparm <o PyUnicode>|fileName||The name of the file
    unsigned long *OUTPUT	// pointer to DWORD to receive high-order doubleword of file size
);

// @pyswig <o PyLARGE_INTEGER>|GetFileSize|Determines the size of a file.
%name(GetFileSize) DWORD MyGetFileSize(
    HANDLE hFile	// @pyparm <o PyHANDLE>|hFile||The handle to the file.
);

%native(ReadFile) MyReadFile;
%native(WriteFile) MyWriteFile;
%native(CloseHandle) MyCloseHandle;

// @pyswig int|GetFileType|Determines the type of a file.
unsigned long GetFileType( // DWORD
    HANDLE hFile // @pyparm <o PyHANDLE>|hFile||The handle to the file.
);
#define FILE_TYPE_UNKNOWN FILE_TYPE_UNKNOWN // The type of the specified file is unknown.
#define FILE_TYPE_DISK FILE_TYPE_DISK // The specified file is a disk file.
#define FILE_TYPE_CHAR FILE_TYPE_CHAR // The specified file is a character file, typically an LPT device or a console.
#define FILE_TYPE_PIPE FILE_TYPE_PIPE // The specified file is either a named or anonymous pipe.
 


// GetFullPathName	

// @pyswig int|GetLogicalDrives|Returns a bitmaks of the logical drives installed.
unsigned long GetLogicalDrives( // DWORD
);

/**
GetLogicalDriveStrings	
GetQueuedCompletionStatus	
GetShortPathName	
GetTempFileName	
GetTempPath	
GetVolumeInformation	
*/

// @pyswig |LockFile|Determines the type of a file.
BOOLAPI LockFile(
    HANDLE hFile,	// @pyparm <o PyHANDLE>|hFile||handle of file to lock 
    DWORD dwFileOffsetLow,	// @pyparm int|offsetLow||low-order word of lock region offset 
    DWORD dwFileOffsetHigh,	// @pyparm int|offsetHigh||high-order word of lock region offset  
    DWORD nNumberOfBytesToLockLow,	// @pyparm int|nNumberOfBytesToLockLow||low-order word of length to lock 
    DWORD nNumberOfBytesToLockHigh 	// @pyparm int|nNumberOfBytesToLockHigh||high-order word of length to lock 
   );


// LockFileEx

// @pyswig |MoveFile|Renames an existing file or a directory (including all its children). 
BOOLAPI MoveFile(
    TCHAR *lpExistingFileName,	// @pyparm <o PyUnicode>|existingFileName||Name of the existing file  
    TCHAR *lpNewFileName 	// @pyparm <o PyUnicode>|newFileName||New name for the file 
);

// @pyswig |MoveFileEx|Renames an existing file or a directory (including all its children). 
BOOLAPI MoveFileEx(
    TCHAR *lpExistingFileName,	// @pyparm <o PyUnicode>|existingFileName||Name of the existing file  
    TCHAR *lpNewFileName, 	// @pyparm <o PyUnicode>|newFileName||New name for the file 
    DWORD dwFlags 	        // @pyparm int|flags||flag to determine how to move file 
);
#define MOVEFILE_COPY_ALLOWED MOVEFILE_COPY_ALLOWED // If the file is to be moved to a different volume, the function simulates the move by using the CopyFile and DeleteFile functions. Cannot be combined with the MOVEFILE_DELAY_UNTIL_REBOOT flag.
#define MOVEFILE_DELAY_UNTIL_REBOOT MOVEFILE_DELAY_UNTIL_REBOOT // Windows NT only: The function does not move the file until the operating system is restarted. The system moves the file immediately after AUTOCHK is executed, but before creating any paging files. Consequently, this parameter enables the function to delete paging files from previous startups.
#define MOVEFILE_REPLACE_EXISTING MOVEFILE_REPLACE_EXISTING // If a file of the name specified by lpNewFileName already exists, the function replaces its contents with those specified by lpExistingFileName.
#define MOVEFILE_WRITE_THROUGH MOVEFILE_WRITE_THROUGH // Windows NT only: The function does not return until the file has actually been moved on the disk. Setting this flag guarantees that a move perfomed as a copy and delete operation is flushed to disk before the function returns. The flush occurs at the end of the copy operation.<nl>This flag has no effect if the MOVEFILE_DELAY_UNTIL_REBOOT flag is set. 

// PostQueuedCompletionStatus	
// QueryDosDevice	
// ReadDirectoryChangesW	
// ReadFile	
// ReadFileEx	

// @pyswig |RemoveDirectory|Removes an existing directory
BOOLAPI RemoveDirectory(
    TCHAR *lpPathName	// @pyparm <o PyUnicode>|lpPathName||Name of the path to remove.
);

//SearchPath	

// @pyswig |SetCurrentDirectory|Sets the current directory.
BOOLAPI SetCurrentDirectory(
    TCHAR *lpPathName	// @pyparm <o PyUnicode>|lpPathName||Name of the path to set current.
);

// @pyswig |SetEndOfFile|Moves the end-of-file (EOF) position for the specified file to the current position of the file pointer. 
BOOL SetEndOfFile(
    HANDLE hFile	// @pyparm <o PyHANDLE>|hFile||handle of file whose EOF is to be set 
);

// @pyswig |SetFileApisToANSI|Causes a set of Win32 file functions to use the ANSI character set code page. This function is useful for 8-bit console input and output operations.
void SetFileApisToANSI(void);

// @pyswig |SetFileApisToOEM|Causes a set of Win32 file functions to use the OEM character set code page. This function is useful for 8-bit console input and output operations.
void SetFileApisToOEM(void);

// @pyswig |SetFileAttributes|Changes a file's attributes.
BOOLAPI SetFileAttributes(
    TCHAR *lpFileName,	// @pyparm <o PyUnicode>|filename||filename 
    DWORD dwFileAttributes 	// @pyparm int|newAttributes||attributes to set 
);	
 

// SetFilePointer	

// @pyswig |SetVolumeLabel|Sets a volume label for a disk drive.
BOOLAPI SetVolumeLabel(
    TCHAR *lpRootPathName,	// @pyparm <o PyUnicode>|rootPathName||address of name of root directory for volume 
    TCHAR *lpVolumeName 	// @pyparm <o PyUnicode>|volumeName||name for the volume 
   );

// @pyswig |UnlockFile|Determines the type of a file.
BOOLAPI UnlockFile(
    HANDLE hFile,	// @pyparm <o PyHANDLE>|hFile||handle of file to unlock 
    DWORD dwFileOffsetLow,	// @pyparm int|offsetLow||low-order word of lock region offset 
    DWORD dwFileOffsetHigh,	// @pyparm int|offsetHigh||high-order word of lock region offset  
    DWORD nNumberOfBytesToUnlockLow,	// @pyparm int|nNumberOfBytesToUnlockLow||low-order word of length to unlock 
    DWORD nNumberOfBytesToUnlockHigh 	// @pyparm int|nNumberOfBytesToUnlockHigh||high-order word of length to unlock 
   );

// UnlockFileEx	
// WriteFile	
// WriteFileEx 

