
#ifndef __PYWINTYPES_H__
#define __PYWINTYPES_H__

#ifdef BUILD_PYWINTYPES
#define PYWINTYPES_EXPORT __declspec(dllexport)
#else
#define PYWINTYPES_EXPORT __declspec(dllimport)
#endif

/*
** Error/Exception handling
*/
extern PYWINTYPES_EXPORT PyObject *PyWinExc_ApiError;

/* A global function that sets an API style error (ie, (code, fn, errTest)) */
PYWINTYPES_EXPORT PyObject *PyWin_SetAPIError(char *fnName, long err = 0);

/* Basic COM Exception handling.  The main COM exception object
   is actually defined here.  However, the most useful functions
   for raising the exception are still in the COM package.  Therefore,
   you can use the fn below to raise a basic COM exception - no fancy error
   messages available, just the HRESULT.  It will, however, _be_ a COM
   exception, and therefore trappable like any other COM exception
*/
extern PYWINTYPES_EXPORT PyObject *PyWinExc_COMError;
PYWINTYPES_EXPORT PyObject *PyWin_SetBasicCOMError(HRESULT hr);

/*
** String/UniCode support
*/
extern PYWINTYPES_EXPORT PyTypeObject PyUnicodeType; // the Type for PyUnicode
#define PyUnicode_Check(ob)	((ob)->ob_type == &PyUnicodeType)

// Given a PyObject (string, Unicode, etc) create a "BSTR" with the value
PYWINTYPES_EXPORT BOOL PyWinObject_AsBstr(PyObject *stringObject, BSTR *pResult, BOOL bNoneOK = FALSE, DWORD *pResultLen = NULL);
// And free it when finished.
PYWINTYPES_EXPORT void PyWinObject_FreeBstr(BSTR pResult);

// Given a PyObject (string, Unicode, etc) create a "char *" with the value
PYWINTYPES_EXPORT BOOL PyWinObject_AsString(PyObject *stringObject, char **pResult, BOOL bNoneOK = FALSE, DWORD *pResultLen = NULL);
// And free it when finished.
PYWINTYPES_EXPORT void PyWinObject_FreeString(char *pResult);

/* ANSI/Unicode Support */
/* If UNICODE defined, will be a BSTR - otherwise a char *
   Either way - PyWinObject_FreeTCHAR() must be called
*/

#ifdef UNICODE
#define PyWinObject_AsTCHAR PyWinObject_AsBstr
#define PyWinObject_FreeTCHAR PyWinObject_FreeBstr
#else
#define PyWinObject_AsTCHAR PyWinObject_AsString
#define PyWinObject_FreeTCHAR PyWinObject_FreeString
#endif


PYWINTYPES_EXPORT PyObject *PyUnicodeObject_FromString(const char *string);
PYWINTYPES_EXPORT PyObject *PyWinObject_FromBstr(const BSTR bstr, BOOL takeOwnership=FALSE);
PYWINTYPES_EXPORT PyObject *PyWinObject_FromOLECHAR(const OLECHAR * str);
PYWINTYPES_EXPORT PyObject *PyWinObject_FromOLECHAR(const OLECHAR * str, int numChars);
#define PyWinObject_FromWCHAR PyWinObject_FromOLECHAR
#ifdef UNICODE
#define PyWinObject_FromTCHAR PyWinObject_FromOLECHAR
#else
#define PyWinObject_FromTCHAR PyString_FromString
#endif

// String conversion - These must also be freed with PyWinObject_FreeString
PYWINTYPES_EXPORT BOOL PyWin_WCHAR_AsString(WCHAR *input, DWORD inLen, char **pResult);
PYWINTYPES_EXPORT BOOL PyWin_Bstr_AsString(BSTR input, char **pResult);

/*
** LARGE_INTEGER objects
*/
//AsLARGE_INTEGER takes either int, or (int, int) - NOT long yet!
PYWINTYPES_EXPORT BOOL PyWinObject_AsLARGE_INTEGER(PyObject *ob, LARGE_INTEGER *pResult);
PYWINTYPES_EXPORT BOOL PyWinObject_AsULARGE_INTEGER(PyObject *ob, ULARGE_INTEGER *pResult);
PYWINTYPES_EXPORT PyObject *PyWinObject_FromLARGE_INTEGER(LARGE_INTEGER &val);
PYWINTYPES_EXPORT PyObject *PyWinObject_FromULARGE_INTEGER(ULARGE_INTEGER &val);
#define PyLong_FromLARGE_INTEGER PyWinObject_FromLARGE_INTEGER
#define PyLong_FromULARGE_INTEGER PyWinObject_FromULARGE_INTEGER

// Should be part of the Python API?
PYWINTYPES_EXPORT PyObject *PyLong_FromTwoInts(int highPart, unsigned lowPart);


/*
** OVERLAPPED Object and API
*/
extern PYWINTYPES_EXPORT PyTypeObject PyOVERLAPPEDType; // the Type for PyOVERLAPPED
#define PyOVERLAPPED_Check(ob)	((ob)->ob_type == &PyOVERLAPPEDType)
PYWINTYPES_EXPORT BOOL PyWinObject_AsOVERLAPPED(PyObject *ob, OVERLAPPED **ppOverlapped, BOOL bNoneOK = TRUE);

// A global function that can work as a module method for making an OVERLAPPED object.
PYWINTYPES_EXPORT PyObject *PyWinMethod_NewOVERLAPPED(PyObject *self, PyObject *args);

/*
** IID/GUID support
*/
extern PYWINTYPES_EXPORT PyTypeObject PyIIDType;		// the Type for PyIID
#define PyIID_Check(ob)		((ob)->ob_type == &PyIIDType)

// Given an object repring a CLSID (either PyIID or string), fill the CLSID.
PYWINTYPES_EXPORT BOOL PyWinObject_AsIID(PyObject *obCLSID, CLSID *clsid);

// return a native PyIID object representing an IID
PYWINTYPES_EXPORT PyObject *PyWinObject_FromIID(const IID &riid);

// return a string/Unicode object representing an IID
PYWINTYPES_EXPORT PyObject *PyWinStringObject_FromIID(const IID &riid);
PYWINTYPES_EXPORT PyObject *PyWinUnicodeObject_FromIID(const IID &riid);

// A global function that can work as a module method for making an IID object.
PYWINTYPES_EXPORT PyObject *PyWinMethod_NewIID( PyObject *self, PyObject *args);

/*
** TIME support
*/
extern PYWINTYPES_EXPORT PyTypeObject PyTimeType;		// the Type for PyTime
#define PyTime_Check(ob)		((ob)->ob_type == &PyTimeType)

PYWINTYPES_EXPORT PyObject *PyWinObject_FromSYSTEMTIME(const SYSTEMTIME &t);
PYWINTYPES_EXPORT PyObject *PyWinObject_FromFILETIME(const FILETIME &t);
PYWINTYPES_EXPORT PyObject *PyWinObject_FromDATE(DATE t);
PYWINTYPES_EXPORT PyObject *PyWinTimeObject_FromLong(long t);

PYWINTYPES_EXPORT BOOL PyWinObject_AsDATE(PyObject *ob, DATE *pDate);
PYWINTYPES_EXPORT BOOL PyWinObject_AsFILETIME(PyObject *ob,	FILETIME *pDate);
PYWINTYPES_EXPORT BOOL PyWinObject_AsSYSTEMTIME(PyObject *ob, SYSTEMTIME *pDate);

// A global function that can work as a module method for making a time object.
PYWINTYPES_EXPORT PyObject *PyWinMethod_NewTime( PyObject *self, PyObject *args);

/*
** SECURITY_ATTRIBUTES support
*/
extern PYWINTYPES_EXPORT PyTypeObject PySECURITY_ATTRIBUTESType;
#define PySECURITY_ATTRIBUTES_Check(ob)		((ob)->ob_type == &PySECURITY_ATTRIBUTESType)

PYWINTYPES_EXPORT PyObject *PyWinMethod_NewSECURITY_ATTRIBUTES(PyObject *self, PyObject *args);
PYWINTYPES_EXPORT BOOL PyWinObject_AsSECURITY_ATTRIBUTES(PyObject *ob, SECURITY_ATTRIBUTES **ppSECURITY_ATTRIBUTES, BOOL bNoneOK = TRUE);


/*
** SID support
*/
extern PYWINTYPES_EXPORT PyTypeObject PySIDType;
#define PySID_Check(ob)		((ob)->ob_type == &PySIDType)

PYWINTYPES_EXPORT PyObject *PyWinMethod_NewSID(PyObject *self, PyObject *args);
PYWINTYPES_EXPORT BOOL PyWinObject_AsSID(PyObject *ob, PSID *ppSID, BOOL bNoneOK = FALSE);

/*
** ACL support
*/
extern PYWINTYPES_EXPORT PyTypeObject PyACLType;
#define PyACL_Check(ob)		((ob)->ob_type == &PyACLType)

PYWINTYPES_EXPORT PyObject *PyWinMethod_NewACL(PyObject *self, PyObject *args);
PYWINTYPES_EXPORT BOOL PyWinObject_AsACL(PyObject *ob, PACL *ppACL, BOOL bNoneOK = FALSE);

/*
** Win32 HANDLE wrapper - any handle closable by "CloseHandle()"
*/
extern PYWINTYPES_EXPORT PyTypeObject PyHANDLEType; // the Type for PyHANDLE
#define PyHANDLE_Check(ob)	((ob)->ob_type == &PyHANDLEType)

PYWINTYPES_EXPORT BOOL PyWinObject_AsHANDLE(PyObject *ob, HANDLE *pRes, BOOL bNoneOK = FALSE);
PYWINTYPES_EXPORT PyObject *PyWinObject_FromHANDLE(HANDLE h);

// A global function that can work as a module method for making a HANDLE object.
PYWINTYPES_EXPORT PyObject *PyWinMethod_NewHANDLE( PyObject *self, PyObject *args);

// A global function that does the right thing wrt closing a "handle".
// The object can be either a PyHANDLE or an integer.
// If result is FALSE, a Python error is all setup (cf PyHANDLE::Close(), which doesnt set the Python error)
PYWINTYPES_EXPORT BOOL PyWinObject_CloseHANDLE(PyObject *obHandle);


/*
** Other Utilities
*/

// ----------------------------------------------------------------------
// WARNING - NEVER EVER USE new() ON THIS CLASS
// This class can be used as a local variable, typically in a Python/C
// function, and can be passed whereever a TCHAR/WCHAR is expected.
// Typical Usage:
// PyWin_AutoFreeBstr arg;
// PyArg_ParseTuple("O", &obStr);
// PyWin_PyObjectAsAutoFreeBstr(obStr, &arg);
// CallTheFunction(arg); // Will correctly pass BSTR/OLECHAR
// -- when the function goes out of scope, the string owned by "arg" will
// -- automatically be freed.
// ----------------------------------------------------------------------
class PyWin_AutoFreeBstr {
public:
	PyWin_AutoFreeBstr( BSTR bstr = NULL ) : m_bstr(bstr) {return;}
	~PyWin_AutoFreeBstr() {SysFreeString(m_bstr);}
	void SetBstr( BSTR bstr ) {
		SysFreeString(m_bstr);
		m_bstr = bstr;
	}
	operator BSTR() {return m_bstr;}
private:
	BSTR m_bstr;
};

inline BOOL PyWinObject_AsAutoFreeBstr(PyObject *stringObject, PyWin_AutoFreeBstr *pResult, BOOL bNoneOK = FALSE)
{
	BSTR bs;
	if (!PyWinObject_AsBstr(stringObject, &bs, bNoneOK))
		return NULL;
	pResult->SetBstr(bs);
	return TRUE;
}


// ----------------------------------------------------------------------
//
// THREAD MANAGEMENT
//

// ### need to rename the PYCOM_ stuff soon...

PYWINTYPES_EXPORT void PyWin_AcquireGlobalLock(void);
PYWINTYPES_EXPORT void PyWin_ReleaseGlobalLock(void);

#ifndef FORCE_NO_FREE_THREAD
# ifdef WITH_FREE_THREAD
#  define PYCOM_USE_FREE_THREAD
# endif
#endif
#ifdef PYCOM_USE_FREE_THREAD
# include <threadstate.h>
# define DLLAcquirePythonLock()
# define DLLReleasePythonLock()
# define AllocThreadState()			BOOL PyCom___ThreadLockCreated = PyThreadState_Ensure()
# define FreeThreadState()			if ( PyCom___ThreadLockCreated ) PyThreadState_Free(); else PyThreadState_ClearExc()
# define PyCom_EnterPython()		AllocThreadState()
# define PyCom_LeavePython()		FreeThreadState()
#else
# define DLLAcquirePythonLock()		DLLAcquireGlobalLock()
# define DLLReleasePythonLock()		DLLReleaseGlobalLock()
# define AllocThreadState()
# define FreeThreadState()
# define PyCom_EnterPython()		DLLAcquirePythonLock()
# define PyCom_LeavePython()		DLLReleasePythonLock()
#endif

// Helper class for Enter/Leave Python
// NEVER new one of these objects - only use on the stack!
class CEnterLeavePython {
public:
	CEnterLeavePython() {
#ifdef PYCOM_USE_FREE_THREAD
		created = PyThreadState_Ensure();
#else
		PyWin_AcquireGlobalLock();
#endif
	}
	~CEnterLeavePython() {
#ifdef PYCOM_USE_FREE_THREAD
		if ( created )
			PyThreadState_Free();
		else
			PyThreadState_ClearExc();
	}
private:
	BOOL created;
#else
		PyWin_ReleaseGlobalLock();
	}
#endif
};


#endif // __PYWINTYPES_H__


