# Utility functions for Python programs which run as services...
#
# Note that most utility functios here will raise win32api.error's when
# things go wrong - eg, not enough permissions to hit the registry etc.

import win32service, win32api, regutil, win32con, winerror
import sys, string, pywintypes

error = "Python Service Utility Error"

def LocatePythonServiceExe():
	# Get the full path to the executable from the registry
	import regutil
	try:
		return regutil.GetRegistryDefaultValue("Software\\Python\\PythonService\\%s" % (sys.winver))
	except win32api.error:
		msg = "PythonService.exe is not correctly registered\nPlease locate and run PythonService.exe, and it will self-register\nThen run this service registration process again."
		raise error, msg

def InstallService(pythonClassString, serviceName, displayName, startType = None, errorControl = None, bRunInteractive = 0, serviceDeps = None, userName = None, password = None):
	# Handle the default arguments.
	if startType is None:
		startType = win32service.SERVICE_DEMAND_START
		if bRunInteractive:
			startType = startType | win32service.SERVICE_INTERACTIVE_PROCESS
	if errorControl is None:
		errorControl = win32service.SERVICE_ERROR_NORMAL

	exeName = LocatePythonServiceExe()
	hscm = win32service.OpenSCManager(None,None,win32service.SC_MANAGER_ALL_ACCESS)
	try:
		hs = win32service.CreateService(hscm,
					serviceName,
					displayName,
					win32service.SERVICE_ALL_ACCESS,         # desired access
		            win32service.SERVICE_WIN32_OWN_PROCESS,  # service type
		            startType,
		            errorControl,       # error control type
		            exeName,
		            None,
		            0,
		            serviceDeps,
		            userName,
		            password)
		win32service.CloseServiceHandle(hs)
	finally:
		win32service.CloseServiceHandle(hscm)
	InstallPythonClassString(pythonClassString, serviceName)
	
def InstallPythonClassString(pythonClassString, serviceName):
	# Now setup our Python specific entries.
	key = win32api.RegCreateKey(win32con.HKEY_LOCAL_MACHINE, "System\\CurrentControlSet\\Services\\%s\\PythonClass" % serviceName)
	try:
		win32api.RegSetValue(key, None, win32con.REG_SZ, pythonClassString);
	finally:
		win32api.RegCloseKey(key)

# Utility functions for Services, to allow persistant properties.
def SetServiceCustomOption(serviceName, option, value):
	try:
		serviceName = serviceName._svc_name_
	except AttributeError:
		pass
	key = win32api.RegCreateKey(win32con.HKEY_LOCAL_MACHINE, "System\\CurrentControlSet\\Services\\%s\\PythonOptions" % serviceName)
	try:
		win32api.RegSetValue(key, option, win32con.REG_SZ, value);
	finally:
		win32api.RegCloseKey(key)

def GetServiceCustomOption(serviceName, option):
	# First param may also be a service class/instance.
	# This allows services to pass "self"
	try:
		serviceName = serviceName._svc_name_
	except AttributeError:
		pass
	key = win32api.RegCreateKey(win32con.HKEY_LOCAL_MACHINE, "System\\CurrentControlSet\\Services\\%s\\PythonOptions" % serviceName)
	try:
		return win32api.RegQueryValue(key, option)
	finally:
		win32api.RegCloseKey(key)


def RemoveService(serviceName):
	hscm = win32service.OpenSCManager(None,None,win32service.SC_MANAGER_ALL_ACCESS)
	try:
		hs = win32service.OpenService(hscm, serviceName, win32service.SERVICE_ALL_ACCESS)
		win32service.DeleteService(hs)
	finally:
		win32service.CloseServiceHandle(hscm)

def StopService(serviceName):
	hscm = win32service.OpenSCManager(None,None,win32service.SC_MANAGER_ALL_ACCESS)
	try:

		hs = win32service.OpenService(hscm, serviceName, win32service.SERVICE_ALL_ACCESS)
		try:
			status = win32service.ControlService(hs, win32service.SERVICE_CONTROL_STOP)
		finally:
			win32service.CloseServiceHandle(hs)
	finally:
		win32service.CloseServiceHandle(hscm)
	return status

def StartService(serviceName, args = None):
	hscm = win32service.OpenSCManager(None,None,win32service.SC_MANAGER_ALL_ACCESS)
	try:

		hs = win32service.OpenService(hscm, serviceName, win32service.SERVICE_ALL_ACCESS)
		try:
			win32service.StartService(hs, args)
		finally:
			win32service.CloseServiceHandle(hs)
	finally:
		win32service.CloseServiceHandle(hscm)

def GetServiceClassString(cls, argv = None):
		if argv is None:
			argv = sys.argv
		import pickle, os
		modName = pickle.whichmodule(cls)
		if modName == '__main__':
			try:
				fname = win32api.GetFullPathName(argv[0])
			except win32api.error:
				raise error, "Could not resolve the path name '%s' to a full path" % (argv[0])
			modName = os.path.splitext(fname)[0]
		return modName + "." + cls.__name__

def usage():
	try:
		import os
		fname = os.path.split(sys.argv[0])[1]
	except:
		fname = sys.argv[0]
	print "Usage: '%s [options] install|remove|start [...]|stop|restart [...]|debug [...]'" % fname
	print "Options for 'install' command only:"
	print " --username domain\username : The Username the service is to run under"
	print " --password password : The password for the username"
	print " --startup [manual|auto|disabled] : How the service starts, default = manual"
	sys.exit(1)

def HandleCommandLine(cls, serviceClassString = None, argv = None, customInstallOptions = "", customOptionHandler = None):
	"""Utility function allowing services to process the command line.
	
	Allows standard commands such as 'start', 'stop', 'debug', 'install' etc.
	
	Install supports 'standard' command line options prefixed with '--', such as
	--username, --password, etc.  In addition,
	the function allows custom command line options to be handled by the calling function.
	"""
	
	if argv is None: argv = sys.argv

	if len(argv)==0:
		usage()

	serviceName = cls._svc_name_
	serviceDisplayName = cls._svc_display_name_
	if serviceClassString is None:
		serviceClassString = GetServiceClassString(cls)

	# First me process all arguments which require access to the
	# arg list directly
	if argv[1]=="start":
		print "Starting service %s" % (serviceName)
		try:		
			StartService(serviceName, argv[2:])
		except win32service.error, (hr, fn, msg):
			print "Error starting service: %s" % msg

	elif argv[1]=="restart":
		print "Restarting service %s" % (serviceName)
		try:
			StopService(serviceName)
		except pywintypes.error, (hr, name, msg):
			# Allow only "service not running" error
			if hr!=winerror.ERROR_SERVICE_NOT_ACTIVE:
				raise win32service.error, (hr, name, msg)

		# Give it a few goes, as the service may take tome to stop
		for i in range(10):
			try:
				StartService(serviceName, argv[2:])
				break
			except pywintypes.error, (hr, name, msg):
				if hr!=winerror.ERROR_SERVICE_ALREADY_RUNNING:
					raise win32service.error, (hr, name, msg)
				win32api.Sleep(500)
		else:
			print "Gave up waiting for the old service to stop!"
	
	elif argv[1]=="debug":
		svcArgs = string.join(sys.argv[2:])
		print "Executing Python Service Mananger for debug session"
		import os
		exeName = LocatePythonServiceExe()
		try:
			os.system("%s -debug %s %s" % (exeName, serviceName, svcArgs))
		# ^C is used to kill the debug service.  Sometimes Python also gets
		# interrupted - ignore it...
		except KeyboardInterrupt:
			pass
	else:
		# Pull apart the command line
		import getopt
		try:
			opts, args = getopt.getopt(argv[1:], customInstallOptions,["password=","username=","startup="])
		except getopt.error, details:
			print details
			usage()
		userName = None
		password = None
		startup = None
		for opt, val in opts:
			if opt=='--username':
				userName = val
			elif opt=='--password':
				password = val
			elif opt=='--startup':
				map = {"manual": win32service.SERVICE_DEMAND_START, "auto" : win32service.SERVICE_AUTO_START, "disabled": win32service.SERVICE_DISABLED}
				try:
					startup = map[string.lower(val)]
				except KeyError:
					print "'%s' is not a valid startup option" % val
		if len(args)<>1:
			usage()
		arg=args[0]
		if arg=="install":
			try:
				serviceDeps = cls._svc_deps_
			except AttributeError:
				serviceDeps = None
			print "Installing service %s to Python class %s" % (serviceName,serviceClassString)
			# Note that we install the service before calling the custom option
			# handler, so if the custom handler fails, we have an installed service (from NT's POV)
			# but is unlikely to work, as the Python code controlling it failed.  Therefore
			# we remove the service if the first bit works, but the second doesnt!
			try:
				InstallService(serviceClassString, serviceName, serviceDisplayName, serviceDeps = serviceDeps, startType=startup, userName=userName,password=password)
				if customOptionHandler:
					apply( customOptionHandler, (opts,) )
				print "Service installed"
			except win32service.error, (hr, fn, msg):
				print "Error installing service: %s" % msg
			except ValueError, msg: # Can be raised by custom option handler.
				print "Error installing service: %s" % str(msg)
				# As we failed here, remove the service, so the next installation
				# attempt works.
				try:
					RemoveService(serviceName)
				except win32api.error:
					print "Warning - could not remove the partially installed service."

			
		elif arg=="remove":
			print "Removing service %s" % (serviceName)
			try:
				RemoveService(serviceName)
				print "Service removed"
			except win32service.error, (hr, fn, msg):
				print "Error removing service: %s" % msg
		elif arg=="stop":
			print "Stopping service %s" % (serviceName)
			try:
				StopService(serviceName)
			except win32service.error, (hr, fn, msg):
				print "Error stopping service: %s" % msg
		else:
			print "Unknown command - '%s'" % arg
			usage()

#
# Useful base class to build services from.
#
class ServiceFramework:
	# _svc_name = The service name
	# _svc_display_name = The service display name
	def __init__(self, args):
		import servicemanager
		self.ssh = servicemanager.RegisterServiceCtrlHandler(args[0], self.ServiceCtrlHandler)
		self.checkPoint = 0

	def GetAcceptedControls(self):
		# Setup the service controls we accept based on our attributes
		accepted = 0
		if hasattr(self, "SvcStop"): accepted = accepted | win32service.SERVICE_ACCEPT_STOP
		if hasattr(self, "SvcPause") and hasattr(self, "SvcContinue"):
			accepted = accepted | win32service.SERVICE_ACCEPT_PAUSE_CONTINUE
		if hasattr(self, "SvcShutdown"): accepted = accepted | win32service.SERVICE_ACCEPT_SHUTDOWN
		return accepted
			
	def ReportServiceStatus(self, serviceStatus, waitHint = 5000, win32ExitCode = 0, svcExitCode = 0):
		if self.ssh is None: # Debugging!
			return
		if serviceStatus == win32service.SERVICE_START_PENDING:
			accepted = 0
		else:
			accepted = self.GetAcceptedControls()

		if serviceStatus in [win32service.SERVICE_RUNNING,  win32service.SERVICE_STOPPED]:
			checkPoint = 0
		else:
			self.checkPoint = self.checkPoint + 1
			checkPoint = self.checkPoint

		# Now report the status to the control manager
		status = (win32service.SERVICE_WIN32_OWN_PROCESS,
		         serviceStatus,
		         accepted, # dwControlsAccepted,
		         win32ExitCode, # dwWin32ExitCode; 
		         svcExitCode, # dwServiceSpecificExitCode; 
		         checkPoint, # dwCheckPoint; 
		         waitHint)
		win32service.SetServiceStatus( self.ssh, status)
  			
  	def SvcInterrogate(self):
  		# Assume we are running, and everyone is happy.
  		self.ReportServiceStatus(win32service.SERVICE_RUNNING)

  	def SvcOther(self, control):
  		print "Unknown control status - %d" % control

	def ServiceCtrlHandler(self, control):
		if control==win32service.SERVICE_CONTROL_STOP:
			self.SvcStop()
		elif control==win32service.SERVICE_CONTROL_PAUSE:
			self.SvcPause()
		elif control==win32service.SERVICE_CONTROL_CONTINUE:
			self.SvcContinue()
		elif control==win32service.SERVICE_CONTROL_INTERROGATE:
			self.SvcInterrogate()
		elif control==win32service.SERVICE_CONTROL_SHUTDOWN:
			self.SvcShutdown()
		else:
			self.SvcOther(control)

	def SvcRun(self):
		self.ReportServiceStatus(win32service.SERVICE_RUNNING)
		self.SvcDoRun()
		self.ReportServiceStatus(win32service.SERVICE_STOP_PENDING)


