# Misc
import traceback, string, os, sys, time

# Stuff for the event log
import win32evtlogutil, win32api, winerror, pywintypes
# Get the stuff we need for MAPI
import olemsg32, win32com.client, pythoncom

error = "Event Log Checker Error"
configError = "Event Log Checker config. error"
actionError = "Event Log Checker action failed"

try:
	iniPath = os.path.split(__file__)[0]	# Imported as other module
except NameError:
	iniPath = os.path.split(sys.argv[0])[0]	# run as script
	
iniName = os.path.join(iniPath, "EvtLogFilters.ini")

class EventLogFilter:
	def __init__(self, name, optionsDict):
		self.name = name
		self.eventCodes = None
		self.eventFacilities = None
		self.eventSeverities = None
		self.eventTypes = None
		self.sources = None
		self.categories = None
		self.actions = None
		self.logType = optionsDict['logType'] # Special case - not a list
		self._setoptions(['eventCodes','eventFacilities','eventSeverities','eventTypes', 'sources', 'categories','actions'],optionsDict)

	def _setoptions(self, options, dict):
		for option in options:
			if dict.has_key(option):
				val = dict[option]
				if type(val) <> type([]):
					val = [val]
				setattr(self, option, val)
	
	def _CheckList(self, item, list):
		return list is None or item in list
		
	def Match(self, eventLogRecord):
		eventId = eventLogRecord.EventID
		if not self._CheckList(winerror.HRESULT_CODE(eventId), self.eventCodes): return 0
		if not self._CheckList(winerror.HRESULT_FACILITY(eventId), self.eventFacilities): return 0
		if not self._CheckList(winerror.HRESULT_SEVERITY(eventId), self.eventSeverities): return 0
		eventType = eventLogRecord.EventType
		if not self._CheckList(eventType, self.eventTypes): return 0
		source = eventLogRecord.SourceName
		if not self._CheckList(source, self.sources): return 0
		category = eventLogRecord.EventCategory
		if not self._CheckList(category, self.categories): return 0
		return 1

class EventLogChecker:
	def __init__(self, filters, actions, logType, server = None):
		self.server = server
		self.actions = actions
		self.logType = logType
		self.filters = {}
		for filter in filters:
			self.filters[filter] = []
	def Go(self):
		win32evtlogutil.FeedEventLogRecords(self.EventItemFeeder, self.server, self.logType)
		return self

	def EventItemFeeder(self, item):
		for filter in self.filters.keys():
			if filter.Match(item):
				self.filters[filter].append(item)
	
	def Process(self):
		for filter, items in self.filters.items():
			if items:
				try:
					for action in filter.actions:
						actionInst = self.actions[action]
						try:
							actionInst.TakeAction(items, filter, self.server)
						except SystemError: # actionError, msg:
							self.ActionFailed(msg, items, filter, actionInst)
						except SystemError:
							msg = "%s: %s" % (sys.exc_type, sys.exc_value)
							self.ActionFailed(msg, items, filter, actionInst)
				finally:
					self.filters[filter] = []

	def ActionFailed(self, msg, items, filter, action):
		print "Action %s failed\n%s" % (action.actionName, `msg`)
		
class ActionTaker:
	def __init__(self, actionName):
		self.actionName = actionName
	def FormatSummary(self):
		if self.server is None: 
			server = win32api.GetComputerName()
		else:
			server = self.server
		return "%s on %s" % (self.filter.name, server)

	def FormatText(self):
		minTimeWritten = maxTimeWritten = 0
		text = ""
		if len(self.items)>1:
			for item in self.items:
				if minTimeWritten==0 or item.TimeWritten < minTimeWritten:
					minTimeWritten = item.TimeWritten
				if item.TimeWritten > maxTimeWritten:
					maxTimeWritten = item.TimeWritten
			text = "There were %d events detected\n" % len(self.items)
			if minTimeWritten:
				text = text + "The earliest event was written at " + time.strftime("%#c", time.localtime(int(minTimeWritten))) + "\n"
			if maxTimeWritten:
				text = text + "The latest event was written at " + time.strftime("%#c", time.localtime(int(maxTimeWritten))) + "\n"
			text = text + "The last event entry is:\n"
		else:
			item = self.items[-1]
		text = "The event was generated at " + time.strftime("%#c", time.localtime(int(item.TimeGenerated))) + "\n"
		text = text + "Source: %s, Event ID: %s\n" % (item.SourceName, item.EventID)
		text = text + "Type: %s, Category: %s\n" % (item.EventType, item.EventCategory)
		text = text + str(win32evtlogutil.SafeFormatMessage(item))
		return text
	def TakeAction(self, items, filter, server = None):
		self.items = items
		self.filter = filter
		self.server = server
		self.DoTakeAction()

class NetSendActionTaker(ActionTaker):
	def LoadIniSettings(self, section, iniName):
		self.to = win32api.GetProfileVal(section, "To", "", iniName)
		if not self.to:
			raise configError, "The NetSend Action requires a 'to' setting"
	
	def DoTakeAction(self):
		import win32net
		try:
			win32net.NetMessageBufferSend(None, self.to, None, self.FormatSummary() + "\n\n" + self.FormatText())
		except win32net.error, details:
			raise actionError, "NetSend to %s failed - %s" % (self.to, `details`)
		return 1
	
class MailActionTaker(ActionTaker):
	def HandleMailComError(self, function, com_exc_details):
		raise actionError, 	"Error %s\n%s" % (function, `com_exc_details[2]`)
	def LoadIniSettings(self, section, iniName):
		self.profile = win32api.GetProfileVal(section, "Profile", "", iniName)
		self.password = win32api.GetProfileVal(section, "Password", "", iniName)
		self.to = win32api.GetProfileVal(section, "To", "", iniName)
		if not self.profile or not self.to:
			raise configError, "The Mail Action requires 'profile' and 'to' settings"
	
	def SubjectLine(self):
		return "EVTLOG: %s" % self.FormatSummary()


class ActiveMessagingMailActionTaker(MailActionTaker):
	def DoTakeAction(self):
		try:
			s = win32com.client.Dispatch("MAPI.Session")
			s.Logon(ProfileName = self.profile, ProfilePassword = self.password, ShowDialog=0,NoMail=1)
		except pythoncom.com_error, details:
			self.HandleMailComError("establishing MAPI session", details)
		try:
			try:
				m = s.Outbox.Messages.Add()
				m.Subject = self.SubjectLine()
				m.Text = self.FormatText()
			except pythoncom.com_error, details:
				self.HandleMailComError("creating message body", details)
				return 0
			try:
				recip = m.Recipients.Add()
				recip.Name = self.to
				recip.Resolve(ShowDialog=0)
			except pythoncom.com_error, details:
				self.HandleMailComError("resolving mail recipients", details)
			try:
				m.Send(ShowDialog=0)
				s.DeliverNow()
			except pythoncom.com_error, details:
				self.HandleMailComError("sending message", details)
		finally:
			s.Logoff()
		return 1

from win32com.mapi import mapi, mapitags, mapiutil

class MAPIMailActionTaker(MailActionTaker):
	def SendMessage(self, session):
		try:
			privateStore = mapi.HrOpenExchangePrivateStore(session)
			hr, data = privateStore.GetProps((mapitags.PR_IPM_OUTBOX_ENTRYID,),0)
			outBox = session.OpenEntry(data[0][1], None, mapi.MAPI_BEST_ACCESS)
			msg = outBox.CreateMessage(None,0)
		except pythoncom.com_error, details:
			self.HandleMailComError("creating message", details)
			
		props = {}
		props[mapitags.PR_SUBJECT] = self.SubjectLine()
		props[mapitags.PR_BODY] = self.FormatText()
		print "Props are", props
		
		recipProps = (
		               ( mapitags.PR_DISPLAY_NAME, self.to),
		               ( mapitags.PR_RECIPIENT_TYPE, mapi.MAPI_TO ),
		             )
		try:		             
			ab = session.OpenAddressBook(0,None,0)
			recipProps = ab.ResolveName(0,mapi.MAPI_UNICODE,"", (recipProps,))
		except pythoncom.com_error, details:
			self.HandleMailComError("resolving mail recipients", details)

		try:
			mapiutil.SetProperties(msg, props)
			msg.ModifyRecipients(mapi.MODRECIP_ADD, recipProps)
			msg.SubmitMessage(0)
		except pythoncom.com_error, details:
			self.HandleMailComError("sending the message", details)
	
	def DoTakeAction(self):
		mapi.MAPIInitialize(None)
		try:
			try:
				s = mapi.MAPILogonEx(0,self.profile, self.password,mapi.MAPI_NO_MAIL | mapi.MAPI_NEW_SESSION )
			except pythoncom.com_error, details:
				self.HandleMailComError("establishing MAPI session", details)
			try:
				try:
					self.SendMessage(s)
				except:
					print "SendMessage failed!"
				sys.exc_type = sys.exc_value = sys.exc_traceback = None
			finally:
				s.Logoff(0,0,0)
			s = None # AHH - needed to stop death!
			print "Interface is", pythoncom._GetInterfaceCount()
		finally:
			mapi.MAPIUninitialize()

def LoadFilterOpt(dict, filterName, optName, iniName, bIsInt = 0):
	values = win32api.GetProfileVal(filterName, optName, "", iniName)
	if not values:
		return
	ret = []
	for value in string.split(values,","):
		if bIsInt:
			try:
				value = string.atoi(value)
			except ValueError:
				raise configError, "The filter option '%s' must be a list of integers" % optName
		else:
			value = string.strip(value)
				
		ret.append(value)
	dict[optName] = ret

def LoadFilterUnicodeOpt(dict, filterName, optName, iniName):
	values = win32api.GetProfileVal(filterName, optName, "", iniName)
	if not values:
		return
	ret = []
	for value in string.split(values,","):
		value = pywintypes.Unicode(string.strip(value))
		ret.append(value)
	dict[optName] = ret

def LoadFilters():
	filterNames = win32api.GetProfileSection("Filters", iniName)
	filters = []
	for filter in filterNames:
		optDict = {}
		LoadFilterOpt(optDict, filter, "eventCodes", iniName, 1)
		LoadFilterOpt(optDict, filter, "eventSeverities", iniName, 1)
		LoadFilterOpt(optDict, filter, "eventFacilities", iniName, 1)
		LoadFilterUnicodeOpt(optDict, filter, "sources", iniName)
		LoadFilterUnicodeOpt(optDict, filter, "eventTypes", iniName)
		LoadFilterUnicodeOpt(optDict, filter, "categories", iniName)
		LoadFilterOpt(optDict, filter, "actions", iniName)
		optDict['logType'] = win32api.GetProfileVal(filter, 'logType', "Application", iniName)

		filters.append(EventLogFilter(filter, optDict))
	return filters

def LoadActions():
	actionNames = win32api.GetProfileSection("Actions", iniName)
	actionTakers = {}
	for actionEntry in actionNames:
		# Indirect to get the actual action definition
		action = win32api.GetProfileVal(actionEntry, "action", "", iniName)
		if action=="Mail":
			actionClass = MAPIMailActionTaker
		elif action=="OLEMSG Mail":
			actionClass = ActiveMessagingMailActionTaker
		elif action=="Net Send":
			actionClass = NetSendActionTaker
		elif action=="":
			raise configError, "The action entry '%s' does not have an action listed." % actionEntry
		else:
			raise configError, "The action '%s' in entry '%s' is not recognised!" % (action, actionEntry)
		actionInst = actionClass(action)
		actionTakers[actionEntry] = actionInst
		actionInst.LoadIniSettings(actionEntry, iniName)
	return actionTakers

def CreateEventLogCheckers(checkerClass = None):
	filters = LoadFilters()
	actions = LoadActions()

	logTypes = {'Application':[], 'System':[], 'Security':[]}

	# Check all actions are known
	if len(filters)==0:
		raise configError, "There are no filters setup"
	
	for filter in filters:
		if not filter.actions:
			raise configError, "The filter '%s' does not have a valid action" % (filter.name)
		for action in filter.actions:
			if not actions.has_key(action):
				raise configError, "The action '%s' for filter '%s' is invalid" % (action, filter.name)

		try:
			logTypes[filter.logType].append(filter)
		except KeyError:
			raise configError, "The filter '%s' has an invalid 'LogType' entry" % (filter.name)

	servers = win32api.GetProfileSection("Servers", iniName)
	if not servers:
		servers = [None]
	rc = []
	if checkerClass is None:
		checkerClass = EventLogChecker
	for logType, filters in logTypes.items():
		for server in servers:
			rc.append(checkerClass(filters, actions, logType, server))
	return rc

if __name__=='__main__':
	for checker in CreateEventLogCheckers():
		try:
			checker.Go().Process()
		except win32evtlogutil.error, details:
			print "Error checking event log for '%s' - %s" % (server, details[2])
