//  MODULE:   PythonService.exe
//
//  PURPOSE:  An executable that hosts Python services.
//
// @doc

#include "windows.h"
#include "objbase.h"
#include "Python.h"
#undef main
#include "tchar.h"

#include "PythonServiceMessages.h"

#include "PyWinTypes.h"

SERVICE_STATUS          ssStatus;       // current status of the service
SERVICE_STATUS_HANDLE   sshStatusHandle = 0;
DWORD                   dwErr = 0;
BOOL                    bServiceDebug = FALSE;
TCHAR                   szErr[256];

#define SZ_EVENT_SOURCE_NAME L"PythonServiceManager"

// internal function prototypes
VOID WINAPI service_main(DWORD dwArgc, LPTSTR *lpszArgv);
BOOL WINAPI ControlHandler ( DWORD dwCtrlType );
static void ReportAPIError(DWORD msgCode, DWORD errCode = 0);
static void ReportPythonError(DWORD);
static void ReportError(DWORD, LPCTSTR *inserts = NULL, WORD errorType = EVENTLOG_ERROR_TYPE);
BOOL RegisterPythonServiceExe(void);
BOOL WINAPI DebugControlHandler ( DWORD dwCtrlType );

BOOL LocatePythonServiceClass( DWORD dwArgc, LPTSTR *lpszArgv, PyObject **result );

SERVICE_STATUS errorStatus = {
	SERVICE_WIN32_OWN_PROCESS,
	SERVICE_STOP_PENDING,
    0, // dwControlsAccepted,
    0, // dwWin32ExitCode; 
    0, // dwServiceSpecificExitCode; 
    0, // dwCheckPoint; 
    5000 };

SERVICE_STATUS startingStatus = {
	SERVICE_WIN32_OWN_PROCESS,
	SERVICE_START_PENDING,
    0, // dwControlsAccepted,
    0, // dwWin32ExitCode; 
    0, // dwServiceSpecificExitCode; 
    0, // dwCheckPoint; 
    5000 };

SERVICE_STATUS stoppedStatus = {
	SERVICE_WIN32_OWN_PROCESS,
	SERVICE_STOPPED,
    0, // dwControlsAccepted,
    0, // dwWin32ExitCode; 
    0, // dwServiceSpecificExitCode; 
    0, // dwCheckPoint; 
    0 };

// The built-in Python module.
static PyObject *servicemanager_module_error;
static PyObject *g_obServiceCtrlHandler = NULL;

VOID WINAPI service_ctrl(DWORD dwCtrlCode)
{
	if (g_obServiceCtrlHandler==NULL) { // Python is in error.
		if (!bServiceDebug)
			SetServiceStatus( sshStatusHandle, &errorStatus );
		return;
	}
	PyObject *args = Py_BuildValue("(l)", dwCtrlCode);
	PyObject *result = PyObject_CallObject(g_obServiceCtrlHandler, args);
	Py_XDECREF(args);
	if (result==NULL)
		ReportPythonError(PYS_E_SERVICE_CONTROL_FAILED);
	else
		Py_DECREF(result);
}

static PyObject *DoLogMessage(WORD errorType, PyObject *obMsg)
{
	BSTR msg;
	if (!PyWinObject_AsBstr(obMsg, &msg))
		return NULL;
	DWORD errorCode = errorType==EVENTLOG_ERROR_TYPE ? PYS_E_GENERIC_ERROR : PYS_E_GENERIC_WARNING;
	LPCTSTR inserts[] = {msg, NULL};
	Py_BEGIN_ALLOW_THREADS
	ReportError(errorCode, inserts, errorType);
	SysFreeString(msg);
	Py_END_ALLOW_THREADS
	Py_INCREF(Py_None);
	return Py_None;
}

// @pymethod |servicemanager|LogInfoMsg|Logs a generic informational message to the event log
static PyObject *PyLogInfoMsg(PyObject *self, PyObject *args)
{
	PyObject *obMsg;
	// @pyparm <o PyUnicode>|msg||The message to write.
	if (!PyArg_ParseTuple(args, "O:LogInfoMsg", &obMsg))
		return NULL;
	return DoLogMessage(EVENTLOG_INFORMATION_TYPE, obMsg);
}

// @pymethod |servicemanager|LogWarningMsg|Logs a generic warning message to the event log
static PyObject *PyLogWarningMsg(PyObject *self, PyObject *args)
{
	PyObject *obMsg;
	// @pyparm <o PyUnicode>|msg||The message to write.
	if (!PyArg_ParseTuple(args, "O:LogWarningMsg", &obMsg))
		return NULL;
	return DoLogMessage(EVENTLOG_WARNING_TYPE, obMsg);
}

// @pymethod |servicemanager|LogErrorMsg|Logs a generic error message to the event log
static PyObject *PyLogErrorMsg(PyObject *self, PyObject *args)
{
	// @pyparm <o PyUnicode>|msg||The message to write.
	PyObject *obMsg;
	if (!PyArg_ParseTuple(args, "O:LogErrorMsg", &obMsg))
		return NULL;
	return DoLogMessage(EVENTLOG_ERROR_TYPE, obMsg);
}

// @pymethod int/None|servicemanager|RegisterServiceCtrlHandler|Registers the Python service control handler function.
static PyObject *PyRegisterServiceCtrlHandler(PyObject *self, PyObject *args)
{
	PyObject *nameOb, *obCallback;
	// @pyparm <o PyUnicode>|serviceName||The name of the service.  This is provided in args[0] of the service class __init__ method.
	// @pyparm object|callback||The Python function that performs as the control function.  This will be called with an integer status argument.
	if (!PyArg_ParseTuple(args, "OO", &nameOb, &obCallback))
		return NULL;
	if (!PyCallable_Check(obCallback)) {
		PyErr_SetString(PyExc_TypeError, "Second argument must be a callable object");
		return NULL;
	}
	BSTR bstrName;
	if (!PyWinObject_AsBstr(nameOb, &bstrName))
		return NULL;
	Py_XDECREF(g_obServiceCtrlHandler);
	g_obServiceCtrlHandler = obCallback;
	Py_INCREF(obCallback);
	if (bServiceDebug) { // If debugging, get out now, and give None back.
		Py_INCREF(Py_None);
		return Py_None;
	}
	sshStatusHandle = RegisterServiceCtrlHandler(bstrName, service_ctrl);
	SysFreeString(bstrName);
	PyObject *rc;
	if (sshStatusHandle==0) {
		Py_DECREF(obCallback);
		obCallback = NULL;
		rc = PyWin_SetAPIError("RegisterServiceCtrlHandler");
	} else {
		rc = PyInt_FromLong((long)sshStatusHandle);
	}
	return rc;
	// @rdesc If the service manager is in debug mode, this returns None, indicating
	// there is no service control manager handle, otherwise the handle to the Win32 service manager.

}

// @pymethod |servicemanager|CoInitializeEx|Initialize OLE with additional options.
static PyObject *PyCoInitializeEx(PyObject *self, PyObject *args)
{
	DWORD flags;
	if (!PyArg_ParseTuple(args, "l:CoInitializeEx", &flags))
		return NULL;
	HRESULT hr = CoInitializeEx(NULL, flags);
	return PyInt_FromLong(hr);
}

// @pymethod |servicemanager|CoUninitialize|Unitialize OLE
static PyObject *PyCoUninitialize(PyObject *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ":CoUninitialize"))
		return NULL;
	CoUninitialize();
	Py_INCREF(Py_None);
	return Py_None;
}

// @pymethod |servicemanager|Debugging|Indicates if the service is running in debug mode.
static PyObject *PyDebugging(PyObject *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ":Debugging"))
		return NULL;
	return PyInt_FromLong(bServiceDebug);
}

// @pymethod int|servicemanager|PumpWaitingMessages|Pumps all waiting messages.
static PyObject *PyPumpWaitingMessages(PyObject *self, PyObject *args)
{
    MSG msg;
	long result = 0;
	// Read all of the messages in this next loop, 
	// removing each message as we read it.
	Py_BEGIN_ALLOW_THREADS
	while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) {
		// If it's a quit message, we're out of here.
		if (msg.message == WM_QUIT) {
			result = 1;
			break;
		}
		// Otherwise, dispatch the message.
		DispatchMessage(&msg); 
	} // End of PeekMessage while loop
	Py_END_ALLOW_THREADS
	return PyInt_FromLong(result);
}

// @module servicemanager|A module built in to PythonService.exe (and therefore only available to Python service programs).
// <nl>The module <o win32service> provides other service facilities.
static struct PyMethodDef servicemanager_functions[] = {
	{"CoInitializeEx", PyCoInitializeEx, 1}, // @pymeth CoInitializeEx|
	{"CoUninitialize", PyCoUninitialize, 1}, // @pymeth CoUninitialize|
	{"RegisterServiceCtrlHandler", PyRegisterServiceCtrlHandler, 1}, // @pymeth RegisterServiceCtrlHandler|Registers a function to retrieve service control notification messages.
	{"LogInfoMsg",                 PyLogInfoMsg, 1},	// @pymeth LogInfoMsg|Write an informational message to the log.
	{"LogErrorMsg",                PyLogErrorMsg, 1},	// @pymeth LogErrorMsg|Write an error message to the log.
	{"LogWarningMsg",              PyLogWarningMsg, 1}, // @pymeth LogWarningMsg|Logs a generic warning message to the event log
	{"PumpWaitingMessages",        PyPumpWaitingMessages, 1},  // @pymeth PumpWaitingMessages|Pumps waiting window messages for the service.
	{"Debugging",                  PyDebugging, 1},  // @pymeth Debugging|Indicates if the service is running in debug mode.
	{NULL}
};

int AddConstant(PyObject *dict, const char *key, long value)
{
	PyObject *oval = PyInt_FromLong(value);
	if (!oval)
	{
		return 1;
	}
	int rc = PyDict_SetItemString(dict, (char*)key, oval);
	Py_DECREF(oval);
	return rc;
}

#define ADD_CONSTANT(tok) AddConstant(dict, #tok, tok)

extern "C" __declspec(dllexport) void
initservicemanager(void)
{
  PyObject *dict, *module;
  module = Py_InitModule("servicemanager", servicemanager_functions);
  dict = PyModule_GetDict(module);
  servicemanager_module_error = PyString_FromString("service manager error");
  PyDict_SetItemString(dict, "error", servicemanager_module_error);

  ADD_CONSTANT(COINIT_MULTITHREADED);
}

// The Service Manager
//


//
//  FUNCTION: main
//
//  PURPOSE: entrypoint for service
//
//  PARAMETERS:
//    argc - number of command line arguments
//    argv - array of command line arguments
//
//  RETURN VALUE:
//    none
//
//  COMMENTS:
//    main() either performs the command line task, or
//    call StartServiceCtrlDispatcher to register the
//    main service thread.  When the this call returns,
//    the service has stopped, so exit.
//

int main(int argc, char **argv)
{
	SERVICE_TABLE_ENTRY   DispatchTable[] = 
    { 
        { TEXT(""), 	service_main      }, 
        { NULL,              NULL         } 
    }; 
    if ( (argc > 1) &&
         ((*argv[1] == '-') || (*argv[1] == '/')) )
    {
        if ( _stricmp( "register", argv[1]+1 ) == 0 ||
             _stricmp( "install", argv[1]+1 ) == 0 )
        {
	        // Get out of here.
			return RegisterPythonServiceExe() ? 0 : 1;
        }
        if ( _stricmp( "debug", argv[1]+1 ) == 0 ) {
        	if (argc<3) {
        		printf("-debug requires a service name");
        		return 1;
        	}
			initservicemanager();
			bServiceDebug = TRUE;
			printf("Debugging service %s\n", argv[2]);
		    int dwArgc;
		    LPTSTR *lpszArgv;

#ifdef UNICODE
		    lpszArgv = CommandLineToArgvW(GetCommandLineW(), &(dwArgc) );
#else
		    dwArgc   = argc;
		    lpszArgv = argv;
#endif
		    
	        SetConsoleCtrlHandler( DebugControlHandler, TRUE );
			service_main(dwArgc-2, lpszArgv+2);
        	return 0; // gotta assume OK...
        }
    }

	// To be friendly, say what we are doing
    printf("%s - Python Service Manager\n", argv[0]);
    printf("Options:\n");
    printf(" -register - register the EXE - this must be done at least once.\n");
    printf(" -debug servicename [parms] - debug the Python service.\n");
    printf("\nStarting service - this may take several seconds - please wait...\n");
	
	initservicemanager();
	
    if (!StartServiceCtrlDispatcher( DispatchTable)) {
    	ReportAPIError(PYS_E_API_CANT_START_SERVICE);
    	printf("Could not start the service - error %d\n", GetLastError());
		RegisterPythonServiceExe();
    }
	return 2;
}

BOOL WINAPI DebugControlHandler ( DWORD dwCtrlType )
{
    switch( dwCtrlType )
    {
        case CTRL_BREAK_EVENT:  // use Ctrl+C or Ctrl+Break to simulate
        case CTRL_C_EVENT:      // SERVICE_CONTROL_STOP in debug mode
            _tprintf(TEXT("Stopping debug service.\n"));
            service_ctrl(SERVICE_CONTROL_STOP);
            return TRUE;
            break;

    }
    return FALSE;
}

//
//  FUNCTION: service_main
//
//  PURPOSE: To perform actual initialization of the service
//
//  PARAMETERS:
//    dwArgc   - number of command line arguments
//    lpszArgv - array of command line arguments
//
//  RETURN VALUE:
//    none
//
//  COMMENTS:
//    This routine performs the service initialization and then calls
//    the user defined ServiceStart() routine to perform majority
//    of the work.
//
void WINAPI service_main(DWORD dwArgc, LPTSTR *lpszArgv)
{
	PyObject *instance = NULL;
	PyObject *start = NULL;
	BOOL bPythonInitedOK = LocatePythonServiceClass(dwArgc, lpszArgv, &instance);

	// If Python has not yet registered the service control handler, then
	// we are in serious trouble - it is likely the service will enter a zombie
	// state, where it wont do anything, but you can not start another.
	if (g_obServiceCtrlHandler==NULL) {
		// If Python seemed to init OK, but hasnt done what we expect,
		// report an error, as we are gunna fail!
		if (bPythonInitedOK) 
			ReportPythonError(E_PYS_NOT_CONTROL_HANDLER);
		if (!bServiceDebug)
			sshStatusHandle = RegisterServiceCtrlHandler( lpszArgv[0], service_ctrl);
	}
	// Not much we can do here.
    if ( !bPythonInitedOK || (!sshStatusHandle && !bServiceDebug) || !instance)
        goto cleanup;

	if (!bServiceDebug)
		if (!SetServiceStatus( sshStatusHandle, &startingStatus ))
			ReportAPIError(PYS_E_API_CANT_SET_PENDING);
			// but I'll keep going, I spose??

	start = PyObject_GetAttrString(instance, "SvcRun");
	if (start==NULL)
		ReportPythonError(E_PYS_NO_RUN_METHOD);
	else {
		PyObject *result = PyObject_CallObject(start, NULL);
		if (result==NULL)
			ReportPythonError(E_PYS_START_FAILED);
		else
			Py_DECREF(result);
	}
	// We are all done.
cleanup:

    // try to report the stopped status to the service control manager.
    //
	Py_XDECREF(start);
	Py_XDECREF(instance);
		
    if (sshStatusHandle) { // Wont be true if debugging.
		if (!SetServiceStatus( sshStatusHandle, &stoppedStatus ))
			ReportAPIError(PYS_E_API_CANT_SET_STOPPED);
    }
    return;
}

BOOL LocatePythonServiceClass( DWORD dwArgc, LPTSTR *lpszArgv, PyObject **result )
{
	char valueBuf[512];
	char keyName[1024];
	LONG valueBufSize = sizeof(valueBuf) * sizeof(TCHAR);
//	char *asciiServiceName = svcName;
	
	char asciiServiceName[256];
	if (WideCharToMultiByte(CP_ACP, 0, lpszArgv[0], -1, asciiServiceName, sizeof(asciiServiceName), NULL, NULL)==0) {
		ReportAPIError(PYS_E_API_CANT_CONVERT_ASCII);
		return FALSE;
	}
	
	wsprintfA(keyName, "System\\CurrentControlSet\\Services\\%s\\PythonClass", asciiServiceName);
	if (RegQueryValueA(HKEY_LOCAL_MACHINE, keyName, valueBuf, &valueBufSize)!=ERROR_SUCCESS) {
		ReportAPIError(PYS_E_API_CANT_LOCATE_PYTHON_CLASS);
		return FALSE;
	}
	// Find the last "\\"
	char *sep = strrchr(valueBuf, '\\');
	char *fname;
	if (sep) {
		*sep = '\0';
		fname = sep+1;
		// Stick the Path on the Python sys.path.
		PyObject *obPath = PySys_GetObject("path");
		if (obPath==NULL) {
				ReportPythonError(PYS_E_NO_SYS_PATH);
				return FALSE;
		}
		PyObject *obNew = PyString_FromString(valueBuf);
		if (obNew==NULL) {
			Py_DECREF(obPath);
			ReportPythonError(PYS_E_NO_MEMORY_FOR_SYS_PATH);
			return FALSE;
		}
			
		PyList_Append(obPath, obNew);
	} else {
		fname = valueBuf;
	}
	
	// Find the last "." in the name, and assume it is a module name.
	char *classNamePos = strrchr(fname, '.');
	if (classNamePos==NULL) {
		ReportError(PYS_E_CANT_LOCATE_MODULE_NAME);
		return FALSE;
	}
	PyObject *module;
	*classNamePos++ = '\0';
	// If we have another '.', then likely a 'ni' package.
	if (strrchr(keyName, '.')!=NULL) {
		module = PyImport_ImportModule("ni");
		Py_XDECREF(module);
	}

	module = PyImport_ImportModule(fname);
	if (module==NULL) {
		ReportPythonError(E_PYS_NO_MODULE);
		return FALSE;
	}
	PyObject *pyclass = PyObject_GetAttrString(module, classNamePos);
	Py_DECREF(module);
	if (pyclass==NULL) {
		ReportPythonError(E_PYS_NO_CLASS);
		return FALSE;
	}
	PyObject *args = PyTuple_New(dwArgc);
	if (args==NULL) {
		ReportPythonError(PYS_E_NO_MEMORY_FOR_ARGS);
		return FALSE;
	}
	for (DWORD i=0;i<dwArgc;i++) {
		PyObject *arg = PyWinObject_FromWCHAR(lpszArgv[i]);
		if (arg==NULL) {
			Py_DECREF(args);
			ReportPythonError(PYS_E_BAD_ARGS);
			return FALSE;
		}
		PyTuple_SET_ITEM(args, i, arg);
	}
	PyObject *realArgs = PyTuple_New(1);
	PyTuple_SET_ITEM(realArgs, 0, args);
	*result = PyObject_CallObject(pyclass, realArgs);
	if (*result==NULL)
		ReportPythonError(PYS_E_BAD_CLASS);
	
	Py_DECREF(pyclass);
	return (*result != NULL);
}

void ReportAPIError(DWORD msgCode, DWORD errCode /*= 0*/)
{
	if (errCode==0)
		errCode = GetLastError();

	const int bufSize = 512;
	TCHAR buf[bufSize];
	BOOL bHaveMessage = (::FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, NULL, errCode, 0, buf, bufSize, NULL )>0);
	if (!bHaveMessage)
		_tcscpy(buf,TEXT("No error message is available"));
	/* strip trailing cr/lf */
	int end = _tcslen(buf)-1;
	if (end>1 && (buf[end-1]==L'\n' || buf[end-1]==L'\r'))
		buf[end-1] = L'\0';
	else
		if (end>0 && (buf[end]==L'\n' || buf[end]==L'\r'))
			buf[end]=L'\0';

	TCHAR cvtBuf[20];
	_stprintf(cvtBuf, L"%d", errCode);
    LPTSTR  lpszStrings[] = {cvtBuf, buf, L'\0'};
    ReportError(msgCode, (LPCTSTR *)lpszStrings);
}

void ReportPythonError(DWORD code)
{
	if (PyErr_Occurred()) {
		LPTSTR inserts[4];
		inserts[3] = NULL; // terminate array
		PyObject *type, *value, *traceback;
		PyErr_Fetch(&type, &value, &traceback);
		inserts[0] = L"Traceback may go here later!";
		PyObject *obStr = PyObject_Str(type);
		PyWinObject_AsBstr(obStr, inserts+1);
		Py_XDECREF(obStr);
		obStr = PyObject_Str(value);
		PyWinObject_AsBstr(PyObject_Str(obStr), inserts+2);
		Py_XDECREF(obStr);
	    ReportError(code, (LPCTSTR *)inserts);
	    SysFreeString(inserts[1]);
	    SysFreeString(inserts[2]);
	    if (bServiceDebug) { // If debugging, restore for traceback print,
		    PyErr_Restore(type, value, traceback);
		} else {	// free em up.
			Py_DECREF(type);
			Py_DECREF(value);
			Py_DECREF(traceback);
		}
	} else {
		LPCTSTR inserts[] = {L"<No Python Error!>", L"", L"", NULL};
    	ReportError(code, inserts);
	}
    if (bServiceDebug) {
	    if (PyErr_Occurred())
		    PyErr_Print();
    }
}

VOID ReportError(DWORD code, LPCTSTR *inserts, WORD errorType /* = EVENTLOG_ERROR_TYPE*/)
{
	WORD numInserts = 0;
	while (inserts && inserts[numInserts]!=NULL)
		numInserts++;
		
    HANDLE  hEventSource;
    // Use event logging to log the error.
	//
    if (bServiceDebug) {
    	LPTSTR buffer;
    	// Get the message text, and just print it.
    	if (FormatMessage(FORMAT_MESSAGE_FROM_HMODULE | FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_ARGUMENT_ARRAY,
    			GetModuleHandle(NULL), code, 0, (LPTSTR)&buffer, 0, (va_list *)inserts)==0) {
    		printf("Error 0x%X - No message available\nMessage inserts were", code);
    		for (int i=0;i<numInserts;i++)
    			_tprintf(L"'%s',", inserts[i]);
    	} else {
    		_tprintf(L"Error 0x%X - %s", code, buffer);
    		LocalFree(buffer);
    	}
    } else {
 	
	    hEventSource = RegisterEventSource(NULL, SZ_EVENT_SOURCE_NAME);
	    if (hEventSource != NULL) {
	        ReportEvent(hEventSource, // handle of event source
	            errorType,  // event type
	            0,                    // event category
	            code,                 // event ID
	            NULL,                 // current user's SID
	            numInserts,           // strings in lpszStrings
	            0,                    // no bytes of raw data
	            inserts,          // array of error strings
	            NULL);                // no raw data

	        (VOID) DeregisterEventSource(hEventSource);
	    }
    }
}

// Register the EXE.
// This writes an entry to the Python registry and also
// to the EventLog so I can stick in messages.
BOOL RegisterPythonServiceExe(void)
{
	printf("Registering the Python Service Manager...\n");
	const int fnameBufSize = MAX_PATH + 1;
	TCHAR fnameBuf[fnameBufSize];
	if (GetModuleFileName( NULL, fnameBuf, fnameBufSize)==0) {
		printf("Registration failed due to GetModuleFileName() failing (error %d)\n", GetLastError());
		return FALSE;
	}
	// Register this specific EXE against this specific DLL version
	PyObject *obVerString = PySys_GetObject("winver");
	if (obVerString==NULL || !PyString_Check(obVerString)) {
		Py_XDECREF(obVerString);
		printf("Registration failed as sys.winver is not available or not a string\n");
		return FALSE;
	}
	char *szVerString = PyString_AsString(obVerString);
	Py_DECREF(obVerString);
	// note wsprintf allows %hs to be "char *" even when UNICODE!
	TCHAR keyBuf[256];
	wsprintf(keyBuf, _T("Software\\Python\\PythonService\\%hs"), szVerString);
	DWORD rc;
	if ((rc=RegSetValue(HKEY_LOCAL_MACHINE,
	                keyBuf, REG_SZ, 
					fnameBuf, _tcslen(fnameBuf)))!=ERROR_SUCCESS) {
		printf("Registration failed due to RegSetValue() of service EXE - error %d\n", rc);
		return FALSE;
	}

	HKEY hkey;
	if ((rc=RegCreateKey(HKEY_LOCAL_MACHINE,
	               L"SYSTEM\\CurrentControlSet\\Services\\EventLog\\Application\\" SZ_EVENT_SOURCE_NAME,
	               &hkey ))!=ERROR_SUCCESS) {
		printf("Registration failed due to RegCreateKey() of EventLog entry - error %d\n", rc);
		return FALSE;
	}
	rc = RegSetValueEx(hkey,
	                TEXT("EventMessageFile"), 0, REG_SZ, 
					(const BYTE *)fnameBuf, (_tcslen(fnameBuf)+1)*sizeof(TCHAR));
	if (rc==ERROR_SUCCESS) {
		DWORD types = EVENTLOG_ERROR_TYPE | EVENTLOG_WARNING_TYPE | EVENTLOG_INFORMATION_TYPE;
		rc = RegSetValueEx(hkey,
	                TEXT("TypesSupported"), 0, REG_DWORD, 
					(const BYTE *)&types, sizeof(types));
	}
	RegCloseKey(hkey);
	if (rc!=ERROR_SUCCESS) {
		printf("Registration failed due to RegSetValue() of EventLog entry - error %d\n", rc);
		return FALSE;
	}
	return TRUE;
}

