/***********************************************************

PyWinTypes.cpp -- implementation of standard win32 types


Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc

******************************************************************/

#include "windows.h"
#include "malloc.h"

#include "Python.h"
#include "PyWinTypes.h"
#include "PyWinObjects.h"
#include "PySecurityObjects.h"

PyObject * PyWinExc_ApiError;
PyObject * PyWinExc_COMError;

/* error helper - GetLastError() is provided, but this is for exceptions */
PyObject *PyWin_SetAPIError(char *fnName, long err /*= 0*/)
{
	const int bufSize = 512;
	char buf[bufSize];
	DWORD errorCode = err == 0 ? GetLastError() : err;
	BOOL bHaveMessage = FALSE;
	if (errorCode)
		bHaveMessage = ::FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, NULL, errorCode, 0, buf, bufSize, NULL )>0;
	if (!bHaveMessage)
		strcpy(buf,"No error message is available");
	/* strip trailing cr/lf */
	int end = strlen(buf)-1;
	if (end>1 && (buf[end-1]=='\n' || buf[end-1]=='\r'))
		buf[end-1] = '\0';
	else
		if (end>0 && (buf[end]=='\n' || buf[end]=='\r'))
			buf[end]='\0';
	PyObject *v = Py_BuildValue("(iss)", errorCode, fnName, buf);
	if (v != NULL) {
		PyErr_SetObject(PyWinExc_ApiError, v);
		Py_DECREF(v);
	}
	return NULL;
}

PyObject *PyWin_SetBasicCOMError(HRESULT hr)
{
	char buf[80];
	wsprintf(buf, "COM Error 0x%x", hr);
	PyObject *evalue = Py_BuildValue("iszz", hr, buf, NULL, NULL);
	PyErr_SetObject(PyWinExc_COMError, evalue);
	Py_XDECREF(evalue);
	return NULL;
}

// @pymethod <o PyUnicode>|pywintypes|Unicode|Creates a new Unicode object
static PyObject *PyWin_NewUnicode(PyObject *self, PyObject *args)
{
	PyObject *obString;
	// @pyparm string|str||The string to convert.
	if (!PyArg_ParseTuple(args, "O", &obString))
		return NULL;

	PyUnicode *result = new PyUnicode(obString);
	if ( result->m_bstrValue )
		return result;
	Py_DECREF(result);
	/* an error should have been raised */
	return NULL;
}

// @pymethod <o PyUnicode>|pywintypes|UnicodeFromRaw|Creates a new Unicode object from raw binary data
static PyObject *PyWin_NewUnicodeFromRaw(PyObject *self, PyObject *args)
{
	const char * value;
	unsigned int numBytes;

	// @pyparm string|str||The string containing the binary data.
	if (!PyArg_ParseTuple(args, "s#", &value, &numBytes))
		return NULL;

	PyUnicode *result = new PyUnicode(value, numBytes);
	if ( result->m_bstrValue )
		return result;
	Py_DECREF(result);
	/* an error should have been raised */
	return NULL;
}

// @pymethod int, int|pywintypes|IsTextUnicode|Determines whether a buffer probably contains a form of Unicode text.
static PyObject *PyWin_IsTextUnicode(PyObject *self, PyObject *args)
{
	const char * value;
	unsigned int numBytes;
	int flags;

	// @pyparm string|str||The string containing the binary data.
	// @pyparm int|flags||Determines the specific tests to make
	if (!PyArg_ParseTuple(args, "s#i", &value, &numBytes, &flags))
		return NULL;

	DWORD rc = IsTextUnicode((LPVOID)value, numBytes, &flags);
	return Py_BuildValue("ii", rc, flags);
	// @rdesc The function returns (result, flags), both integers.
	// <nl>result is nonzero if the data in the buffer passes the specified tests.
	// <nl>result is zero if the data in the buffer does not pass the specified tests.
	// <nl>In either case, flags contains the results of the specific tests the function applied to make its determination.
}

// @pymethod <o PyIID>|pywintypes|CreateGuid|Creates a new, unique GUIID.
static PyObject *PyWin_CreateGuid(PyObject *self, PyObject *args)
{
	if (!PyArg_ParseTuple(args, ":CreateGuid"))
		return NULL;
	GUID guid;
	CoCreateGuid(&guid);
	return PyWinObject_FromIID(guid);
}


/* List of functions exported by this module */
// @module pywintypes|A module which supports common Windows types.
static struct PyMethodDef pywintypes_functions[] = {
	{"Unicode",     PyWin_NewUnicode, 1}, 	// @pymeth Unicode|Creates a new <o PyUnicode> object
	{"UnicodeFromRaw",     PyWin_NewUnicodeFromRaw, 1}, 	// @pymeth UnicodeFromRaw|Creates a new <o PyUnicode> object from raw binary data
	{"IsTextUnicode",      PyWin_IsTextUnicode, 1}, // @pymeth IsTextUnicode|Determines whether a buffer probably contains a form of Unicode text.
	{"OVERLAPPED",  PyWinMethod_NewOVERLAPPED, 1}, 	// @pymeth OVERLAPPED|Creates a new <o PyOVERLAPPED> object
	{"IID",			PyWinMethod_NewIID, 1 },         // @pymeth IID|Makes an <o PyIID> object from a string.
	{"Time",		PyWinMethod_NewTime, 1 },		// @pymeth Time|Makes a <o PyTime> object from the argument.  Argument can be an integer/float or a tuple (as returned by time module functions).
	{"CreateGuid",  PyWin_CreateGuid, 1 },      // @pymeth CreateGuid|Creates a new, unique GUIID.
	{"ACL",         PyWinMethod_NewACL, 1 },      // @pymeth ACL|Creates a new <o PyACL> object.
	{"SID",         PyWinMethod_NewSID, 1 },      // @pymeth SID|Creates a new <o PySID> object.
	{"SECURITY_ATTRIBUTES",         PyWinMethod_NewSECURITY_ATTRIBUTES, 1 },      // @pymeth SECURITY_ATTRIBUTES|Creates a new <o PySECURITY_ATTRIBUTES> object.
	{"HANDLE",      PyWinMethod_NewHANDLE, 1 },      // @pymeth HANDLE|Creates a new <o PyHANDLE> object object.
	{NULL,			NULL}
};


static CRITICAL_SECTION g_csMain;
#ifdef _DEBUG
static DWORD g_cGlobalLocks = 0;
#endif

__declspec(dllexport) void PyWin_AcquireGlobalLock(void)
{
	EnterCriticalSection(&g_csMain);
#ifdef _DEBUG
	++g_cGlobalLocks;
#endif
//	LogEvent("add a lock");
}
__declspec(dllexport) void PyWin_ReleaseGlobalLock(void)
{
//	LogEvent("remove a lock");
#ifdef _DEBUG
	--g_cGlobalLocks;
#endif
	LeaveCriticalSection(&g_csMain);
}

extern "C" __declspec(dllexport) void
initpywintypes(void)
{
	// do nothing - DLLMain has already initialised this module.
}

void realinitpywintypes(void)
{
  // when a COM object is created, this DLL is loaded automatically; we need
  // a thread state to execute the code below.
  CEnterLeavePython _celp;

  PyObject *dict, *module;
  module = Py_InitModule("pywintypes", pywintypes_functions);
  dict = PyModule_GetDict(module);
  PyWinExc_ApiError = PyString_FromString("Win32 API error");
  PyWinExc_COMError = PyString_FromString("Win32 COM error");
  if (PyWinExc_ApiError == NULL || PyWinExc_COMError==NULL) {
    PyErr_SetString(PyExc_MemoryError, "can't define ole_error");
    return;
  }

  PyDict_SetItemString(dict, "error", PyWinExc_ApiError);
  PyDict_SetItemString(dict, "com_error", PyWinExc_COMError);

  // Add a few types.
  PyDict_SetItemString(dict, "TimeType", (PyObject *)&PyTimeType);
  PyDict_SetItemString(dict, "IIDType", (PyObject *)&PyIIDType);
  PyDict_SetItemString(dict, "UnicodeType", (PyObject *)&PyUnicodeType);
  PyDict_SetItemString(dict, "SECURITY_ATTRIBUTESType", (PyObject *)&PySECURITY_ATTRIBUTESType);
  PyDict_SetItemString(dict, "SIDType", (PyObject *)&PySIDType);
  PyDict_SetItemString(dict, "ACLType", (PyObject *)&PyACLType);
  PyDict_SetItemString(dict, "HANDLEType", (PyObject *)&PyHANDLEType);
  PyDict_SetItemString(dict, "OVERLAPPEDType", (PyObject *)&PyHANDLEType);
}

BOOL WINAPI DllMain(HINSTANCE hInstance, DWORD dwReason, LPVOID lpReserved)
{
	if ( dwReason == DLL_PROCESS_ATTACH )
	{
		/*
		** For various process-level global locks.  Strictly speaking, this
		** could throw a C++ exception, but we don't care to trap that.
		*/
		InitializeCriticalSection(&g_csMain);

		realinitpywintypes();
	}
	else if ( dwReason == DLL_PROCESS_DETACH )
	{
		DeleteCriticalSection(&g_csMain);

#ifdef _DEBUG
		if ( g_cGlobalLocks )
		{
// ### need to fix up with some "correct" code ...
#if 0
			char buf[100];
			wsprintf(buf, "non-zero global lock count: %ld", g_cGlobalLocks);
			LogEvent(buf);
#endif
		}
#endif
	}
	return TRUE;    // ok
}

