/*

	win32 view data type

	Created Dec 1995, Mark Hammond (MHammond@skippinet.com.au)

Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc

*/
#include "stdafx.h"

#include "win32win.h"
#include "win32doc.h"
#include "win32dc.h"
#include "win32control.h"
#include "win32ctrlList.h"
#include "win32ctrlTree.h"

CPythonEditView *GetEditViewPtr(PyObject *self)
{
	// need to only rtti check on CView, as CPythonEditView is not derived from CPythonView.
	return (CPythonEditView *)PyCWnd::GetPythonGenericWnd(self, &PyCView::type);
}

CView *PyCView::GetViewPtr(PyObject *self)
{
	// ditto!
	return (CView *)PyCWnd::GetPythonGenericWnd(self, &PyCView::type);
}

CPythonView *PyCScrollView::GetViewPtr(PyObject *self)
{
	// ditto!
	return (CPythonView *)PyCWnd::GetPythonGenericWnd(self, &PyCScrollView::type);
}

CListView *GetListViewPtr(PyObject *self)
{
	// ditto!
	return (CListView *)PyCWnd::GetPythonGenericWnd(self, &PyCView::type);
}

CTreeView *GetTreeViewPtr(PyObject *self)
{
	// ditto!
	return (CTreeView *)PyCWnd::GetPythonGenericWnd(self, &PyCView::type);
}

// @pymethod |PyCView|CreateWindow|Creates the window for a view.
static PyObject *
PyCView_create_window(PyObject *self, PyObject *args)
{
	CView *pView = PyCView::GetViewPtr(self);
	if (!pView) {
		// PyCView::GetViewPtr will trace to the debug device - just so I know it is an OK one!
		TRACE("  ignore warning - RTTI Error detected and handled!");
		PyErr_Clear();
		pView = GetEditViewPtr(self);
	}
	if (!pView)
		return NULL;

	if (pView->m_hWnd!=NULL)
		RETURN_ERR("The view already has a window");

	PyObject *parent;
	int id= AFX_IDW_PANE_FIRST;
	int style= AFX_WS_DEFAULT_VIEW;
	CRect rect(0,0,0,0);
	if (!PyArg_ParseTuple(args, "O|ii(iiii);Must pass parent window and optionally ID, style and (def rect)", 
	          &parent,  // @pyparm <o PyCWnd>|parent||The parent window (usually a frame)
	          &id,      // @pyparm int|id|win32ui.AFX_IDW_PANE_FIRST|The child ID for the view
	          &style,   // @pyparm int|style|win32ui.AFX_WS_DEFAULT_VIEW|The style for the view
			  // @pyparm (left, top, right, bottom)|rect|(0,0,0,0)|The default position of the window.
	          &rect.left, &rect.top, &rect.right, &rect.bottom))
		return NULL;

	if (!ui_base_class::is_uiobject(parent, &PyCWnd::type))
		RETURN_TYPE_ERR("Argument must be a PyCWnd");
	CWnd *pWnd = GetWndPtr( parent );
	if (pWnd==NULL)
		return NULL;

	CCreateContext context;
	context.m_pCurrentDoc = pView->GetDocument();
//	if (!context.m_pCurrentDoc)
//		RETURN_ERR("There is no document attached to the view");

	// must reset doc to NULL, else MFC asserts all over the place!
	// Create() resets this value (via the CreateContext)
	((CProtectedView *)pView)->SetDocument( NULL );

    BOOL ok;
	GUI_BGN_SAVE;
    ok = pView->Create(NULL, NULL, style, rect, pWnd, id, &context);
	GUI_END_SAVE;
	if (!ok)
		RETURN_ERR("Create() view failed\n");
	RETURN_NONE;
}

// @pymethod <o PyCDocument>|PyCView|GetDocument|Returns the document for a view.
static PyObject *
PyCView_get_document (PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CView *view = PyCView::GetViewPtr (self);
	if (view == NULL)
		return NULL;
	return ui_assoc_object::make(PyCDocument::type, view->GetDocument())->GetGoodRet();
}

// @pymethod |PyCView|OnInitialUpdate|Calls the underlying MFC OnInitialUpdate method.
PyObject *
PyCView_on_initial_update(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS2(args, OnInitialUpdate);
	CView *view = PyCView::GetViewPtr (self);
	if (view == NULL)
		return NULL;
	// @xref <vm PyCView.OnInitialUpdate>
	view->CView::OnInitialUpdate();
	RETURN_NONE;
}


// @pymethod int|PyCView|OnActivateView|Calls the underlying MFC OnActivateView method.
PyObject *
PyCView_on_activate_view(PyObject *self, PyObject *args)
{
	// @pyparm int|activate||Indicates whether the view is being activated or deactivated.
	// @pyparm <o PyCView>|activateView||The view object that is being activated.
	// @pyparm <o PyCView>|DeactivateView||The view object that is being deactivated.
	int activate;
	PyObject *obActivate, *obDeactivate;
	if (!PyArg_ParseTuple (args, "iOO:OnActivateView", &activate, &obActivate, &obDeactivate))
		return NULL;
	CView	*view = PyCView::GetViewPtr (self);
	if (view == NULL)
		return NULL;

	PyErr_Clear();
	CView	*pActivate = obActivate==Py_None ? NULL : PyCView::GetViewPtr (obActivate);
	CView	*pDevactive = obDeactivate==Py_None ? NULL : PyCView::GetViewPtr (obDeactivate);
	if (PyErr_Occurred())
		return NULL;
	((CProtectedView *)view)->BaseOnActivateView(activate, pActivate, pDevactive);
	RETURN_NONE;
}

// @pymethod int|PyCView|OnQueryNewPalette|Calls the underlying MFC OnQueryNewPalette method.
PyObject *
PyCView_on_query_new_palette(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS2(args, OnQueryNewPalette);
	CProtectedView	*view = (CProtectedView*)PyCView::GetViewPtr (self);
	if (view == NULL)
		return NULL;
	return Py_BuildValue("i", view->BaseOnQueryNewPalette());
}

// @pymethod tuple|PyCView|PreCreateWindow|Calls the underlying MFC PreCreateWindow method.
PyObject *
PyCView_pre_create_window(PyObject *self, PyObject *args)
{
	CProtectedView	*view = (CProtectedView*)PyCView::GetViewPtr (self);
	if (view == NULL)
		return NULL;
	CREATESTRUCT cs;
	//@pyparm tuple|createStruct||A tuple representing a CREATESTRUCT structure.
	if (!CreateStructFromPyObject( &cs, args, "PreCreateWindow", TRUE))
		return NULL;

	if (!view->BasePreCreateWindow(cs))
		RETURN_ERR("CView::PreCreateWindow failed");
	return PyObjectFromCreateStruct(&cs);
}

// @object PyCView|A class which implements a generic CView.  Derived from a <o PyCWnd> object.
static struct PyMethodDef PyCView_methods[] = {
	{"CreateWindow",			PyCView_create_window,	1}, // @pymeth CreateWindow|Create the window for a view.
	{"GetDocument",             PyCView_get_document,           1}, // @pymeth GetDocument|Returns the document for a view.
	{"OnActivateView",          PyCView_on_activate_view, 1}, // @pymeth OnActivateView|Calls the underlying MFC OnActivateView method.
	{"OnInitialUpdate",         PyCView_on_initial_update, 1}, // @pymeth OnInitialUpdate|Calls the underlying MFC OnInitialUpdate method.
	{"OnQueryNewPalette",       PyCView_on_query_new_palette, 1}, // @pymeth OnQueryNewPalette|Calls the underlying MFC OnQueryNewPalette method.
	{"PreCreateWindow",         PyCView_pre_create_window, 1}, // @pymeth PreCreateWindow|Calls the underlying MFC PreCreateWindow method.
	{NULL,			NULL}
};

// View type
ui_type_CObject PyCView::type("PyCView", 
							  &PyCWnd::type, 
							  RUNTIME_CLASS(CView), 
							  sizeof(PyCView), 
							  PyCView_methods, 
							  NULL);

// @pymethod <o PyCScrollView>|win32ui|CreateView|Creates a generic view object.
PyObject * PyCScrollView::create(PyObject *self, PyObject *args)
{
	PyObject *doc;
	// @pyparm <o PyCDocument>|doc||The document to use with the view.
	if (!PyArg_ParseTuple(args, "O:CreateView", &doc))
		return NULL;
	if (!ui_base_class::is_uiobject(doc, &PyCDocument::type))
		RETURN_TYPE_ERR("Argument must be a PyCDocument");
	CDocument *pDoc = PyCDocument::GetDoc( doc );
	CPythonView *pView = new CPythonView();
	((CProtectedView *)pView)->SetDocument(pDoc);
	return ui_assoc_object::make( PyCScrollView::type, pView );
}

// @pymethod <o PyCDC>|PyCScrollView|GetDC|Gets the view's current DC.
static PyObject *
ui_view_get_dc (PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CScrollView	*view = (CScrollView *) PyCView::GetViewPtr (self);
	if (view == NULL)
		return NULL;

	// create MFC device context
	CDC *pDC = view->GetDC();
	if (pDC==NULL)
		RETURN_ERR ("Could not get the DC for the view.");

	// update logical 0,0 position based on scroll position
	CPoint offset = view->GetDeviceScrollPosition();
	// really should be SetDeviceOrgEx (bad MS names, don'cha know)
	SetViewportOrgEx (pDC->GetSafeHdc(), -offset.x, -offset.y, NULL);

	// create Python device context
	ui_dc_object *dc =
	(ui_dc_object *) ui_assoc_object::make (ui_dc_object::type, pDC);
	return dc;
}

// @pymethod (x,y)|PyCScrollView|GetDeviceScrollPosition|Returns the positon of the scroll bars in device units.
static PyObject *
ui_view_get_dscroll_pos (PyObject *self, PyObject *args)
{
  CHECK_NO_ARGS(args);
  CScrollView *view = (CScrollView *) PyCView::GetViewPtr (self);
  if (view == NULL)
	return NULL;

  CPoint pos = view->GetDeviceScrollPosition();
  return Py_BuildValue ("(ii)", pos.x, pos.y);
}

// @pymethod |PyCScrollView|ScrollToPosition|Scrolls to a given point in the view.
static PyObject *
ui_view_scroll_to_position (PyObject *self, PyObject *args)
{
  CScrollView	*view = (CScrollView *) PyCView::GetViewPtr (self);
  if (view == NULL)
    return NULL;

  POINT position;
  // @pyparm (x,y)|position||The position to scroll to.
  if (!PyArg_ParseTuple (args, "(ii):ScrollToPosition", &position.x, &position.y))
    return NULL;

  view->ScrollToPosition (position);

  RETURN_NONE;
}


// @pymethod |PyCScrollView|SetDynamicScrollBars|Turns dynamic scroll bars on or off
static PyObject *
ui_view_set_dynamic_scrollbars (PyObject *self, PyObject *args)
{
  CPythonView	*view = (CPythonView *) PyCView::GetViewPtr (self);
  if (view == NULL)
    return NULL;

  int dynamic = TRUE;
  // @pyparm int|bDynamic|1|The flag to indicate the state of the dynamic scroll bars.
  if (!PyArg_ParseTuple (args, "|i:SetDynamicScrollBars", &dynamic))
    return NULL;

  view->SetDynamicScrollBars ((BOOL) dynamic);

  RETURN_NONE;
}

// @pymethod tuple|PyCScrollView|ResizeParentToFit|Lets the size of a view dictate the size of its frame window.
PyObject *
ui_view_resize_parent_to_fit(PyObject *self, PyObject *args)
{
	CPythonView	*view = (CPythonView *) PyCView::GetViewPtr (self);
	if (view == NULL)
		return NULL;
	BOOL bShrink = TRUE;
	//@pyparm int|bShrinkOnly|1|The kind of resizing to perform. The default value, TRUE, shrinks the frame window if appropriate. 
	if (!PyArg_ParseTuple(args, "|i:ResizeParentToFit", &bShrink))
		return NULL;
	view->ResizeParentToFit(bShrink);
	// @comm This is recommended only for views in MDI child frame windows.
	// <nl>Use ResizeParentToFit in the OnInitialUpdate handler function of your View class.
	// <nl>You must ensure the parent's <om PyCFrameWnd.RecalcLayout> is called before using this method.
	RETURN_NONE;
}

// @pymethod |PyCScrollView|SetScrollSizes|Sets the sizes of the scroll bars
static PyObject *
ui_view_set_scroll_sizes (PyObject *self, PyObject *args)
{
  CPythonView	*view = (CPythonView *) PyCView::GetViewPtr (self);
  if (view == NULL)
	return NULL;
  int map_mode;
  CSize total,
        page = CScrollView::sizeDefault,
        line = CScrollView::sizeDefault;
  if (!PyArg_ParseTuple (args, "i(ii)|(ii)(ii):SetScrollSizes", 
                &map_mode, // @pyparm int|mapMode||The mapping mode for this view.
                &total.cx, &total.cy, // @pyparm (x,y)|sizeTotal||The total size of the view.  Sizes are in logical units.  Both x and y must be greater than zero.
				&page.cx, &page.cy,   // @pyparm (x,y)|sizePage|win32ui.rectDefault|The number of untils to scroll in response to a page-down command.
				&line.cx, &line.cy))  // @pyparm (x,y)|sizePage|win32ui.rectDefault|The number of untils to scroll in response to a line-down command.
	return NULL;
  
  BOOL save = view->SetDynamicScrollBars (TRUE);
  view->SetScrollSizes (map_mode, total, page, line);
  view->SetDynamicScrollBars (save);
  RETURN_NONE;
}
// @pymethod (x,y)|PyCScrollView|GetScrollPosition|Returns the current position of the scroll bars (in logical units).
static PyObject *
PyCView_get_scroll_pos (PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CScrollView	*view = (CScrollView *) PyCView::GetViewPtr (self);
	if (view == NULL)
		return NULL;
	CPoint pos = view->GetScrollPosition();
	return Py_BuildValue ("(ii)", pos.x, pos.y);
}
// @pymethod (x,y)|PyCScrollView|GetTotalSize|Returns the total size of the view in logical units.
static PyObject *
PyCView_get_total_size (PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CScrollView	*view = (CScrollView *) PyCView::GetViewPtr (self);
	if (view == NULL)
		return NULL;
	CSize size = view->GetTotalSize();
	return Py_BuildValue ("(ii)", size.cx, size.cy);
}

// @object PyCScrollView|A class which implements a generic CScrollView.  Derived from a <o PyCView> object.
static struct PyMethodDef PyCScrollView_methods[] = {
	{"GetDeviceScrollPosition",	ui_view_get_dscroll_pos,		1}, // @pymeth GetDeviceScrollPosition|Return the position of the scroll bars (device units).
	{"GetDC",					ui_view_get_dc,					1}, // @pymeth GetDC|Get the views current <o PyCDC>
	{"GetScrollPosition",		PyCView_get_scroll_pos,	1}, // @pymeth GetScrollPosition|Return the position of the scroll bars (logical units).
	{"GetTotalSize",			PyCView_get_total_size,	1}, // @pymeth GetTotalSize|Return the total size of the views.
	{"ResizeParentToFit",       ui_view_resize_parent_to_fit, 1}, // @pymeth ResizeParentToFit|Call ResizeParentToFit to let the size of your view dictate the size of its frame window.
	{"ScrollToPosition",		ui_view_scroll_to_position,		1}, // @pymeth ScrollToPosition|Scroll to a specified point.
	{"SetDynamicScrollBars",	ui_view_set_dynamic_scrollbars,	1}, // @pymeth SetDynamicScrollBars|Set the state of dynamic scroll bars.
	{"SetScrollSizes",			ui_view_set_scroll_sizes,		1}, // @pymeth SetScrollSizes|Set the scrolling sizes.
	{NULL,			NULL}
};

// View type
ui_type_CObject PyCScrollView::type("PyCScrollView", 
							  &PyCView::type, 
							  RUNTIME_CLASS(CScrollView), 
							  sizeof(PyCScrollView), 
							  PyCScrollView_methods, 
							  GET_PY_CTOR(PyCScrollView));

/////////////////////////////////////////////////////////////////////
//
// Edit View object
//
//////////////////////////////////////////////////////////////////////
// @pymethod <o PyCEditView>|win32ui|CreateEditView|Creates a PyEditView object.
PyObject * PyCEditView::create(PyObject *self, PyObject *args)
{
	PyObject *doc;
	// @pyparm <o PyCDocument>|doc||The document to use with the view.
	if (!PyArg_ParseTuple(args, "O:CreateEditView", &doc))
		return NULL;
	if (!ui_base_class::is_uiobject(doc, &PyCDocument::type))
		RETURN_TYPE_ERR("Argument must be a PyCDocument");
	CDocument *pDoc = PyCDocument::GetDoc( doc );
	CPythonEditView *pView = new CPythonEditView();
	pView->SetDocument(pDoc);
	return ui_assoc_object::make( PyCEditView::type, pView );
}


// @pymethod <o PyCEditCtrl>|PyCEditView|GetEditCtrl|returns the underlying edit control object.
static PyObject *
ui_edit_get_edit_ctrl(PyObject *self, PyObject *args)
{
	CPythonEditView *pView = GetEditViewPtr(self);
	if (!pView)
		return NULL;
	CEdit &ed = pView->GetEditCtrl();
	return ui_assoc_object::make( UITypeFromCObject(&ed), &ed )->GetGoodRet();
}

// @pymethod |PyCEditView|SetModifiedFlag|Sets the modified flag for the view's document.
static PyObject *
ui_edit_window_set_modified_flag(PyObject *self, PyObject *args)
{
	CPythonEditView *pView = GetEditViewPtr(self);
	if (!pView)
		return NULL;
	BOOL bState = TRUE;
	// @pyparm int|bModified|1|The modified state to set.
	if (!PyArg_ParseTuple(args, "|i:SetModifiedFlag", &bState))
		return NULL;
	pView->GetDocument()->SetModifiedFlag(bState);
	RETURN_NONE;
}

// @pymethod int|PyCEditView|IsModified|Indicates if the view's document has the modified flag set.
static PyObject *
ui_edit_window_is_modified(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CPythonEditView *pView = GetEditViewPtr(self);
	if (!pView)
		return NULL;
	return Py_BuildValue("i",pView->GetDocument()->IsModified());
}

// @pymethod |PyCEditView|LoadFile|Loads a file into the view.
static PyObject *
ui_edit_window_load_file(PyObject *self, PyObject *args)
{
	char *fileName;
	// @pyparm string|fileName||The name of the file to be loaded.
	if (!PyArg_ParseTuple(args, "s:LoadFile", &fileName))
		return NULL;

	CPythonEditView *pView;
	if (!(pView=GetEditViewPtr(self)))
		return NULL;

	CFile file;
	CFileException fe;
	if (!file.Open(fileName, CFile::modeRead | CFile::shareDenyWrite, &fe)) {
		long errCode = fe.m_lOsError;
		CString csMessage = GetAPIErrorString(errCode);
		if (csMessage.GetLength())
			PyErr_SetString(PyExc_IOError, (char *)(const char *)csMessage);
		else
			PyErr_SetString(PyExc_IOError, "Unknown IO error?");
		return NULL;
	}
	pView->DeleteContents();
	CArchive loadArchive(&file, CArchive::load | CArchive::bNoFlushOnDelete);
	TRY
	{
		pView->BeginWaitCursor();
		pView->SerializeRaw(loadArchive);     // load me
		loadArchive.Close();
		file.Close();
	}
	CATCH_ALL(e)
	{
		file.Abort(); // will not throw an exception
		pView->EndWaitCursor();
		pView->DeleteContents();   // remove failed contents
		PyErr_SetString(PyExc_IOError, "File error reading file");
		e->Delete();
		return NULL;
	}
	END_CATCH_ALL

	pView->EndWaitCursor();
	CDocument *pDocument = pView->GetDocument();
	if (pDocument)
		pDocument->SetModifiedFlag(FALSE);     // start off with unmodified
	RETURN_NONE;
}
// @pymethod |PyCEditView|SaveFile|Saves the view to a file.
static PyObject *
ui_edit_window_save_file(PyObject *self, PyObject *args)
{
	char *fileName;
	// @pyparm string|fileName||The name of the file to be written.
	if (!PyArg_ParseTuple(args, "s:SaveFile", &fileName))
		return NULL;

	CPythonEditView *pView;
	if (!(pView=GetEditViewPtr(self)))
		return NULL;

	CFile file;
	CFileException fe;

	if (!file.Open(fileName, CFile::modeCreate |
	  CFile::modeReadWrite | CFile::shareExclusive, &fe)) {
  		long errCode = fe.m_lOsError;
		CString csMessage = GetAPIErrorString(errCode);
		if (csMessage.GetLength())
			PyErr_SetString(PyExc_IOError, (char *)(const char *)csMessage);
		else
			PyErr_SetString(PyExc_IOError, "Unknown IO error?");
		return NULL;
	}
	CArchive saveArchive(&file, CArchive::store | CArchive::bNoFlushOnDelete);
	TRY
	{
		pView->BeginWaitCursor();
		pView->SerializeRaw(saveArchive);     // save me
		saveArchive.Close();
		file.Close();
	}
	CATCH_ALL(e)
	{
		file.Abort(); // will not throw an exception
		pView->EndWaitCursor();
		PyErr_SetString(PyExc_IOError, "File error saving file");
		e->Delete();
		return NULL;
	}
	END_CATCH_ALL

	pView->EndWaitCursor();
	CDocument *pDocument = pView->GetDocument();
	if (pDocument)
		pDocument->SetModifiedFlag(FALSE);     // start off with unmodified
	RETURN_NONE;
}
// @pymethod tuple|PyCEditView|PreCreateWindow|Calls the underlying MFC PreCreateWindow method.
PyObject *
PyCEditView_pre_create_window(PyObject *self, PyObject *args)
{
	CPythonEditView *pView;
	if (!(pView=GetEditViewPtr(self)))
		return NULL;
	CREATESTRUCT cs;
	//@pyparm tuple|createStruct||A tuple representing a CREATESTRUCT structure.
	if (!CreateStructFromPyObject( &cs, args, "PreCreateWindow", TRUE))
		return NULL;

	if (!pView->PreCreateWindow(cs)) // WARNING - If CPythonEditView::PreCreateWindow gets
		RETURN_ERR("CEditView::PreCreateWindow failed"); // fixed, this will break.
	return PyObjectFromCreateStruct(&cs);
}

///////////////////////////////////////
//
// Edit View Methods
//
// inherited from view
//
///////////////////////////////////////
// @object PyCEditView|A class which implementes a CView of a text file.  Derived from <o PyCView> and <o PyCEdit> objects.
static struct PyMethodDef ui_edit_window_methods[] = {
	{"IsModified", 		ui_edit_window_is_modified,			1}, // @pymeth IsModified|Indicates if the view's document is modified.
	{"LoadFile",		ui_edit_window_load_file,			1},	// @pymeth LoadFile|Loads a named file into the view.
	{"SetModifiedFlag",	ui_edit_window_set_modified_flag,	1},	// @pymeth SetModifiedFlag|Sets the view's document modified flag.
	{"GetEditCtrl",     ui_edit_get_edit_ctrl,              1}, // @pymeth GetEditCtrl|Returns the underlying <o PyCEdit> object
	{"PreCreateWindow", PyCEditView_pre_create_window, 1}, // @pymeth PreCreateWindow|Calls the underlying MFC PreCreateWindow method.
	{"SaveFile",		ui_edit_window_save_file,			1}, // @pymeth SaveFile|Saves the view to a named file.
	{NULL, NULL}
};

PyCCtrlView_Type PyCEditView::type("PyCEditView", &PyCCtrlView::type, &PyCEdit::type, RUNTIME_CLASS(CCtrlView), sizeof(PyCEditView), ui_edit_window_methods, GET_PY_CTOR(PyCEditView));

/////////////////////////////////////////////////////////////////////
//
// List View object
//
//////////////////////////////////////////////////////////////////////
// @pymethod <o PyCListView>|win32ui|CreateListView|Creates a PyCListView object.
PyObject * PyCListView::create(PyObject *self, PyObject *args)
{
	PyObject *doc;
	// @pyparm <o PyCDocument>|doc||The document to use with the view.
	if (!PyArg_ParseTuple(args, "O:CreateListView", &doc))
		return NULL;
	if (!ui_base_class::is_uiobject(doc, &PyCDocument::type))
		RETURN_TYPE_ERR("Argument must be a PyCDocument");
	CDocument *pDoc = PyCDocument::GetDoc( doc );
	CListView *pView = new CPythonListView();
	((CProtectedView *)pView)->SetDocument(pDoc);
	return ui_assoc_object::make( PyCListView::type, pView );
}

// @pymethod tuple|PyCListView|PreCreateWindow|Calls the underlying MFC PreCreateWindow method.
PyObject *
PyCListView_pre_create_window(PyObject *self, PyObject *args)
{
	CListView *pView;
	if (!(pView=GetListViewPtr(self)))
		return NULL;
	CREATESTRUCT cs;
	//@pyparm tuple|createStruct||A tuple representing a CREATESTRUCT structure.
	if (!CreateStructFromPyObject( &cs, args, "PreCreateWindow", TRUE))
		return NULL;

	if (!pView->CListView::PreCreateWindow(cs))
		RETURN_ERR("CListView::PreCreateWindow failed");
	return PyObjectFromCreateStruct(&cs);
}

// @pymethod <o PyCListCtrl>|PyCListView|GetListCtrl|Returns the underlying list control object.
static PyObject *
PyCListView_get_list_ctrl(PyObject *self, PyObject *args)
{
	CListView *pView;
	if (!(pView=GetListViewPtr(self)))
		return NULL;
	CListCtrl &lc = pView->GetListCtrl();
	return ui_assoc_object::make( UITypeFromCObject(&lc), &lc )->GetGoodRet();
}

///////////////////////////////////////
// @object PyCListView|A class which implementes a CListView.  Derived from <o PyCView> and <o PyCListCtrl> objects.
static struct PyMethodDef ui_list_view_methods[] = {
	{"PreCreateWindow", PyCListView_pre_create_window, 1}, // @pymeth PreCreateWindow|Calls the underlying MFC PreCreateWindow method.
	{"GetListCtrl", PyCListView_get_list_ctrl, 1}, // @pymeth GetListCtrl|Returns the underlying list control object.
	{NULL, NULL}
};

PyCCtrlView_Type PyCListView::type("PyCListView", 
								   &PyCCtrlView::type, 
								   &PyCListCtrl::type, 
								   RUNTIME_CLASS(CListView), 
								   sizeof(PyCListView), 
								   ui_list_view_methods, 
								   GET_PY_CTOR(PyCListView));

/////////////////////////////////////////////////////////////////////
//
// Tree View object
//
//////////////////////////////////////////////////////////////////////
// @pymethod <o PyCTreeView>|win32ui|CreateTreeView|Creates a PyCTreeView object.
PyObject * PyCTreeView::create(PyObject *self, PyObject *args)
{
	PyObject *doc;
	// @pyparm <o PyCDocument>|doc||The document to use with the view.
	if (!PyArg_ParseTuple(args, "O:CreateTreeView", &doc))
		return NULL;
	if (!ui_base_class::is_uiobject(doc, &PyCDocument::type))
		RETURN_TYPE_ERR("Argument must be a PyCDocument");
	CDocument *pDoc = PyCDocument::GetDoc( doc );
	CTreeView *pView = new CPythonTreeView();
	((CProtectedView *)pView)->SetDocument(pDoc);
	return ui_assoc_object::make( PyCTreeView::type, pView );
}
// @pymethod tuple|PyCTreeView|PreCreateWindow|Calls the underlying MFC PreCreateWindow method.
PyObject *
PyCTreeView_pre_create_window(PyObject *self, PyObject *args)
{
	CTreeView *pView;
	if (!(pView=GetTreeViewPtr(self)))
		return NULL;
	CREATESTRUCT cs;
	//@pyparm tuple|createStruct||A tuple representing a CREATESTRUCT structure.
	if (!CreateStructFromPyObject( &cs, args, "PreCreateWindow", TRUE))
		return NULL;

	if (!pView->CTreeView::PreCreateWindow(cs))
		RETURN_ERR("CTreeView::PreCreateWindow failed");
	return PyObjectFromCreateStruct(&cs);
}

// @pymethod <o PyCTreeCtrl>|PyCTreeView|GetTreeCtrl|Returns the underlying tree control object.
static PyObject *
PyCTreeView_get_tree_ctrl(PyObject *self, PyObject *args)
{
	CTreeView *pView;
	if (!(pView=GetTreeViewPtr(self)))
		return NULL;
	CTreeCtrl &lc = pView->GetTreeCtrl();
	return ui_assoc_object::make( UITypeFromCObject(&lc), &lc )->GetGoodRet();
}

///////////////////////////////////////
// @object PyCTreeView|A class which implementes a CTreeView.  Derived from <o PyCView> and <o PyCTreeCtrl> objects.
static struct PyMethodDef ui_tree_view_methods[] = {
	{"PreCreateWindow", PyCTreeView_pre_create_window, 1}, // @pymeth PreCreateWindow|Calls the underlying MFC PreCreateWindow method.
	{"GetTreeCtrl", PyCTreeView_get_tree_ctrl, 1}, // @pymeth GetTreeCtrl|Returns the underlying tree control object.
	{NULL, NULL}
};

PyCCtrlView_Type PyCTreeView::type("PyCTreeView", 
								   &PyCCtrlView::type, 
								   &PyCTreeCtrl::type, 
								   RUNTIME_CLASS(CTreeView), 
								   sizeof(PyCTreeView), 
								   ui_tree_view_methods, 
								   GET_PY_CTOR(PyCTreeView));


///////////////////////////////////////
//
// Form View Methods
//
// inherited from view
//
///////////////////////////////////////

// @pymethod <o PyCFormView>|win32ui|CreateFormView|Creates a form view object.
PyObject * PyCFormView::create(PyObject *self, PyObject *args)
{
	PyObject *doc;
	char *szTemplate = NULL;
	int iTemplate;
	// @pyparm <o PyCDocument>|doc||The document to use with the view.
	// @pyparm int|iTemplate||The ID of the dialog control.
	if (!PyArg_ParseTuple(args, "Oi:CreateFormView", &doc, &iTemplate)) {
		PyErr_Clear();
		// @pyparmalt1 <o PyCDocument>|doc||The document to use with the view.
		// @pyparmalt1 string|template||The ID of the dialog control.
		if (!PyArg_ParseTuple(args, "Os:CreateFormView", &doc, &szTemplate))
			return NULL;
	}
	if (!ui_base_class::is_uiobject(doc, &PyCDocument::type))
		RETURN_TYPE_ERR("Argument must be a PyCDocument");
	CDocument *pDoc = PyCDocument::GetDoc( doc );
	CFormView *pView;
	if (szTemplate)
		pView = new CPythonFormView(szTemplate);
	else
		pView = new CPythonFormView(iTemplate);
	((CProtectedView *)pView)->SetDocument(pDoc);
	return ui_assoc_object::make( PyCFormView::type, pView );
}

// @object PyCFormView|A class which implementes a CFormView (ie, a view based on a dialog resource.
static struct PyMethodDef PyCFormView_methods[] = {
	{NULL, NULL}
};

ui_type_CObject PyCFormView::type("PyCFormView", 
								&PyCView::type,
								RUNTIME_CLASS(CFormView), 
								sizeof(PyCFormView), 
								PyCFormView_methods, 
								GET_PY_CTOR(PyCFormView));
