# VSSEditor - Source Control Editor.

# Like a normal editor, but checks if the file is read-only, and first
# modification it will offer to check the file out (just like MSVC5 and VSS)

# Also provides an auto-reload facility, to automatically check if the file has
# been modified outside the editor, and will reload it (see below).

# To use this editor:
# * In each directory, create a Mssccprj.scc filem which is an INI file.
#   (This is the same name as VB uses - should be OK!)
#   Add a section "[Python]" and an entry "Project=projectname" (without the $)
# * In the interactive window, type the following:
#   >>> import win32ui
#   >>> win32ui.WriteProfileVal("Editor", "Module", "vsseditor")
#   and if you want
#   >>> win32ui.WriteProfileVal("Editor", "Auto Reload", 0)
#   which will prompt each time the file is reloaded  (Note that it will
#   _always_ prompt you if the file in the editor has been modified.
# * Restart Pythonwin.  Open a "read-only" file.  If the title-bar has
#   "(read-only)" in it, you have the new editor. Now try changing the file!


import editor, win32ui, win32api, win32con, afxres, os, string, sys

import traceback

g_iniName = "Mssccprj.scc" # Use the same INI name as VB!

g_sourceSafe = None

def FindVssProjectInfo(fullfname):
	"""Looks up the file system for an INI file describing the project.
	
	Looking up the tree is for ni style packages.
	
	Returns (projectName, pathToFileName) where pathToFileName contains
	the path from the ini file to the actual file.
	"""
	path, fnameonly = os.path.split(fullfname)
	origPath = path
	project = ""
	retPaths = [fnameonly]
	while not project:
		iniName = os.path.join(path, g_iniName)
		database = win32api.GetProfileVal("Python","Database", "", iniName)
		project = win32api.GetProfileVal("Python","Project", "", iniName)
		if project:
			break;
		# No valid INI file in this directory - look up a level.
		path, addpath = os.path.split(path)
		if not addpath: # Root?
			break
		retPaths.insert(0, addpath)
	if not project:
		win32ui.MessageBox("%s\r\n\r\nThis directory is not configured for Python/VSS" % origPath)
		return
	return project, string.join(retPaths, "/")
		
	
def CheckoutFile(fileName):
	global g_sourceSafe
	import pythoncom
	ok = 0
	# Assumes the fileName has a complete path,
	# and that the INI file can be found in that path
	# (or a parent path if a ni style package)
	try:
		rc = FindVssProjectInfo(fileName)
		if rc is None:
			return
		project, vssFname = rc
		if g_sourceSafe is None:
			import ni,win32com.client
			g_sourceSafe=win32com.client.Dispatch("SourceSafe")
			g_sourceSafe.Open()
		item = g_sourceSafe.VSSItem("$/%s/%s" % (project, vssFname))
		item.Checkout(None, fileName)
		ok = 1
	except pythoncom.com_error, (hr, msg, exc, arg):
		if exc:
			msg = exc[2]
		win32ui.MessageBox(msg, "Error checking out file")
	except:
		typ, val = sys.exc_type, sys.exc_value
		traceback.print_exc()
		win32ui.MessageBox("%s - %s" % (str(typ), str(val)),"Error checking out file")
	return ok
	
class EditorDocument(editor.EditorDocument):
	def __init__(self, template):
		editor.EditorDocument.__init__(self, template)
		self.fileStat = None
		self.filename = None
		self.bAutoReload = win32ui.GetProfileVal("Editor","Auto Reload", 1)
		self.bDeclinedReload = 0 # Has the user declined to reload.

	def _ChangeReadonlyTitle(self):
		if not self.filename: return # New file - nothing to do
		try:
			# This seems necessary so the internal state of the window becomes
			# "visible".  without it, it is still shown, but certain functions
			# (such as updating the title) dont immediately work?
			self.GetFirstView().ShowWindow(win32con.SW_SHOW)
			title = win32ui.GetFileTitle(self.filename)
		except win32ui.error:
			title = self.filename
		if self._IsReadOnly():
			title = title + " (read-only)"
		self.SetTitle(title)
	def _IsReadOnly(self):
		return self.fileStat is not None and (self.fileStat[0] & 128)==0

	def OnOpenDocument(self, filename):
		if not editor.EditorDocument.OnOpenDocument(self, filename):
			return 0
		if filename:
			self.filename = win32api.GetFullPathName(filename)
			self.fileStat = os.stat(self.filename)
			self._ChangeReadonlyTitle()
		return 1

	def OnSaveDocument( self, fileName ):
		if not editor.EditorDocument.OnSaveDocument(self, fileName):
			return 0
		if self.filename:
			self.filename = win32api.GetFullPathName(self.filename)
			self.fileStat = os.stat(self.filename)
		return 1

	def OnCloseDocument(self):
		return self._obj_.OnCloseDocument()
		
	def ReloadDocument(self):
		self.SetModifiedFlag(0)
		# Loop over all views, saving their state, then reload the document
		views = self.GetAllViews()
		states = []
		for view in views:
			try:
				info = view._PrepareUserStateChange()
			except AttributeError: # Not our editor view?
				info = None
			states.append(info)
		self.OnOpenDocument(self.filename)
		for view, info in map(None, views, states):
			if info is not None:
				view._EndUserStateChange(info)
		
	def CheckExternalFileUpdated(self):
		if self.bDeclinedReload or not self.filename:
			return
		try:
			newstat = os.stat(self.filename)
		except os.error, (code, msg):
			print "Warning on file %s - %s" % (self.filename, msg)
			self.bDeclinedReload = 1
			return
		if self.fileStat[0] != newstat[0] or \
		   self.fileStat[6] != newstat[6] or \
		   self.fileStat[8] != newstat[8] or \
		   self.fileStat[9] != newstat[9]:
			question = None
			if self.IsModified():
				question = "%s\r\n\r\nThis file has been modified outside of the source editor.\r\nDo you want to reload it and LOSE THE CHANGES in the source editor?" % self.filename
			else:
				if not self.bAutoReload:
					question = "%s\r\n\r\nThis file has been modified outside of the source editor.\r\nDo you want to reload it?" % self.filename
			if question:
				rc = win32ui.MessageBox(question, None, win32con.MB_YESNO)
				if rc!=win32con.IDYES:
					self.bDeclinedReload = 1
					return
			self.ReloadDocument()

class EditorView(editor.EditorView):
	def __init__(self, doc):
		self.idleHandlerSet = 0
		editor.EditorView.__init__(self, doc)
		self.bCheckingFile = 0
	def HookHandlers(self):
		editor.EditorView.HookHandlers(self)
		self.HookAllKeyStrokes(self.OnKey)
		self.HookMessage(self.OnKillFocus, win32con.WM_KILLFOCUS)
		self.HookMessage(self.OnSetFocus, win32con.WM_SETFOCUS)
		self.HookMessage(self.OnKeyDown, win32con.WM_KEYDOWN)
		self.HookCommand(self.OnEditPaste, afxres.ID_EDIT_PASTE)
	
	def _DeleteIdleHandler(self):
		if self.idleHandlerSet:
			self.idleHandlerSet = 0
			try:
				win32ui.GetApp().DeleteIdleHandler(self.CheckFileUpdate)
			except:
				pass

	def OnEditPaste(self, id, code):
		if self.GetDocument()._IsReadOnly():
			# Return 1 if we can make the file editable.
			return self.MakeFileWritable()
		return 1

	def OnKeyDown(self, msg):
		key = msg[2]
		if key in [win32con.VK_DELETE, win32con.VK_BACK]:
			if self.GetDocument()._IsReadOnly():
				# Return 1 if we can make the file editable.
				return self.MakeFileWritable()
		return 1 # Pass it on OK
		
	def OnDestroy(self, msg):
		self._DeleteIdleHandler()
		return editor.EditorView.OnDestroy(self, msg)
		
	def MakeFileWritable(self):
		msg = "Would you like to check this file out?"
		if self.GetModify(): msg = msg + "\r\n\r\nAll changes will be lost!"
		if self.MessageBox(msg, None, win32con.MB_YESNO)==win32con.IDYES:
			if CheckoutFile(self.GetDocument().GetPathName()):
				self.GetDocument().ReloadDocument()
				return 1
		return 0
	
	def OnKey(self, key):
		if self.GetDocument()._IsReadOnly():
			# Return 1 if we can make the file editable.
			return self.MakeFileWritable()
		else:
			return 1 # Pass it on OK
	def OnKillFocus(self,msg):
		self._DeleteIdleHandler()
	def OnSetFocus(self,msg):
		self.CheckFileUpdate(self.CheckFileUpdate,0)
		win32ui.GetApp().AddIdleHandler(self.CheckFileUpdate)
		self.idleHandlerSet = 1
	def CheckFileUpdate(self, handler, count):
		if self._obj_ is None or self.bCheckingFile: return
		self.bCheckingFile = 1
		try:
			self.GetDocument().CheckExternalFileUpdated()
		except:
			traceback.print_exc()
			print "Idle handler failed!"
			self._DeleteIdleHandler()
		self.bCheckingFile = 0
		return 0 # No more idle handling required.

class EditorTemplate(editor.EditorTemplate):
	def InitialUpdateFrame(self, frame, doc, makeVisible=1):
		self._obj_.InitialUpdateFrame(frame, doc, makeVisible) # call default handler.
		fileName = doc.GetPathName()
		doc._ChangeReadonlyTitle()

if __name__==win32ui.GetProfileVal("Editor","Module", "editor"):
	# For debugging purposes, when this module may be reloaded many times.
	try:
		win32ui.GetApp().RemoveDocTemplate(editorTemplate)
	except (NameError, win32ui.error):
		pass

	editorTemplate = EditorTemplate(win32ui.IDR_TEXTTYPE, EditorDocument, None, EditorView)
	win32ui.GetApp().AddDocTemplate(editorTemplate)

#def test(fname = None):
#	EditorTemplate(win32ui.IDR_TEXTTYPE, EditorDocument, None, EditorView).OpenDocumentFile(fname)
