"""
Various utilities for running/importing a script
"""
import app
import sys
import win32ui
import win32con
import __main__
import dialog
import os
import string
import traceback

from cmdline import ParseArgs

class DlgRunScript(dialog.Dialog):
	"A class for the 'run script' dialog"
	def __init__(self):
		dialog.Dialog.__init__(self, win32ui.IDD_RUN_SCRIPT )
		self.AddDDX(win32ui.IDC_EDIT1, "script")
		self.AddDDX(win32ui.IDC_EDIT2, "args")
		self.HookCommand(self.OnBrowse, win32ui.IDC_BUTTON2)
		
	def OnBrowse(self, id, cmd):
		openFlags = win32con.OFN_OVERWRITEPROMPT|win32con.OFN_FILEMUSTEXIST
		dlg = win32ui.CreateFileDialog(1,None,None,openFlags, "Python Scripts (*.py)|*.py||", self)
		dlg.SetOFNTitle("Run Script")
		if dlg.DoModal()!=win32con.IDOK:
			return 0
		self['script'] = dlg.GetPathName()
		self.UpdateData(0)
		return 0
		
lastScript = ''
lastArgs = ''

def IsOnPythonPath(path):
	"Given a path only, see if it is on the Pythonpath.  Assumes path is a full path spec."
	# must check that the command line arg's path is in sys.path
	for syspath in sys.path:
		try:
			if win32ui.FullPath(syspath)==path:
				return 1
		except win32ui.error, details:
			print "Warning: The sys.path entry '%s' is invalid\n%s" % (syspath, details)
	return 0

def GetPackageModuleName(fileName):
	"""Given a filename, return (module name, new path).
	   eg - given "c:\a\b\c\my.py", return ("b.c.my",None) if "c:\a" is on sys.path.
	   If no package found, will return ("my", "c:\a\b\c")
	"""
	import os
	path, fname = os.path.split(fileName)
	path=origPath=win32ui.FullPath(path)
	fname = os.path.splitext(fname)[0]
	modBits = []
	newPathReturn = None
	if not IsOnPythonPath(path):
		# Module not directly on the search path - see if under a package.
		while len(path)>3: # ie 'C:\'
			path, modBit = os.path.split(path)
			modBits.append(modBit)
			# If on path, _and_ existing package of that name loaded.
			if IsOnPythonPath(path) and sys.modules.has_key(modBit):
				modBits.reverse()
				return string.join(modBits, ".") + "." + fname, newPathReturn
			# Not found - look a level higher
		else:
			newPathReturn = origPath
		
	return fname, newPathReturn

def RunScript(defName=None, defArgs=None, bShowDialog = 1):
	import linecache
	global lastScript, lastArgs
	if defName is None:
		pathName = ""
		try:
			active = win32ui.GetMainFrame().GetWindow(win32con.GW_CHILD).GetWindow(win32con.GW_CHILD)
			doc = active.GetActiveDocument()
			pathName = doc.GetPathName()
			
			if len(pathName)>0 or \
			    doc.GetTitle()[:8]=="Untitled" or \
			    doc.GetTitle()[:6]=="Script": # if not a special purpose window
				if doc.IsModified():
					try:
						doc.OnSaveDocument(pathName)
					except:
						return	# user cancelled
				# clear the linecache buffer
				linecache.clearcache()
		except (win32ui.error, AttributeError):
			pass
	else:
		pathName = defName
	if defArgs is None:
		args = ''
		if len(pathName)==0:
			pathName = lastScript
		if pathName==lastScript:
			args = lastArgs		
	else:
		args = defArgs

	if not pathName or bShowDialog:
		dlg = DlgRunScript()
		dlg['script'] = pathName
		dlg['args'] = args
		if dlg.DoModal() != win32con.IDOK:
			return
		script=dlg['script']
		args=dlg['args']
		if not script: return
	else:
		script = pathName
	lastScript = script
	lastArgs = args
	# try and open the script.
	if len(os.path.splitext(script)[1])==0:	# check if no extension supplied, and give one.
			script = script + '.py'
	# If no path specified, try and locate the file
	path, fnameonly = os.path.split(script)
	if len(path)==0:
		try:
			os.stat(fnameonly) # See if it is OK as is...
			script = fnameonly
		except SystemError: #os.error:
			fullScript = app.LocatePythonFile(script)
			if fullScript is None:
				win32ui.MessageBox("The file '%s' can not be located" % script )
				return
			script = fullScript
	else:
		path = win32ui.FullPath(path)
		if not IsOnPythonPath(path): sys.path.append(path)
		
	try:
		f = open(script)
	except IOError, (code, msg):
		win32ui.MessageBox("The file could not be opened - %s (%d)" % (msg, code))
		return

	oldArgv = sys.argv
	sys.argv = ParseArgs(args)
	sys.argv.insert(0, script)
	bWorked = 0
	win32ui.DoWaitCursor(1)
	base = os.path.split(script)[1]
	win32ui.PumpWaitingMessages()
	win32ui.SetStatusText('Running script %s...' % base,1 )
	exitCode = 0
	oldFlag = None
	try:
		oldFlag = sys.stdout.template.writeQueueing
		sys.stdout.template.writeQueueing = 0
	except (NameError, AttributeError):
		pass
	try:
		exec f in __main__.__dict__
		bWorked = 1
	except SystemExit, code:
		exitCode = code
		bWorked = 1
	except:
		# Reset queueing before exception for nice clean printing.
		if not oldFlag is None:
			sys.stdout.writeQueueing = oldFlag
			oldFlag = None
		traceback.print_exception(sys.exc_type, sys.exc_value, sys.exc_traceback)
	sys.argv = oldArgv
	f.close()
	if not oldFlag is None:
		sys.stdout.template.writeQueueing = oldFlag
		
	if bWorked:
		win32ui.SetStatusText("Script '%s' returned exit code %d" %(script, exitCode))
	else:
		win32ui.SetStatusText('Exception raised while running script  %s' % base)
	try:
		sys.stdout.flush()
	except:
		pass

	win32ui.DoWaitCursor(0)
		
def ImportFile():
	import linecache
	""" This code looks for the current window, and determines if it can be imported.  If not,
	it will prompt for a file name, and allow it to be imported. """
	doc = None
	try:
		active = win32ui.GetMainFrame().GetWindow(win32con.GW_CHILD).GetWindow(win32con.GW_CHILD)
		doc = active.GetActiveDocument()
	except:
		pass
		
	if not doc is None:
		if doc.GetTitle()[:8]=="Untitled" or doc.GetTitle()[:6]=="Script":
			try:
				doc.DoFileSave()
			except:
				return 0

	if doc is not None:
		pathName = doc.GetPathName()
		if string.lower(os.path.splitext(pathName)[1]) <> ".py":
			doc = None
			
	if doc is None:
		openFlags = win32con.OFN_OVERWRITEPROMPT|win32con.OFN_FILEMUSTEXIST
		dlg = win32ui.CreateFileDialog(1,None,None,openFlags, "Python Scripts (*.py)|*.py||")
		dlg.SetOFNTitle("Import Script")
		if dlg.DoModal()!=win32con.IDOK:
			return 0

		pathName = dlg.GetPathName()
		
	if doc is not None and doc.IsModified():
		try:
			doc.OnSaveDocument(pathName)
		except:
			return	# user cancelled
		# clear the linecache buffer
		linecache.clearcache()
		
	# If already imported, dont look for package
	path, modName = os.path.split(pathName)
	modName, modExt = os.path.splitext(modName)
	newPath = None
	for key in sys.modules.keys():
		if string.lower(key)==string.lower(modName):
			modName = key
			bNeedReload = 1
			what = "reload"
			break
	else: # for not broken
		what = "import"
		bNeedReload = 0
		modName, newPath = GetPackageModuleName(pathName)
		if newPath: sys.path.append(newPath)
	
	win32ui.SetStatusText(string.upper(what[0])+what[1:]+'ing module...',1)
	# always do an import, as it is cheap is already loaded.  This ensures
	# it is in our name space.
	bWorked = bSyntaxError = 0
	win32ui.DoWaitCursor(1)
#	win32ui.GetMainFrame().BeginWaitCursor()
	try:
		codeObj = compile('import '+modName,'<auto import>','exec')
		exec codeObj in __main__.__dict__
		if bNeedReload:
			reload(sys.modules[modName])
#			codeObj = compile('reload('+modName+')','<auto import>','eval')
#			exec codeObj in __main__.__dict__
		bWorked = 1
	except SyntaxError, details:
		bSyntaxError = 1
	except:
		traceback.print_exception(sys.exc_type, sys.exc_value, sys.exc_traceback)
	
	if bWorked:
		win32ui.SetStatusText('Successfully ' + what + "ed module '"+modName+"'")
	elif bSyntaxError:
		try:
			msg, (fileName, line, col, text) = details
			view = win32ui.GetApp().OpenDocumentFile(fileName).GetFirstView()
			charNo = view.LineIndex(line-1)
			view.SetSel(charNo + col - 1)
		except TypeError:
			msg = `details`
		win32ui.SetStatusText('Failed to ' + what + ' - syntax error - %s' % msg)
	else:
		win32ui.SetStatusText('Failed to ' + what + " module '"+modName+"'")
	win32ui.DoWaitCursor(0)
#		win32ui.GetMainFrame().EndWaitCursor()
	return None
