/*
  long awaited python toolbar class

  by Dave Brennan (brennan@hal.com)

Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc

*/

#include "stdafx.h"
#include "win32win.h"
#include "win32toolbar.h"


CControlBar *PyCControlBar::GetControlBar (PyObject *self)
{
	return (CControlBar *)ui_assoc_object::GetGoodCppObject( self, &type);
}

// @pymethod <o PyCFrameWnd>|PyCControlBar|GetDockingFrame|Returns the frame window to which a control bar is docked.
PyObject *PyCControlBar_GetDockingFrame( PyObject *self, PyObject *args ) 
{
	CControlBar *pCtlBar = PyCControlBar::GetControlBar(self);
	if (!pCtlBar) return NULL;
	if (!PyArg_ParseTuple( args, ":GetDockingFrame")) return NULL;
	CWnd *pWnd = pCtlBar->GetDockingFrame();
	if (pWnd==NULL)
		RETURN_ERR("There is no docking frame window");
	return ui_assoc_object::make(PyCFrameWnd::type, pWnd)->GetGoodRet();
}

// @pymethod int|PyCControlBar|GetCount|Returns the number of non-HWND elements in the control bar.
PyObject *PyCControlBar_GetCount( PyObject *self, PyObject *args ) 
{
	CControlBar *pCtlBar = PyCControlBar::GetControlBar(self);
	if (!pCtlBar) return NULL;
	if (!PyArg_ParseTuple( args, ":GetCount")) return NULL;
	return Py_BuildValue("i", pCtlBar->GetCount());
}
// @pymethod int|PyCControlBar|IsFloating|Returns a nonzero value if the control bar in question is a floating control bar.
PyObject *PyCControlBar_IsFloating( PyObject *self, PyObject *args ) 
{
	CControlBar *pCtlBar = PyCControlBar::GetControlBar(self);
	if (!pCtlBar) return NULL;
	if (!PyArg_ParseTuple( args, ":IsFloating")) return NULL;
	return Py_BuildValue("i", pCtlBar->IsFloating());
}


// @pymethod int|PyCControlBar|GetBarStyle|Retrieves the control bar style settings.
PyObject *PyCControlBar_GetBarStyle( PyObject *self, PyObject *args ) 
{
	CControlBar *pCtlBar = PyCControlBar::GetControlBar(self);
	if (!pCtlBar) return NULL;
	if (!PyArg_ParseTuple( args, ":GetBarStyle")) return NULL;
	return Py_BuildValue("i", pCtlBar->GetBarStyle());
}

// @pymethod |PyCControlBar|SetBarStyle|Modifies the control bar style settings.
PyObject *PyCControlBar_SetBarStyle( PyObject *self, PyObject *args ) 
{
	CControlBar *pCtlBar = PyCControlBar::GetControlBar(self);
	if (!pCtlBar) return NULL;
	int style;
	// @pyparm int|style||The new style
	if (!PyArg_ParseTuple( args, "i:SetBarStyle", &style))
		return NULL;
	pCtlBar->SetBarStyle(style);
	RETURN_NONE;
}

// @pymethod |PyCControlBar|EnableDocking|pecifies whether the control bar supports docking and the sides of its parent window.
PyObject *PyCControlBar_EnableDocking( PyObject *self, PyObject *args ) 
{
	CControlBar *pCtlBar = PyCControlBar::GetControlBar(self);
	if (!pCtlBar) return NULL;
	int style;
	// @pyparm int|style||Enables a control bar to be docked.
	if (!PyArg_ParseTuple( args, "i:EnableDocking", &style))
		return NULL;
	pCtlBar->EnableDocking(style);
	RETURN_NONE;
}

// @pymethod int|PyCControlBar|ShowWindow|Shows the toolbar, and recalculates the button layout.
static PyObject *
PyCControlBar_ShowWindow (PyObject *self, PyObject *args)
{
  // proto the base class method here
  extern PyObject *ui_window_show_window(PyObject *self, PyObject *args);

  // @comm This method is provided for convenience.  For further details, see
  // <om PyCWnd.ShowWindow> and <om PyCFrameWnd.RecalcLayout>

  PyObject *ret = ui_window_show_window(self, args);
  // call base first
  CWnd *pWnd = GetWndPtr (self);
  if (!pWnd)
    return NULL;

  CWnd *parent = pWnd->GetParent();
  if (parent && parent->IsKindOf (RUNTIME_CLASS(CFrameWnd)))
    ((CFrameWnd *) parent)->RecalcLayout();
  return ret;
  // @rdesc The return value is that returned from <om PyCWnd.ShowWindow>
}

// @object PyCControlBar|A class which encapsulates an MFC <o CControlBar>.  Derived from a <o PyCWnd> object.
static struct PyMethodDef 
PyCControlBar_methods[] =
{
	{"EnableDocking",	PyCControlBar_EnableDocking,1}, // @pymeth EnableDocking|Specifies whether the control bar supports docking and the sides of its parent window.
	{"GetBarStyle",	    PyCControlBar_GetBarStyle,1}, // @pymeth GetBarStyle|Retrieves the control bar style settings.
	{"GetCount",	    PyCControlBar_GetCount,1}, // @pymeth GetCount|Returns the number of non-HWND elements in the control bar.
	{"GetDockingFrame",	PyCControlBar_GetDockingFrame,1}, // @pymeth GetDockingFrame|Returns the frame window to which a control bar is docked.
	{"IsFloating",	    PyCControlBar_IsFloating,1}, // @pymeth IsFloating|Returns a nonzero value if the control bar in question is a floating control bar.
	{"SetBarStyle",	    PyCControlBar_SetBarStyle,1}, // @pymeth SetBarStyle|Modifies the control bar style settings.
	{"ShowWindow",		PyCControlBar_ShowWindow,1}, // @pymeth ShowWindow|Shows the window, and recalculates the toolbar layout.
  { NULL,			NULL }
};

ui_type_CObject PyCControlBar::type ("PyCControlBar",
					&PyCWnd::type, 
					RUNTIME_CLASS(CControlBar),
					sizeof(PyCControlBar),
					PyCControlBar_methods,
					NULL);

/* ToolBar Wish, er, ToDo List:

   change toolbar button styles (SetButtonInfo)

*/
#define MAKE_GET_INT_INT_METH(fnname, mfcName) \
PyObject *fnname( PyObject *self, PyObject *args ) { \
	CToolBar *pToolBar = PyCToolBar::GetToolBar(self); \
	if (!pToolBar) return NULL; \
	int val; \
	if (!PyArg_ParseTuple( args, "i:" #mfcName, &val)) \
		return NULL; \
	return Py_BuildValue("i", pToolBar->mfcName(val)); \
}

#define MAKE_SETVOID_INT_METH(fnname, mfcName) \
PyObject *fnname( PyObject *self, PyObject *args ) { \
	CToolBar *pToolBar = PyCToolBar::GetToolBar(self); \
	if (!pToolBar) return NULL; \
	int val; \
	if (!PyArg_ParseTuple( args, "i:" #mfcName, &val)) \
		return NULL; \
	pToolBar->mfcName(val); \
	RETURN_NONE; \
}


/* static */ CToolBar *
PyCToolBar::GetToolBar (PyObject *self)
{
  return (CToolBar *)ui_assoc_object::GetGoodCppObject( self, &type);
}

// @pymethod <o PyCToolBar>|win32ui|CreateToolBar|Creates a toolbar object.
PyObject *
PyCToolBar::create (PyObject *self, PyObject *args)
{
  PyObject *parent;
  int style;
  int id = AFX_IDW_TOOLBAR;
  if (!PyArg_ParseTuple (args,"Oi|i:CreateToolBar", 
            &parent, 	// @pyparm <o PyCWnd>|parent||The parent window for the toolbar.
            &style,     // @pyparm int|style||The style for the toolbar.
			&id))        // @pyparm int|windowId|afxres.AFX_IDW_TOOLBAR|The child window ID.
    return NULL;
  if (!ui_base_class::is_uiobject (parent, &PyCWnd::type))
    {
      RETURN_ERR("The parent param must be a window object.");
    }

  // @comm You must ensure no 2 toolbars share the same ID.
  CString error;
  CToolBar *tb = new CToolBar();
  CFrameWnd *frame = (CFrameWnd *)
	  PyCWnd::GetPythonGenericWnd (parent, &PyCFrameWnd::type);
  if (frame == NULL)
    return NULL;

  if (!tb->Create (frame, style, id)) {
	  delete tb;
	  RETURN_API_ERR("PyCToolBar.Create");
  }
  tb->m_bAutoDelete = TRUE;  // let MFC handle deletion
  return ui_assoc_object::make (PyCToolBar::type, tb)->GetGoodRet();
}

// @pymethod |PyCToolBar|SetButtons|Sets button styles and an index of button images within the bitmap.
PyObject *
PyCToolBar_SetButtons(PyObject *self, PyObject *args)
{
	PyObject *buttons;
	if (!PyArg_ParseTuple (args,"O:SetButtons", 
			&buttons)   // @pyparm tuple|buttons||A tuple containing the ID's of the buttons.
	    ||
	    !PyTuple_Check (buttons))
	  RETURN_ERR("SetButtons requires a tuple of IDs");
	CToolBar *pToolBar = PyCToolBar::GetToolBar(self);
	if (!pToolBar)
		return NULL;
	// convert button tuple to array
	int num_buttons = PyTuple_Size (buttons);
	UINT *button_list = new UINT[num_buttons];
	PyObject *o;
	for (int i = 0; i < num_buttons; i++) {
		o = PyTuple_GetItem (buttons, i);
		if (!PyInt_Check(o)) {
			delete button_list;
			RETURN_ERR ("SetButtons expected integer button ids.");
		}
		button_list[i] = PyInt_AsLong (o);
	}
	BOOL rc = pToolBar->SetButtons(button_list, num_buttons);
	delete button_list;
	if (!rc) RETURN_API_ERR("PyCToolBar.SetButtons");
	RETURN_NONE;
}

// @pymethod |PyCToolBar|LoadBitmap|Loads the bitmap containing bitmap-button images.
static PyObject *
PyCToolBar_LoadBitmap (PyObject *self, PyObject *args)
{
	BOOL rc;
	int id;
	CToolBar *pToolBar = PyCToolBar::GetToolBar(self);
	if (!pToolBar)
		return NULL;
	if (PyArg_ParseTuple(args,"i", 
	          &id )) // @pyparm int|id||The bitmap ID.
		rc = pToolBar->LoadBitmap(id);
	else {
		char *szId;
		PyErr_Clear();
		if (PyArg_ParseTuple(args,"i:LoadBitmap", 
	          &szId )) // @pyparmalt1 string|id||The bitmap ID.
			rc = pToolBar->LoadBitmap(szId);
		else
			RETURN_ERR("LoadBitmap requires an integer or string argument");
	}

	if (!rc)
		RETURN_ERR("LoadBitmap failed");
	// @comm The bitmap should contain one image for each toolbar button. If the 
	// images are not of the standard size (16 pixels wide and 15 pixels high), 
	// call <om PyCToolBar.SetSizes> to set the button sizes and their images.
	RETURN_NONE;
}
// @pymethod |PyCToolBar|LoadToolBar|Loads a toolbar from a toolbar resource.
static PyObject *
PyCToolBar_LoadToolBar (PyObject *self, PyObject *args)
{
	BOOL rc;
	int id;
	CToolBar *pToolBar = PyCToolBar::GetToolBar(self);
	if (!pToolBar)
		return NULL;
	if (PyArg_ParseTuple(args,"i", 
	          &id )) // @pyparm int|id||The bitmap ID.
		rc = pToolBar->LoadToolBar(id);
	else {
		char *szId;
		PyErr_Clear();
		if (PyArg_ParseTuple(args,"i:LoadBitmap", 
	          &szId )) // @pyparmalt1 string|id||The bitmap ID.
			rc = pToolBar->LoadToolBar(szId);
		else
			RETURN_ERR("LoadToolBar requires an integer or string argument");
	}

	if (!rc)
		RETURN_ERR("LoadBitmap failed");
	// @comm The bitmap should contain one image for each toolbar button. If the 
	// images are not of the standard size (16 pixels wide and 15 pixels high), 
	// call <om PyCToolBar.SetSizes> to set the button sizes and their images.
	RETURN_NONE;
}


// @pymethod |PyCToolBar|SetHeight|Sets the height of the toolbar.
// @pyparm int|height||The height in pixels of the toolbar.
MAKE_SETVOID_INT_METH(PyCToolBar_SetHeight, SetHeight )

// @pymethod |PyCToolBar|GetItemID|Returns the command ID of a button or separator at the given index.
// @pyparm int|index||Index of the item whose ID is to be retrieved.
MAKE_GET_INT_INT_METH(PyCToolBar_GetItemID, GetItemID )

// @pymethod |PyCToolBar|GetButtonStyle|Retrieves the style for a button.
// @pyparm int|index||Index of the item whose style is to be retrieved.
MAKE_GET_INT_INT_METH(PyCToolBar_GetButtonStyle, GetButtonStyle )

// @pymethod |PyCToolBar|SetBitmap|Sets a bitmapped image.
PyObject *PyCToolBar_SetBitmap( PyObject *self, PyObject *args ) 
{
	CToolBar *pToolBar = PyCToolBar::GetToolBar(self);
	if (!pToolBar) return NULL;
	int val;
	// @pyparm int|hBitmap||The handle to a bitmap resource.
	// @comm Call this method to set the bitmap image for the toolbar. For example, 
	// call SetBitmap to change the bitmapped image after the user takes an action on 
	// a document that changes the action of a button.
	if (!PyArg_ParseTuple( args, "i:SetBitmap", &val))
		return NULL;
	if (!IsWin32s() && ::GetObjectType((HBITMAP)val) != OBJ_BITMAP)
		RETURN_ERR("The bitmap handle is invalid");
	if (!pToolBar->SetBitmap((HBITMAP)val))
		RETURN_ERR("SetBitmap failed");
	RETURN_NONE;
}


// @pymethod |PyCToolBar|SetSizes|Sets the size of each button.
static PyObject *
PyCToolBar_SetSizes (PyObject *self, PyObject *args)
{
	SIZE sizeBut, sizeBmp;
	CToolBar *pToolBar = PyCToolBar::GetToolBar(self);
	if (!pToolBar)
		return NULL;
  
	if (!PyArg_ParseTuple(args,"(ii)(ii)", 
	          &sizeBut.cx, &sizeBut.cy,	 // @pyparm (cx, cy)|sizeButton||The size of each button.
		      &sizeBmp.cx, &sizeBmp.cy)) // @pyparm (cx, cy)|sizeButton||The size of each bitmap.
		return NULL;
	pToolBar->SetSizes(sizeBut, sizeBmp);
	RETURN_NONE;
}

// @pymethod |PyCToolBar|SetButtonStyle|Sets the style for a button.
static PyObject *
PyCToolBar_SetButtonStyle (PyObject *self, PyObject *args)
{
	int index, style;
	CToolBar *pToolBar = PyCToolBar::GetToolBar(self);
	if (!pToolBar)
		return NULL;
	if (!PyArg_ParseTuple(args,"ii", 
			&index, // @pyparm int|index||Index of the item whose style is to be set
			&style))// @pyparm int|style||The new style
		return NULL;
	pToolBar->SetButtonStyle(index, style);
	RETURN_NONE;
}

// @pymethod string|PyCToolBar|GetButtonText|Gets the text for a button.
PyObject *PyCToolBar_GetButtonText( PyObject *self, PyObject *args ) 
{
	CToolBar *pToolBar = PyCToolBar::GetToolBar(self);
	if (!pToolBar) return NULL;
	int index;
	// @pyparm int|index||Index of the item whose text is to be retrieved.
	if (!PyArg_ParseTuple( args, "i:GetButtonText", &index))
		return NULL;
	return Py_BuildValue("s", (const char *)pToolBar->GetButtonText(index));
}

// @pymethod |PyCToolBar|SetButtonText|Sets the text for a button.
static PyObject *
PyCToolBar_SetButtonText(PyObject *self, PyObject *args)
{
	int index;
	char *text;
	CToolBar *pToolBar = PyCToolBar::GetToolBar(self);
	if (!pToolBar)
		return NULL;
	if (!PyArg_ParseTuple(args,"is", 
			&index, // @pyparm int|index||Index of the item whose style is to be set
			&text))// @pyparm string|text||The new text
		return NULL;
	pToolBar->SetButtonText(index, text);
	RETURN_NONE;
}

// @object PyCToolBar|A class which encapsulates an MFC <o CToolBar>.  Derived from a <o PyCControlBar> object.
static struct PyMethodDef 
PyCToolBar_methods[] =
{
	{"GetButtonStyle",  PyCToolBar_GetButtonStyle,1}, // @pymeth GetButtonStyle|Retrieves the style for a button.
	{"GetButtonText",   PyCToolBar_GetButtonText,1}, // @pymeth GetButtonText|Gets the text for a button.
	{"GetItemID",       PyCToolBar_GetItemID,1}, // @pymeth GetItemID|Returns the command ID of a button or separator at the given index.
	{"LoadBitmap",      PyCToolBar_LoadBitmap,1}, // @pymeth LoadBitmap|Loads the bitmap containing bitmap-button images.
	{"LoadToolBar",     PyCToolBar_LoadToolBar,1}, // @pymeth LoadToolBar|Loads a toolbar from a Toolbar resource.
	{"SetBitmap",       PyCToolBar_SetBitmap,1}, // @pymeth SetBitmap|Sets a bitmapped image.
	{"SetButtons",      PyCToolBar_SetButtons,1}, // @pymeth SetButtons|Sets button styles and an index of button images within the bitmap.
	{"SetButtonStyle",  PyCToolBar_SetButtonStyle,1}, // @pymeth SetButtonStyle|Sets the style for a button
	{"SetHeight",       PyCToolBar_SetHeight,1},  // @pymeth SetHeight|Sets the height of the toolbar.
	{"SetSizes",		PyCToolBar_SetSizes,1},   // @pymeth SetSizes|Sets the sizes for the toolbar items.
  { NULL,			NULL }
};

ui_type_CObject PyCToolBar::type ("PyCToolBar",
					&PyCControlBar::type, 
					RUNTIME_CLASS(CToolBar),
					sizeof(PyCToolBar),
					PyCToolBar_methods,
					GET_PY_CTOR(PyCToolBar));
