/* win32ctllist : implementation file

	List control object.  

	Created May 1996, Mark Hammond (MHammond@skippinet.com.au)

Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc

*/
#include "stdafx.h"

#include "win32win.h"
#include "win32dc.h"
#include "win32control.h"
#include "win32ctrlList.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

const char *szErrListRequiresWindow = "The list must have a window object for this operation";

PyCListCtrl::PyCListCtrl()
{
}
PyCListCtrl::~PyCListCtrl()
{
}
CListCtrl *GetListCtrl(PyObject *self)
{
	extern CListView *GetListViewPtr(PyObject *self);
	CListCtrl *rc;

	if (ui_base_class::is_uiobject(self, &PyCListView::type)) {
		CListView *pView = GetListViewPtr(self);
		if (pView)
			rc = &(pView->GetListCtrl());
		else
			rc = NULL;
	} else
		rc = (CListCtrl *)PyCWnd::GetPythonGenericWnd(self, &PyCListCtrl::type);
	if (rc && !::IsWindow(rc->m_hWnd))
		RETURN_ERR((char *)szErrListRequiresWindow);
	return rc;
}

#define MAKE_GET_INT_METH(fnname, mfcName) \
PyObject *fnname( PyObject *self, PyObject *args ) { \
	CHECK_NO_ARGS2(args,mfcName); \
	CListCtrl *pList = GetListCtrl(self); \
	if (!pList) return NULL; \
	int ret = pList->mfcName(); \
	return Py_BuildValue("i",ret); \
}

#define MAKE_SETBOOL_INT_METH(fnname, mfcName) \
PyObject *fnname( PyObject *self, PyObject *args ) { \
	CListCtrl *pList = pList=GetListCtrl(self); \
	if (!pList) return NULL; \
	int val; \
	if (!PyArg_ParseTuple( args, "i:" #mfcName, &val)) \
		return NULL; \
	if (!pList->mfcName(val)) \
		RETURN_ERR(#mfcName "failed"); \
	RETURN_NONE; \
}
#define MAKE_SETVOID_INT_METH(fnname, mfcName) \
PyObject *fnname( PyObject *self, PyObject *args ) { \
	CListCtrl *pList = pList=GetListCtrl(self); \
	if (!pList) return NULL; \
	int val; \
	if (!PyArg_ParseTuple( args, "i:" #mfcName, &val)) \
		return NULL; \
	pList->mfcName(val); \
	RETURN_NONE; \
}

// @pymethod int|PyCListCtrl|GetBkColor|Retrieves the background color of the control.
MAKE_GET_INT_METH(PyCListCtrl_GetBkColor, GetBkColor )

// @pymethod int|PyCListCtrl|GetTextColor|Retrieves the text color of a list view control.
MAKE_GET_INT_METH(PyCListCtrl_GetTextColor, GetTextColor )
// @pymethod |PyCListCtrl|SetTextColor|Sets the text color of a list view control.
// @pyparm int|color||The new color.
MAKE_SETBOOL_INT_METH(PyCListCtrl_SetTextColor, SetTextColor )

// @pymethod |PyCListCtrl|Update|Forces the control to repaint a specified item.
// @pyparm int|item||The new color.
MAKE_SETBOOL_INT_METH(PyCListCtrl_Update, Update )

// @pymethod |PyCListCtrl|Arrange|Aligns items on a grid.
// @pyparm int|code||Specifies the alignment style for the items
MAKE_SETBOOL_INT_METH(PyCListCtrl_Arrange, Arrange )

// @pymethod int|PyCListCtrl|GetTextBkColor|Retrieves the text background color of a list view control.
MAKE_GET_INT_METH(PyCListCtrl_GetTextBkColor, GetTextBkColor )
// @pymethod |PyCListCtrl|SetTextBkColor|Sets the text background color of a list view control.
// @pyparm int|color||The new background color.
MAKE_SETBOOL_INT_METH(PyCListCtrl_SetTextBkColor, SetTextBkColor )

// @pymethod int|PyCListCtrl|GetItemCount|Retrieves the number of items in a list view control.
MAKE_GET_INT_METH(PyCListCtrl_GetItemCount, GetItemCount )
// @pymethod |PyCListCtrl|SetItemCount|Prepares a list view control for adding a large number of items.
// @pyparm int|count||Number of items that the control will ultimately contain.
// @comm By calling this function before adding a large number of items,
// you enable a list view control to reallocate its internal data structures
// only once rather than every time you add an item.
MAKE_SETVOID_INT_METH(PyCListCtrl_SetItemCount, SetItemCount )


// @pymethod |PyCListCtrl|SetBkColor|Sets the background color of the control.
// @pyparm int|color||The new background color.
MAKE_SETBOOL_INT_METH(PyCListCtrl_SetBkColor, SetBkColor)

// @pymethod int|PyCListCtrl|GetTopIndex|Retrieves the index of the topmost visible item.
MAKE_GET_INT_METH(PyCListCtrl_GetTopIndex, GetTopIndex )

// @pymethod int|PyCListCtrl|GetCountPerPage|Calculates the number of items that can fit vertically in a list view control.
MAKE_GET_INT_METH(PyCListCtrl_GetCountPerPage, GetCountPerPage )

// @pymethod int|PyCListCtrl|GetSelectedCount|Retrieves the number of selected items in the list view control.
MAKE_GET_INT_METH(PyCListCtrl_GetSelectedCount, GetSelectedCount )

// @pymethod |PyCListCtrl|DeleteAllItems|Deletes all items from the list.
PyObject *PyCListCtrl_DeleteAllItems( PyObject *self, PyObject *args )
{
	CListCtrl *pList;
	if (!(pList=GetListCtrl(self)))
		return NULL;
	CHECK_NO_ARGS2(args, "DeleteAllItems");
	if (!pList->DeleteAllItems())
		RETURN_ERR("DeleteAllItems failed");
	RETURN_NONE;
}

// @pymethod <o PyCImageList>|PyCListCtrl|GetImageList|Retrieves the current image list.
PyObject *PyCListCtrl_GetImageList( PyObject *self, PyObject *args )
{
	CListCtrl *pList;
	if (!(pList=GetListCtrl(self)))
		return NULL;
	int nList;
	// @pyparm int|nImageList||Value specifying which image list to retrieve. It can be one of:
	// <nl>-	commctrl.LVSIL_NORMAL   Image list with large icons.
	// <nl>-	commctrl.LVSIL_SMALL   Image list with small icons.
	// <nl>-	commctrl.LVSIL_STATE   Image list with state images.
	if (!PyArg_ParseTuple(args, "i:GetImageList", &nList))
		return NULL;
	CImageList *ret = pList->GetImageList(nList);
	if (ret==NULL)
		RETURN_ERR("There is no image list available");
	return ui_assoc_object::make( PyCImageList::type, ret)->GetGoodRet();
}

// @pymethod int|PyCListCtrl|InsertColumn|Inserts a column into a list control when in report view.
PyObject *PyCListCtrl_InsertColumn( PyObject *self, PyObject *args )
{
	CListCtrl *pList;
	int iColNo;
	PyObject *obLVCol;
	
	if (!(pList=GetListCtrl(self)))
		return NULL;

	if (!PyArg_ParseTuple(args, "iO:InsertColumn",
			             &iColNo,   // @pyparm int|colNo||The new column number
		                 &obLVCol)) // @pyparm <om PyCListCtrl.LV_COLUMN tuple>|item||A tuple describing the new column.
		return NULL;
	LV_COLUMN lvCol;
	if (!ParseLV_COLUMNTuple(obLVCol, &lvCol))
		return NULL;
	int ret = pList->InsertColumn(iColNo, &lvCol);
	if (ret==-1)
		RETURN_ERR("InsertColumn failed");
	return Py_BuildValue("i",ret);
}

// @pymethod int|PyCListCtrl|InsertItem|Inserts an item into the list.
PyObject *PyCListCtrl_InsertItem( PyObject *self, PyObject *args )
{
	CListCtrl *pList;
	int ret;
	int item;
	char *text;
	int image;
	PyObject *obLVItem;
	if (!(pList=GetListCtrl(self)))
		return NULL;

	if (PyArg_ParseTuple(args, "isi:InsertItem", 
		                 &item, // @pyparmalt1 int|item||The index of the item.
						 &text, // @pyparmalt1 string|text||The text of the item.
						 &image)) // @pyparmalt1 int|image||The index of the image to use.
		ret = pList->InsertItem(item, text, image );

	else {
		PyErr_Clear();
		if (PyArg_ParseTuple(args, "is:InsertItem", 
		                 &item, // @pyparmalt2 int|item||The index of the item.
						 &text))// @pyparmalt2 string|test||The text of the item.
			ret = pList->InsertItem(item, text );
		else {
			PyErr_Clear();
			if (PyArg_ParseTuple(args, "O:InsertItem",
							 &obLVItem)) { // @pyparm <om PyCListCtrl.LV_ITEM tuple>|item||A tuple describing the new item.
				LV_ITEM lvItem;
				if (!ParseLV_ITEMTuple(obLVItem, &lvItem))
					return NULL;
				ret = pList->InsertItem(&lvItem);
			} else {
				PyErr_Clear();
				RETURN_ERR("InsertItem requires (item, text, image), (item, text), or (itemObject)");
			}
		}
	}
	if (ret==-1)
		RETURN_ERR("InsertItem failed");
	return Py_BuildValue("i",ret);
}

// @pymethod int|PyCListCtrl|SetItem|Sets some of all of an items attributes.
PyObject *PyCListCtrl_SetItem( PyObject *self, PyObject *args )
{
	CListCtrl *pList;
	PyObject *obLVItem;
	if (!(pList=GetListCtrl(self)))
		return NULL;
	if (!PyArg_ParseTuple(args, "O:SetItem",
		                 &obLVItem)) // @pyparm <om PyCListCtrl.LV_ITEM tuple>|item||A tuple describing the new item.
		return NULL;
	LV_ITEM lvItem;
	if (!ParseLV_ITEMTuple(obLVItem, &lvItem))
		return NULL;
	if (!pList->SetItem(&lvItem))
		RETURN_ERR("SetItem failed");
	RETURN_NONE;
}

// @pymethod int|PyCListCtrl|SetImageList|Assigns an image list to a list view control.
PyObject *PyCListCtrl_SetImageList( PyObject *self, PyObject *args )
{
	CListCtrl *pList;
	PyObject *obList;
	int imageType;
	if (!(pList=GetListCtrl(self)))
		return NULL;
	if (!PyArg_ParseTuple(args, "Oi:SetImageList", 
		                  &obList, // @pyparm <o PyCImageList>|imageList||The Image List to use.
						  &imageType )) // @pyparm int|imageType||Type of image list. It can be one of (COMMCTRL.) LVSIL_NORMAL, LVSIL_SMALL or LVSIL_STATE
		return NULL;
	CImageList *pImageList = PyCImageList::GetImageList(obList);
	if (pImageList==NULL) return NULL;
	CImageList *pOldList = pList->SetImageList( pImageList, imageType );
	if (pOldList==NULL)
		RETURN_NONE;
	return ui_assoc_object::make( PyCImageList::type, pOldList )->GetGoodRet();
}

// @pymethod <om PyCListCtrl.LV_COLUMN tuple>|PyCListCtrl|GetColumn|Retrieves the details of a column in the control.
PyObject *PyCListCtrl_GetColumn( PyObject *self, PyObject *args )
{
	int col;
	if (!PyArg_ParseTuple( args, "i:GetColumn", 
	                   &col)) // @pyparm int|column||The index of the column whose attributes are to be retrieved.
		return NULL;
	CListCtrl *pList = GetListCtrl(self);
	if (!pList) return NULL;
	char textBuf[256];
	LV_COLUMN lvItem;
	lvItem.pszText = textBuf;
	lvItem.cchTextMax = sizeof(textBuf);
	lvItem.mask = LVCF_FMT | LVCF_SUBITEM | LVCF_TEXT | LVCF_WIDTH;
	if (!pList->GetColumn( col, &lvItem) )
		RETURN_ERR("GetColumn failed");
	return MakeLV_COLUMNTuple(&lvItem);
}

// @pymethod <om PyCListCtrl.LV_ITEM tuple>|PyCListCtrl|GetItem|Retrieves the details of an items attributes.
PyObject *PyCListCtrl_GetItem( PyObject *self, PyObject *args )
{
	int item, sub = 0;
	if (!PyArg_ParseTuple( args, "i|i:GetItem", 
	                   &item, // @pyparm int|item||The index of the item whose attributes are to be retrieved.
					   &sub)) // @pyparm int|sub||Specifies the subitem whose text is to be retrieved.
		return NULL;

	CListCtrl *pList = GetListCtrl(self);
	if (!pList) return NULL;
	char textBuf[256];
	LV_ITEM lvItem;
	lvItem.iItem = item;
	lvItem.iSubItem = sub;
	lvItem.pszText = textBuf;
	lvItem.cchTextMax = sizeof(textBuf);
	lvItem.mask = LVIF_TEXT | LVIF_IMAGE | LVIF_PARAM | LVIF_STATE;
	if (!pList->GetItem( &lvItem) )
		RETURN_ERR("GetItem failed");
	return MakeLV_ITEMTuple(&lvItem);
}

// @pymethod int|PyCListCtrl|GetItemText|Retrieves the text of a list view item or subitem.
PyObject *PyCListCtrl_GetItemText( PyObject *self, PyObject *args )
{
	int item, sub;
	char buf[256];
	if (!PyArg_ParseTuple( args, "ii:GetItemText", 
	                   &item, // @pyparm int|item||The index of the item whose text is to be retrieved.
					   &sub)) // @pyparm int|sub||Specifies the subitem whose text is to be retrieved.
		return NULL;
	CListCtrl *pList = GetListCtrl(self);
	if (!pList) return NULL;
	if (!pList->GetItemText(item, sub, buf, sizeof(buf)))
		RETURN_ERR("GetItemText failed");
	return Py_BuildValue("s",buf);
}

// @pymethod int|PyCListCtrl|SetItemText|Changes the text of a list view item or subitem.
PyObject *PyCListCtrl_SetItemText( PyObject *self, PyObject *args )
{
	CListCtrl *pList = pList=GetListCtrl(self);
	if (!pList) return NULL;
	int item, sub;
	char *text;
	if (!PyArg_ParseTuple( args, "iis:SetItemText", 
		                   &item, // @pyparm int|item||Index of the item whose text is to be set.
						   &sub, // @pyparm int|sub||Index of the subitem, or zero to set the item label.
						   &text)) // @pyparm string|text||String that contains the new item text.

		return NULL;
	if (!pList->SetItemText(item, sub, text))
		RETURN_ERR("SetItemText failed");
	RETURN_NONE;
}

// @pymethod int|PyCListCtrl|GetItemState|Retrieves the state of a list view item.
PyObject *PyCListCtrl_GetItemState( PyObject *self, PyObject *args )
{
	int item, mask;
	if (!PyArg_ParseTuple( args, "ii:GetItemState", 
	                   &item, // @pyparm int|item||The index of the item whose position is to be retrieved.
					   &mask)) // @pyparm int|mask||Mask specifying which of the items state flags to return.

		return NULL;
	CListCtrl *pList = GetListCtrl(self);
	if (!pList) return NULL;
	return Py_BuildValue("i", pList->GetItemState(item, mask));
}

// @pymethod int|PyCListCtrl|SetItemState|Changes the state of an item in a list view control.
PyObject *PyCListCtrl_SetItemState( PyObject *self, PyObject *args )
{
	CListCtrl *pList = pList=GetListCtrl(self);
	if (!pList) return NULL;
	int item, state, mask;
	if (!PyArg_ParseTuple( args, "iii:SetItemState", 
		                   &item, // @pyparm int|item||Index of the item whose state is to be set.
						   &state, // @pyparm int|state||New values for the state bits.
						   &mask)) // @pyparm int|mask||Mask specifying which state bits to change.
		return NULL;
	if (!pList->SetItemState(item, state, mask))
		RETURN_ERR("SetItemState failed");
	RETURN_NONE;
}
// @pymethod object|PyCListCtrl|GetItemData|Retrieves the application-specific value associated with an item.
PyObject *PyCListCtrl_GetItemData( PyObject *self, PyObject *args )
{
	int item;
	if (!PyArg_ParseTuple( args, "i:GetItemData", 
	                   &item)) // @pyparm int|item||The index of the item whose data is to be retrieved.

		return NULL;
	CListCtrl *pList = GetListCtrl(self);
	if (!pList) return NULL;
	PyObject *ret = PyWin_GetPythonObjectFromLong(pList->GetItemData(item));
	// inc ref count for return value.
	Py_INCREF(ret);
	return ret;
}

// @pymethod int|PyCListCtrl|SetItemData|Sets the items application-specific value.
PyObject *PyCListCtrl_SetItemData( PyObject *self, PyObject *args )
{
	CListCtrl *pList = pList=GetListCtrl(self);
	if (!pList) return NULL;
	int item;
	PyObject *data;
	if (!PyArg_ParseTuple( args, "iO:SetItemData", 
		                   &item, // @pyparm int|item||Index of the item whose Data is to be set.
						   &data)) // @pyparm object|Data||New value for the data.
		return NULL;
	if (data==Py_None) data = NULL;
	if (!pList->SetItemData(item, (DWORD)data))
		RETURN_ERR("SetItemData failed");
	// @comm Note that a reference count is not added to the object.  This it is your
	// responsibility to make sure the object remains alive while in the list.
	RETURN_NONE;
}

// @pymethod int|PyCListCtrl|GetNextItem|Searches for a list view item with specified properties and with specified relationship to a given item.
PyObject *PyCListCtrl_GetNextItem( PyObject *self, PyObject *args )
{
	int item, flags;
	if (!PyArg_ParseTuple( args, "ii:GetNextItem", 
	                   &item, // @pyparm int|item||The index of the item whose text is to be retrieved.
					   &flags)) // @pyparm int|flags||Geometric relation of the requested item to the specified item, 
					            // and the state of the requested item. The geometric relation can be one of these values:
								// <nl>LVNI_ABOVE<nl>LVNI_ALL<nl>LVNI_BELOW<nl>LVNI_TOLEFT<nl>LVNI_TORIGHT<nl>
								// The state can be zero, or it can be one or more of these values:
								// <nl>LVNI_DROPHILITED<nl>LVNI_FOCUSED<nl>LVNI_HIDDEN<nl>LVNI_MARKED<nl>LVNI_SELECTED<nl>
								// If an item does not have all of the specified state flags set, the search continues with the next item.
		return NULL;
	CListCtrl *pList = GetListCtrl(self);
	if (!pList) return NULL;
	int rc = pList->GetNextItem(item, flags);
	if (rc==(int)-1)
		RETURN_ERR("GetNextItem failed");
	return Py_BuildValue("i",rc);
}

// @pymethod int|PyCListCtrl|RedrawItems|Forces a listview to repaint a range of items.
PyObject *PyCListCtrl_RedrawItems( PyObject *self, PyObject *args )
{
	int first, last;
	if (!PyArg_ParseTuple( args, "ii:RedrawItems", 
	                   &first, // @pyparm int|first||Index of the first item to be repainted.
					   &last)) // @pyparm int|first||Index of the last item to be repainted.
		return NULL;
	CListCtrl *pList = GetListCtrl(self);
	if (!pList) return NULL;
	if (!pList->RedrawItems(first, last))
		RETURN_ERR("RedrawItems failed");
	RETURN_NONE;
	// @comm The specified items are not actually repainted until the list view window receives a WM_PAINT message.
	// To repaint immediately, call the Windows UpdateWindow function after using this function.
}

// @pymethod (int, int, int, int)|PyCListCtrl|GetItemRect|Retrieves the bounding rectangle of a tree view item.
PyObject *PyCListCtrl_GetItemRect( PyObject *self, PyObject *args )
{
	CListCtrl *pList = pList=GetListCtrl(self);
	if (!pList) return NULL;
	int item;
	RECT rect;
	BOOL bTextOnly;
	if (!PyArg_ParseTuple( args, "ii:GetItemRect", 
		                   &item, // @pyparm int|item||Index of the item whose Data is to be set.
						   &bTextOnly)) // @pyparm int|bTextOnly||f this parameter is nonzero, the bounding rectangle includes only the text of the item. Otherwise it includes the entire line that the item occupies in the tree view control.
		return NULL;
	if (!pList->GetItemRect(item, &rect, bTextOnly))
		RETURN_ERR("GetItemRect failed");
	return Py_BuildValue("(iiii)",rect.left, rect.top, rect.right, rect.bottom);
}

// @pymethod <o PyCEdit>|PyCListCtrl|GetEditControl|Retrieves the handle of the edit control used to edit the specified tree view item.
PyObject *PyCListCtrl_GetEditControl( PyObject *self, PyObject *args )
{
	CListCtrl *pList = pList=GetListCtrl(self);
	if (!pList) return NULL;
	if (!PyArg_ParseTuple( args, ":GetEditControl"))
		return NULL;
	CEdit *pEdit = pList->GetEditControl();
	if (pEdit==NULL)
		RETURN_ERR("GetEditControl failed");
	return ui_assoc_object::make(UITypeFromCObject(pEdit), pEdit)->GetGoodRet();
}

// @pymethod <o PyCEdit>|PyCListCtrl|EditLabel|Edits a specified tree view item in-place.
PyObject *PyCListCtrl_EditLabel( PyObject *self, PyObject *args )
{
	CListCtrl *pList = pList=GetListCtrl(self);
	if (!pList) return NULL;
	int item;
	// @pyparm int|item||The index of item to edit.
	if (!PyArg_ParseTuple( args, "i:EditLabel", &item))
		return NULL;
	CEdit *pEdit = pList->EditLabel(item);
	if (pEdit==NULL)
		RETURN_ERR("EditLabel failed");
	return ui_assoc_object::make(UITypeFromCObject(pEdit), pEdit)->GetGoodRet();
}

// @pymethod int|PyCListCtrl|EnsureVisible|Ensures that a tree view item is visible in its tree view control.
PyObject *PyCListCtrl_EnsureVisible( PyObject *self, PyObject *args )
{
	CListCtrl *pList = pList=GetListCtrl(self);
	if (!pList) return NULL;
	int item;
	BOOL bPartialOK;
	// @pyparm int|item||The index of item to edit.
	// @pyparm int|bPartialOK||Specifies whether partial visibility is acceptable.
	if (!PyArg_ParseTuple( args, "ii:EnsureVisible", &item, &bPartialOK))
		return NULL;
	if (!pList->EnsureVisible(item,bPartialOK))
		RETURN_ERR("EnsureVisible failed");
	RETURN_NONE;
}

// @pymethod <o PyCImageList>,(x,y)|PyCListCtrl|CreateDragImage|Creates a dragging bitmap for the specified tree view item.
PyObject *PyCListCtrl_CreateDragImage( PyObject *self, PyObject *args )
{
	CListCtrl *pList = pList=GetListCtrl(self);
	if (!pList) return NULL;
	int item;
	// @pyparm int|item||The index of the item to edit.
	if (!PyArg_ParseTuple( args, "i:CreateDragImage", &item))
		return NULL;
	POINT pt;
	CImageList *pIL = pList->CreateDragImage(item, &pt);
	if (pIL==NULL)
		RETURN_ERR("CreateDragImage failed");
	PyObject *newOb = ui_assoc_object::make(PyCImageList::type, pIL)->GetGoodRet();
	PyObject *ret = Py_BuildValue("O(ii)", newOb, pt.x, pt.y);
	Py_DECREF(newOb);
	return ret;
}


// @object PyCListCtrl|A class which encapsulates an MFC CListCtrl object.  Derived from a <o PyCWnd> object.
static struct PyMethodDef PyCListCtrl_methods[] = {
	{"Arrange",        PyCListCtrl_Arrange, 1}, // @pymeth Arrange|Aligns items on a grid.
	{"DeleteAllItems", PyCListCtrl_DeleteAllItems, 1}, // @pymeth DeleteAllItems|Deletes all items from the list.
	{"GetTextColor",   PyCListCtrl_GetTextColor, 1}, // @pymeth GetTextColor|Retrieves the text color of a list view control.
	{"SetTextColor",   PyCListCtrl_SetTextColor, 1}, // @pymeth SetTextColor|Sets the text color of a list view control.
	{"GetBkColor",     PyCListCtrl_GetBkColor, 1}, // @pymeth GetBkColor|Retrieves the background color of the control.
	{"SetBkColor",     PyCListCtrl_SetBkColor,  1}, // @pymeth SetBkColor|Sets the background color of the control.
	{"GetItem",        PyCListCtrl_GetItem,  1}, // @pymeth GetItem|Retrieves the details of an items attributes.
	{"GetItemCount",   PyCListCtrl_GetItemCount,  1}, // @pymeth GetItemCount|Retrieves the number of items in a list view control.
	{"GetItemRect",    PyCListCtrl_GetItemRect, 1}, // @pymeth GetItemRect|Retrieves the bounding rectangle of a tree view item.
	{"GetEditControl",   PyCListCtrl_GetEditControl, 1}, // @pymeth GetEditControl|Retrieves the handle of the edit control used to edit the specified tree view item.
	{"EditLabel",      PyCListCtrl_EditLabel,  1}, // @pymeth EditLabel|Edits a specified tree view item in-place.
	{"EnsureVisible",  PyCListCtrl_EnsureVisible,  1}, // @pymeth EnsureVisible|Ensures that a tree view item is visible in its tree view control.
	{"CreateDragImage",PyCListCtrl_CreateDragImage,  1}, // @pymeth CreateDragImage|Creates a dragging bitmap for the specified tree view item.
	{"GetImageList",   PyCListCtrl_GetImageList,  1}, // @pymeth GetImageList|Retrieves the current image list.
	{"GetNextItem",    PyCListCtrl_GetNextItem,  1}, // @pymeth GetNextItem|Searches for a list view item with specified properties and with specified relationship to a given item.
	{"InsertColumn",   PyCListCtrl_InsertColumn,  1}, // @pymeth InsertColumn|Inserts a column into a list control when in report view.
	{"InsertItem",     PyCListCtrl_InsertItem,  1}, // @pymeth InsertItem|Inserts an item into the list.
	{"SetImageList",   PyCListCtrl_SetImageList, 1}, // @pymeth SetImageList|Assigns an image list to a list view control.
	{"GetColumn",      PyCListCtrl_GetColumn, 1}, // @pymeth GetColumn|Retrieves the details of a column in the control.
	{"GetTextBkColor", PyCListCtrl_GetTextBkColor, 1}, // @pymeth GetTextBkColor|Retrieves the text background color of a list view control.
	{"SetTextBkColor", PyCListCtrl_SetTextBkColor, 1}, // @pymeth SetTextBkColor|Sets the text background color of a list view control.
	{"GetTopIndex",    PyCListCtrl_GetTopIndex, 1}, // @pymeth GetTopIndex|Retrieves the index of the topmost visible item.
	{"GetCountPerPage",PyCListCtrl_GetCountPerPage, 1}, // @pymeth GetCountPerPage|Calculates the number of items that can fit vertically in a list view control.
	{"GetSelectedCount",PyCListCtrl_GetSelectedCount, 1}, // @pymeth GetSelectedCount|Retrieves the number of selected items in the list view control.
	{"SetItem",        PyCListCtrl_SetItem, 1}, // @pymeth SetItem|Sets some of all of an items attributes.
	{"SetItemState",   PyCListCtrl_SetItemState, 1}, // @pymeth SetItemState|Changes the state of an item in a list view control.
	{"GetItemState",   PyCListCtrl_GetItemState, 1}, // @pymeth GetItemState|Retrieves the state of a list view item.
	{"SetItemData",    PyCListCtrl_SetItemData, 1}, // @pymeth SetItemData|Sets the items application-specific value.
	{"GetItemData",    PyCListCtrl_GetItemData, 1}, // @pymeth GetItemData|Retrieves the application-specific value associated with an item.
	{"SetItemCount",   PyCListCtrl_SetItemCount, 1}, // @pymeth SetItemCount|Prepares a list view control for adding a large number of items.
	{"GetItemCount",   PyCListCtrl_GetItemCount, 1}, // @pymeth GetItemCount|Retrieves the number of items in a list view control.
	{"SetItemText",    PyCListCtrl_SetItemText, 1}, // @pymeth SetItemText|Changes the text of a list view item or subitem.
	{"GetItemText",    PyCListCtrl_GetItemText, 1}, // @pymeth GetItemText|Retrieves the text of a list view item or subitem.
	{"RedrawItems",    PyCListCtrl_RedrawItems, 1}, // @pymeth RedrawItems|Redraws a range of items
	{"Update",         PyCListCtrl_Update, 1}, // @pymeth Update|Forces the control to repaint a specified item.
	{NULL,			NULL}
};


ui_type_CObject PyCListCtrl::type("PyCListCtrl", 
									 &PyCWnd::type, 
									 RUNTIME_CLASS(CListCtrl), 
									 sizeof(PyCListCtrl), 
									 PyCListCtrl_methods, 
									 GET_PY_CTOR(PyCListCtrl));
