/* win32control : implementation file

	Control object - base class for listboxes, editbox's, prompts, etc.

	Created August 1994, Mark Hammond (MHammond@skippinet.com.au)

Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc

*/
#include "stdafx.h"

#include "win32win.h"
#include "win32control.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

#define IS_LB_ERR(rc) (rc==LB_ERR||rc==LB_ERRSPACE)
#define IS_CB_ERR(rc) (rc==CB_ERR||rc==CB_ERRSPACE)


///////////////////////////////////////////////////////////////////////////
//
// The control objects.
//
/* Implement an psuedo-inheritance for ControlView */
PyObject *
PyCCtrlView::getattr(char *name)
{
	// implement inheritance.
	PyObject *retMethod = PyCView::getattr(name);
	if (!retMethod) {
		PyErr_Clear();
		PyCCtrlView_Type *thisType = (PyCCtrlView_Type *)ob_type;
		if (thisType)
			retMethod = Py_FindMethod(thisType->control->methods, (PyObject *)this, name);
	}
	return retMethod;
}

// Control view.
// @object PyCCtrlView|A windows Rich Text edit control.  Encapsulates an MFC <c CRichEditCtrl> class.  Derived from a <o PyCControl> object.
static struct PyMethodDef PyCCtrlView_methods[] = {
	{ NULL, NULL }
};

ui_type_CObject PyCCtrlView::type("PyCCtrlView",
								  &PyCView::type, 
								  RUNTIME_CLASS(CCtrlView), 
								  sizeof(PyCCtrlView), 
								  PyCCtrlView_methods, 
								  NULL);

///////////////////////////////////////////////////////////////////////////
//
// The control objects.
//
ui_control_object::ui_control_object()
{
}
ui_control_object::~ui_control_object()
{
}

/////////////////////////////////////////////////////////////////////
//
// ui_control_object
//
// @object PyCControl|A windows abstract control.  Derived from a <o PyCWnd> object.
static struct PyMethodDef ui_control_object_methods[] = {
	{NULL,			NULL}		/* sentinel */
};

ui_type_CObject ui_control_object::type("PyCControl", 
										&PyCWnd::type, 
										RUNTIME_CLASS(CObject), 
										sizeof(ui_control_object), 
										ui_control_object_methods, 
										NULL);


/////////////////////////////////////////////////////////////////////
//
// PyCButton
//
static CButton *GetButton(PyObject *self)
{
	// note we can only ask for a CWnd, if the LB is created from a resource based
	// dialog.  This is also the technique MFC uses (specifically appdlg.cpp)
	return (CButton *)PyCWnd::GetPythonGenericWnd(self);
}
PyCButton::PyCButton()
{
}
PyCButton::~PyCButton()
{
}
// @pymethod <o PyCButton>|win32ui|CreateButton|Creates a button object.  <om PyCButton.CreateWindow> creates the actual control.
PyObject *
PyCButton_create(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CButton *pBut = new CButton();
	return ui_assoc_object::make( PyCButton::type, pBut );
}

// @pymethod |PyCButton|CreateWindow|Creates the window for a new button object.
static PyObject *
PyCButton_create_window(PyObject *self, PyObject *args)
{
	char *caption;
	int style, id;
	PyObject *obParent;
	RECT rect;
	if (!PyArg_ParseTuple(args, "si(iiii)Oi:CreateWindow", 
		       &caption, // @pyparm string|caption||The caption (text) for the button.
			   &style, // @pyparm int|style||The style for the button.  Use any of the win32con.BS_* constants.
			   &rect.left,&rect.top,&rect.right,&rect.bottom,
			   // @pyparm (left, top, right, bottom)|rect||The size and position of the button.
			   &obParent, // @pyparm <o PyCWnd>|parent||The parent window of the button.  Usually a <o PyCDialog>.
			   &id )) // @pyparm int|id||The buttons control ID. 
		return NULL;

	if (!ui_base_class::is_uiobject(obParent, &PyCWnd::type))
		RETURN_TYPE_ERR("parent argument must be a window object");
	CWnd *pParent = GetWndPtr( obParent );
	if (pParent==NULL)
		return NULL;
	CButton *pBut = GetButton(self);
	if (!pBut)
		return NULL;

	if (!pBut->Create(caption, style, rect, pParent, id ))
		RETURN_ERR("CButton::Create");
	RETURN_NONE;
}

// @pymethod int|PyCButton|GetCheck|Retrieves the check state of a radio button or check box.
static PyObject *
PyCButton_get_check(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CButton *pBut = GetButton(self);
	if (!pBut)
		return NULL;
	return Py_BuildValue("i", pBut->GetCheck());
}
// @pymethod |PyCButton|SetCheck|Sets or resets the state of a radio button or check box.
static PyObject *
PyCButton_set_check(PyObject *self, PyObject *args)
{
	int check;
	if (!PyArg_ParseTuple(args, "i", &check)) // @pyparm int|idCheck||The ID of the button.
		return NULL;
	CButton *pBut = GetButton(self);
	if (!pBut)
		return NULL;
	pBut->SetCheck(check);
	RETURN_NONE;
}
// @pymethod int|PyCButton|GetState|Returns the state of the button.
static PyObject *
PyCButton_get_state(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CButton *pBut = GetButton(self);
	if (!pBut)
		return NULL;
	return Py_BuildValue("i", pBut->GetState()); 
}
// @pymethod |PyCButton|SetState|Sets the state of the button.
static PyObject *
PyCButton_set_state(PyObject *self, PyObject *args)
{
	int state;
	if (!PyArg_ParseTuple(args, "i", &state))  // @pyparm int|state||The new state for the button.
		return NULL;
	CButton *pBut = GetButton(self);
	if (!pBut)
		return NULL;
	pBut->SetState(state);
	return Py_BuildValue("i", state);
}
// @pymethod int|PyCButton|GetButtonStyle|Gets the style of the button.
static PyObject *
PyCButton_get_style(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CButton *pBut = GetButton(self);
	if (!pBut)
		return NULL;
	return Py_BuildValue("i", pBut->GetButtonStyle());
}
// @pymethod int|PyCButton|SetButtonStyle|Sets the style of the button.
static PyObject *
PyCButton_set_style(PyObject *self, PyObject *args)
{
	int style;
	BOOL bRedraw = TRUE;
	if (!PyArg_ParseTuple(args, "i|i", 
	          &style, // @pyparm int|style||The new style for the button.
	          &bRedraw))// @pyparm int|bRedraw|1|Should the button be redrawn?
		return NULL;
	CButton *pBut = GetButton(self);
	if (!pBut)
		return NULL;
	pBut->SetButtonStyle(style, bRedraw);
	return Py_BuildValue("i", style);
}

// @object PyCButton|A windows button.  Encapsulates an MFC <c CButton> class.  Derived from <o PyCControl>.
static struct PyMethodDef PyCButton_methods[] = {
	{"CreateWindow",    PyCButton_create_window,1}, // @pymeth CreateWindow|Creates the window for a new button object.
	{"GetCheck",		PyCButton_get_check,	1}, // @pymeth GetCheck|Retrieves the check state of a radio button or check box.
	{"SetCheck",		PyCButton_set_check,	1},	// @pymeth SetCheck|Sets the check state of a radio button or check box.
	{"GetState",		PyCButton_get_state,	1},	// @pymeth GetState|Retrieves the state of a radio button or check box.
	{"SetState",		PyCButton_set_state,	1},	// @pymeth SetState|Sets the state of a radio button or check box.
	{"GetButtonStyle",	PyCButton_get_style,	1},	// @pymeth GetButtonStyle|Retrieves the style of a radio button or check box.
	{"SetButtonStyle",	PyCButton_set_style,	1},	// @pymeth SetButtonStyle|Sets the state of a radio button or check box.
	{NULL,				NULL}
};

ui_type_CObject PyCButton::type("PyCButton", 
								&ui_control_object::type, 
								RUNTIME_CLASS(CButton), 
								sizeof(PyCButton), 
								PyCButton_methods, 
								GET_PY_CTOR(PyCButton));

/////////////////////////////////////////////////////////////////////
//
// PyCListBox
//
static CListBox *GetListBox(PyObject *self)
{
	// note we can only ask for a CWnd, if the LB is created from a resource based
	// dialog.  This is also the technique MFC uses (specifically appdlg.cpp)
	return (CListBox *)PyCWnd::GetPythonGenericWnd(self);
}
PyCListBox::PyCListBox()
{
}
PyCListBox::~PyCListBox()
{
}
// @pymethod int|PyCListBox|AddString|Adds a string to a listbox.
static PyObject *
PyCListBox_add_string(PyObject *self, PyObject *args)
{
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	PyObject *ob;
	if (!PyArg_ParseTuple(args, "O", &ob )) // @pyparm any|object||Any object.  If not a string, __str__, __repr__ or a default repr() will be used
		return NULL;
	//@pyseemfc CListBox|AddString
	int rc=pLB->AddString( GetReprText(ob) );
	if (IS_LB_ERR(rc))
		RETURN_ERR("PyCListBox.AddString failed");
	return Py_BuildValue("i", rc);
	//@rdesc The zero based index of the new string.
}
// @pymethod int|PyCListBox|DeleteString|Deletes an item from a listbox.
static PyObject *
PyCListBox_delete_string(PyObject *self, PyObject *args)
{
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	int pos;
	if (!PyArg_ParseTuple(args,"i",&pos)) // @pyparm int|pos||The zero based index of the item to delete.
		return NULL;
	int rc=pLB->DeleteString(pos);	// @pyseemfc CListBox|DeleteString
	if (IS_LB_ERR(rc))
		RETURN_ERR("PyCListBox.DeleteString failed");
	return Py_BuildValue("i", rc);
	// @rdesc The count of the items remaining in the list.
}
// @pymethod int|PyCListBox|Dir|Fills a listbox with a directory listing.
static PyObject *
PyCListBox_dir(PyObject *self, PyObject *args)
{
	int attr;
	char *szWild;
	if (!PyArg_ParseTuple(args,"is",
	          &attr,    // @pyparm int|attr||The attributes of the files to locate
	          &szWild)) // @pyparm string|wild||A file specification string - eg, *.*
		return NULL;
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	int rc=pLB->Dir(attr, szWild);	// @pyseemfc CListBox|Dir
	if (IS_LB_ERR(rc))
		RETURN_ERR("PyCListBox.Dir failed");
	return Py_BuildValue("i", rc);
	// @rdesc The index of the last file name added to the list.
}

// @pymethod int|PyCListBox|InsertString|Insert a string into a listbox.
static PyObject *
PyCListBox_insert_string(PyObject *self, PyObject *args)
{
	int pos;
	PyObject *ob;
	if (!PyArg_ParseTuple(args, "iO",
	          &pos, // @pyparm int|pos||The zero based index in the listbox to insert the new string
	          &ob)) // @pyparm any|object||The object to be added to the listbox
		return NULL;
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	int rc=pLB->InsertString( pos, GetReprText(ob) ); // @pyseemfc CListBox|InsertString
	if (IS_LB_ERR(rc))
		RETURN_ERR("PyCListBox.InsertString failed");
	return Py_BuildValue("i", rc);
	// @rdesc The zero based index of the new string added.
}
// @pymethod |PyCListBox|ResetContent|Clear all the items from a listbox.
static PyObject *
PyCListBox_reset_content(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	pLB->ResetContent(); // @pyseemfc CListBox|ResetContent
	RETURN_NONE;
}
// @pymethod int|PyCListBox|GetCaretIndex|Returns the index of the item which has focus.
static PyObject *
PyCListBox_get_caret_index(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CListBox *pLB = GetListBox(self); //pyseemfc CListBox|GetCaretIndex
	if (!pLB)
		return NULL;
	return Py_BuildValue("i", pLB->GetCaretIndex()); 
	// @rdesc The zero-based index of the item that has the focus rectangle in a list box.  
	//If the list box is a single-selection list box, the return value is the index of the item that is selected, if any.

}

// @pymethod int|PyCListBox|GetCount|Returns the count of items in the listbox.
static PyObject *
PyCListBox_get_count(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	return Py_BuildValue("i", pLB->GetCount()); // @pyseemfc CListBox|GetCount
	// @rdesc Returns the number of items currently in the listbox.
}
// @pymethod int|PyCListBox|GetCurSel|Returns the index of the currently selected item.
static PyObject *
PyCListBox_get_cur_sel(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	return Py_BuildValue("i", pLB->GetCurSel()); // @pyseemfc CListBox|GetCurSel
	//@comm Should not be called for a multiple selection listbox.
}
// @pymethod object|PyCListBox|GetItemData|Retrieves the application-specific object associated with an item.
PyObject *PyCListBox_GetItemData( PyObject *self, PyObject *args )
{
	int item;
	if (!PyArg_ParseTuple( args, "i:GetItemData", 
	                   &item)) // @pyparm int|item||The index of the item whose data is to be retrieved.

		return NULL;
	CListBox *pLB = GetListBox(self);
	if (!pLB) return NULL;
	PyObject *ret = PyWin_GetPythonObjectFromLong(pLB->GetItemData(item));
	// inc ref count for return value.
	Py_INCREF(ret);
	return ret;
}

// @pymethod int|PyCListBox|GetItemValue|Retrieves the application-specific value associated with an item.
PyObject *PyCListBox_GetItemValue( PyObject *self, PyObject *args )
{
	int item;
	if (!PyArg_ParseTuple( args, "i:GetItemValue", 
	                   &item)) // @pyparm int|item||The index of the item whose data is to be retrieved.

		return NULL;
	CListBox *pLB = GetListBox(self);
	if (!pLB) return NULL;
	return PyInt_FromLong((long)pLB->GetItemData(item));
}

// @pymethod int|PyCListBox|GetSel|Returns the selection state of a specified item.
static PyObject *
PyCListBox_get_sel(PyObject *self, PyObject *args)
{
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	int pos;
	if (!PyArg_ParseTuple(args,"i",&pos)) // @pyparm int|index||The index of the item to return the state for.
		return NULL;
	return Py_BuildValue("i", pLB->GetSel(pos)); // @pyseemfc CListBox|GetSel
	//@rdesc A +ve number if the item is selected, else zero. 
}
// @pymethod int|PyCListBox|GetSelCount|Returns the number of selected items in a multiple selection listbox.
static PyObject *
PyCListBox_get_sel_count(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	int ret=pLB->GetSelCount(); // @pyseemfc CListBox|GetSelCount
	if (ret==LB_ERR)
		RETURN_ERR("Listbox is a single selection listbox");
	return Py_BuildValue("i", ret);
}
// @pymethod list|PyCListBox|GetSelItems|Returns a list of the indexes of the currently selected items in a multiple selection listbox.
static PyObject *
PyCListBox_get_sel_items(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	int numItems = pLB->GetSelCount(); // @pyseemfc CListBox|GetSelCount
	if (numItems==0)
		return PyList_New(0);	// return an empty list
	if (numItems==LB_ERR)
		RETURN_ERR("Listbox is a single selection listbox");
	int *rgItems = new int[numItems];
	if (rgItems==NULL)
		RETURN_ERR("Memory error");
	if (pLB->GetSelItems(numItems,rgItems)!=numItems) { // @pyseemfc CListBox|GetSelItems
		delete rgItems;
		RETURN_ERR("GetSelItems failed!");
	}
	PyObject *list;
	if ((list = PyList_New(numItems)) == NULL) {
		delete rgItems;
		return NULL;
	}
	for (int i=0;i<numItems;i++)
		PyList_SetItem( list, i, Py_BuildValue("i", rgItems[i]));

	delete rgItems;
	return list;
}
// @pymethod list|PyCListBox|GetSelTextItems|Returns a list of the strings of the currently selected items in a multiple selection listbox.
static PyObject *
PyCListBox_get_sel_text_items(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	int numItems = pLB->GetSelCount(); // @pyseemfc CListBox|GetSelCount
	if (numItems==0)
		return PyList_New(0);	// return an empty list
	if (numItems==LB_ERR)
		RETURN_ERR("Listbox is a single selection listbox");
	int *rgItems = new int[numItems];
	if (rgItems==NULL)
		RETURN_ERR("Memory error");
	if (pLB->GetSelItems(numItems,rgItems)!=numItems) { // @pyseemfc CListBox|GetSelItems
		delete rgItems;
		RETURN_ERR("GetSelItems failed!");
	}
	PyObject *list;
	if ((list = PyList_New(numItems)) == NULL) {
		delete rgItems;
		return NULL;
	}
	for (int i=0;i<numItems;i++) {
		CString value;
		pLB->GetText(rgItems[i], value); // @pyseemfc CListBox|GetText
		PyList_SetItem( list, i, Py_BuildValue("s", value));
	}

	delete rgItems;
	return list;
}

// @pymethod string|PyCListBox|GetText|Returns the string for a specified item.
static PyObject *
PyCListBox_get_text(PyObject *self, PyObject *args)
{
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	int pos;
	if (!PyArg_ParseTuple(args,"i",&pos)) //@pyparm int|index||The index of the item to retrieve the text of
		return NULL;
	CString cs;
	pLB->GetText(pos, cs); // @pyseemfc CListBox|GetText
    return Py_BuildValue("s", (const char *)cs);
}
// @pymethod int|PyCListBox|GetTextLen|Returns the length of the string for a specified item.
static PyObject *
PyCListBox_get_text_len(PyObject *self, PyObject *args)
{
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	int pos;
	if (!PyArg_ParseTuple(args,"i",&pos)) //@pyparm int|index||The index of the item to retrieve the length of the text.
		return NULL;
	return Py_BuildValue("i", pLB->GetTextLen(pos)); // @pyseemfc CListBox|GetTextLen
}

// @pymethod int|PyCListBox|GetTopIndex|Returns the index of the top most visible item.
static PyObject *
PyCListBox_get_top_index(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	return Py_BuildValue("i", pLB->GetTopIndex()); // @pyseemfc CListBox|GetTopIndex
	// @rdesc The zero based index of the top most visible item.
}

// @pymethod |PyCListBox|SelectString|Searches for a list-box item that matches the specified string, and selects it.
static PyObject *
PyCListBox_select_string(PyObject *self, PyObject *args)
{
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	char *string;
	int after;
	if (!PyArg_ParseTuple(args,"is",
	          &after, 	// @pyparm int|after||Contains the zero-based index of the item before the first item to be searched, or -1 for the entire listbox.
	          &string)) // @pyparm string|string||The string to search for.
		return NULL;
	if (pLB->SelectString(after, string)==LB_ERR) // @pyseemfc CListBox|SelectString
		RETURN_ERR("The string does not exist");
	RETURN_NONE;
	// @rdesc The return value is always None - an exception is raised if the string can not be located.
}

// @pymethod |PyCListBox|SelItemRange|Selects an item range.
static PyObject *
PyCListBox_sel_item_range(PyObject *self, PyObject *args)
{
	int bSel, start, end;
	if (!PyArg_ParseTuple(args, "iii",
	          &bSel,  // @pyparm int|bSel||Should the selection specified be set or cleared?
	          &start, // @pyparm int|start||The zero based index of the first item to select.
	          &end))  // @pyparm int|end||The zero based index of the last item to select.
		return NULL;
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	if (pLB->SelItemRange(bSel, start, end)==LB_ERR)
		RETURN_ERR("SelItemRange failed");
	RETURN_NONE;
}

// @pymethod |PyCListBox|SetCaretIndex|Sets the focus rectange to a specified item.
static PyObject *
PyCListBox_set_caret_index(PyObject *self, PyObject *args)
{
	int index;
	BOOL bScroll = TRUE;
	if (!PyArg_ParseTuple(args,"i|i",
	          &index,  // @pyparm int|index||The zero based index of the item.
	          &bScroll))// @pyparm int|bScroll|1|Should the listbox scroll to the item?
		return NULL;
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	if (pLB->SetCaretIndex(index, bScroll)==LB_ERR) // @pyseemfc CListBox|SetCaretIndex
		RETURN_ERR("SetCaretIndex failed");
	RETURN_NONE;
}
// @pymethod |PyCListBox|SetSel|Selects an item in a multiple selection listbox.
static PyObject *
PyCListBox_set_sel(PyObject *self, PyObject *args)
{
	int index;
	BOOL bSel = TRUE;
	if (!PyArg_ParseTuple(args,"i|i",
	          &index, // @pyparm int|index||The zero based index of the item to select.
	          &bSel)) // @pyparm int|bSel|1|Should the item be selected or deselected?
		return NULL;
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	if (pLB->SetSel(index, bSel)==LB_ERR)	// @pyseemfc CListBox|SetSel
		RETURN_ERR("SetSel failed");
	RETURN_NONE;
}
// @pymethod |PyCListBox|SetCurSel|Selects an item in a single selection listbox.
static PyObject *
PyCListBox_set_cur_sel(PyObject *self, PyObject *args)
{
	int index;
	if (!PyArg_ParseTuple(args,"i",&index)) // @pyparm int|index||The zero based index of the item to select.
		return NULL;
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	if (pLB->SetCurSel(index)==LB_ERR && index!=LB_ERR) // @pyseemfc CListBox|SetCurSel
		RETURN_ERR("SetCurSel failed");
	RETURN_NONE;
}
// @pymethod int|PyCListBox|SetItemData|Sets the items application-specific object value.
PyObject *PyCListBox_SetItemData( PyObject *self, PyObject *args )
{
	CListBox *pLB = GetListBox(self);
	if (!pLB) return NULL;
	int item;
	PyObject *data;
	if (!PyArg_ParseTuple( args, "iO:SetItemData", 
		                   &item, // @pyparm int|item||Index of the item whose Data is to be set.
						   &data)) // @pyparm object|Data||New value for the data.
		return NULL;
	if (data==Py_None) data = NULL;
	if (!pLB->SetItemData(item, (DWORD)data))
		RETURN_ERR("SetItemData failed");
	// @comm Note that a reference count is not added to the object.  This it is your
	// responsibility to make sure the object remains alive while in the list.
	RETURN_NONE;
}

// @pymethod int|PyCListBox|SetItemValue|Sets the items application-specific value.
PyObject *PyCListBox_SetItemValue( PyObject *self, PyObject *args )
{
	CListBox *pLB = GetListBox(self);
	if (!pLB) return NULL;
	int item;
	int data;
	if (!PyArg_ParseTuple( args, "ii:SetItemValue", 
		                   &item, // @pyparm int|item||Index of the item whose Data is to be set.
						   &data)) // @pyparm int|data||New value for the data.
		return NULL;
	if (!pLB->SetItemData(item, (DWORD)data))
		RETURN_ERR("SetItemValue failed");
	RETURN_NONE;
}

// @pymethod |PyCListBox|SetTabStops|Sets the tab stops for a listbox.
static PyObject *
PyCListBox_set_tab_stops(PyObject *self, PyObject *args)
{
	CListBox *pLB = GetListBox(self);
	int index;
	BOOL rc;
	if (!pLB)
		return NULL;
	// @pyparm int|eachTabStop||The position for each tab stop.
	if (PyArg_ParseTuple(args,"i",&index)) {
		rc = pLB->SetTabStops(index);
	}
	else {
		PyObject *listOb;
		PyErr_Clear();
		// @pyparmalt1 list of integers|tabStops||Each individual tab stop.
		if (!PyArg_ParseTuple(args,"O",&listOb))
			return NULL;
		if (!PyList_Check(listOb))
			RETURN_TYPE_ERR("Param must be a list object");
		int numChildren = PyList_Size(listOb);
		int *pArray = new int[numChildren];
		int tabVal;
		for (int child=0;child<numChildren;child++) {
			PyObject *obChild = PyList_GetItem(listOb, child );
			if (!PyArg_Parse( obChild, "i", &tabVal)) {
				delete pArray;
				RETURN_TYPE_ERR("List param must be a list of integers");
			}
			pArray[child] = tabVal;
		}
		rc = pLB->SetTabStops( numChildren, pArray );
		delete pArray;
	}

	if (!rc)
		RETURN_ERR("SetTabStops failed");
	RETURN_NONE;
}


// @pymethod |PyCListBox|SetTopIndex|Sets the top index (top most visible item) of the listbox.
static PyObject *
PyCListBox_set_top_index(PyObject *self, PyObject *args)
{
	int index;
	if (!PyArg_ParseTuple(args,"i",&index)) // @pyparm int|index||The zero based index of the item to place at the top of the list.
		return NULL;
	CListBox *pLB = GetListBox(self);
	if (!pLB)
		return NULL;
	if (pLB->SetTopIndex(index)==LB_ERR)	// @pyseemfc CListBox|SetTopIndex
		RETURN_ERR("SetTopIndex failed");
	RETURN_NONE;
}

// @object PyCListBox|A windows listbox control.  Encapsulates an MFC <c CListBox> class.  Derived from a <o PyCControl> object.
static struct PyMethodDef PyCListBox_methods[] = {
	{"AddString",			PyCListBox_add_string,	1}, // @pymeth AddString|Add a string to the listbox.
	{"DeleteString",		PyCListBox_delete_string,	1}, // @pymeth DeleteString|Delete a string from the listbox.
	{"Dir",					PyCListBox_dir,	1}, // @pymeth Dir|Fill a listbox with a file specification.
	{"GetCaretIndex",		PyCListBox_get_caret_index,	1}, // @pymeth GetCaretIndex|Get the index of the item with the focus rectangle.
	{"GetCount",			PyCListBox_get_count,	1}, // @pymeth GetCount|Get the count of items in the listbox.
	{"GetCurSel",			PyCListBox_get_cur_sel,	1}, // @pymeth GetCurSel|Get the current selection in a single selection listbox.
	{"GetItemData",			PyCListBox_GetItemData, 1}, // @pymeth GetItemData|Retrieves the application-specific object associated with a listbox entry
	{"GetItemValue",		PyCListBox_GetItemValue, 1}, // @pymeth GetItemValue|Retrieves the application-specific value associated with a listbox entry
	{"GetSel",				PyCListBox_get_sel,	1}, // @pymeth GetSel|Get the selected items in a multiple selection listbox.
	{"GetSelCount",			PyCListBox_get_sel_count,	1}, // @pymeth GetSelCount|Get the number of selected items in a multtiple selection listbox.
	{"GetSelItems",			PyCListBox_get_sel_items,	1},	 // @pymeth GetSelItems|Get the index of the selected items in a multiple selection listbox.
	{"GetSelTextItems",		PyCListBox_get_sel_text_items,	1}, // @pymeth GetSelTextItems|Get the text of the selected items in a multiple selection listbox.
	{"GetTopIndex",			PyCListBox_get_top_index,	1}, // @pymeth GetTopIndex|Get the index of the topmost item.
	{"GetText",				PyCListBox_get_text,	1}, // @pymeth GetText|Get the text associated with an item.
	{"GetTextLen",			PyCListBox_get_text_len,	1}, // @pymeth GetTextLen|Get the length of an item
	{"InsertString",		PyCListBox_insert_string,	1}, // @pymeth InsertString|Insert a string into the listbox.
	{"ResetContent",		PyCListBox_reset_content,	1}, // @pymeth ResetContent|Remove all items from a listbox.
	{"SetCaretIndex",		PyCListBox_set_caret_index,	1}, // @pymeth SetCaretIndex|Set the focus rectange to a specified item.
	{"SelectString",		PyCListBox_select_string,	1}, // @pymeth SelectString|Select an item, based on a string.
	{"SelItemRange",		PyCListBox_sel_item_range,	1}, // @pymeth SelItemRange|Select a range of items in a multiple selection listbox.
	{"SetCurSel",			PyCListBox_set_cur_sel,	1}, // @pymeth SetCurSel|Set the current selection in a single selection listbox.
	{"SetItemData",			PyCListBox_SetItemData, 1}, // @pymeth SetItemData|Sets the application-specific object associated with a listbox entry
	{"SetItemValue",		PyCListBox_SetItemValue, 1}, // @pymeth SetItemValue|Sets the application-specific value associated with a listbox entry
	{"SetSel",				PyCListBox_set_sel,	1}, // @pymeth SetSel|Set the selection.
	{"SetTabStops",			PyCListBox_set_tab_stops,	1}, // @pymeth SetTabStops|Set the tab stops for a listbox.
	{"SetTopIndex",			PyCListBox_set_top_index,	1}, // @pymeth SetTopIndex|Set the top most visible item in a listbox.
	{NULL,			NULL}
};

ui_type_CObject PyCListBox::type("PyCListBox", 
								 &ui_control_object::type, 
								 RUNTIME_CLASS(CListBox), 
								 sizeof(PyCListBox), 
								 PyCListBox_methods, 
								 GET_PY_CTOR(PyCListBox));

/////////////////////////////////////////////////////////////////////
//
// PyCComboBox
//
static CComboBox *GetCombo(PyObject *self)
{
	// note we can only ask for a CWnd, if the LB is created from a resource based
	// dialog.  This is also the technique MFC uses (specifically appdlg.cpp)
	return (CComboBox *)PyCWnd::GetPythonGenericWnd(self);
}
PyCComboBox::PyCComboBox()
{
}
PyCComboBox::~PyCComboBox()
{
}
// @pymethod int|PyCComboBox|AddString|Adds a string to a combobox.
static PyObject *
PyCComboBox_add_string(PyObject *self, PyObject *args)
{
	CComboBox *pCB = GetCombo(self);
	if (!pCB)
		return NULL;
	PyObject *ob;
	if (!PyArg_ParseTuple(args, "O", &ob ))	// @pyparm any|object||Any object.  If not a string, __str__, __repr__ or a default repr() will be used
		return NULL;
	int rc=pCB->AddString( GetReprText(ob) );
	//@pyseemfc CComboBox|AddString
	if (IS_CB_ERR(rc))
		RETURN_ERR("PyCComboBox.AddString failed");
	return Py_BuildValue("i", rc);
	//@rdesc The zero based index of the new string.
}
// @pymethod int|PyCComboBox|DeleteString|Deletes an item from a combobox.
static PyObject *
PyCComboBox_delete_string(PyObject *self, PyObject *args)
{
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	int pos;
	if (!PyArg_ParseTuple(args,"i",&pos)) // @pyparm int|pos||The zero based index of the item to delete.
		return NULL;
	return Py_BuildValue("i", pLB->DeleteString(pos)); // @pyseemfc CComboBox|DeleteString
	// @rdesc The count of the items remaining in the list.
}
// @pymethod int|PyCComboBox|Dir|Fills the list portion of a combobox with a directory listing.
static PyObject *
PyCComboBox_dir(PyObject *self, PyObject *args)
{
	int attr;
	char *szWild;
	if (!PyArg_ParseTuple(args,"is",
	          &attr,    // @pyparm int|attr||The attributes of the files to locate
	          &szWild))	// @pyparm string|wild||A file specification string - eg, *.*
		return NULL;
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	return Py_BuildValue("i", pLB->Dir(attr, szWild)); // @pyseemfc CComboBox|Dir
	// @rdesc The index of the last file name added to the list.
}

// @pymethod int|PyCComboBox|InsertString|Insert a string into a combobox.
static PyObject *
PyCComboBox_insert_string(PyObject *self, PyObject *args)
{
	int pos;
	PyObject *ob;
	if (!PyArg_ParseTuple(args, "iO",
	          &pos, // @pyparm int|pos||The zero based index in the combobox to insert the new string
	          &ob))	// @pyparm any|object||The object to be added to the combobox
		return NULL;
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	return Py_BuildValue("i", pLB->InsertString( pos, GetReprText(ob) ));	// @pyseemfc CComboBox|InsertString
	// @rdesc The zero based index of the new string added.
}
// @pymethod |PyCComboBox|ResetContent|Clear all the items from a combobox.
static PyObject *
PyCComboBox_reset_content(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	pLB->ResetContent(); // @pyseemfc CComboBox|ResetContent
	RETURN_NONE;
}
// @pymethod int|PyCComboBox|GetCount|Returns the count of items in the combobox.
static PyObject *
PyCComboBox_get_count(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CComboBox *pCB = GetCombo(self);
	if (!pCB)
		return NULL;
	return Py_BuildValue("i", pCB->GetCount()); // @pyseemfc CListBox|GetCount
	// @rdesc Returns the number of items currently in the combobox.
}
// @pymethod int|PyCComboBox|GetCurSel|Returns the index of the currently selected item.
static PyObject *
PyCComboBox_get_cur_sel(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	return Py_BuildValue("i", pLB->GetCurSel()); // @pyseemfc CComboBox|GetCurSel
	//@comm Should not be called for a multiple selection listbox.
}
// @pymethod int|PyCComboBox|GetEditSel|Returns the selection of the edit control portion of a combo box.
static PyObject *
PyCComboBox_get_edit_sel(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	int rc=pLB->GetEditSel();// @pyseemfc CComboBox|GetEditSel
	if (IS_CB_ERR(rc))
		RETURN_ERR("GetEditSel failed");

	return Py_BuildValue("i", rc); 
	// @rdesc A 32-bit value that contains the starting position in the low-order word and 
	// the position of the first nonselected character after the end of
	// the selection in the high-order word. If this function is used on a combo box 
	// without an edit control, an exception is raised.
}
// @pymethod int|PyCComboBox|GetExtendedUI|Indicates if the combo has the extended interface.
static PyObject *
PyCComboBox_get_extended_ui(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	return Py_BuildValue("i", pLB->GetExtendedUI()); // @pyseemfc CComboBox|GetExtendedUI
	// @rdesc Nonzero if the combo box has the extended user interface; otherwise 0.
}

// @pymethod object|PyCComboBox|GetItemData|Retrieves the application-specific object associated with an item.
PyObject *PyCComboBox_GetItemData( PyObject *self, PyObject *args )
{
	int item;
	if (!PyArg_ParseTuple( args, "i:GetItemData", 
	                   &item)) // @pyparm int|item||The index of the item whose data is to be retrieved.

		return NULL;
	CComboBox *pLB = GetCombo(self);
	if (!pLB) return NULL;
	PyObject *ret = PyWin_GetPythonObjectFromLong(pLB->GetItemData(item));
	// inc ref count for return value.
	Py_INCREF(ret);
	return ret;
}

// @pymethod int|PyCComboBox|GetItemValue|Retrieves the application-specific value associated with an item.
PyObject *PyCComboBox_GetItemValue( PyObject *self, PyObject *args )
{
	int item;
	if (!PyArg_ParseTuple( args, "i:GetItemValue", 
	                   &item)) // @pyparm int|item||The index of the item whose data is to be retrieved.

		return NULL;
	CComboBox *pLB = GetCombo(self);
	if (!pLB) return NULL;
	return PyInt_FromLong((long)pLB->GetItemData(item));
}

// @pymethod string|PyCComboBox|GetLBText|Gets the string from the list of a combo box.
static PyObject *
PyCComboBox_get_lb_text(PyObject *self, PyObject *args)
{
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	int pos; // @pyparm int|index||The index of the item to return the string for.
	if (!PyArg_ParseTuple(args,"i",&pos)) 
		return NULL;
	CString cs;
	pLB->GetLBText(pos, cs); // @pyseemfc CComboBox|GetLBText
    return Py_BuildValue("s", (const char *)cs);
	// @rdesc The requested string. If index does 
	// not specify a valid index, no exception is raised.
}
// @pymethod int|PyCComboBox|GetLBTextLen|Returns the length of a string in the list of a combobox.
static PyObject *
PyCComboBox_get_lb_text_len(PyObject *self, PyObject *args)
{
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	int pos; // @pyparm int|index||The index of the item to return the length of.
	if (!PyArg_ParseTuple(args,"i",&pos))
		return NULL;
	int rc=pLB->GetLBTextLen(pos);
	if (IS_CB_ERR(rc))
		RETURN_ERR("PyCComboBox.GetLBTextLen failed");
	return Py_BuildValue("i", rc); // @pyseemfc CComboBox|GetLBTextLen
	// @ rdesc Returns the length of the string (in bytes), or raises an exception on error.
}
// @pymethod int|PyCComboBox|LimitText|Limits the amount of text the edit portion of a combo box can hold.
static PyObject *
PyCComboBox_limit_text(PyObject *self, PyObject *args)
{
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	int pos;
	if (!PyArg_ParseTuple(args,"i",&pos)) // @pyparm int|max||The maximum number of characters the user can enter.  If zero, the size is set to (virtually) unlimited.
		return NULL;
	if (pLB->LimitText(pos)==CB_ERR) // @pyseemfc CComboBox|LimitText
		RETURN_ERR("Combo does not have an edit box");
	RETURN_NONE;
}

// @pymethod |PyCComboBox|SelectString|Searches for a combobox item that matches the specified string, and selects it.
static PyObject *
PyCComboBox_select_string(PyObject *self, PyObject *args)
{
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	char *string;
	int after;
	if (!PyArg_ParseTuple(args,"is",
	          &after, 	 // @pyparm int|after||Contains the zero-based index of the item before the first item to be searched, or -1 for the entire combobox.
	          &string))	 // @pyparm string|string||The string to search for.
		return NULL;
	if (pLB->SelectString(after, string)==CB_ERR) // @pyseemfc CComboBoxBox|SelectString
		RETURN_ERR("The string does not exist");
	RETURN_NONE;
	// @rdesc The return value is always None - an exception is raised if the string can not be located.
}

// @pymethod |PyCComboBox|SetCurSel|Selects an item in a combobox.
static PyObject *
PyCComboBox_set_cur_sel(PyObject *self, PyObject *args)
{
	int index;
	if (!PyArg_ParseTuple(args,"i",&index))	// @pyparm int|index||The zero based index of the item to select.
		return NULL;
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	if (pLB->SetCurSel(index)==CB_ERR && index!=CB_ERR)	// @pyseemfc CComboBox|SetCurSel
		RETURN_ERR("SetCurSel failed");
	RETURN_NONE;
}
// @pymethod |PyCComboBox|SetEditSel|Sets the selection in the edit control portion of a combo box.
static PyObject *
PyCComboBox_set_edit_sel(PyObject *self, PyObject *args)
{
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	int start, end;
	if (!PyArg_ParseTuple(args,"ii",
		&start, // @pyparm int|start||Specifies the starting position. If the starting position is set to -1, then any existing selection is removed.
		&end))  // @pyparm int|end||Specifies the ending position. If the ending position is set to -1, then all text from the starting position to the last character in the edit control is selected.
		return NULL;
	if (pLB->SetEditSel(start, end)==CB_ERR) // @pyseemfc PyCComboBox|SetEditSel
		RETURN_ERR("Combo is dropdown, or does not have an edit box");
	RETURN_NONE;
	// @rdesc The return value is always None - an exception is raised if the combo is a dropdown style, or does not have an edit control.
}

// @pymethod |PyCComboBox|SetExtendedUI|Selects the Extended UI mode for a combo box.
static PyObject *
PyCComboBox_set_extended_ui(PyObject *self, PyObject *args)
{
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	int flag = TRUE; // @pyparm int|bExtended|1|Indicates if the combo should have the extended user interface.
	if (!PyArg_ParseTuple(args,"|i",&flag))
		return NULL;
	if (pLB->SetExtendedUI(flag)==CB_ERR) // @pyseemfc CListBox|SetExtendedUI
		RETURN_ERR("SetExtendedUI failed");
	// @comm A combo box with the Extended UI flag set can be identified in the following ways:~
	// * Clicking the static control displays the list box only for combo boxes with the CBS_DROPDOWNLIST style.~
	// * Pressing the DOWN ARROW key displays the list box (F4 is disabled).~
	// * Scrolling in the static control is disabled when the item list is not visible (the arrow keys are disabled).
	RETURN_NONE;
}
// @pymethod int|PyCComboBox|SetItemData|Sets the items application-specific object value.
PyObject *PyCComboBox_SetItemData( PyObject *self, PyObject *args )
{
	CComboBox *pLB = GetCombo(self);
	if (!pLB) return NULL;
	int item;
	PyObject *data;
	if (!PyArg_ParseTuple( args, "iO:SetItemData", 
		                   &item, // @pyparm int|item||Index of the item whose Data is to be set.
						   &data)) // @pyparm object|Data||New value for the data.
		return NULL;
	if (data==Py_None) data = NULL;
	if (!pLB->SetItemData(item, (DWORD)data))
		RETURN_ERR("SetItemData failed");
	// @comm Note that a reference count is not added to the object.  This it is your
	// responsibility to make sure the object remains alive while in the list.
	RETURN_NONE;
}

// @pymethod int|PyCComboBox|SetItemValue|Sets the items application-specific value.
PyObject *PyCComboBox_SetItemValue( PyObject *self, PyObject *args )
{
	CComboBox *pLB = GetCombo(self);
	if (!pLB) return NULL;
	int item;
	int data;
	if (!PyArg_ParseTuple( args, "ii:SetItemValue", 
		                   &item, // @pyparm int|item||Index of the item whose Data is to be set.
						   &data)) // @pyparm int|data||New value for the data.
		return NULL;
	if (!pLB->SetItemData(item, (DWORD)data))
		RETURN_ERR("SetItemValue failed");
	RETURN_NONE;
}
static PyObject *
// @pymethod |PyCComboBox|ShowDropDown|Shows or hides the listbox portion of a combo box.
PyCComboBox_show_drop_down(PyObject *self, PyObject *args)
{
	CComboBox *pLB = GetCombo(self);
	if (!pLB)
		return NULL;
	int flag = TRUE; // @pyparm int|bShowIt|1|Indicates if the listbox should be shown or hidden.
	if (!PyArg_ParseTuple(args,"|i",&flag))
		return NULL;
	pLB->ShowDropDown(flag);
	RETURN_NONE;
}
// @object PyCComboBox|A windows combo control.  Encapsulates an MFC <c CComboBox> class.  Derived from a <o PyCControl> object.
static struct PyMethodDef PyCComboBox_methods[] = {
	{"AddString",			PyCComboBox_add_string,	1}, // @pymeth AddString|Add a string to the listbox portion of a combo box.
	{"DeleteString",		PyCComboBox_delete_string,	1}, // @pymeth DeleteString|Delete a string to the listbox portion of a combo box.
	{"Dir",					PyCComboBox_dir,			1}, // @pymeth Dir|Fill the listbox portion of a combo with a file specification.
	{"GetCount",			PyCComboBox_get_count,		1}, // @pymeth GetCount|Get the count of items in the listbox portion of a combo box.
	{"GetCurSel",			PyCComboBox_get_cur_sel,	1}, // @pymeth GetCurSel|Get the current selection in the listbox portion of a combo box.
	{"GetEditSel",			PyCComboBox_get_edit_sel,	1}, // @pymeth GetEditSel|Gets the edit control selection from a combo box.
	{"GetExtendedUI",		PyCComboBox_get_extended_ui,	1}, // @pymeth GetExtendedUI|Gets the ExtendedUI flag for a combo box.
	{"GetItemData",			PyCComboBox_GetItemData, 1}, // @pymeth GetItemData|Retrieves the application-specific object associated with a combobox entry
	{"GetItemValue",		PyCComboBox_GetItemValue, 1}, // @pymeth GetItemValue|Retrieves the application-specific value associated with a combobox entry
	{"GetLBText",			PyCComboBox_get_lb_text,	1}, // @pymeth GetLBText|Gets the text from the edit control in a combo box.
	{"GetLBTextLen",		PyCComboBox_get_lb_text_len,	1}, // @pymeth GetLBTextLen|Gets the length of the text in the edit control of a combo box.
	{"InsertString",		PyCComboBox_insert_string,	1}, // @pymeth InsertString|Inserts a string into the listbox portion of a combo box.
	{"LimitText",			PyCComboBox_limit_text,	1}, // @pymeth LimitText|Limit the length of text in the edit control portion of a combo box.
	{"ResetContent",		PyCComboBox_reset_content,	1}, // @pymeth ResetContent|Remove all items from the listbox portion of a combo box.
	{"SelectString",		PyCComboBox_select_string,	1}, // @pymeth SelectString|Select a string in the listbox portion of a combo box.
	{"SetCurSel",			PyCComboBox_set_cur_sel,		1}, // @pymeth SetCurSel|Sets the current selection in the listbox portion of a combo box.
	{"SetEditSel",			PyCComboBox_set_edit_sel,	1}, // @pymeth SetEditSel|Sets the current selection in the edit control portion of a combo box.
	{"SetExtendedUI",		PyCComboBox_set_extended_ui,	1}, // @pymeth SetExtendedUI|Sets the ExtendedUI flag for a combo box.
	{"SetItemData",			PyCComboBox_SetItemData, 1}, // @pymeth SetItemData|Sets the application-specific object associated with a combobox entry
	{"SetItemValue",		PyCComboBox_SetItemValue, 1}, // @pymeth SetItemValue|Sets the application-specific value associated with a combobox entry
	{"ShowDropDown",		PyCComboBox_show_drop_down,1}, // @pymeth ShowDropDown|Shows the listbox portion of a combo box.
	{NULL,			NULL}
};

ui_type_CObject PyCComboBox::type("PyCComboBox", 
								  &ui_control_object::type, 
								  RUNTIME_CLASS(CComboBox), 
								  sizeof(PyCComboBox), 
								  PyCComboBox_methods, 
								  GET_PY_CTOR(PyCComboBox));
