/* win32ImageList : implementation file

	Image List object.  

	Created Feb 1997, Mark Hammond (MHammond@skippinet.com.au)

Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc

*/
#include "stdafx.h"
#include "win32win.h"
#include "win32ImageList.h"

/////////////////////////////////////////////////////////////////////////
//
// ImageList
PythonImageList::PythonImageList()
{
}
PythonImageList::~PythonImageList()
{
	Python_delete_assoc(this);
}
#ifdef _DEBUG
void PythonImageList::Dump( CDumpContext &dc ) const
{
	CImageList::Dump(dc);
	DumpAssocPyObject(dc, (void *)this);
}
#endif

PyCImageList::PyCImageList()
{
}
PyCImageList::~PyCImageList()
{
}

CImageList *PyCImageList::GetImageList(PyObject *self)
{
	return (CImageList *)PyCWnd::GetPythonGenericWnd(self, &PyCImageList::type);
}

// @pymethod int|win32ui|CreateImageList|Creates an image list.
PyObject *PyCImageList_Create( PyObject *self, PyObject *args )
{
	BOOL bRet;
	int cx, cy, nInitial, nGrow, nBitmap;
	char *strBitmap;
	COLORREF crMask;
	BOOL bMask;
	CImageList *pList = new PythonImageList();
	if (PyArg_ParseTuple(args, "iiiii", 
		&cx, // @pyparm int|cx||Dimension of each image, in pixels.
		&cy, // @pyparm int|cy||Dimension of each image, in pixels.
		&bMask, // @pyparm int|mask||TRUE if the image contains a mask; otherwise FALSE.
		&nInitial, // @pyparm int|initial||Number of images that the image list initially contains.
		&nGrow)) {// @pyparm int|grow||Number of images by which the image list can grow when the system needs to resize the list to make room for new images. This parameter represents the number of new images the resized image list can contain.
		bRet = pList->Create(cx, cy, bMask, nInitial, nGrow);
	} else if (PyArg_ParseTuple(args, "iiii", 
		&nBitmap, // @pyparmalt1 int|bitmapId||Resource ID of the bitmap to be associated with the image list.
		&cx, // @pyparmalt1 int|cx||Dimension of each image, in pixels.
		&nGrow, // @pyparmalt1 int|grow||Number of images by which the image list can grow when the system needs to resize the list to make room for new images. This parameter represents the number of new images the resized image list can contain.
		&crMask)) { // @pyparmalt1 int|crMask||Color used to generate a mask. Each pixel of this color in the specified bitmap is changed to black, and the corresponding bit in the mask is set to one.
		bRet = pList->Create(nBitmap,cx,nGrow,crMask);
	} else if (PyArg_ParseTuple(args, "siii", 
		    &strBitmap, // @pyparmalt2 int|bitmapId||Resource ID of the bitmap to be associated with the image list.
			&cx, // @pyparmalt2 int|cx||Dimension of each image, in pixels.
			&nGrow, // @pyparmalt2 int|grow||Number of images by which the image list can grow when the system needs to resize the list to make room for new images. This parameter represents the number of new images the resized image list can contain.
			&crMask)) { // @pyparmalt2 int|crMask||Color used to generate a mask. Each pixel of this color in the specified bitmap is changed to black, and the corresponding bit in the mask is set to one.
		bRet = pList->Create(strBitmap,cx,nGrow,crMask);
	} else {
		delete pList;
		RETURN_ERR("PyCImageList::Create() - bad argument list");
	}
	if (!bRet) {
		delete pList;
		RETURN_ERR("PyCImage::Create failed");
	}
	return ui_assoc_object::make( PyCImageList::type, pList )->GetGoodRet();
}

// @pymethod |PyCImageList|DeleteImageList|Deletes an image list.
PyObject *PyCImageList_DeleteImageList( PyObject *self, PyObject *args )
{
	// @comm This frees all resources associated with an image list.
	// No further operations on the object will be allowed.
	CImageList *pList;
	if (!(pList=PyCImageList::GetImageList(self)))
		return NULL;
	CHECK_NO_ARGS2(args, "DeleteImageList");
	// Kill the C++ object.
	Python_delete_assoc(pList);
	if (!pList->DeleteImageList())
		RETURN_ERR("DeleteImageList failed");
	delete pList;
	RETURN_NONE;
}

// @pymethod int|PyCImageList|Add|Adds an image to the list.
PyObject *PyCImageList_Add( PyObject *self, PyObject *args )
{
	HICON hIcon;
	int bmp1, bmp2, mask;
	CImageList *pList = PyCImageList::GetImageList(self);
	int rc;
	if (!pList)
		return NULL;
	if (PyArg_ParseTuple(args, "(ii)", 
			&bmp1, &bmp2)) {// @pyparm (int,int)|bitmap, bitmapMask||2 Bitmaps to use (primary and mask)
		if (!IsGdiHandleValid((HANDLE)bmp1) || !IsGdiHandleValid((HANDLE)bmp2))
			RETURN_ERR("One of the bitmap handles is invalid");
		rc = pList->Add(CBitmap::FromHandle((HBITMAP)bmp1), CBitmap::FromHandle((HBITMAP)bmp2));
	} else if (PyArg_ParseTuple(args, "ii", 
			&bmp1, 	// @pyparmalt1 int|bitmap||Bitmap to use
			&mask)) { // @pyparmalt1 int|color||Color to use for the mask.
		if (!IsGdiHandleValid((HANDLE)bmp1))
			RETURN_ERR("The bitmap handle is invalid");
		rc = pList->Add(CBitmap::FromHandle((HBITMAP)bmp1), (COLORREF)mask);
	} else if (PyArg_ParseTuple(args, "i", 
			&hIcon)) {// @pyparmalt2 int|hIcon||Handle of an icon to add.
		rc = pList->Add(hIcon);
	} else {
		RETURN_ERR("Add requires '(hbitmap, hbitmap)', 'hbitmap, color' or 'hicon'");
	}
	if (rc==-1)
		RETURN_ERR("Add failed");
	return Py_BuildValue("i", rc);
	// @rdesc Zero-based index of the first new image.
}
// @pymethod |PyCImageList|Destroy|Destroys the underlying CImageList
PyObject *PyCImageList_Destroy( PyObject *self, PyObject *args )
{
	CImageList *pList;
	if (!(pList=PyCImageList::GetImageList(self)))
		return NULL;
	CHECK_NO_ARGS2(args, "Destroy");
	delete pList;
	// @comm This method actually calls delete() on the CImageList - you
	// should ensure that no controls still require access to this list.
	RETURN_NONE;
}

// @pymethod int|PyCImageList|GetBkColor|Retrieves the background color of an Image List.
PyObject *PyCImageList_GetBkColor( PyObject *self, PyObject *args )
{
	CImageList *pList;
	if (!(pList=PyCImageList::GetImageList(self)))
		return NULL;
	CHECK_NO_ARGS2(args, "GetBkColor");
	return Py_BuildValue("i", pList->GetBkColor());
}

// @pymethod |PyCImageList|SetBkColor|Sets the background color for an Image List.
PyObject *PyCImageList_SetBkColor( PyObject *self, PyObject *args )
{
	CImageList *pList;
	if (!(pList=PyCImageList::GetImageList(self)))
		return NULL;
	int col;
	// @pyparm int|color||The new background color.
	if (!PyArg_ParseTuple( args, "i:SetBkColor", &col))
		return NULL;
	if (!pList->SetBkColor(col))
		RETURN_ERR("SetBkColor failed");
	RETURN_NONE;
}

// @pymethod int|PyCImageList|GetSafeHandle|Retrieves the HIMAGELIST for the object
PyObject *PyCImageList_GetSafeHandle( PyObject *self, PyObject *args )
{
	CImageList *pList;
	if (!(pList=PyCImageList::GetImageList(self)))
		return NULL;
	CHECK_NO_ARGS2(args, "GetSafeHandle");
	return Py_BuildValue("i", pList->GetSafeHandle());
}

// @pymethod int|PyCImageList|GetImageCount|Retrieves the number of images in an image list.
PyObject *PyCImageList_GetImageCount( PyObject *self, PyObject *args )
{
	CImageList *pList;
	if (!(pList=PyCImageList::GetImageList(self)))
		return NULL;
	CHECK_NO_ARGS2(args, "GetImageCount");
	return Py_BuildValue("i", pList->GetImageCount());
}

// @pymethod iiii(iiii)|PyCImageList|GetImageInfo|Retrieves information about an image.
PyObject *PyCImageList_GetImageInfo( PyObject *self, PyObject *args )
{
	CImageList *pList;
	int nIndex;
	if (!(pList=PyCImageList::GetImageList(self)))
		return NULL;
	// @pyparm int|index||Index of image.
	if (!PyArg_ParseTuple(args, "i:GetImageInfo", &nIndex))
		return NULL;
	IMAGEINFO info;
	if (!pList->GetImageInfo(nIndex, &info))
		RETURN_ERR("GetImageInfo failed");
	return Py_BuildValue("iiii(iiii)", info.hbmImage, info.hbmMask,
		info.Unused1, info.Unused2,
		info.rcImage.left,info.rcImage.top,info.rcImage.right,info.rcImage.bottom );
	// @rdesc The return info is a tuple describing an IMAGELIST structure.
}

// @object PyCImageList|A Python type encapsulating an MFC CImageList class.
static struct PyMethodDef PyCImageList_methods[] = {
	{"Add",            PyCImageList_Add, 1 }, // @pymeth Add|Adds an icon to the image list.
	{"Destroy",        PyCImageList_Destroy, 1}, // @pymeth Destroy|Destroys the underlying MFC imagelist object.
	{"DeleteImageList",PyCImageList_DeleteImageList, 1}, // @pymeth DeleteImageList|Deletes an image list.
	{"GetBkColor",     PyCImageList_GetBkColor, 1}, // @pymeth GetBkColor|Retrieves the background color of an Image List.
	{"GetSafeHandle",  PyCImageList_GetSafeHandle, 1}, // @pymeth GetSafeHandle|Retrieves the HIMAGELIST for the object
	{"GetImageCount",  PyCImageList_GetImageCount, 1}, // @pymeth GetImageCount|Retrieves the number of images in an image list.
	{"GetImageInfo",   PyCImageList_GetImageInfo, 1}, // @pymeth GetImageInfo|Retrieves information about an image.
	{"SetBkColor",     PyCImageList_SetBkColor, 1}, // @pymeth SetBkColor|Sets the background color for an Image List.
	{NULL, NULL }
};

ui_type_CObject PyCImageList::type("PyCImageList", 
								   &ui_assoc_CObject::type, 
								   RUNTIME_CLASS(CImageList), 
								   sizeof(PyCImageList), 
								   PyCImageList_methods, 
								   GET_PY_CTOR(PyCImageList));
