/*

	win32 app data type

	Created July 1994, Mark Hammond (MHammond@skippinet.com.au)

Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc

*/
#include "stdafx.h"

#include "win32ui.h"
#include "win32doc.h"
#include "win32template.h"

extern CWnd *GetWndPtr(PyObject *self);

PyObject *PyCWinApp::pExistingAppObject = NULL;
char *errmsgAlreadyInit = "The application has already been initialised";

/////////////////////////////////////////////////////////////////////
//
// CProtectedWinApp Application helpers.
//
//////////////////////////////////////////////////////////////////////
CString CProtectedWinApp::GetRecentFileName(int index)
{
	if (index>=0 && index < _AFX_MRU_MAX_COUNT) {
		return (*m_pRecentFileList)[index];
	}
	else {
		ASSERT(0);
		return CString();
	}
}

PyObject *CProtectedWinApp::MakePyDocTemplateList()
{
	PyObject *retList = PyList_New(0);
	if (retList==NULL)
		return NULL;
	POSITION posTempl = m_pDocManager->GetFirstDocTemplatePosition();
	while (posTempl) {
		CDocTemplate* pTemplate = m_pDocManager->GetNextDocTemplate(posTempl);
		ASSERT(pTemplate->IsKindOf(RUNTIME_CLASS(CDocTemplate)));
		PyObject *newListItem = ui_assoc_object::make(PyCDocTemplate::type, pTemplate)->GetGoodRet();
		if (newListItem==NULL) {
			Py_DECREF(retList);
			return NULL;
		}
		PyList_Append(retList, newListItem);
		Py_DECREF(newListItem);
	}
	return retList;
}

// FindOpenDocument - if the C++ framework has a document with this name open,
// then return a pointer to it, else NULL.
CDocument *CProtectedWinApp::FindOpenDocument(const char *lpszFileName)
{
	POSITION posTempl = m_pDocManager->GetFirstDocTemplatePosition();
	CDocument* pOpenDocument = NULL;

	char szPath[_MAX_PATH];
	if (!GetFullPath(szPath, lpszFileName))
		strcpy(szPath, lpszFileName);

	while (posTempl) {
		CDocTemplate* pTemplate = m_pDocManager->GetNextDocTemplate(posTempl);
		ASSERT(pTemplate->IsKindOf(RUNTIME_CLASS(CDocTemplate)));
		// go through all documents
		POSITION posDoc = pTemplate->GetFirstDocPosition();
		while (posDoc) {
			CDocument* pDoc = pTemplate->GetNextDoc(posDoc);
			if (lstrcmpi(pDoc->GetPathName(), szPath) == 0)
				return pDoc;
		}
	}
	return NULL;
}

CProtectedDocManager *CProtectedWinApp::GetDocManager()
{
	CProtectedDocManager *ret = (CProtectedDocManager *)m_pDocManager;
	if (!ret->IsKindOf(RUNTIME_CLASS(CDocManager)))
		RETURN_ERR("There is not a valid Document Manager");
	return ret;
}

void CProtectedWinApp::PumpIdle()
{
	long lIdleCount = 0;
	while (OnIdle(lIdleCount++))
		;
	return;
}

extern BOOL bDebuggerPumpStopRequested;
void CProtectedWinApp::PumpForDebugger()
{
/*
	// Acquire and dispatch messages until a WM_QUIT message is received.
	for (; ;)
	{
		if (!PumpMessage()) {
#ifdef _DEBUG
			m_nDisablePumpCount--; // application must NOT die
#endif
			break;
		}
	}
*******/
		ASSERT_VALID(this);

	// for tracking the idle time state
	BOOL bIdle = TRUE;
	LONG lIdleCount = 0;

	// acquire and dispatch messages until a WM_QUIT message is received.
	for (;;)
	{
		// phase1: check to see if we can do idle work
		while (bIdle &&
			!::PeekMessage(&m_msgCur, NULL, NULL, NULL, PM_NOREMOVE))
		{
			// call OnIdle while in bIdle state
			if (!OnIdle(lIdleCount++))
				bIdle = FALSE; // assume "no idle" state
		}
		// phase2: pump messages while available
		do
		{
			// pump message, but quit on WM_QUIT
			if (!PumpMessage()) {
#ifdef _DEBUG
				m_nDisablePumpCount--; // application must NOT die
#endif
				// If told to stop pump, then just stop it.
				// Otherwise user may have shut down the entire application,
				// and therefore we stop this pump, and tell the outer pump it 
				// should also stop.
				if (!bDebuggerPumpStopRequested)
						PostQuitMessage(0);
				bDebuggerPumpStopRequested = FALSE;
				return;
			}

			// reset "no idle" state after pumping "normal" message
			if (IsIdleMessage(&m_msgCur))
			{
				bIdle = TRUE;
				lIdleCount = 0;
			}

		} while (::PeekMessage(&m_msgCur, NULL, NULL, NULL, PM_NOREMOVE));
	}

	ASSERT(FALSE);  // not reachable
}

void CProtectedWinApp::PumpWaitingMessages()
{
	MSG msg;
	if (::PeekMessage(&msg, NULL, WM_PAINT, WM_PAINT, PM_REMOVE))
	{
		::DispatchMessage(&msg);
	}
/**************88
	while(::PeekMessage(&m_msgCur, NULL, NULL, NULL, PM_NOREMOVE))
		if (!PumpMessage()) {
			// if got a close message, must send it back down.
#ifdef _DEBUG
			m_nDisablePumpCount--; // hack!
#endif
			PostQuitMessage(0);
			return;
		}
	    // let MFC do its idle processing
    LONG lIdle = 0;
    while ( AfxGetApp()->OnIdle(lIdle++ ) )
        ;  
********/
}


/////////////////////////////////////////////////////////////////////
//
// Application object
//
//////////////////////////////////////////////////////////////////////
PyCWinApp::PyCWinApp()
{
	ASSERT(pExistingAppObject== NULL);
}

PyCWinApp::~PyCWinApp()
{
	XDODECREF(pExistingAppObject);
	pExistingAppObject = NULL;
}

// @pymethod |PyCWinApp|AddDocTemplate|Adds a template to the application list.
static PyObject *
ui_app_add_doc_template(PyObject *self, PyObject *args)
{
	PyObject *obTemplate;
	if (!PyArg_ParseTuple(args,"O:AddDocTemplate",
	                      &obTemplate))     // @pyparm <o PyCDocTemplate>|template||The template to be added.
		return NULL;

	if (!ui_base_class::is_uiobject(obTemplate, &PyCDocTemplate::type))
		RETURN_TYPE_ERR("The paramater must be a template object");

	CDocTemplate *pTempl = PyCDocTemplate::GetTemplate(obTemplate);
	if (pTempl==NULL)
		return NULL;
	CWinApp *pApp = GetApp();
	if (!pApp) return NULL;
	// walk all templates in the application looking for it.
	CDocTemplate* pTemplate;
	POSITION pos = pApp->m_pDocManager ? pApp->m_pDocManager->GetFirstDocTemplatePosition() : NULL;
	while (pos != NULL) {
		pTemplate = pApp->m_pDocManager->GetNextDocTemplate(pos);
		if (pTemplate==pTempl)
			RETURN_ERR("The template is already in the application list");
	}
	pApp->AddDocTemplate(pTempl);
	RETURN_NONE;
}

// @pymethod |PyCWinApp|RemoveDocTemplate|Removes a template to the application list.
static PyObject *
ui_app_remove_doc_template(PyObject *self, PyObject *args)
{
	// @comm Note that MFC does not provide an equivilent function.
	PyObject *obTemplate;
	if (!PyArg_ParseTuple(args,"O:RemoveDocTemplate",
	                      &obTemplate))     // @pyparm <o PyCDocTemplate>|template||The template to be removed.  Must have previously been added by <om PyCWinApp.AddDocTemplate>.
		return NULL;

	if (!ui_base_class::is_uiobject(obTemplate, &PyCDocTemplate::type))
		RETURN_TYPE_ERR("The paramater must be a template object");

	CDocTemplate *pTempl = PyCDocTemplate::GetTemplate(obTemplate);
	if (pTempl==NULL)
		return NULL;
	if (!PyCDocTemplate::RemoveDocTemplateFromApp( pTempl ))
		RETURN_ERR("The template is not in the application template list");
	RETURN_NONE;
}


// @pymethod |PyCWinApp|InitMDIInstance|Calls critical InitInstance code for an MDI application.  This should only be called during InitInstance handling.
static PyObject *
ui_init_mdi_instance(PyObject *self, PyObject *args)
{
	CProtectedWinApp *pApp = GetProtectedApp();
	if (!pApp) return NULL;

	CHECK_NO_ARGS2(args,InitMDIInstance);
// Anyone who calls this should know what they are doing!
//	if (pApp->GetMainFrame())
//		RETURN_ERR(errmsgAlreadyInit);
	RETURN_NONE;
}

// @pymethod |PyCWinApp|OpenDocumentFile|Opens a document file by name.
static PyObject *
ui_open_document_file(PyObject *self, PyObject *args)
{
	char *fileName;
	if (!PyArg_ParseTuple(args, "s:OpenDocumentFile",
	                       &fileName )) // @pyparm string|fileName||The name of the document to open.
		return NULL;
	CWinApp *pApp = GetApp();
	if (!pApp) return NULL;

	if (((CProtectedWinApp *)pApp)->GetMainFrame()->GetSafeHwnd()==0)
		RETURN_ERR("There is no main frame in which to create the document");

	CDocument *pDoc = pApp->OpenDocumentFile(fileName);
	if (PyErr_Occurred())
		return NULL;
	if (pDoc==NULL)
		RETURN_NONE;
	return ui_assoc_object::make(PyCDocument::type, pDoc)->GetGoodRet();
}

// @pymethod <o PyCDocument>|PyCWinApp|FindOpenDocument|Returns an existing document with the specified file name.
static PyObject *
ui_find_open_document(PyObject *self, PyObject *args)
{
	char *fileName;
	// @pyparm string|fileName||The fully qualified filename to search for.
	if (!PyArg_ParseTuple(args,"s", &fileName))
		return NULL;
	CProtectedWinApp *pApp = GetProtectedApp();
	if (!pApp) return NULL;
	// Let MFC framework search for a filename for us.
	CDocument *pDoc=pApp->FindOpenDocument(fileName);
	if (pDoc==NULL)
		RETURN_NONE;
	return ui_assoc_object::make(PyCDocument::type, pDoc)->GetGoodRet();
}

// @pymethod |PyCWinApp|SetMainFrame|Sets the C++ applications main frame
static PyObject *
ui_app_set_main_frame(PyObject *self, PyObject *args)
{
	PyObject *wndObject;
	if (!PyArg_ParseTuple(args, "O:SetMainFrame",
	                       &wndObject )) // @pyparm <o PyCWnd>|mainFrame||The applications main frame.
		return NULL;
	CWinApp *pApp = GetApp();
	if (!pApp) return NULL;

	CWnd *pMainWnd = GetWndPtr(wndObject);
	if (!pMainWnd) return NULL;
	pApp->m_pMainWnd = pMainWnd;
	RETURN_NONE;
}

// @pymethod <o PyCWinApp>|win32ui|GetApp|Retrieves the application object.
PyObject *
ui_get_app(PyObject *self, PyObject *args)
{
	// @comm There will only ever be one application object per application.
	CHECK_NO_ARGS2(args,GetApp);
	CWinApp *pApp = GetApp();
	if (pApp==NULL) return NULL;
	return ui_assoc_object::make(PyCWinApp::type, pApp)->GetGoodRet();
}

// @pymethod |PyCWinApp|OnFileNew|Calls the underlying OnFileNew MFC method.
static PyObject *
ui_on_file_new(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS2(args,OnFileNew);
	CProtectedWinApp *pApp = GetProtectedApp();
	if (!pApp) return NULL;
	pApp->OnFileNew();
	RETURN_NONE;
}
// @pymethod |PyCWinApp|OnFileOpen|Calls the underlying OnFileOpen MFC method.
static PyObject *
ui_on_file_open(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS2(args,OnFileNew);
	CProtectedWinApp *pApp = GetProtectedApp();
	if (!pApp) return NULL;
	pApp->OnFileOpen();
	RETURN_NONE;
}

// @pymethod int|PyCWinApp|LoadIcon|Loads an icon resource.
static PyObject *
ui_load_icon(PyObject *self, PyObject *args)
{
	int idResource;
	// @pyparm int|idResource||The ID of the icon to load.
	if (!PyArg_ParseTuple(args,"i:LoadIcon", &idResource))
		return NULL;
	CWinApp *pApp = GetApp();
	if (!pApp) return NULL;
	return Py_BuildValue("i", pApp->LoadIcon(idResource));
}

// @pymethod int|PyCWinApp|LoadStandardIcon|Loads an icon resource.
static PyObject *
ui_load_standard_icon(PyObject *self, PyObject *args)
{
	char *resName;
	// @pyparm string|resourceName||The name of the standard icon to load.
	if (!PyArg_ParseTuple(args,"s:LoadStandardIcon", &resName))
		return NULL;
	CWinApp *pApp = GetApp();
	if (!pApp) return NULL;
	return Py_BuildValue("i", pApp->LoadStandardIcon(resName));
}


// @pymethod int|PyCWinApp|Run|Starts the message pump.  Advanced users only
static PyObject *
ui_app_run(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS2(args, "Run");
	return PyInt_FromLong(AfxGetApp()->Run());
}

// @pymethod int|PyCWinApp|IsInproc|Returns a flag to indicate if the created CWinApp was in the DLL, or an external EXE.
static PyObject *
ui_app_isinproc(PyObject *self, PyObject *args)
{
	extern CWinApp *pCreatedApp;
	CHECK_NO_ARGS2(args, IsInproc);
	return PyInt_FromLong(pCreatedApp!=NULL);
}

// @pymethod [<o PyCDocTemplate>,...]|PyCWinApp|GetDocTemplateList|Returns a list of all document templates.
static PyObject *
ui_app_get_doc_template_list(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS2(args, GetDocTemplateList);
	CProtectedWinApp *pApp = GetProtectedApp();
	if (!pApp) return NULL;
	return pApp->MakePyDocTemplateList();
}

extern PyObject *ui_init_dlg_instance(PyObject *self, PyObject *args);

// @object PyCWinApp|An application class.  Encapsulates an MFC <c CWinApp> class
static struct PyMethodDef PyCWinApp_methods[] = {
	{"AddDocTemplate",    		ui_app_add_doc_template, 1 }, // @pymeth AddDocTemplate|Adds a template to the application list.
	{"FindOpenDocument",		ui_find_open_document,	1}, // @pymeth FindOpenDocument|Returns an existing document with the specified file name.
	{"GetDocTemplateList",      ui_app_get_doc_template_list, 1}, // @pymeth GetDocTemplateList|Returns a list of all document templates in use.
    {"InitMDIInstance",			ui_init_mdi_instance,	1}, // @pymeth InitMDIInstance|Calls critical InitInstance processing for an MDI application.
    {"InitDlgInstance",			ui_init_dlg_instance,	1}, // @pymeth InitDlgInstance|Calls critical InitInstance processing for a dialog based application.
	{"LoadIcon",				ui_load_icon,			1}, // @pymeth LoadIcon|Loads an icon resource.
	{"LoadStandardIcon",		ui_load_standard_icon,  1}, // @pymeth LoadStandardIcon|Loads an icon resource.
    {"OpenDocumentFile",		ui_open_document_file,	1}, // @pymeth OpenDocumentFile|Opens a document file by name.
	{"OnFileNew",               ui_on_file_new,         1}, // @pymeth OnFileNew|Calls the underlying OnFileNew MFC method.
	{"OnFileOpen",              ui_on_file_open,         1}, // @pymeth OnFileOpen|Calls the underlying OnFileOpen MFC method.
	{"RemoveDocTemplate",    	ui_app_remove_doc_template, 1}, // @pymeth RemoveDocTemplate|Removes a template to the application list.
	{"SetMainFrame",    	    ui_app_set_main_frame, 1}, // @pymeth SetMainFrame|Sets the C++ applications main frame
	{"Run",    					ui_app_run, 1}, // @pymeth Run|Starts the main application message pump.
	{"IsInproc",    			ui_app_isinproc, 1}, // @pymeth IsInproc|Returns a flag to indicate if the created CWinApp was in the DLL, or an external EXE.
	{NULL,			NULL}
};
ui_type_CObject PyCWinApp::type("PyCWinApp", 
								&PyCCmdTarget::type,
								RUNTIME_CLASS(CWinApp), 
								sizeof(PyCWinApp), 
								PyCWinApp_methods, 
								GET_PY_CTOR(PyCWinApp) );

void PyCWinApp::cleanup()
{
	// total hack!
	while (pExistingAppObject)
		DODECREF(pExistingAppObject); // this may delete it.
}
