/*

	device context data types

	Created July 1994, Mark Hammond (MHammond@cmutual.com.au)

	These are implemented using CDC's, and hDC's in the map

Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc

*/
#include "stdafx.h"

#include <winspool.h>

#include "win32dc.h"
#include "win32gdi.h"
#include "win32brush.h"
#include "win32font.h"
#include "win32pen.h"
#include "win32bitmap.h"

// this returns a pointer that should not be stored.
void *ui_dc_object::GetGoodCppObject(ui_type *ui_type_check) const
{
	HDC handle = (HDC)ui_assoc_object::GetGoodCppObject(ui_type_check);
	if (handle==0)
    	RETURN_ERR("There is no DC associated with the object");
	/* Need a hdc validity test - maybe try scroll zero pixels, or something...
	if (!::IsWindow(handle))
    	RETURN_ERR("The window associated with the object has been destroyed");
	*/
	CDC *pDC = CDC::FromHandle(handle);
	if (pDC==NULL)
    	RETURN_ERR("CDC from handle creation failed");

	ASSERT_VALID(pDC);
	if (!pDC->IsKindOf(RUNTIME_CLASS(CDC))) {
		TRACE1("ui_dc_object::GetGoodCppObject fails due to RTTI - got %s\n",
			   pDC->GetRuntimeClass()->m_lpszClassName);
		RETURN_ERR("win32ui: Internal error - C++ RTTI failed");
	}
	return pDC;
}

// this returns a pointer that should not be stored.
CDC *ui_dc_object::GetDC(PyObject *self)
{
	return (CDC *)ui_assoc_object::GetGoodCppObject( self, &type);
}

void ui_dc_object::SetAssocInvalid()
{
	return;	// do nothing.  Dont call base as dont want my handle wiped.
}
void ui_dc_object::DoKillAssoc( BOOL bDestructing /*= FALSE*/ )
{
  if (m_deleteDC) {
    CDC *pDC = GetDC(this);
	if (pDC)
	  ::DeleteDC (pDC->m_hDC);
  }
  ui_assoc_object::DoKillAssoc(bDestructing);
}

ui_dc_object::~ui_dc_object()
{
  DoKillAssoc(TRUE);
}
// @pymethod |PyCDC|BitBlt|Copies a bitmap from the source device context to this device context. 
static PyObject *
ui_dc_bitblt (PyObject *self, PyObject *args)
{
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	int x, y, width, height, xsrc, ysrc;
	DWORD rop;
	PyObject *dc_ob;
	if (!PyArg_ParseTuple (args, "(ii)(ii)O(ii)i", 
	          &x, &y,          // @pyparm (x,y)-ints|destPos||The logical x,y coordinates of the upper-left corner of the destination rectangle.
	          &width, &height, // @pyparm (width, height)-ints|size||Specifies the width and height (in logical units) of the destination rectangle and source bitmap.
	          &dc_ob,          // @pyparm <o PyCDC>|dc||Specifies the PyCDC object from which the bitmap will be copied. It must be None if rop specifies a raster operation that does not include a source.
	          &xsrc, &ysrc,    // @pyparm (xSrc, ySrc)-ints|srcPos||Specifies the logical x,y coordinates of the upper-left corner of the source bitmap.
			  &rop))           // @pyparm int|rop||Specifies the raster operation to be performed. See the win32 api documentation for details.
		return NULL;
	if (!ui_base_class::is_uiobject (dc_ob, &ui_dc_object::type))
		RETURN_TYPE_ERR("The 'O' param must be a PyCDC object");
	CDC *pSrcDC = NULL;
	if (dc_ob!=Py_None) {
		pSrcDC = ui_dc_object::GetDC(dc_ob);
		if (!pSrcDC)
			RETURN_ERR("The source DC is invalid");
	}
	if (!pDC->BitBlt(x, y, width, height, pSrcDC, xsrc, ysrc, rop)) // @pyseemfc CDC|BitBlt
		RETURN_ERR("BitBlt failed");
	RETURN_NONE;
}

// @pymethod |PyCDC|PatBlt|Creates a bit pattern on the device.
static PyObject *
ui_dc_patblt (PyObject *self, PyObject *args)
{
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	int x, y, width, height;
	DWORD rop;
	if (!PyArg_ParseTuple (args, "(ii)(ii)i", 
	          &x, &y,          // @pyparm (x,y)-ints|destPos||The logical x,y coordinates of the upper-left corner of the destination rectangle.
	          &width, &height, // @pyparm (width, height)-ints|size||Specifies the width and height (in logical units) of the destination rectangle and source bitmap.
			  &rop))           // @pyparm int|rop||Specifies the raster operation to be performed. See the win32 api documentation for details.
		return NULL;
	if (!pDC->PatBlt(x, y, width, height, rop)) // @pyseemfc CDC|BitBlt
		RETURN_ERR("PatBlt failed");
	RETURN_NONE;
}

// @pymethod |win32ui|CreateDC|Creates an uninitialised device context.
PyObject *ui_dc_object::create_dc( PyObject *self, PyObject *args )
{
	CHECK_NO_ARGS2(args, CreateDC);
	// create Python device context
	CDC *pDC = new CDC;
    ui_dc_object *dc =
      (ui_dc_object *) ui_assoc_object::make (ui_dc_object::type, pDC)->GetGoodRet();
    return dc;
}

// @pymethod |PyCDC|CreateCompatibleDC|Creates a memory device context that is compatible with this DC.
PyObject *ui_dc_object::create_compatible_dc( PyObject *self, PyObject *args )
{
	PyObject *obDCFrom;
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	if (!PyArg_ParseTuple(args, "|O:CreateCompatibleDC", &obDCFrom ))
		return NULL; // @pyparm <o PyCDC>|dcFrom|None|The source DC, or None to make a screen compatible DC.
	CDC *dcFrom = NULL;
	if (obDCFrom!=Py_None)
		dcFrom = GetDC(obDCFrom);
	if (!pDC->CreateCompatibleDC(dcFrom)) // @pyseemfc CDC|CreateCompatibleDC
		RETURN_ERR("CreateCompatibleDC failed");
	RETURN_NONE;
}

// @pymethod |PyCDC|CreatePrinterDC|Creates a device context for a specific printer
PyObject *ui_dc_object::create_printer_dc( PyObject *self, PyObject *args )
{
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	char *printerName = NULL;
	if (!PyArg_ParseTuple(args, "|z:CreatePrinterDC", &printerName ))
		return NULL; // @pyparm string|printerName|None|The printer name, or None for the default printer

	BOOL result;
	if (printerName == NULL)
	{
		// Attempt to open the default printer
		CPrintInfo info;
		if (!AfxGetApp()->GetPrinterDeviceDefaults(&(info.m_pPD->m_pd)))
		{
			RETURN_ERR("No default printer found");
			return NULL;
		}

		if (info.m_pPD->m_pd.hDC == NULL && !info.m_pPD->CreatePrinterDC())
		{
			result = FALSE;
		}
		else
		{
			result = pDC->Attach(info.m_pPD->m_pd.hDC);
			info.m_pPD->m_pd.hDC = NULL; // Prevent this DC from being deleted
		}
	}
	else
	{
		// Attempt to open a specific printer
		HANDLE hPrinter;
		if (!::OpenPrinter(printerName, &hPrinter, NULL))
		{
			RETURN_ERR("Unable to open printer");
			return NULL;
		}

		DWORD len;
		unsigned char buf;
		::GetPrinter(hPrinter, 2, &buf, 1, &len);
		unsigned char *buffer = new unsigned char[len];
		result = ::GetPrinter(hPrinter, 2, buffer, len, &len);
		::ClosePrinter(hPrinter);
		if (!result)
		{
			RETURN_ERR("Unable to get printer info");
			delete [] buffer;
			return NULL;
		}

		PRINTER_INFO_2 *pinfo = (PRINTER_INFO_2*)buffer;
		result = pDC->CreateDC(pinfo->pDriverName, pinfo->pPrinterName, NULL, NULL);  // @pyseemfc CDC|CreateDC
		delete [] buffer;
	}
	
	if (!result)
		RETURN_ERR("CreateDC failed");
	RETURN_NONE;
}

// @pymethod |PyCDC|DeleteDC|Deletes all resources associated with a device context.
static PyObject *
ui_dc_delete_dc (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  CHECK_NO_ARGS2(args, DeleteDC);
  if (!pDC->DeleteDC())
	  RETURN_ERR("DeleteDC failed");
  RETURN_NONE;
  // @comm In general, do not call this function; the destructor will do it for you. 
  // <nl>An application should not call DeleteDC if objects have been selected into the device context. Objects must first be selected out of the device context before it it is deleted. 
  // <nl>An application must not delete a device context whose handle was obtained by calling CWnd::GetDC. Instead, it must call CWnd::ReleaseDC to free the device context.
  // <nl>The DeleteDC function is generally used to delete device contexts created with CreateDC, CreateIC, or CreateCompatibleDC.
}

// @pymethod |PyCDC|DrawIcon|Draws an icon on the DC.
static PyObject *
ui_dc_draw_icon (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;

  // @pyparm (x,y)|point||The point coordinate to draw to.
  // @pyparm int|hIcon||The handle of the icon to draw.
  int x, y;
  HICON hIcon;
  if (!PyArg_ParseTuple (args, "(ii)i", &x, &y, &hIcon))
    return NULL;
  
  if (!pDC->DrawIcon (x,y, hIcon)) // @pyseemfc CDC|DrawIcon
	RETURN_ERR ("DrawIcon failed");
  else
 	RETURN_NONE;
}
// @pymethod |PyCDC|DrawFocusRect|Draws a rectangle in the style used to 
// indicate the rectangle has focus
static PyObject *
ui_dc_draw_focus_rect (PyObject *self, PyObject *args) 
{
	CDC *pDC = ui_dc_object::GetDC(self);
	CRect rect;
	if (!pDC) {
		return NULL;
	// @pyparm (left, top, right, bottom)|rect||The coordinates of the
	// rectangle
	} else if (!PyArg_ParseTuple (args,
						"(iiii)",
						&rect.left, &rect.top,
						&rect.right, &rect.bottom)) {
		return NULL;
	} else {
	// it's a void function
		pDC->DrawFocusRect (rect); // @pyseemfc CDC|DrawFocusRect
		RETURN_NONE;
	}
}

// @pymethod int|PyCDC|RectVisible|Determines whether any part of the given rectangle lies within the clipping region of the display context.
static PyObject *ui_dc_rect_visible( PyObject *self, PyObject *args )
{
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	CRect rect;
    // @pyparm (left, top, right, bottom)|rect||The coordinates of the reactangle to be checked.
	if (!PyArg_ParseTuple(args,"(iiii)", &rect.left, &rect.top, &rect.right, &rect.bottom))
		return NULL;
	return Py_BuildValue("i",pDC->RectVisible(&rect)); // @pyseemfc CDC|RectVisible
	// @rdesc Non zero if any part of the rectangle lies within the clipping region, else zero.
}

// @pymethod |PyCDC|Arc|Draws an eliptical arc. 
static PyObject *ui_dc_arc (PyObject *self, PyObject *args)
{
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	CRect rect;
	POINT pts, pte;
	if (!PyArg_ParseTuple(args,"(iiii)(ii)(ii):Arc", 
 	           // @pyparm (left, top, right, bottom)|rect||Specifies the ellipses bounding rectangle
	           &rect.left, &rect.top, &rect.right, &rect.bottom,
			   // @pyparm (x,y)|pointStart||Specifies the x- and y-coordinates
			   // of the point that defines the arcs starting point (in logical units).
			   // This point does not have to lie exactly on the arc.
			   &pts.x, &pts.y,
			   // @pyparm (x,y)|pointEnd||Specifies the x- and y-coordinates
			   // of the point that defines the arcs ending point (in logical units).
			   // This point does not have to lie exactly on the arc.
			   &pte.x, &pte.y ))
	  return NULL;
	BOOL ret = pDC->Arc(&rect, pts, pte); // @pyseemfc CDC|Arc
	if (!ret)
		RETURN_API_ERR("CDC::Arc");
	RETURN_NONE;
	// @rdesc Always none.  If the function fails, an exception is raised.
	// @comm The arc drawn by using the function is a segment of the ellipse defined by the specified bounding rectangle.
	// The actual starting point of the arc is the point at which a ray drawn
	// from the center of the bounding rectangle through the specified starting
	// point intersects the ellipse. The actual ending point of the arc is the
	// point at which a ray drawn from the center of the bounding rectangle through
	// the specified ending point intersects the ellipse. The arc is drawn in a
	// counterclockwise direction. Since an arc is not a closed figure, it is
	// not filled. Both the width and height of the rectangle must be greater
	// than 2 units and less than 32,767 units.
}

// @pymethod |PyCDC|Chord|Draws a chord. 
static PyObject *ui_dc_chord (PyObject *self, PyObject *args)
{
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	CRect rect;
	POINT pts, pte;
	if (!PyArg_ParseTuple(args,"(iiii)(ii)(ii):Chord", 
 	           // @pyparm (left, top, right, bottom)|rect||Specifies the ellipses bounding rectangle
	           &rect.left, &rect.top, &rect.right, &rect.bottom,
			   // @pyparm (x,y)|pointStart||Specifies the x- and y-coordinates
			   // of the point that defines the arcs starting point (in logical units).
			   // This point does not have to lie exactly on the arc.
			   &pts.x, &pts.y,
			   // @pyparm (x,y)|pointEnd||Specifies the x- and y-coordinates
			   // of the point that defines the arcs ending point (in logical units).
			   // This point does not have to lie exactly on the arc.
			   &pte.x, &pte.y ))
	  return NULL;
	BOOL ret = pDC->Chord(&rect,pts,pte); // @pyseemfc CDC|Chord
	if (!ret)
		RETURN_API_ERR("CDC::Chord");
	RETURN_NONE;
	// @rdesc Always none.  If the function fails, an exception is raised.
	// @comm Draws a chord (a closed figure bounded by the intersection
	// of an ellipse and a line segment). The rect parameter specify the
	// upper-left and lower-right corners, respectively, of a rectangle
	// bounding the ellipse that is part of the chord.
	// The pointStart and pointEnd parameters specify
	// the endpoints of a line that intersects the ellipse.
	// The chord is drawn by using the selected pen and filled
	// by using the selected brush. 
}

// @pymethod |PyCDC|Ellipse|Draws an Ellipse.
static PyObject *ui_dc_ellipse (PyObject *self, PyObject *args)
{
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	CRect rect;
	// @pyparm (left, top, right, bottom)|rect||Specifies the ellipses bounding rectangle
	if (!PyArg_ParseTuple(args,"(iiii):Ellipse", &rect.left, &rect.top, &rect.right, &rect.bottom))
	  return NULL;
	BOOL ret = pDC->Ellipse(rect); // @pyseemfc CDC|Ellipse
	if (!ret)
		RETURN_API_ERR("CDC::Ellipse");
	RETURN_NONE;
	// @rdesc Always none.  If the function fails, an exception is raised.
	// @comm The center of the ellipse is the center of the bounding rectangle
	// specified by rect. The ellipse is drawn with the current pen, and its
	// interior is filled with the current brush. 
}



// @pymethod |PyCDC|Polygon|Draws an Polygon.
static PyObject *ui_dc_polygon (PyObject *self, PyObject *args)
{
  PyObject * point_list;
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  if (!PyArg_ParseTuple(args,
						"O:Polygon",
						&point_list)) {
	return NULL;
  } else if (!PyList_Check(point_list)) {
	return NULL;
  } else {
	// Convert the list of point tuples into an array of POINT structs
	int num = PyList_Size (point_list);
	POINT * point_array = new POINT[num];
	for (int i=0; i < num; i++) {
	  PyObject * point_tuple = PyList_GetItem (point_list, i);
	  if (!PyTuple_Check (point_tuple) || PyTuple_Size (point_tuple) != 2) {
		PyErr_SetString (PyExc_ValueError,
						 "point list must be a list of (x,y) tuples");
		delete[] point_array;
		return NULL;
	  } else {
		long x, y;
		PyObject *px, *py;
		px = PyTuple_GetItem (point_tuple, 0);
		py = PyTuple_GetItem (point_tuple, 1);
		if ((!PyInt_Check(px)) || (!PyInt_Check(py))) {
		  PyErr_SetString (PyExc_ValueError,
						   "point list must be a list of (x,y) tuples");
		  delete[] point_array;
		  return NULL;
		} else {
		  x = PyInt_AsLong (px);
		  y = PyInt_AsLong (py);
		  point_array[i].x = x;
		  point_array[i].y = y;
		}
	  }
	}
	// we have an array of POINT structs, now we
	// can finally draw the polygon.
	BOOL ret = pDC->Polygon (point_array, num);
	delete[] point_array;
	if (!ret) {
	  RETURN_API_ERR("CDC::Polygon");
	} else {
	  RETURN_NONE;
	}
  }
}

// @pymethod |PyCDC|PolyBezier|Draws one or more Bezier splines.
static PyObject *ui_dc_poly_bezier (PyObject *self, PyObject *args)
{
  PyObject * triple_list;
  int do_to=0;
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  if (!PyArg_ParseTuple(args,
						"O|i:PolyBezier[To]",
						&triple_list,
						&do_to)) {
	return NULL;
  } else if (!PyList_Check(triple_list)) {
	return NULL;
  } else {
	int index = 0;
	int num = PyList_Size (triple_list);

#define HURL																\
	do {																	\
	  PyErr_SetString (PyExc_ValueError,									\
					   "arg must be a list of 3-tuples of (x,y) tuples");	\
	  delete[] point_array;													\
	  return NULL;															\
	}																		\
	while (0)

	POINT * point_array = new POINT[num*3];
	for (int i=0; i < num; i++) {
	  PyObject * triplet = PyList_GetItem (triple_list, i);
	  if (!PyTuple_Check (triplet) || PyTuple_Size (triplet) != 3) {
		HURL;
	  } else {
		for (int j=0; j < 3; j++) {
		  PyObject *point = PyTuple_GetItem (triplet, j);
		  if (!PyTuple_Check (point) || PyTuple_Size (point) != 2) {
			HURL;
		  } else {
			PyObject *px, *py;
			px = PyTuple_GetItem (point, 0);
			py = PyTuple_GetItem (point, 1);
			if (!PyInt_Check(px) || !PyInt_Check(py)) {
			  HURL;
			} else {
			  point_array[index].x = PyInt_AsLong (px);
			  point_array[index].y = PyInt_AsLong (py);
			  index++;
			}
		  }
		}
	  }
	}
	// we have an array of POINT structs, now we
	// can finally draw the splines..
	BOOL result;
	if (do_to) {
	  result  = pDC->PolyBezierTo (point_array, index);
	} else {
	  result  = pDC->PolyBezier (point_array, index);
	}
	delete[] point_array;
	if (!result) {
	  RETURN_API_ERR("CDC::PolyBezier[To]");
	} else {
	  RETURN_NONE;
	}
  }
}

// @pymethod |PyCDC|FillRect|Fills a given rectangle with the specified brush
static PyObject *
ui_dc_fillrect (PyObject *self, PyObject *args)
{
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	RECT rect;
	PyObject *obBrush;
	if (!PyArg_ParseTuple (args, "(iiii)O:FillRect", 
	          &rect.left, &rect.top, &rect.right, &rect.bottom,
			  // @pyparm (left, top, right, bottom|rect||Specifies the bounding rectangle, in logical units.
	          &obBrush)) // @pyparm <o PyCBrush>|brush||Specifies the brush to use.
		return NULL;
	if (!ui_base_class::is_uiobject (obBrush, &PyCBrush::type))
		RETURN_TYPE_ERR("The 'O' param must be a PyCBrush object");
	CBrush *pBrush = PyCBrush::GetBrush(obBrush);
	if (!pBrush)
		return NULL;
	pDC->FillRect( &rect, pBrush );
	// @pyseemfc CDC|FillRect
	RETURN_NONE;
}

// @pymethod |PyCDC|FillSolidRect|Fills the given rectangle with the specified solid color.
static PyObject *
ui_dc_fillsolidrect (PyObject *self, PyObject *args)
{
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	RECT rect;
	int col;
	if (!PyArg_ParseTuple (args, "(iiii)i:FillSolidRect", 
	          &rect.left, &rect.top, &rect.right, &rect.bottom,
			  // @pyparm (left, top, right, bottom|rect||Specifies the bounding rectangle, in logical units.
	          &col)) // @pyparm int|color||Specifies the color to use.
		return NULL;
	pDC->FillSolidRect( &rect, (COLORREF)col );
	// @pyseemfc CDC|FillSolidRect
	RETURN_NONE;
}

// @pymethod |PyCDC|FrameRect|Draws a border around the rectangle specified by rect
static PyObject *
ui_dc_framerect (PyObject *self, PyObject *args)
{
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	RECT rect;
	PyObject *obBrush;
	if (!PyArg_ParseTuple (args, "(iiii)O:FrameRect", 
	          &rect.left, &rect.top, &rect.right, &rect.bottom,
			  // @pyparm (left, top, right, bottom|rect||Specifies the bounding rectangle, in logical units.
	          &obBrush)) // @pyparm <o PyCBrush>|brush||Specifies the brush to use.
		return NULL;
	if (!ui_base_class::is_uiobject (obBrush, &PyCBrush::type))
		RETURN_TYPE_ERR("The 'O' param must be a PyCBrush object");
	CBrush *pBrush = PyCBrush::GetBrush(obBrush);
	if (!pBrush)
			RETURN_ERR("The PyCBrush parameter is invalid.");
	pDC->FrameRect( &rect, pBrush );
	// @pyseemfc CDC|FrameRect
	RETURN_NONE;
}

// @pymethod int|PyCDC|GetNearestColor|Returns the closest color a device can map.
static PyObject *ui_dc_get_nearest_color (PyObject *self, PyObject *args)
{
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	int col;
	// @pyparm int|color||Specifies the color to be matched.
	if (!PyArg_ParseTuple (args, "i:GetNearestColor", &col))
	  return NULL;
	return Py_BuildValue ("i", pDC->GetNearestColor(col));
}

// @pymethod (x,y)|PyCDC|GetTextExtentPoint|An alias for <om PyCDC.GetTextExtent>. 
// GetTextExtentPoint is the preferred win32api name, but GetTextExtent is the MFC name.<nl>
// Calculates the width and height of a line of text using the current font to determine the dimensions. 
// @pyparm string|text||The text to calculate for.
// @rdesc A tuple of integers with the size of the string, in logical units.

// @pymethod (x,y)|PyCDC|GetTextExtent|Calculates the width and height of a line of text using the current font to determine the dimensions. 
static PyObject *ui_dc_get_text_extent (PyObject *self, PyObject *args)
{
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	char *text;
	int strLen;
	// @pyparm string|text||The text to calculate for.
	if (!PyArg_ParseTuple (args, "s#", &text, &strLen))
	  return NULL;
	CSize sz = pDC->GetTextExtent(text, strLen); // @pyseemfc CFC|GetTextExtent
	return Py_BuildValue ("(ii)", sz.cx, sz.cy);
	// @rdesc A tuple of integers with the size of the string, in logical units.
}

// @pymethod int|PyCDC|SetTextColor|Sets the text color to the specified color.
static PyObject *
ui_dc_set_text_color (PyObject *self, PyObject *args)
{
	// @comm This text color is used when writing text to this device context and also when converting bitmaps between color and monochrome device contexts. 
	// If the device cannot represent the specified color, the system sets the text color to the nearest physical color.
	// The background color for a character is specified by the SetBkColor and SetBkMode member functions.
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;

  int new_color;
  // @pyparm int|color||A windows color specification.  See the win32api documentation for details.
  if (!PyArg_ParseTuple (args, "i", &new_color))
	return NULL;

  int old_color = pDC->SetTextColor (new_color); // @pyseemfc CDC|SetTextColor
  return Py_BuildValue ("i", old_color);
  // @rdesc The return value is the previous text color.
}

// @pymethod int|PyCDC|SetBkColor|Sets the current background color to the specified color.
static PyObject *
ui_dc_set_bk_color (PyObject *self, PyObject *args)
{
  // @comm If the background mode is OPAQUE, the system uses the background color 
  // to fill the gaps in styled lines, the gaps between hatched lines in brushes, and 
  // the background in character cells.
  // The system also uses the background color when converting bitmaps between color and 
  // monochrome device contexts. 
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  int new_color;
  if (!PyArg_ParseTuple (args, "i", &new_color))  // @pyparm int|color||A windows color specification.  See the win32api documentation for details.
	return NULL;
  int old_color = pDC->SetBkColor (new_color); // @pyseemfc CDC|SetBkColor
  return Py_BuildValue ("i", old_color);
  // @rdesc The return value is the previous background color.
}

// @pymethod int|PyCDC|SetBkMode|Sets the current background mode to the specified mode.
static PyObject *
ui_dc_set_bk_mode (PyObject *self, PyObject *args)
{
  // @comm Specifies the mode to be set.  This parameter can be either OPAQUE or TRANSPARENT 
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  int new_mode;
  if (!PyArg_ParseTuple (args, "i", &new_mode))  // @pyparm int|mode||A background mode.  May be either TRANSPARENT or OPAQUE.
	return NULL;
  int old_mode = pDC->SetBkMode(new_mode); // @pyseemfc CDC|SetBkMode
  return Py_BuildValue ("i", old_mode);
  // @rdesc The return value is the previous background mode.
}

// @pymethod (int, int)|PyCDC|SetBrushOrg|Specifies the origin that GDI will assign to the next brush that the application selects into the device context.
static PyObject *
ui_dc_set_brush_org (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  int x, y;
  // @pyparm (x,y)|point||The new origin in device units.
  if (!PyArg_ParseTuple (args, "(ii)", &x, &y))
	return NULL;
  CPoint pt = pDC->SetBrushOrg( x, y ); // @pyseemfc CDC|SetBrushOrg
  return Py_BuildValue ("(ii)", pt.x, pt.y);
  // @rdesc The previous origin in device units.
}

// @pymethod (int,int)|PyCDC|GetBrushOrg|Retrieves the origin (in device units) of the brush currently selected for the device context. 
static PyObject *
ui_dc_get_brush_org(PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  int item;
  if (!PyArg_ParseTuple (args, ":GetBrushOrg", &item))
	return NULL;
  CPoint pt = pDC->GetBrushOrg();// @pyseemfc CDC|GetBrushOrg
  return Py_BuildValue ("ii", pt.x, pt.y);
}


// @pymethod int|PyCDC|GetDeviceCaps|Retrieves a capability of the device context.
static PyObject *
ui_dc_get_device_caps (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  int item;
  if (!PyArg_ParseTuple (args, "i", &item))  // @pyparm int|index||The information requested.  See the win32api documentation for details.
	return NULL;
  int value = pDC->GetDeviceCaps (item); // @pyseemfc CDC|GetDeviceCaps
  return Py_BuildValue ("i", value);
  // @rdesc The value of the requested capability
}

// @pymethod int|PyCDC|SetMapMode|Sets the mapping mode for the device context.
static PyObject *
ui_dc_set_map_mode (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  int new_mode;
  if (!PyArg_ParseTuple (args, "i", &new_mode))
  // @pyparm int|newMode||The new mode.  Can be one of 
  // MM_ANISOTROPIC, MM_HIENGLISH, MM_HIMETRIC, MM_ISOTROPIC, MM_LOENGLISH, MM_LOMETRIC, MM_TEXT, MM_TWIPS
	return NULL;
  int old_mode = pDC->SetMapMode (new_mode); // @pyseemfc CDC|SetMapMode
  if (old_mode == 0)
	RETURN_ERR ("SetMapMode failed");
  else
	return Py_BuildValue ("i", old_mode);
  // @rdesc The previous mapping mode.
}

static PyObject *
ui_dc_set_window_org (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC) {
	return NULL;
  } else {
	int x, y;
	if (!PyArg_ParseTuple (args, "(ii)", &x, &y)) {
	return NULL;
	} else {
	  CSize old_size = pDC->SetWindowOrg (x, y);
	  return Py_BuildValue ("(ii)", old_size.cx, old_size.cy);
	}
  }
}

static PyObject *
ui_dc_get_window_org (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC) {
	return NULL;
  } else {
	if (!PyArg_ParseTuple (args, "")) {
	return NULL;
	} else {
	  CSize org = pDC->GetWindowOrg ();
	  return Py_BuildValue ("(ii)", org.cx, org.cy);
	}
  }
}

static PyObject *
ui_dc_set_viewport_org (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC) {
	return NULL;
  } else {
	int x, y;
	if (!PyArg_ParseTuple (args, "(ii)", &x, &y)) {
	return NULL;
	} else {
	  CSize old_size = pDC->SetViewportOrg (x, y);
	  return Py_BuildValue ("(ii)", old_size.cx, old_size.cy);
	}
  }
}

static PyObject *
ui_dc_get_viewport_org (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC) {
	return NULL;
  } else {
	if (!PyArg_ParseTuple (args, "")) {
	return NULL;
	} else {
	  CSize org = pDC->GetViewportOrg ();
	  return Py_BuildValue ("(ii)", org.cx, org.cy);
	}
  }
}

static PyObject *
ui_dc_get_viewport_ext (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC) {
	return NULL;
  } else {
	if (!PyArg_ParseTuple (args, "")) {
	return NULL;
	} else {
	  CSize ext = pDC->GetViewportExt ();
	  return Py_BuildValue ("(ii)", ext.cx, ext.cy);
	}
  }
}

static PyObject *
ui_dc_get_window_ext (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC) {
	return NULL;
  } else {
	if (!PyArg_ParseTuple (args, "")) {
	return NULL;
	} else {
	  CSize ext = pDC->GetWindowExt ();
	  return Py_BuildValue ("(ii)", ext.cx, ext.cy);
	}
  }
}

static PyObject *
ui_dc_set_graphics_mode (PyObject * self, PyObject * args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC) {
	return NULL;
  } else {
	int mode;
	if (!PyArg_ParseTuple (args, "i", &mode)) {
	  return NULL;
	} else {
	  return Py_BuildValue ("i", SetGraphicsMode (pDC->GetSafeHdc(), mode));
	}
  }
}

static PyObject *
ui_dc_set_world_transform (PyObject * self, PyObject * args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC) {
	return NULL;
  } else {
	XFORM xf;
	if (!PyArg_ParseTuple (args,
						   "ffffff",
						   &xf.eM11,
						   &xf.eM12,
						   &xf.eM21,
						   &xf.eM22,
						   &xf.eDx,
						   &xf.eDy
						   )) {
	  return NULL;
	} else {
	  return Py_BuildValue ("i", SetWorldTransform (pDC->GetSafeHdc(), &xf));
	}
  }
}

// @pymethod (x,y)|PyCDC|SetWindowExt|Sets the x,y extents of the window associated with the device context.
static PyObject *
ui_dc_set_window_ext (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  int x, y;
  // @pyparm (x,y)|size||The new size.
  if (!PyArg_ParseTuple (args, "(ii)", &x, &y))
	return NULL;
  CSize old_size = pDC->SetWindowExt (x, y); // @pyseemfc CDC|SetWindowExt
  if (old_size.cx == 0 && old_size.cy == 0)
	RETURN_ERR ("SetWindowExt failed");
  else
	return Py_BuildValue ("(ii)", old_size.cx, old_size.cy);
  // @rdesc The previous extents of the window (in logical units).
}

// @pymethod (x,y)|PyCDC|SetViewportExt|Sets the x,y extents of the viewport of the device context.
static PyObject *
ui_dc_set_viewport_ext (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  int x, y;
  // @pyparm (x,y)|size||The new size.
  if (!PyArg_ParseTuple (args, "(ii)", &x, &y))
	return NULL;
  CSize old_size = pDC->SetViewportExt (x, y); // @pyseemfc CDC|SetViewportExt
  if (old_size.cx == 0 && old_size.cy == 0)
	RETURN_ERR ("SetViewportExt failed");
  else
	return Py_BuildValue ("(ii)", old_size.cx, old_size.cy);
  // @rdesc The previous extents of the viewport (in logical units).
}

// @pymethod int|PyCDC|SetTextAlign|Sets the text-alignment flags.
static PyObject *
ui_dc_set_text_align (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  int new_flags;
  // @pyparm int|newFlags||The new alignment flags.  Can be a combination of (TA_CENTER, TA_LEFT, TA_RIGHT), (TA_BASELINE, TA_BOTTOM, TA_TOP) and (TA_NOUPDATECP, TA_UPDATECP)<nl>
  // The default is TA_LEFT\|TA_TOP\|TA_NOUPDATECP
  if (!PyArg_ParseTuple (args, "i", &new_flags))
	return NULL;
  int old_flags = pDC->SetTextAlign (new_flags); // @pyseemfc CDC|SetTextAlign
  return Py_BuildValue ("i", old_flags);
  // @rdesc The old alignment flags.
}

// @pymethod object|PyCDC|SelectObject|Selects an object into the device context.<nl>
// Currently, only <o PyCFont>, <o PyCBitMap>, and <o PyCPen> objects are supported.
static PyObject *
ui_dc_select_object (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  PyObject *v;
  // @pyparm object|ob||The object to select.
  if (!PyArg_ParseTuple (args, "O", &v))
	return NULL;

  if (ui_base_class::is_uiobject (v, &PyCFont::type)) {
    PyCFont *new_font = (PyCFont *) v;
    CFont *cFont = pDC->SelectObject (new_font->GetFont());
    if (cFont == NULL)
      RETURN_ERR ("Select font object failed");
    else {
      PyCFont *ret = (PyCFont *)ui_assoc_object::make (PyCFont::type, cFont );
      if (ret && ret->ob_refcnt == 1) // only set m_delete if new object
	ret->m_deleteObject = FALSE;
      return ret;
    }
  } else if (ui_base_class::is_uiobject (v, &ui_bitmap::type)) {
    ui_bitmap *new_bitmap = (ui_bitmap *) v;
    CBitmap *pbm = pDC->SelectObject (new_bitmap->GetBitmap());
    if (pbm == NULL)
      RETURN_ERR ("Select bitmap object failed");
    else {
      ui_bitmap *ret = (ui_bitmap *)ui_assoc_object::make (ui_bitmap::type, pbm );
      if (ret && ret->ob_refcnt == 1) // only set m_delete if new object
	ret->m_deleteObject = FALSE;
      return ret;
    }
  } else if (ui_base_class::is_uiobject (v, &PyCBrush::type)) {
    PyCBrush *new_brush = (PyCBrush *) v;
    CBrush *pbm = pDC->SelectObject (PyCBrush::GetBrush(new_brush));
    if (pbm == NULL)
      RETURN_ERR ("Select brush object failed");
    else {
      PyCBrush *ret = (PyCBrush *)ui_assoc_object::make (PyCBrush::type, pbm );
      if (ret && ret->ob_refcnt == 1) // only set m_delete if new object
	ret->m_deleteObject = FALSE;
      return ret;
    }
  } else if (ui_base_class::is_uiobject (v, &ui_pen_object::type)) {
    ui_pen_object * new_pen = (ui_pen_object *) v;
    CPen * cPen = pDC->SelectObject (new_pen->GetPen());
    if (cPen == NULL) {
      RETURN_ERR ("Select pen object failed");
      } else {
	ui_pen_object *ret = (ui_pen_object *)ui_assoc_object::make (ui_pen_object::type, cPen );
	if (ret) {
	  ret->m_deleteObject = FALSE;
	}
	return ret;
      }
  }
  RETURN_ERR ("Attempt to select unsupported object type.");
  // @pyseemfc CDC|SelectObject
  // @rdesc The previously selected object.  This will be the same type as the object parameter.
}

// @pymethod int|PyCDC|SelectPalette|Sets the logical palette.
static PyObject *
ui_dc_select_palette (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  HPALETTE hPal;
  BOOL bForceBG;
  // @pyparm int|hPalette||The handle to the palette
  // @pyparm int|forceBackground||Specifies whether the logical palette is forced to be a background palette.
  if (!PyArg_ParseTuple (args, "i:SelectPalette", &hPal, &bForceBG))
	return NULL;
  HPALETTE ret = ::SelectPalette(pDC->GetSafeHdc(), hPal, bForceBG); // @pyseemfc CDC|SelectePalette
  return Py_BuildValue ("i", (int)ret);
  // @rdesc The previous palette handle.
}

// @pymethod int|PyCDC|RealizePalette|Maps palette entries in the current logical palette to the system palette.
static PyObject *
ui_dc_realize_palette (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;
  CHECK_NO_ARGS2(args, RealizePalette);
  UINT ret = pDC->RealizePalette();
  return Py_BuildValue ("i", ret);
  // @rdesc Indicates how many entries in the logical palette were mapped to different entries 
  // in the system palette. This represents the number of entries that this function 
  // remapped to accommodate changes in the system palette since the logical palette 
  // was last realized.
}


// @pymethod dict|PyCDC|SetROP2|Sets the current drawing mode.
static PyObject *
ui_dc_set_rop2 (PyObject * self, PyObject * args)
{
  int mode, old_mode;
  CDC *pDC = ui_dc_object::GetDC (self);
  if (!pDC) {
    return NULL;
  }
  // @pyparm int|mode||The new drawing mode.
  if (!PyArg_ParseTuple (args, "i", &mode)) {
    return NULL;
  }
  old_mode = pDC->SetROP2 (mode);// @pyseemfc CDC|SetROP2
  return Py_BuildValue ("i", old_mode);
}


// @pymethod |PyCDC|TextOut|Outputs text to the display context, using the currently selected font.
static PyObject *ui_dc_text_out (PyObject *self, PyObject *args)
{
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	char *text;
	int strLen, x, y;
	if (!PyArg_ParseTuple (args, "iis#", 
	          &x,        // @pyparm x|int||The x coordinate to write the text to.
	          &y,        // @pyparm y|int||The y coordinate to write the text to.
	          &text,     // @pyparm text|string||The text to write.
	          &strLen))
	  return NULL;
	BOOL ret = pDC->TextOut (x, y, text, strLen); // @pyseemfc CDC|TextOut
	if (!ret)
		RETURN_API_ERR("CDC::TextOut");
	RETURN_NONE;
	// @rdesc Always none.  If the function fails, an exception is raised.
}

/* struct to dict macro (alpha version)
   move to win32ui_int eventually */
#define DICTADD(D,ST,M,TYPE) PyDict_SetItemString (D, #M, Py_BuildValue (TYPE, ST.M))

// @pymethod dict|PyCDC|GetTextMetrics|Retrieves the metrics for the current font in this device context.
static PyObject *
ui_dc_get_text_metrics (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;

  CHECK_NO_ARGS(args);

  TEXTMETRIC tm;

  if (!pDC->GetTextMetrics (&tm)) // @pyseemfc CDC|GetTextMetrics
	RETURN_ERR ("GetTextMetrics failed");

  PyObject *d = PyDict_New();

  // @rdesc A dictionary of integers, keyed by the following strings:<nl>
  DICTADD (d, tm, tmHeight, "i"); // tmHeight<nl>
  DICTADD (d, tm, tmAscent, "i"); // tmAscent<nl>
  DICTADD (d, tm, tmDescent, "i"); // tmDescent<nl>
  DICTADD (d, tm, tmInternalLeading, "i"); // tmInternalLeading<nl>
  DICTADD (d, tm, tmExternalLeading, "i"); // tmExternalLeading<nl>
  DICTADD (d, tm, tmAveCharWidth, "i"); // tmAveCharWidth<nl>
  DICTADD (d, tm, tmMaxCharWidth, "i"); // tmMaxCharWidth<nl>
  DICTADD (d, tm, tmWeight, "i"); // tmWeight<nl>
  DICTADD (d, tm, tmItalic, "i"); // tmItalic<nl>
  DICTADD (d, tm, tmUnderlined, "i"); // tmUnderlined<nl>
  DICTADD (d, tm, tmStruckOut, "i"); // tmStruckOut<nl>
  DICTADD (d, tm, tmFirstChar, "i"); // tmFirstChar<nl>
  DICTADD (d, tm, tmLastChar, "i"); // tmLastChar<nl>
  DICTADD (d, tm, tmDefaultChar, "i"); // tmDefaultChar<nl>
  DICTADD (d, tm, tmBreakChar, "i"); // tmBreakChar<nl>
  DICTADD (d, tm, tmPitchAndFamily, "i"); // tmPitchAndFamily<nl>
  DICTADD (d, tm, tmCharSet, "i"); // tmCharSet<nl>
  DICTADD (d, tm, tmOverhang, "i"); // tmOverhang<nl>
  DICTADD (d, tm, tmDigitizedAspectX, "i"); // tmDigitizedAspectX<nl>
  DICTADD (d, tm, tmDigitizedAspectY, "i"); // tmDigitizedAspectY<nl>
  
  return d;
}

// @pymethod string|PyCDC|GetTextFace|Returns typeface name of the current font.
static PyObject *
ui_dc_get_text_face (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;

  if (!PyArg_ParseTuple (args, ""))
	return NULL;

  char buf[LF_FACESIZE];

  int ret = pDC->GetTextFace (LF_FACESIZE, buf); // @pyseemfc CDC|GetTextFace
  if (ret == 0)
	buf[0] = '\0';

  return Py_BuildValue ("s", buf);
}


// @pymethod int|PyCDC|SaveDC|Saves the current state of the device context.  Windows manages a stack of state information.
// The saved device context can later be restored by using <om CDC.RestoreDC>
static PyObject *
ui_dc_save_dc (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;

  if (!PyArg_ParseTuple (args, ""))
	return NULL;

  int ret = pDC->SaveDC(); // @pyseemfc CDC|SaveDC
  if (ret == 0)
	RETURN_ERR ("SaveDC failed");
  else
	return Py_BuildValue ("i", ret);
  // @rdesc An integer identifying the context, which can be used by <om PyCDC.RestoreDC>.
  // An exception is raised if this function fails.
}

// @pymethod |PyCDC|RestoreDC|Restores the state of the device context.
static PyObject *
ui_dc_restore_dc (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;

  int saved;
  // @pyparm int|saved||The id of a previously saved device context.  See <om PyCDC.SaveDC>
  if (!PyArg_ParseTuple (args, "i", &saved))
	return NULL;

  if (!pDC->RestoreDC (saved)) // @pyseemfc CDC|RestoreDC
	RETURN_ERR ("RestoreDC failed");
  else
	RETURN_NONE;
}

// @pymethod (x,y)|PyCDC|MoveTo|Moves the current position to a specified point.
static PyObject *
ui_dc_move_to (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;

  // @pyparm (x,y)|point||The point coordinate to move to.
  // @pyparmalt1 int|x||The x coordinate to move to.
  // @pyparmalt1 int|y||The y coordinate to move to.
  int x, y;
  CPoint prev;
  if (!PyArg_ParseTuple (args, "ii", &x, &y)) {
    PyErr_Clear();
	if (!PyArg_ParseTuple (args, "(ii)", &x, &y))
	  return NULL;
  }

  prev = pDC->MoveTo (x,y); // @pyseemfc CDC|MoveTo

  return Py_BuildValue ("(ii)", prev.x, prev.y);
  // @rdesc The previous position.
}

// @pymethod |PyCDC|LineTo|Draws a line to a specified point, using the currently selected pen.
static PyObject *
ui_dc_line_to (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;

  // @pyparm (x,y)|point||The point coordinate to draw to.
  // @pyparmalt1 int|x||The x coordinate to draw to.
  // @pyparmalt1 int|y||The y coordinate to draw to.
  int x, y;
  CPoint prev;
  if (!PyArg_ParseTuple (args, "ii", &x, &y)) {
    PyErr_Clear();
	if (!PyArg_ParseTuple (args, "(ii)", &x, &y))
	  return NULL;
  }

  if (!pDC->LineTo (x,y)) // @pyseemfc CDC|LineTo
	RETURN_ERR ("LineTo failed");
  else
 	RETURN_NONE;
}

// @pymethod (x,y)|PyCDC|DPtoLP|Converts device units into logical units.
static PyObject *
ui_dc_dp_to_lp (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;

  CPoint pt;
  // @todo Should really handle list of (x,y) points
  // @pyparm (x,y)|point||The point to convert
  // @pyparmalt1 int|x||The x coordinate to convert.
  // @pyparmalt1 int|y||The y coordinate to convert.
  if (!PyArg_ParseTuple (args, "ii", &pt.x, &pt.y)) {
    PyErr_Clear();
	if (!PyArg_ParseTuple (args, "(ii)", &pt.x, &pt.y))
	  return NULL;
  }

  pDC->DPtoLP (&pt, 1); // @pyseemfc CDC|DPtoLP
  return (Py_BuildValue ("(ii)", pt.x, pt.y));
  // @rdesc The converted coordinates.
}

// @pymethod (x,y)|PyCDC|LPtoDP|Converts logical units into device units.
static PyObject *
ui_dc_lp_to_dp (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;

  CPoint pt;
  // should really handle array of (x,y) points
  // @pyparm (x,y)|point||The point coordinate to convert.
  // @pyparmalt1 int|x||The x coordinate to convert.
  // @pyparmalt1 int|y||The y coordinate to convert.
  if (!PyArg_ParseTuple (args, "ii", &pt.x, &pt.y)) {
    PyErr_Clear();
	if (!PyArg_ParseTuple (args, "(ii)", &pt.x, &pt.y))
	  return NULL;
  }

  pDC->LPtoDP (&pt, 1); // @pyseemfc CDC|LPtoDP
  return (Py_BuildValue ("(ii)", pt.x, pt.y));
  // @rdesc The converted coordinates.

}

// @pymethod (left, top, right, bottom)|PyCDC|GetClipBox|Retrieves the dimensions of the smallest bounding rectangle around the current clipping boundary.
static PyObject *
ui_dc_get_clip_box (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;

  if (!PyArg_ParseTuple (args, ""))
	return NULL;

  CRect rect;
  int ret = pDC->GetClipBox (&rect); // @pyseemfc CDC|GetClipBox
  if (ret == ERROR)
	RETURN_ERR ("GetClipBox failed");
  else
	return Py_BuildValue ("(iiii)", rect.left, rect.top, rect.right, rect.bottom);
	// @rdesc A tuple of integers specifying the rectangle.
}

// @pymethod int|PyCDC|GetHandleAttrib|Retrieves the handle of the attribute device context.
static PyObject *
ui_dc_get_handle_attrib (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;

  if (!PyArg_ParseTuple (args, ":GetHandleAttrib"))
	return NULL;

  return Py_BuildValue ("i", pDC->m_hAttribDC);
}
// @pymethod int|PyCDC|GetHandleOutput|Retrieves the handle of the output device context.
static PyObject *
ui_dc_get_handle_output (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC)
	return NULL;

  if (!PyArg_ParseTuple (args, ":GetHandleOutput"))
	return NULL;

  return Py_BuildValue ("i", pDC->m_hDC);
}

// Path methods:
// BeginPath
// EndPath
// StrokePath
// FillPath
// StrokeAndFillPath

static PyObject *
ui_dc_begin_path (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC) {
	return NULL;
  } else if (!PyArg_ParseTuple (args, "")) {
	return NULL;
  } else {
	if (!pDC->BeginPath()) {
	  RETURN_API_ERR ("CDC::BeginPath");
	} else {
	  RETURN_NONE;
	}
  }
}
	
static PyObject *
ui_dc_end_path (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC) {
	return NULL;
  } else if (!PyArg_ParseTuple (args, "")) {
	return NULL;
  } else {
	if (!pDC->EndPath()) {
	  RETURN_API_ERR ("CDC::EndPath");
	} else {
	  RETURN_NONE;
	}
  }
}
	
static PyObject *
ui_dc_fill_path (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC) {
	return NULL;
  } else if (!PyArg_ParseTuple (args, "")) {
	return NULL;
  } else {
	if (!pDC->FillPath()) {
	  RETURN_API_ERR ("CDC::FillPath");
	} else {
	  RETURN_NONE;
	}
  }
}

static PyObject *
ui_dc_stroke_path (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC) {
	return NULL;
  } else if (!PyArg_ParseTuple (args, "")) {
	return NULL;
  } else {
	if (!pDC->StrokePath()) {
	  RETURN_API_ERR ("CDC::StrokePath");
	} else {
	  RETURN_NONE;
	}
  }
}

static PyObject *
ui_dc_stroke_and_fill_path (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC) {
	return NULL;
  } else if (!PyArg_ParseTuple (args, "")) {
	return NULL;
  } else {
	if (!pDC->StrokeAndFillPath()) {
	  RETURN_API_ERR ("CDC::StrokeAndFillPath");
	} else {
	  RETURN_NONE;
	}
  }
}

static PyObject *
ui_dc_is_printing (PyObject *self, PyObject *args)
{
  CDC *pDC = ui_dc_object::GetDC(self);
  if (!pDC) {
	return NULL;
  } else {
	return Py_BuildValue ("i", pDC->IsPrinting());
  }
}


static PyObject *
ui_dc_scale_window_ext (PyObject * self, PyObject * args)
{
  CDC *pDC = ui_dc_object::GetDC (self);
  if (!pDC) {
	return NULL;
  } else {
	int xn,xd,yn,yd;
	if (!PyArg_ParseTuple (args, "iiii", &xn, &xd, &yn, &yd)) {
	  return NULL;
	} else {
	  CSize r = pDC->ScaleWindowExt (xn, xd, yn, yd);
	  return Py_BuildValue ("(ii)", r.cx, r.cy);
	}
  }
}


static PyObject *
ui_dc_scale_viewport_ext (PyObject * self, PyObject * args)
{
  CDC *pDC = ui_dc_object::GetDC (self);
  if (!pDC) {
	return NULL;
  } else {
	int xn,xd,yn,yd;
	if (!PyArg_ParseTuple (args, "iiii", &xn, &xd, &yn, &yd)) {
	  return NULL;
	} else {
	  CSize r = pDC->ScaleViewportExt (xn, xd, yn, yd);
	  return Py_BuildValue ("(ii)", r.cx, r.cy);
	}
  }
}

// Printing functions

// @pymethod |PyCDC|AbortDoc|Aborts a print job
static PyObject *ui_dc_abort_doc(PyObject * self, PyObject * args)
{
	CHECK_NO_ARGS2(args, AbortDoc);
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	pDC->AbortDoc();
	RETURN_NONE;
}

// @pymethod |PyCDC|EndDoc|Finishes spooling the document and starts printing it
static PyObject *ui_dc_end_doc(PyObject * self, PyObject * args)
{
	CHECK_NO_ARGS2(args, EndDoc);
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	int err = pDC->EndDoc();
	if (err < 0)
	{
		char msg[64];
		sprintf(msg, "EndDoc failed (error code %d)", err);
		PyErr_SetString(ui_module_error, msg);
		return NULL;
	}
	RETURN_NONE;
}

// @pymethod |PyCDC|EndPage|Finishes a page on a printer DC
static PyObject *ui_dc_end_page(PyObject * self, PyObject * args)
{
	CHECK_NO_ARGS2(args, EndPage);
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	int err = pDC->EndPage();
	if (err < 0)
	{
		char msg[64];
		sprintf(msg, "EndDoc failed (error code %d)", err);
		PyErr_SetString(ui_module_error, msg);
		return NULL;
	}
	RETURN_NONE;
}

// @pymethod |PyCDC|StartDoc|Starts spooling a document to a printer DC
static PyObject *ui_dc_start_doc(PyObject * self, PyObject * args)
{
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;

	// @pyparm string|docName||The document name
	// @pyparm string|outputFile||The output file name. Use this to spool to a file. Omit to send to the printer.
	char *docName, *outputFile = NULL;
	if (!PyArg_ParseTuple(args, "s|z:StartDoc", &docName, &outputFile))
		return NULL;

	DOCINFO info;
	info.cbSize = sizeof(DOCINFO);
	memset(&info, 0, sizeof(DOCINFO));
	info.lpszDocName = docName;
	info.lpszOutput = outputFile;

	if (pDC->StartDoc(&info) < 0)
	{
		RETURN_ERR("StartDoc failed");
	}

	RETURN_NONE;
}

// @pymethod |PyCDC|StartPage|Starts a new page on a printer DC
static PyObject *ui_dc_start_page(PyObject * self, PyObject * args)
{
	CHECK_NO_ARGS2(args, StartPage);
	CDC *pDC = ui_dc_object::GetDC(self);
	if (!pDC)
		return NULL;
	if (pDC->StartPage() <= 0)
		RETURN_ERR("StartPage failed");
	RETURN_NONE;
}

// DC Methods
// @object PyCDC|A Device Context.  Encapsulates an MFC <c CDC> class.
static struct PyMethodDef ui_dc_methods[] = {
	{"AbortDoc",			ui_dc_abort_doc,		1}, // @pymeth AbortDoc|Aborts a print job
	{"Arc",					ui_dc_arc,	1}, // @pymeth Arc|Draws an arc.
	{"BitBlt",				ui_dc_bitblt,	1}, // @pymeth BitBlt|Copies a bitmap
	{"Chord",				ui_dc_chord,	1}, // @pymeth Chord|Draws a chord.
    {"CreateCompatibleDC",	ui_dc_object::create_compatible_dc,	1}, // @pymeth CreateCompatibleDC|Creates a memory DC compatible with this DC.
    {"CreatePrinterDC",		ui_dc_object::create_printer_dc,	1}, // @pymeth CreatePrinterDC|Creates a device context for a specific printer
	{"DeleteDC",            ui_dc_delete_dc, 1}, // @pymeth DeleteDC|Deletes all resources associated with a device context.
    {"DPtoLP",				ui_dc_dp_to_lp,	1}, // @pymeth DPtoLP|Convert from device points to logical points.
	{"DrawFocusRect",       ui_dc_draw_focus_rect, 1}, // @pymeth DrawFocusRect|Draws a rectangle in the style used to indicate the rectangle has focus
	{"DrawIcon",			ui_dc_draw_icon, 1}, // @pymeth DrawIcon|Draws an icon on the DC.
	{"LPtoDP",				ui_dc_lp_to_dp,	1}, // @pymeth LPtoDP|Convert from logical points to device points
	{"Ellipse",             ui_dc_ellipse,  1}, // @pymeth Ellipse|Draws an Ellipse.
	{"EndDoc",				ui_dc_end_doc,		1}, // @pymeth EndDoc|Finishes spooling the document and starts printing it
	{"EndPage",				ui_dc_end_page,		1}, // @pymeth EndPage|Finishes a page on a printer DC
	{"FillRect",            ui_dc_fillrect, 1}, // @pymeth FillRect|Fills a given rectangle with the specified brush
	{"FillSolidRect",       ui_dc_fillsolidrect, 1}, // @pymeth FillSolidRect|Fills the given rectangle with the specified solid color.
	{"FrameRect",           ui_dc_framerect, 1}, // @pymeth FrameRect|Draws a border around the rectangle specified by rect
	{"GetBrushOrg",         ui_dc_get_brush_org, 1}, // @pymeth GetBrushOrg|Retrieves the origin (in device units) of the brush currently selected for the device context. 
	{"GetClipBox",			ui_dc_get_clip_box,	1}, // @pymeth GetClipBox|Retrives the current clipping region.
	{"GetDeviceCaps",		ui_dc_get_device_caps,	1}, // @pymeth GetDeviceCaps|Retrieves current device capabilities.
	{"GetHandleAttrib",     ui_dc_get_handle_attrib, 1}, // @pymeth GetHandleAttrib|Retrieves the handle of the attribute device context.
	{"GetHandleOutput",     ui_dc_get_handle_output, 1}, // @pymeth GetHandleOutput|Retrieves the handle of the output device context.
	{"GetNearestColor",     ui_dc_get_nearest_color, 1}, // @pymeth GetNearestColor|Returns the closest color a device can map.
	{"GetTextExtent",		ui_dc_get_text_extent,	1}, // @pymeth GetTextExtent|Calculates the size of the string.
	{"GetTextExtentPoint",	ui_dc_get_text_extent,	1}, // @pymeth GetTextExtentPoint|Alias for GetTextExtent - Calculates the size of the string.
	{"GetTextFace",         ui_dc_get_text_face,	1}, // @pymeth GetTextFace|Retrieves the name of the current font.
	{"GetTextMetrics",		ui_dc_get_text_metrics,	1}, // @pymeth GetTextMetrics|Retrieves the metrics for the current font.
	{"LineTo",				ui_dc_line_to,	1}, // @pymeth LineTo|Draws a line to a specified point.
	{"MoveTo",				ui_dc_move_to,	1}, // @pymeth MoveTo|Moves the current position to a specifed point.
	{"PatBlt",              ui_dc_patblt, 1}, // @pymeth PatBlt|Creates a bit pattern on the device.
	{"RealizePalette",		ui_dc_realize_palette,	1}, // @pymeth RealizePalette|Maps palette entries in the current logical palette to the system palette.
	{"RectVisible",			ui_dc_rect_visible,	1}, // @pymeth RectVisible|Determines if a rectangle is currently visisble in the viewport.
	{"RestoreDC",			ui_dc_restore_dc,	1}, // @pymeth RestoreDC|Restores a saved DC.
	{"SaveDC",				ui_dc_save_dc,	1}, // @pymeth SaveDC|Saves a DC.
	{"SelectObject",		ui_dc_select_object,	1}, // @pymeth SelectObject|Selects an object into the DC.
	{"SelectPalette",		ui_dc_select_palette,	1}, // @pymeth SelectObject|Selects the logical palette.
	{"SetBkColor",			ui_dc_set_bk_color,	1}, // @pymeth SetBkColor|Sets the background color.
	{"SetBkMode",			ui_dc_set_bk_mode,	1}, // @pymeth SetBkMode|Sets the background mode.
	{"SetBrushOrg",         ui_dc_set_brush_org, 1}, // @pymeth SetBrushOrg|Specifies the origin that GDI will assign to the next brush that the application selects into the device context.
	{"SetMapMode",			ui_dc_set_map_mode,	1}, // @pymeth SetMapMode|Sets the device mapping mode.
	{"SetROP2",				ui_dc_set_rop2,		1}, // @pymeth SetROP2|Sets the current drawing mode.
	{"SetTextAlign",		ui_dc_set_text_align,	1}, // @pymeth SetTextAlign|Sets the text alignment.
	{"SetTextColor",		ui_dc_set_text_color,	1}, // @pymeth SetTextColor|Sets the text foreground color.
	{"SetWindowExt",		ui_dc_set_window_ext,	1}, // @pymeth SetWindowExt|Sets the extents of the window.
	{"SetViewportExt",		ui_dc_set_viewport_ext,	1}, // @pymeth SetViewportExt|Sets the extents of the window's viewport.
	{"SetWindowOrg",		ui_dc_set_window_org,		1},
	{"SetWorldTransform",	ui_dc_set_world_transform,	1},
	{"SetViewportOrg",		ui_dc_set_viewport_org,		1},
	{"StartDoc",			ui_dc_start_doc,		1}, // @pymeth StartDoc|Starts spooling a document to a printer DC
	{"StartPage",			ui_dc_start_page,		1}, // @pymeth StartPage|Starts a new page on a printer DC
	{"TextOut",				ui_dc_text_out,	1}, // @pymeth TextOut|Writes text to the DC.
	{"BeginPath",			ui_dc_begin_path,			1},
	{"EndPath",				ui_dc_end_path,				1},
	{"StrokePath",			ui_dc_stroke_path,			1},
	{"FillPath",			ui_dc_fill_path,			1},
	{"StrokeAndFillPath",	ui_dc_stroke_and_fill_path,	1},
	{"GetWindowOrg",		ui_dc_get_window_org,		1},
	{"GetViewportOrg",		ui_dc_get_viewport_org,		1},
	{"GetWindowExt",		ui_dc_get_window_ext,		1},
	{"GetViewportExt",		ui_dc_get_viewport_ext,		1},
	{"IsPrinting",			ui_dc_is_printing,			1},
	{"GetViewportExt",		ui_dc_get_viewport_ext,		1},
	{"GetWindowOrg",		ui_dc_get_window_org,		1},
	{"GetViewportOrg",		ui_dc_get_viewport_org,		1},
	{"GetWindowExt",		ui_dc_get_window_ext,		1},
	{"Polygon",				ui_dc_polygon,				1},
	{"PolyBezier",			ui_dc_poly_bezier,			1},
	{"ScaleWindowExt",		ui_dc_scale_window_ext,		1},
	{"ScaleViewportExt",	ui_dc_scale_viewport_ext,	1},
	{NULL,			NULL}
};

ui_type ui_dc_object::type("PyCDC", 
						   &ui_assoc_object::type, 
						   sizeof(ui_dc_object), 
						   ui_dc_methods, 
						   GET_PY_CTOR(ui_dc_object));

