/*

	win32 module utilities

	Created January 1996, Mark Hammond (MHammond@skippinet.com.au)

Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc

*/
#include "stdafx.h"

#include "win32win.h"
#include "win32doc.h"
#include "win32control.h"
#include "win32menu.h"
#include "win32dlg.h"
#include "win32dc.h"
#include "win32gdi.h"
#include "win32bitmap.h"
#include "win32font.h"
#include "win32dll.h"
#include "win32splitter.h"
#include "win32toolbar.h"
#include "win32prop.h"
#include "win32template.h"
#include "win32ctrlList.h"
#include "win32ctrlTree.h"
#include "win32RichEdit.h"
#ifdef HIER_LIST
#include "win32hl.h"
#endif

// The CREATESTRUCT just has pointers (no buffers) for the name
// and classname.  Therefore, I dont treat them as strings, just 
// pointers (via long casts)
PyObject *PyObjectFromCreateStruct(LPCREATESTRUCT lpcs)
{
	return Py_BuildValue("(iiii(iiii)illi)",
	    lpcs->lpCreateParams,
   		lpcs->hInstance, 
   		lpcs->hMenu,
   		lpcs->hwndParent,
   		lpcs->cy,
   		lpcs->cx,
   		lpcs->y,
   		lpcs->x,
   		lpcs->style,
   		(long)lpcs->lpszName,
   		(long)lpcs->lpszClass,
   		lpcs->dwExStyle);
}

BOOL CreateStructFromPyObject(LPCREATESTRUCT lpcs, PyObject *ob, const char *fnName, BOOL bFromTuple)
{
	char argBuf[80];
	if (fnName==NULL) fnName = "CREATESTRUCT value";
	if (bFromTuple)
		sprintf(argBuf, "(iiii(iiii)illi):%s", fnName);
	else
		sprintf(argBuf, "iiii(iiii)illi:%s", fnName);
	long name, className;
	BOOL ret =  PyArg_ParseTuple(ob, argBuf,
	    &lpcs->lpCreateParams,
   		&lpcs->hInstance, 
   		&lpcs->hMenu,
   		&lpcs->hwndParent,
   		&lpcs->cy,
   		&lpcs->cx,
   		&lpcs->y,
   		&lpcs->x,
   		&lpcs->style,
   		&name,
   		&className,
   		&lpcs->dwExStyle);
	// CCreateStruct
	lpcs->lpszName = (LPCTSTR)name;
	lpcs->lpszClass = (LPCTSTR)className;
	return ret;
}
/////////////////////////////////////////////////////////////////////
//
// Font conversion utilities
//
//
static const char *szFontName = "name";
static const char *szFontWeight = "weight";
static const char *szFontWidth = "width";
static const char *szFontHeight = "height";
static const char *szFontItalic = "italic";
static const char *szFontUnderline = "underline";
static const char *szFontPitch = "pitch and family";
static const char *szFontCharSet = "charset";

PyObject *LogFontToDict(const LOGFONT &lf)
{
	PyObject *ret = PyDict_New();
	PyMapping_SetItemString( ret, (char *)szFontName, PyString_FromString((char *)lf.lfFaceName) );
	PyMapping_SetItemString( ret, (char *)szFontHeight, PyInt_FromLong(-lf.lfHeight));
	PyMapping_SetItemString( ret, (char *)szFontWidth, PyInt_FromLong(lf.lfWidth));
	PyMapping_SetItemString( ret, (char *)szFontWeight, PyInt_FromLong(lf.lfWeight));
	PyMapping_SetItemString( ret, (char *)szFontPitch, PyInt_FromLong(lf.lfPitchAndFamily));
	PyMapping_SetItemString( ret, (char *)szFontCharSet, PyInt_FromLong(lf.lfCharSet));
	PyMapping_SetItemString( ret, (char *)szFontUnderline, lf.lfUnderline?PyInt_FromLong(1):Py_None);
	PyMapping_SetItemString( ret, (char *)szFontItalic, lf.lfItalic?PyInt_FromLong(1):Py_None);
	return ret;
}

BOOL DictToLogFont(PyObject *font_props, LOGFONT *pLF)
{
  ZeroMemory (pLF, sizeof(LOGFONT));

  // font default values
  pLF->lfCharSet = DEFAULT_CHARSET; // dont use ANSI_CHARSET to support Japanese charset.
  pLF->lfQuality = PROOF_QUALITY;  // don't scale raster fonts
  PyObject *v;


  v = PyDict_GetItemString (font_props, (char *)szFontName);
  if (v != NULL)
	if (PyString_Check(v))
	  strncpy (pLF->lfFaceName, PyString_AsString(v), LF_FACESIZE - 1);
	else
	  RETURN_ERR ("Expected string value for font name property");

  v = PyDict_GetItemString (font_props, (char *)szFontHeight);
  if (v != NULL)
	if (PyInt_Check (v))
	  pLF->lfHeight = -PyInt_AsLong(v);
	else
	  RETURN_ERR ("Expected integer value for font height property");

  v = PyDict_GetItemString (font_props, (char *)szFontWidth);
  if (v != NULL)
	if (PyInt_Check (v))
	  pLF->lfWidth = PyInt_AsLong(v);
	else
	  RETURN_ERR ("Expected integer value for font width property");

  v = PyDict_GetItemString (font_props, (char *)szFontPitch);
  if (v != NULL)
	if (PyInt_Check (v))
	  pLF->lfPitchAndFamily = (BYTE)PyInt_AsLong(v);
	else
	  RETURN_ERR ("Expected integer value for font 'pitch and family' property");

  v = PyDict_GetItemString (font_props, (char *)szFontCharSet);
  if (v != NULL)
	if (PyInt_Check (v))
	  pLF->lfCharSet = (BYTE)PyInt_AsLong(v);
	else
	  RETURN_ERR ("Expected integer value for font 'charset' property");

  v = PyDict_GetItemString (font_props, (char *)szFontWeight);
  if (v != NULL)
	if (PyInt_Check (v))
	  pLF->lfWeight = PyInt_AsLong(v);
	else
	  RETURN_ERR ("Expected integer value for font weight property");

  v = PyDict_GetItemString (font_props, (char *)szFontItalic);
  if (v != NULL && v != Py_None)
	pLF->lfItalic = TRUE;

  v = PyDict_GetItemString (font_props, (char *)szFontUnderline);
  if (v != NULL && v != Py_None)
	pLF->lfUnderline = TRUE;
  return TRUE;
}

/////////////////////////////////////////////////////////////////////
//
//  ListView conversion utilities
//
//
// LV_ITEM 
PyObject *MakeLV_ITEMTuple(LV_ITEM *item)
{
	PyObject *ret = PyTuple_New(7);
	if (ret==NULL) return NULL;
	PyTuple_SET_ITEM(ret, 0, PyInt_FromLong(item->iItem));
	PyTuple_SET_ITEM(ret, 1, PyInt_FromLong(item->iSubItem));
	if (item->mask & LVIF_STATE) {
		PyTuple_SET_ITEM(ret, 2, PyInt_FromLong(item->state));
		PyTuple_SET_ITEM(ret, 3, PyInt_FromLong(item->stateMask));
	} else {
		Py_INCREF(Py_None);
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 2, Py_None);
		PyTuple_SET_ITEM(ret, 3, Py_None);
	}
	if ((item->mask & LVIF_TEXT) && (item->pszText != NULL)) {
		PyTuple_SET_ITEM(ret, 4, PyString_FromString(item->pszText));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 4, Py_None);
	}
	if (item->mask & LVIF_IMAGE) {
		PyTuple_SET_ITEM(ret, 5, PyInt_FromLong(item->iImage));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 5, Py_None);
	}
	if (item->mask & LVIF_PARAM && item->lParam) {
		PyObject *ob = PyInt_FromLong(item->lParam);
		PyTuple_SET_ITEM(ret, 6, ob);
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 6, Py_None);
	}
	return ret;
}

// @pymethod |PyCListCtrl|LV_ITEM tuple|Describes an LV_ITEM tuple, used by the <o PyCListCtrl> object.
// @pyparm int|item||The item number.
// @pyparm int|subItem||The sub-item number.
// @pyparm int|state||The items state.
// @pyparm int|stateMask||A mask indicating which of the state bits are valid..
// @pyparm string|text||The text for the item
// @pyparm int|iImage||The image offset for the item
// @pyparm int|userObject||Any integer to be associated with the item.
// @comm When passed to Python, will always be a tuple of size 7, and items may be None if not available.
// <nl>When passed from Python, the tuple must be at least 2 items long, and any item may be None.
// <nl>userob is any Python object at all, but no reference count is kept, so you must ensure the object remains referenced throught the lists life.
BOOL ParseLV_ITEMTuple( PyObject *args, LV_ITEM *pItem)
{
	PyObject *ob;
	pItem->mask = 0;
	int len = PyTuple_Size(args);
	if (len<2 || len > 7) {
		PyErr_SetString(PyExc_TypeError, "LV_ITEM tuple has invalid size");
		return FALSE;
	}
	PyErr_Clear(); // clear any errors, so I can detect my own.
	// 0 - iItem.
	if ((ob=PyTuple_GetItem(args, 0))==NULL)
		return FALSE;
	pItem->iItem = (UINT)PyInt_AsLong(ob);
	if (PyErr_Occurred()) return FALSE;
	// 1 - iSubItem
	if ((ob=PyTuple_GetItem(args, 1))==NULL)
		return FALSE;
	pItem->iSubItem = (UINT)PyInt_AsLong(ob);
	if (PyErr_Occurred()) return FALSE;
	// 1/2 - state/stateMask
	if (len<4) return TRUE;
	if ((ob=PyTuple_GetItem(args, 2))==NULL)
		return FALSE;
	if (ob != Py_None) {
		pItem->state = (UINT)PyInt_AsLong(ob);
		if (PyErr_Occurred() || (ob=PyTuple_GetItem(args, 3))==NULL)
			return FALSE;
		pItem->stateMask = (UINT)PyInt_AsLong(ob);
		if (PyErr_Occurred()) return FALSE;
		pItem->mask |= LVIF_STATE;
	}
	if (len<5) return TRUE;
	if ((ob=PyTuple_GetItem(args, 4))==NULL)
		return FALSE;
	if (ob != Py_None) {
		if (!PyString_Check(ob)) RETURN_TYPE_ERR("The text item must be a string or None");
		pItem->mask |= LVIF_TEXT;
		pItem->pszText = PyString_AsString(ob);
		pItem->cchTextMax = strlen(pItem->pszText)+1;
	}
	if (len<6) return TRUE;
	if ((ob=PyTuple_GetItem(args, 5))==NULL)
		return FALSE;
	if (ob != Py_None) {
		pItem->mask |= LVIF_IMAGE;
		pItem->iImage = PyInt_AsLong(ob);
		if (PyErr_Occurred())
			return FALSE;
	}
	if (len<7) return TRUE;
	if ((ob=PyTuple_GetItem(args, 6))==NULL)
		return FALSE;
	if (ob != Py_None) {
		pItem->mask |= LVIF_PARAM;
		pItem->lParam = PyInt_AsLong(ob);
	}
	return TRUE;
}

//
// LV_COLUMN
PyObject *MakeLV_COLUMNTuple(LV_COLUMN *item)
{
	PyObject *ret = PyTuple_New(4);
	if (ret==NULL) return NULL;
	if (item->mask & LVCF_FMT) {
		PyTuple_SET_ITEM(ret, 0, PyInt_FromLong(item->fmt));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 0, Py_None);
	}
	if (item->mask & LVCF_WIDTH) {
		PyTuple_SET_ITEM(ret, 1, PyInt_FromLong(item->cx));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 1, Py_None);
	}
	if ((item->mask & LVCF_TEXT) && (item->pszText != NULL)) {
		PyTuple_SET_ITEM(ret, 2, PyString_FromString(item->pszText));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 2, Py_None);
	}
	if (item->mask & LVCF_SUBITEM) {
		PyTuple_SET_ITEM(ret, 3, PyInt_FromLong(item->iSubItem));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 3, Py_None);
	}
	return ret;
}
// @pymethod |PyCListCtrl|LV_COLUMN tuple|Describes an LV_COLUMN tuple, used by the <o PyCListCtrl> object.
// A tuple of 4 items, being fmt, cx, pszText, iSubItem
// <nl>When passed to Python, will always be a tuple of size 4, and items may be None if not available.
// <nl>When passed from Python, the tuple may be any length up to 4, and any item may be None.
BOOL ParseLV_COLUMNTuple( PyObject *args, LV_COLUMN *pItem)
{
	PyObject *ob;
	pItem->mask = 0;
	int len = PyTuple_Size(args);
	if (len > 4) {
		PyErr_SetString(PyExc_TypeError, "LV_COLUMN tuple has invalid size");
		return FALSE;
	}
	PyErr_Clear(); // clear any errors, so I can detect my own.
	// 0 - fmt
	if (len<1) return TRUE;
	if ((ob=PyTuple_GetItem(args, 0))==NULL)
		return FALSE;
	if (ob != Py_None) {
		pItem->fmt = (int)PyInt_AsLong(ob);
		if (PyErr_Occurred()) return FALSE;
		pItem->mask |= LVCF_FMT;
	}
	// 1 - cx
	if (len<2) return TRUE;
	if ((ob=PyTuple_GetItem(args, 1))==NULL)
		return FALSE;
	if (ob != Py_None) {
		pItem->cx = (int)PyInt_AsLong(ob);
		if (PyErr_Occurred()) return FALSE;
		pItem->mask |= LVCF_WIDTH;
	}
	// 2 - text
	if (len<3) return TRUE;
	if ((ob=PyTuple_GetItem(args, 2))==NULL)
		return FALSE;
	if (ob != Py_None) {
		if (!PyString_Check(ob)) RETURN_TYPE_ERR("The text item must be a string or None");
		pItem->mask |= LVCF_TEXT;
		pItem->pszText = PyString_AsString(ob);
		pItem->cchTextMax = strlen(pItem->pszText)+1;
	}
	// 3 - subitem
	if (len<4) return TRUE;
	if ((ob=PyTuple_GetItem(args, 3))==NULL)
		return FALSE;
	if (ob != Py_None) {
		pItem->mask |= LVCF_SUBITEM;
		pItem->iSubItem = PyInt_AsLong(ob);
		if (PyErr_Occurred())
			return FALSE;
	}
	return TRUE;
}

/////////////////////////////////////////////////////////////////////
//
//  TreeView conversion utilities
//
//
// TV_ITEM 
PyObject *MakeTV_ITEMTuple(TV_ITEM *item)
{
	PyObject *ret = PyTuple_New(8);
	if (ret==NULL) return NULL;
	if (item->mask & TVIF_HANDLE)
		PyTuple_SET_ITEM(ret, 0, PyInt_FromLong((long)item->hItem));
	else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 0, Py_None);
	}
	if (item->mask & TVIF_STATE) {
		PyTuple_SET_ITEM(ret, 1, PyInt_FromLong(item->state));
		PyTuple_SET_ITEM(ret, 2, PyInt_FromLong(item->stateMask));
	} else {
		Py_INCREF(Py_None);
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 1, Py_None);
		PyTuple_SET_ITEM(ret, 2, Py_None);
	}
	if ((item->mask & TVIF_TEXT) && (item->pszText != NULL)) {
		PyTuple_SET_ITEM(ret, 3, PyString_FromString(item->pszText));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 3, Py_None);
	}
	if (item->mask & TVIF_IMAGE) {
		PyTuple_SET_ITEM(ret, 4, PyInt_FromLong(item->iImage));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 4, Py_None);
	}
	if (item->mask & TVIF_SELECTEDIMAGE) {
		PyTuple_SET_ITEM(ret, 5, PyInt_FromLong(item->iSelectedImage));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 5, Py_None);
	}

	if (item->mask & TVIF_CHILDREN) {
		PyTuple_SET_ITEM(ret, 6, PyInt_FromLong(item->cChildren));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 6, Py_None);
	}
	if (item->mask & TVIF_PARAM && item->lParam) {
		// assume lParam is an object
		PyTuple_SET_ITEM(ret, 7, PyInt_FromLong(item->lParam));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 7, Py_None);
	}
	return ret;
}

// @pymethod |PyCTreeCtrl|TV_ITEM tuple|Describes a TV_ITEM tuple, used by the <o PyCListCtrl> object.
// A tuple of 8 items:
// <nl>When passed to Python, will always be a tuple of size 8, and items may be None if not available.
// <nl>When passed from Python, the tuple may be any length up to 8, and any item may be None.
BOOL ParseTV_ITEMTuple( PyObject *args, TV_ITEM *pItem)
{
	PyObject *ob;
	PyObject *ob2;
	pItem->mask = 0;
	int len = PyTuple_Size(args);
	if (len > 8) {
		PyErr_SetString(PyExc_TypeError, "TV_ITEM tuple has invalid size");
		return FALSE;
	}
	PyErr_Clear(); // clear any errors, so I can detect my own.
	// 0 - hItem
	if (len<1) return TRUE;
	if ((ob=PyTuple_GetItem(args, 0))==NULL)
		return FALSE;
	if (ob != Py_None) {
		// @pyparm int|hItem||Item handle
		pItem->hItem = (HTREEITEM)PyInt_AsLong(ob);
		if (PyErr_Occurred()) return FALSE;
		pItem->mask |= TVIF_HANDLE;
	}
	// 1,2 - state/stateMask
	if (len<2) return TRUE;
	if (len<3) {
		PyErr_SetString(PyExc_TypeError, "TV_ITEM - state and stateMask must be provided");
		return FALSE;
	}
	if ((ob=PyTuple_GetItem(args, 1))==NULL)
		return FALSE;
	if ((ob2=PyTuple_GetItem(args, 2))==NULL)
		return FALSE;
	if (ob==Py_None && ob2==Py_None)
		;
	else if (ob==Py_None) {
		PyErr_SetString(PyExc_TypeError, "TV_ITEM - state and stateMask must both be None, or both not Noned");
		return FALSE;
	} else {
		// @pyparm int|state||Item state
		// @pyparm int|stateMask||Item state mask
		pItem->state = (int)PyInt_AsLong(ob);
		pItem->stateMask = (int)PyInt_AsLong(ob2);
		if (PyErr_Occurred()) return FALSE;
		pItem->mask |= TVIF_STATE;
	}
	// 3 - text
	if (len<4) return TRUE;
	if ((ob=PyTuple_GetItem(args, 3))==NULL)
		return FALSE;
	if (ob != Py_None) {
		// @pyparm string|text||Item text
		if (!PyString_Check(ob)) RETURN_TYPE_ERR("The text item must be a string or None");
		pItem->mask |= TVIF_TEXT;
		pItem->pszText = PyString_AsString(ob);
		pItem->cchTextMax = strlen(pItem->pszText)+1;
	}
	// 4 - image
	if (len<5) return TRUE;
	if ((ob=PyTuple_GetItem(args, 4))==NULL)
		return FALSE;
	if (ob != Py_None) {
		pItem->mask |= TVIF_IMAGE;
		// @pyparm int|iImage||Offset of items image.
		pItem->iImage = (int)PyInt_AsLong(ob);
	}
	// 5 - imageSelected
	if (len<6) return TRUE;
	if ((ob=PyTuple_GetItem(args, 5))==NULL)
		return FALSE;
	if (ob != Py_None) {
		// @pyparm int|iSelectedImage||Offset of items selected image.
		pItem->mask |= TVIF_SELECTEDIMAGE;
		pItem->iSelectedImage = (int)PyInt_AsLong(ob);
	}
	// 6 - cChildren
	if (len<7) return TRUE;
	if ((ob=PyTuple_GetItem(args, 6))==NULL)
		return FALSE;
	if (ob != Py_None) {
		// @pyparm int|cChildren||Number of child items.
		pItem->mask |= TVIF_CHILDREN;
		pItem->cChildren = (int)PyInt_AsLong(ob);
	}
	// 7 - object
	if (len<8) return TRUE;
	if ((ob=PyTuple_GetItem(args, 7))==NULL)
		return FALSE;
	if (ob != Py_None) {
		// @pyparm int|lParam||User defined integer param.
		pItem->mask |= LVIF_PARAM;
		pItem->lParam = PyInt_AsLong(ob);
	}
	return !PyErr_Occurred();
}

/////////////////////////////////////////////////////////////////////
//
//  Header Control conversion utilities
//
//
// HD_ITEM 
//HDI_BITMAP, HDI_FORMAT, HDI_HEIGHT, HDI_LPARAM, HDI_TEXT, HDI_WIDTH
//fmt is HDF_CENTER, HDF_LEFT, HDF_RIGHT, HDF_BITMAP, HDF_OWNERDRAW, HDF_STRING
PyObject *MakeHD_ITEMTuple(HD_ITEM *item)
{
	PyObject *ret = PyTuple_New(5);
	if (ret==NULL) return NULL;
	if (item->mask & HDI_HEIGHT)
		PyTuple_SET_ITEM(ret, 0, PyInt_FromLong((long)0));
	else if (item->mask & HDI_WIDTH)
		PyTuple_SET_ITEM(ret, 0, PyInt_FromLong((long)1));
	if ((item->mask & HDI_HEIGHT) || (item->mask & HDI_WIDTH)) 
		PyTuple_SET_ITEM(ret, 1, PyInt_FromLong((long)item->cxy));
	else {
		Py_INCREF(Py_None);
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 0, Py_None);
		PyTuple_SET_ITEM(ret, 1, Py_None);
	}
	if ((item->mask & HDI_TEXT) && (item->pszText != NULL) ) {
		PyTuple_SET_ITEM(ret, 2, PyString_FromString(item->pszText));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 2, Py_None);
	}
	if (item->mask & HDI_BITMAP) {
		// Should this support a bitmap object?
		PyTuple_SET_ITEM(ret, 3, PyInt_FromLong((long)item->hbm));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 3, Py_None);
	}
	if (item->mask & HDI_FORMAT) {
		PyTuple_SET_ITEM(ret, 4, PyInt_FromLong(item->fmt));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 4, Py_None);
	}
	if (item->mask & TVIF_PARAM && item->lParam) {
		// assume lParam is an object
		PyTuple_SET_ITEM(ret, 5, PyInt_FromLong(item->lParam));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 5, Py_None);
	}
	return ret;
}

// *** When PyCHeaderCtrl is implemented, return the '@' to the next line _and_ the parm!
// pymethod |PyCHeaderCtrl|HD_ITEM tuple|Describes a HD_ITEM tuple, used by the <o PyCHeaderCtrl> object.
// A tuple of 6 items:
// <nl>When passed to Python, will always be a tuple of size 6, and items may be None if not available.
// <nl>When passed from Python, the tuple may be any length up to 6, and any item may be None.
BOOL ParseHD_ITEMTuple( PyObject *args, HD_ITEM *pItem)
{
	PyObject *ob;
	pItem->mask = 0;
	int len = PyTuple_Size(args);
	if (len > 6) {
		PyErr_SetString(PyExc_TypeError, "HD_ITEM tuple has invalid size");
		return FALSE;
	}
	PyErr_Clear(); // clear any errors, so I can detect my own.
	// 0 - mask
	if (len<1) return TRUE;
	if ((ob=PyTuple_GetItem(args, 0))==NULL)
		return FALSE;
	if (ob != Py_None) {
		// pyparm int|<none>||Specifies if cxy is width (0) or height (1)
		if (ob)
			pItem->mask |= HDI_HEIGHT;
		else
			pItem->mask |= HDI_WIDTH;
	}
	// 1 - is cxy width or height of item
	if (len<2) return TRUE;
	if ((ob=PyTuple_GetItem(args, 1))==NULL)
		return FALSE;
	if (ob != Py_None) {
		// @pyparm int|cxy||Width or height of item
		pItem->cxy = (int)PyInt_AsLong(ob);
		if (PyErr_Occurred()) return FALSE;
		//mask updated above
	}
	// 2 - cxy (measurement of width or height depending on previous arg)
	
	// 3 - pszText address of item string
	if (len<3) return TRUE;
	if ((ob=PyTuple_GetItem(args, 2))==NULL)
		return FALSE;
	if (ob !=Py_None) {
		// @pyparm string|pszText||Item text
		if (!PyString_Check(ob)) RETURN_TYPE_ERR("The text item must be a string or None");
		pItem->pszText = PyString_AsString(ob);
		if (PyErr_Occurred()) return FALSE;
		pItem->cchTextMax = strlen(pItem->pszText)+1;
		pItem->mask |= HDI_TEXT;
	}
	// 3 - hbm handle of item bitmap
	if (len<4) return TRUE;
	if ((ob=PyTuple_GetItem(args, 3))==NULL)
		return FALSE;
	if (ob != Py_None) {
		// @pyparm string|text||Item text
		pItem->mask |= HDI_BITMAP;
		pItem->hbm = (HBITMAP)PyInt_AsLong(ob);
		if (PyErr_Occurred()) return FALSE;
	}
	// 4 - fmt of item string
	if (len<5) return TRUE;
	if ((ob=PyTuple_GetItem(args, 4))==NULL)
		return FALSE;
	if (ob != Py_None) {
		pItem->mask |= HDI_FORMAT;
		// @pyparm int|fmt||code for centering etc of string
		pItem->fmt = (int)PyInt_AsLong(ob);
	}
	// 5 - object
	if (len<6) return TRUE;
	if ((ob=PyTuple_GetItem(args, 5))==NULL)
		return FALSE;
	if (ob != Py_None) {
		// @pyparm int|lParam||User defined integer param.
		pItem->mask |= LVIF_PARAM;
		pItem->lParam = PyInt_AsLong(ob);
	}
	return !PyErr_Occurred();
}


/////////////////////////////////////////////////////////////////////
//
// CHARFORMAT and PARAFORMAT conversion utilities
//
//
// @pymethod |win32ui|CHARFORMAT tuple|Describes a CHARFORMAT tuple
BOOL ParseCharFormatTuple( PyObject *args, CHARFORMAT *pFmt)
{
	char *fnt = NULL;
	BOOL rc = PyArg_ParseTuple(args, "|iillibbs:CHARFORMAT tuple", 
		       &pFmt->dwMask, // @pyparm int|mask|0|The mask to use.  Bits in this mask indicate which of the following paramaters are interpreted.  Must be a combination the win32con.CFM_* constants.
			   &pFmt->dwEffects, // @pyparm int|effects|None|The effects to use.  Must be a combination the win32con.CFE_* constants.
			   &pFmt->yHeight, // @pyparm int|yHeight|None|The y height.
			   &pFmt->yOffset, // @pyparm int|yOffset|None|Character offset from the baseline. If this member is positive, the character is a superscript; if it is negative, the character is a subscript.
			   &pFmt->crTextColor,// @pyparm int|colorText|None|The color to use.
			   &pFmt->bCharSet,// @pyparm int|bCharSet|None|The charset.  See the LOGFONT structure for details.
			   &pFmt->bPitchAndFamily,// @pyparm int|bPitchAndFamily|None|The charset.  See the LOGFONT structure for details.
			   &fnt)!=NULL; // @pyparm string|faceName|None|The font name.
	if (rc && fnt) strncpy(pFmt->szFaceName, fnt, sizeof(pFmt->szFaceName));
	// @comm  Executing d=win32ui.CreateFontDialog(); d.DoModal(); print d.GetCharFormat()
	// will print a valid CHARFORMAT tuple.
	return rc;
}

PyObject *MakeCharFormatTuple(CHARFORMAT *pFmt)
{
	return Py_BuildValue("iillibbs",
		       pFmt->dwMask,
			   pFmt->dwEffects,
			   pFmt->yHeight,
			   pFmt->yOffset,
			   pFmt->crTextColor,
			   pFmt->bCharSet,
			   pFmt->bPitchAndFamily,
			   pFmt->szFaceName);
}

// @pymethod|win32ui|PARAFORMAT tuple|Describes a PARAFORMAT tuple
BOOL ParseParaFormatTuple( PyObject *args, PARAFORMAT *pFmt)
{
	PyObject *obTabStops = Py_None;
	pFmt->cTabCount = 0;
	BOOL rc = PyArg_ParseTuple(args, "|iiiiiiiO:PARAFORMAT tuple", 
		       &pFmt->dwMask, // @pyparm int|mask|0|The mask to use.  Bits in this mask indicate which of the following paramaters are interpreted.  Must be a combination the win32con.PFM_* constants.
			   &pFmt->wNumbering, // @pyparm int|numbering|None|The numbering style to use.
			   &pFmt->wReserved, // @pyparm int|yHeight|None|Reserved
			   &pFmt->dxStartIndent, // @pyparm int|dxStartIndext|None|Indentation of the first line.
			   &pFmt->dxRightIndent,// @pyparm int|dxRightIndent|None|Indentation from the right.
			   &pFmt->dxOffset,// @pyparm int|dxOffset|None|The indent of second and subsequent lines.
			   &pFmt->wAlignment,// @pyparm int|wAlignment|None|The alignment of the paragraph.
			   &obTabStops); // @pyparm sequence of ints|tabStops|None|The tabstops to use.
	if (rc && obTabStops != Py_None) {
		if (!PySequence_Check(obTabStops))
			RETURN_ERR("tabStops object must be None or a sequence");
		int tabCount = PyObject_Length(obTabStops);
		tabCount = min(MAX_TAB_STOPS, tabCount);
		for (int i=0;rc && i<tabCount;i++) {
			pFmt->rgxTabs[i] = PyInt_AsLong( PySequence_GetItem(obTabStops, i) );
			rc = PyErr_Occurred()==FALSE;
			if (!rc) break;
			pFmt->cTabCount++;
		}
	}
	return rc;
}

PyObject *MakeParaFormatTuple(PARAFORMAT *pFmt)
{
	PyObject *obTabs;
	if (pFmt->cTabCount==0) {
		Py_INCREF(Py_None);
		obTabs = Py_None;
	} else {
		obTabs = PyTuple_New(pFmt->cTabCount);
		for (int i=0;i<pFmt->cTabCount;i++)
			PyTuple_SetItem( obTabs, i, PyInt_FromLong(pFmt->rgxTabs[i]));
	}
	PyObject *ret = Py_BuildValue("iiiiiiiO", 
		       pFmt->dwMask,
			   pFmt->wNumbering,
			   pFmt->wReserved,
			   pFmt->dxStartIndent,
			   pFmt->dxRightIndent,
			   pFmt->dxOffset,
			   pFmt->wAlignment,
			   obTabs);
	Py_DECREF(obTabs); // ref added by BuildValue
//	Py_DECREF(obTabs); // reference I created.
	return ret;
}

/////////////////////////////////////////////////////////////////////
//
// Other utilities
//
//
PyObject *PyWin_GetPythonObjectFromLong(long val)
{
	PyObject *ret = (PyObject *)val;
	if (ret==NULL)
		RETURN_NONE;
	BOOL ok;
	__try {
		ok = ret->ob_refcnt != 0;
		ok = ok && ret->ob_type->tp_name[0] != 0;
	}
	__except (EXCEPTION_ACCESS_VIOLATION) {
		ok = FALSE;
	}
	if (!ok)
		RETURN_ERR("The object is invalid");
	return ret;
}

CString GetAPIErrorString(char *fnName)
{
	CString csBuf = fnName + CString(" failed - ");
	DWORD errorCode = GetLastError();
	if (errorCode) {
		CString message = GetAPIErrorString(errorCode);
		if (message.GetLength()>0)
			csBuf += message;
		else {
			char buf[80];
			sprintf(buf,"error code was %d - no error message is available",errorCode);
			csBuf += buf;
		}
	}
	else
		csBuf += "no error code is available";
	return csBuf;
}

CString GetAPIErrorString(DWORD errCode)
{
	CString csBuf;
	const int bufSize = 512;
	char *buf = csBuf.GetBuffer( bufSize );
	::FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, NULL, errCode, 0, buf, bufSize, NULL );
	csBuf.ReleaseBuffer (-1);
	return csBuf;
}

ui_type_CObject &UITypeFromCObject( CObject *ob )
{
	ui_type_CObject *ret;
	// need some sort of table lookup here!

	// Note that you must check derived classes before parents, as you
	// require the most specific type possible.
	if (ob->IsKindOf(RUNTIME_CLASS(CMDIChildWnd)))
		ret = &PyCMDIChildWnd::type;
	else if (ob->IsKindOf(RUNTIME_CLASS(CPythonEditView)))
		ret = &PyCEditView::type;
	else if (ob->IsKindOf(RUNTIME_CLASS(CPythonView)))
		ret = &PyCView::type;
	else if (ob->IsKindOf(RUNTIME_CLASS(CStatic)))
		ret = &ui_control_object::type;
	else if (ob->IsKindOf(RUNTIME_CLASS(CRichEditView)))
		ret = &PyCRichEditView::type;
	else if (ob->IsKindOf(RUNTIME_CLASS(CListView)))
		ret = &PyCListView::type;
	else if (ob->IsKindOf(RUNTIME_CLASS(CTreeView)))
		ret = &PyCTreeView::type;
	else if (ob->IsKindOf(RUNTIME_CLASS(CRichEditCtrl)))
		ret = &PyCRichEditCtrl::type;
	else if (ob->IsKindOf(RUNTIME_CLASS(CDialog)))
		ret = &PyCDialog::type;
	else if (ob->IsKindOf(RUNTIME_CLASS(CMDIFrameWnd)))
		ret = &PyCMDIFrameWnd::type;
	else if (ob->IsKindOf(RUNTIME_CLASS(CWnd))) {
		ret = &UITypeFromHWnd(((CWnd *)ob)->GetSafeHwnd());
	}
	else {
		TRACE("Warning - unknown class type in UITypeFromCObject");
		ret = &ui_assoc_CObject::type;	// will have no methods - really an error
	}
	return *ret;
}

ui_type_CObject &UITypeFromHWnd( HWND hwnd )
{
	ui_type_CObject *ret;
	// generic window - see if class name can help us.
	char szClassName[64];
	::GetClassName( hwnd, szClassName, sizeof(szClassName));
	// getting really lazy here.
	if (strcmp(szClassName, "ListBox")==0)
		ret = &PyCListBox::type;
	else if (strcmp(szClassName, "ComboBox")==0)
		ret = &PyCComboBox::type;
	else if (strcmp(szClassName, "Button")==0)
		ret = &PyCButton::type;
	else if (strcmp(szClassName, "Edit")==0)
		ret = &PyCEdit::type;
	else if (strcmp(szClassName, "RICHEDIT")==0)
		ret = &PyCRichEditCtrl::type;
	else if (strcmp(szClassName, "SysListView32")==0)
		ret = &PyCListCtrl::type;
	else if (strcmp(szClassName, "SysTreeView32")==0)
		ret = &PyCTreeCtrl::type;
	// now handle some special cases to avoid warning below!
	else if (strcmp(szClassName, "MDIClient")==0)
		ret = &PyCWnd::type;
	else {
		TRACE("Generic window returned for class name '%s'\n", szClassName);
		ret = &PyCWnd::type;
	}
	return *ret;
}

// utility to get a nice printable string from any object.
// reference neutral.
CString GetReprText( PyObject *objectUse )
{
	// special case for integers first.
	if (PyInt_Check(objectUse))
		return (int)PyInt_AsLong(objectUse);
	PyObject *s;
	if (PyString_Check(objectUse))  // if it is a string, then no need to repr it
		s = objectUse;              // and repr on a string may mangle it (eg
	else {                          // turn '\t' into a literal "\t"
		s = PyObject_Str(objectUse);
		if (s==NULL) {
			PyErr_Clear();
			s = PyObject_Repr(objectUse);
		}
		if (s==NULL) {
			PyErr_Clear();
			return CString("No str() or repr()?!");
		}
	}
	const char *szRepr = PyString_AsString(s);
	int len=strlen(szRepr);
	if (len && strchr("\"'[(", *szRepr)) {
		if (szRepr[len-1]==*szRepr) {
			++szRepr;
			len-=2;	// drop first and last chars.
		}
	}
	CString csRes = CString( szRepr, len );
	return csRes;
}

