/*

	propsheet data type

	Created July 1994, Mark Hammond (MHammond@skippinet.com.au)

	propsheet is derived from window.

Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc

*/
#include "stdafx.h"

#include "win32win.h"
#include "win32dlg.h"
#include "win32prop.h"
#include <afxcmn.h>

extern HGLOBAL MakeResourceFromDlgList(PyObject *tmpl);

CTabCtrl *PyGetTabCtrl(PyObject *self)
{
	return (CTabCtrl *)PyCWnd::GetPythonGenericWnd(self, &ui_tabctrl_object::type);
}
CTabCtrl *PyGetTabCtrlWithWnd(PyObject *self)
{
	CTabCtrl *pTab = PyGetTabCtrl(self);
	if (pTab->m_hWnd==NULL || !::IsWindow(pTab->m_hWnd))
		RETURN_ERR("The tab control has no window");
	return pTab;
}

//
// property sheet helpers
//

CPropertySheet *GetPropSheet(PyObject *self)
{
	return (CPropertySheet *)PyCWnd::GetPythonGenericWnd(self, &PyCPropertySheet::type);
}
CPythonPropertySheet *GetPythonPropSheet(PyObject *self)
{
	CPythonPropertySheet *ret = 
		(CPythonPropertySheet *)PyCWnd::GetPythonGenericWnd(self, &PyCPropertySheet::type);
	if (!ret->IsKindOf(RUNTIME_CLASS(CPythonPropertySheet)))
		RETURN_TYPE_ERR("Object is not of the correct type");
	return ret;
}
//
// property page helpers
//
CPythonPropertyPage *GetPropPage(PyObject *self)
{
	return (CPythonPropertyPage *)PyCWnd::GetPythonGenericWnd(self, &PyCPropertyPage::type);
}

BOOL PropSheetCheckForPageCreate( UINT id )
{
//	if (!CPythonPropertyPage::CheckTemplate(id))
//		RETURN_ERR("The property page can not be located");
	return TRUE;
}
BOOL PropSheetCheckForPageCreate( LPCTSTR id )
{
//	if (!CPythonPropertyPage::CheckTemplate(id))
//		RETURN_ERR("The property page can not be located");
	return TRUE;
}

BOOL PropSheetCheckForDisplay( CPropertySheet *pSheet )
{
	int max = pSheet->GetPageCount();
	if (max==0)
		RETURN_ERR("The property sheet has no pages");
	for (int i=0; i<max; i++) {
		CPythonPropertyPage *pPage = (CPythonPropertyPage *)pSheet->GetPage(i);
		// use help ID here
//		if (!pPage->CheckTemplate()) {
//			char buf[80];
//			sprintf(buf, "Property Page %d can not be located", i );
//			RETURN_ERR(buf);
//		}
	}
	return TRUE;
}

/////////////////////////////////////////////////////////////////////
//
// PropSheet object
//
//////////////////////////////////////////////////////////////////////
PyCPropertySheet::PyCPropertySheet()
{
//	bManualDelete = TRUE;
}
PyCPropertySheet::~PyCPropertySheet()
{
}
/********
	TRACE("PropSheet object destructing\n");
	CPythonPropertySheet *pSheet = GetPythonPropSheet(this);
	ASSERT(pSheet);
	PyCWnd::DoKillAssoc(TRUE);
	ui_assoc_object::SetAssocInvalid();	// must call this explicitely, as I ignore SetAssocInvalid
}

void PyCPropertySheet::DoKillAssoc(BOOL bDestructing )
{
	return;
}
*************/
// @pymethod <o PyCPropertySheet>|win32ui|CreatePropertySheet|Creates a property sheet object.
PyObject *PyCPropertySheet::create( PyObject *self, PyObject *args )
{
	PyObject 	*obParent = NULL,
			*obCaption;
	CWnd *pParent = NULL;
	int iSelect = 0;
	if (!PyArg_ParseTuple(args,"O|Oi", 
	          &obCaption, // @pyparm string|caption||The caption for the property sheet.
	          &obParent,  // @pyparm <o PyCWnd>|parent|None|The parent window of the property sheet.
	          &iSelect))  // @pyparm int|select|0|The index of the first page to be selected.
		return NULL;
	if (obParent) {
		if (!ui_base_class::is_uiobject(obParent, &PyCWnd::type))
			RETURN_TYPE_ERR("parameter 2 must be a PyCWnd object");
  		pParent = (CWnd *)PyCWnd::GetPythonGenericWnd(obParent);
	}
	CPythonPropertySheet *pPS;
	if (PyInt_Check(obCaption)) {
		int id = (int)PyInt_AsLong(obCaption);
		pPS = new CPythonPropertySheet(id, pParent, iSelect);
	}
	else if (PyString_Check(obCaption)) {
		char *value = PyString_AsString(obCaption);
		pPS = new CPythonPropertySheet(value, pParent, iSelect);
	}
	else
		RETURN_TYPE_ERR("parameter 1 must be an integer or string object");
	PyCPropertySheet *ret = (PyCPropertySheet *)ui_assoc_object::make( PyCPropertySheet::type, pPS);
	return ret;
}

///////////////////////////////////////
//
// PropSheet Methods
//
// @pymethod |PyCPropertySheet|AddPage|Adds the supplied page with the rightmost tab in the property sheet. 
PyObject *ui_propsheet_add_page( PyObject *self, PyObject *args )
{
	PyObject *obPage;
	CPythonPropertyPage *pPage;
	if (!PyArg_ParseTuple(args,"O", &obPage))
		// @pyparm <o PyCPropertyPage>|page||The page to be added.
		return NULL;
	if (!ui_base_class::is_uiobject(obPage, &PyCPropertyPage::type)) {
		RETURN_TYPE_ERR("passed object must be a PyCPropertyPage object");
	}
  	pPage = GetPropPage(obPage);
	if (!pPage)
		return NULL;

	CPythonPropertySheet *pPS;
	if (!(pPS=GetPythonPropSheet(self)))
		return NULL;
	pPS->AddPage(pPage); // @pyseemfc PyCPropertySheet|AddPage
	// @comm Add pages to the property sheet in the left-to-right order you want them to appear.
	RETURN_NONE;
}
// @pymethod int|PyCPropertySheet|GetActiveIndex|Retrieves the index of the active page of the property sheet.
PyObject *ui_propsheet_get_active_index( PyObject *self, PyObject *args )
{
	CHECK_NO_ARGS(args);
	CPythonPropertySheet *pPS;
	if (!(pPS=GetPythonPropSheet(self)))
		return NULL;
	return Py_BuildValue("i", pPS->GetActiveIndex()); 
}

// @pymethod int|PyCPropertySheet|GetPageIndex|Retrieves the index of the specified page of the property sheet.
PyObject *ui_propsheet_get_page_index( PyObject *self, PyObject *args )
{
	PyObject *obPage;
	// @pyparm <o PyCPropertyPage>|page||The page.
	if (!PyArg_ParseTuple(args, "O:GetPageIndex", &obPage))
		return NULL;
	CPythonPropertySheet *pPS = GetPythonPropSheet(self);
	if (!pPS) return NULL;
	CPythonPropertyPage *pPage = GetPropPage(obPage);
	if (!pPage) return NULL;
	return Py_BuildValue("i", pPS->GetPageIndex(pPage));
}

// @pymethod <o PyCPropertyPage>|PyCPropertySheet|EnableStackedTabs|Enables or disables stacked tabs.
PyObject *ui_propsheet_enable_stacked_tabs( PyObject *self, PyObject *args )
{
	BOOL stacked;
	if (!PyArg_ParseTuple(args,"i", &stacked))
		// @pyparm int|stacked||A boolean flag
		return NULL;
	CPythonPropertySheet *pPS;
	if (!(pPS=GetPythonPropSheet(self)))
		return NULL;
  	pPS->EnableStackedTabs(stacked);
	RETURN_NONE;
}


// @pymethod <o PyCPropertyPage>|PyCPropertySheet|GetPage|Returns the specified property page.
PyObject *ui_propsheet_get_page( PyObject *self, PyObject *args )
{
	int pagenum;
	CPropertyPage *pPage;
	if (!PyArg_ParseTuple(args,"i", &pagenum))
		// @pyparm int|pageNo||The index of the page toretrieve.
		return NULL;
	CPythonPropertySheet *pPS;
	if (!(pPS=GetPythonPropSheet(self)))
		return NULL;
  	pPage = pPS->GetPage(pagenum);
	if (!pPage)
		RETURN_ERR("The page does not exist");
	// @pyseemfc PyCPropertySheet|GetPage
	PyCPropertyPage *ret = (PyCPropertyPage *)ui_assoc_object::make( PyCPropertyPage::type, pPage)->GetGoodRet();
	return ret;
}

// @pymethod <o PyCPropertyPage>|PyCPropertySheet|GetActivePage|Returns the currently active property page.
PyObject *ui_propsheet_get_active_page( PyObject *self, PyObject *args )
{
	CPropertyPage *pPage;
	if (!PyArg_ParseTuple(args,":GetActivePage"))
		return NULL;
	CPythonPropertySheet *pPS;
	if (!(pPS=GetPythonPropSheet(self)))
		return NULL;
  	pPage = pPS->GetActivePage();
	if (!pPage)
		RETURN_ERR("The page does not exist");
	// @pyseemfc PyCPropertySheet|GetActivePage
	PyCPropertyPage *ret = (PyCPropertyPage *)ui_assoc_object::make( PyCPropertyPage::type, pPage)->GetGoodRet();
	return ret;
}

// @pymethod |PyCPropertySheet|SetActivePage|Programmatically sets the active page object.
PyObject *ui_propsheet_set_active_page( PyObject *self, PyObject *args )
{
	PyObject *obPage;
	// @pyparm <o PyCPropertyPage>|page||The page.
	if (!PyArg_ParseTuple(args, "O:SetActivePage", &obPage))
		return NULL;
	CPythonPropertySheet *pPS = GetPythonPropSheet(self);
	if (!pPS) return NULL;
	CPythonPropertyPage *pPage = GetPropPage(obPage);
	if (!pPage) return NULL;
	if (!pPS->SetActivePage(pPage))
		RETURN_ERR("SetActivePage failed");
	RETURN_NONE;
}

// @pymethod int|PyCPropertySheet|GetPageCount|Returns the number of pages.
PyObject *ui_propsheet_get_page_count( PyObject *self, PyObject *args )
{
	CHECK_NO_ARGS(args);
	CPythonPropertySheet *pPS;
	if (!(pPS=GetPythonPropSheet(self)))
		return NULL;
  	return Py_BuildValue("i", pPS->GetPageCount());
}

// @pymethod <o PyCTabCtrl>|PyCPropertySheet|GetTabCtrl|Returns the tab control used by the sheet.
PyObject *ui_propsheet_get_tab_ctrl( PyObject *self, PyObject *args )
{
	CHECK_NO_ARGS(args);
	CPythonPropertySheet *pPS;
	if (!(pPS=GetPythonPropSheet(self)))
		return NULL;
	CTabCtrl *pTab = pPS->GetTabControl();
	if (pTab==NULL)
		RETURN_ERR("The property page does not have a tab control");
	return ui_assoc_object::make( ui_tabctrl_object::type, pTab)->GetGoodRet();
}

// @pymethod int|PyCPropertySheet|DoModal|Displays the property sheet as a modal dialog.
PyObject *ui_propsheet_do_modal( PyObject *self, PyObject *args )
{
	CHECK_NO_ARGS(args);
	CPropertySheet *pPS;
	if (!(pPS=GetPropSheet(self)))
		return NULL;
	if (!PropSheetCheckForDisplay(pPS))
		return NULL;
	Py_INCREF(self);	// make sure Python doesnt kill the object while in a modal call.
					// really only for the common dialog(!?), and other non CPythonPropSheet's
	int ret;
	GUI_BGN_SAVE;
	ret = pPS->DoModal();
	GUI_END_SAVE;
	DODECREF(self);
	return Py_BuildValue("i", ret);
}

// @pymethod int|PyCPropertySheet|CreateWindow|Displays the property sheet as a modeless dialog.
PyObject *ui_propsheet_create_window( PyObject *self, PyObject *args )
{
	PyObject 	*obParent = NULL;
	int dwStyle = WS_SYSMENU | WS_POPUP | WS_CAPTION | DS_MODALFRAME | WS_VISIBLE;
	int dwExStyle = WS_EX_DLGMODALFRAME;
	CWnd *pParent = NULL;
	// @pyparm <o PyCWnd>|parent|None|The parent of the dialog.
	// @pyparm int|style|WS_SYSMENU\|WS_POPUP\|WS_CAPTION\|DS_MODALFRAME\|WS_VISIBLE|The style for the window.
	// @pyparm int|exStyle|WS_EX_DLGMODALFRAME|The extended style for the window.
	if (!PyArg_ParseTuple(args,"|Oll", &obParent, &dwStyle, &dwExStyle))
		return NULL;
	if (obParent && obParent!=Py_None) {
		if (!ui_base_class::is_uiobject(obParent, &PyCWnd::type))
			RETURN_TYPE_ERR("parameter 1 must be a PyCWnd object");
  		pParent = (CWnd *)PyCWnd::GetPythonGenericWnd(obParent);
		if (!pParent)
			return NULL;
	}
	CPythonPropertySheet *pPS;
	if (!(pPS=GetPythonPropSheet(self)))
		return NULL;
	if (!PropSheetCheckForDisplay(pPS))
		return NULL;
	int rc;
	const char *failMsg = "Create() failed";
	GUI_BGN_SAVE;
	try {
		rc=pPS->Create(pParent, dwStyle, dwExStyle );
	}
	catch (...) {
		rc = NULL;
		failMsg = "Create() caused an exception - it is likely that the specified template can not be located";
	}
	GUI_END_SAVE;
	if (!rc)
		RETURN_ERR((char *)failMsg);
	RETURN_NONE;
}

// @pymethod |PyCPropertySheet|EndDialog|Closes the dialog, with the specified result.
PyObject *ui_propsheet_end_dialog( PyObject *self, PyObject *args )
{
	CPropertySheet *pPS = pPS=GetPropSheet(self);
	if (!pPS) return NULL;
	int result;
	// @pyparm int|result||The result to be returned by DoModal.
	if (!PyArg_ParseTuple( args, "i", &result))
		return NULL;
	pPS->EndDialog(result);
	RETURN_NONE;
}
// @pymethod |PyCPropertySheet|RemovePage|Removes the specified page from the sheet.
PyObject *ui_propsheet_remove_page( PyObject *self, PyObject *args )
{
	CPropertySheet *pPS;
	if (!(pPS=GetPropSheet(self)))
		return NULL;
	PyObject *ob;
	// @pyparm int|offset||The page number to remove
	// @pyparmalt1 <o PyCPropertyPage>|page||The page to remove
	if (!PyArg_ParseTuple( args, "O", &ob))
		return NULL;
	if (PyInt_Check(ob)) {
		int id = (int)PyInt_AsLong(ob);
		pPS->RemovePage(id);
	} 
	else if (ui_base_class::is_uiobject(ob, &PyCPropertyPage::type)) {
  		CPythonPropertyPage * pPage = GetPropPage(ob);
		if (!pPage)
			return NULL;
		pPS->RemovePage(pPage);
	} else
		RETURN_TYPE_ERR("passed object must be an integer or PyCPropertyPage object");
	RETURN_NONE;
}

// @pymethod int|PyCPropertySheet|SetTitle|Sets the caption for the property sheet.
PyObject *ui_propsheet_set_title( PyObject *self, PyObject *args )
{
	char *caption;
	// @pyparm string|title||The new caption
	if (!PyArg_ParseTuple(args, "s:SetTitle", &caption))
		return NULL;
	CPythonPropertySheet *pPS = GetPythonPropSheet(self);
	if (!pPS) return NULL;
	pPS->SetTitle(caption);
	RETURN_NONE;
}

// @pymethod |PyCPropertySheet|PressButton|Simulates the choice of the specified button in a property sheet.
PyObject *ui_propsheet_press_button( PyObject *self, PyObject *args )
{
	CPropertySheet *pPS = pPS=GetPropSheet(self);
	if (!pPS) return NULL;
	int button;
	// @pyparm int|button||The button to press
	if (!PyArg_ParseTuple( args, "i", &button))
		return NULL;
	if (!pPS->PressButton(button))
		RETURN_ERR("PressButton failed");
	RETURN_NONE;
}

// @pymethod |PyCPropertySheet|SetWizardButtons|Enables the wizard buttons
PyObject *ui_propsheet_set_wizard_buttons( PyObject *self, PyObject *args )
{
	CPropertySheet *pPS = pPS=GetPropSheet(self);
	if (!pPS) return NULL;
	int flags;
	// @pyparm int|flags||The wizard flags
	if (!PyArg_ParseTuple( args, "i", &flags))
		return NULL;
	pPS->SetWizardButtons(flags);
	RETURN_NONE;
}

// @pymethod |PyCPropertySheet|SetWizardMode|Enables the wizard mode
PyObject *ui_propsheet_set_wizard_mode( PyObject *self, PyObject *args )
{
	CPropertySheet *pPS = pPS=GetPropSheet(self);
	if (!pPS) return NULL;
	CHECK_NO_ARGS2(args,SetWizardMode);
	pPS->SetWizardMode();
	RETURN_NONE;
}

// @pymethod |PyCPropertySheet|SetFinishText|Sets the text for the Finish button
PyObject *ui_propsheet_set_finish_text( PyObject *self, PyObject *args )
{
	CPropertySheet *pPS = pPS=GetPropSheet(self);
	if (!pPS) return NULL;
	char *text;
	// @pyparm string|text||The next for the button
	if (!PyArg_ParseTuple( args, "s", &text))
		return NULL;
	pPS->SetFinishText(text);
	RETURN_NONE;
}

// inherited from window
//
///////////////////////////////////////
// @object PyCPropertySheet|A class which encapsulates an MFC CPropertySheet object.  Derived from a <o PyCWnd> object.
static struct PyMethodDef ui_propsheet_methods[] = {
	{"AddPage",			ui_propsheet_add_page,			1}, // @pymeth AddPage|Adds the supplied page with the rightmost tab in the property sheet. 
	{"CreateWindow",	ui_propsheet_create_window, 	1}, // @pymeth CreateWindow|Displays the property sheet as a modeless dialog.
	{"DoModal",			ui_propsheet_do_modal,			1}, // @pymeth DoModal|Displays the property sheet as a modal dialog.
	{"EnableStackedTabs",ui_propsheet_enable_stacked_tabs,1}, // @pymeth EnableStackedTabs|Enables or disables stacked tabs.
	{"EndDialog",		ui_propsheet_end_dialog, 		1}, // @pymeth EndDialog|Closes the dialog, with the specified result.
	{"GetActiveIndex",	ui_propsheet_get_active_index,	1}, // @pymeth GetActiveIndex|Retrieves the index of the active page of the property sheet.
	{"GetActivePage",	ui_propsheet_get_active_page,	1}, // @pymeth GetActivePage|Returns the currently active property page.
	{"GetPage",			ui_propsheet_get_page,			1}, // @pymeth GetPage|Returns the specified property page.
	{"GetPageIndex",	ui_propsheet_get_page_index,	1}, // @pymeth GetPageIndex|Retrieves the index of the specified page of the property sheet.
	{"GetPageCount",	ui_propsheet_get_page_count,	1}, // @pymeth GetPageCount|Returns the number of pages.
	{"GetTabCtrl",      ui_propsheet_get_tab_ctrl,      1}, // @pymeth GetTabCtrl|Returns the tab control used by the sheet.
	{"PressButton",     ui_propsheet_press_button,      1}, // @pymeth PressButton|Simulates the choice of the specified button in a property sheet.
	{"RemovePage",		ui_propsheet_remove_page,		1}, // @pymeth RemovePage|Removes the specified page from the sheet.
	{"SetActivePage",	ui_propsheet_set_active_page,	1}, // @pymeth SetActivePage|Programmatically sets the active page object.
	{"SetTitle",		ui_propsheet_set_title,			1}, // @pymeth SetTitle|Sets the caption for the property sheet.
	{"SetFinishText",   ui_propsheet_set_finish_text,   1}, // @pymeth SetFinishText|Sets the text for the Finish button
	{"SetWizardMode",   ui_propsheet_set_wizard_mode,   1}, // @pymeth SetWizardMode|Enables the wizard mode
	{"SetWizardButtons",ui_propsheet_set_wizard_buttons,1}, // @pymeth SetWizardButtons|Enables the wizard buttons
	{NULL,			NULL}
};

ui_type_CObject PyCPropertySheet::type("PyCPropertySheet", 
										  &PyCWnd::type, 
										  RUNTIME_CLASS(CPropertySheet), 
										  sizeof(PyCPropertySheet), 
										  ui_propsheet_methods, 
										  GET_PY_CTOR(PyCPropertySheet));

/////////////////////////////////////////////////////////////////////
//
// Property Page object
//
//////////////////////////////////////////////////////////////////////
PyCPropertyPage::PyCPropertyPage()
{
	bManualDelete = FALSE; // dont "delete" the CWnd.
}
PyCPropertyPage::~PyCPropertyPage()
{
//	CPythonPropertyPage *pPage = GetPropPage(this);
//	delete pPage;
}
// @pymethod <o PyCPropertyPage>|win32ui|CreatePropertyPage|Creates a property page object.
PyObject *PyCPropertyPage::create( PyObject *self, PyObject *args )
{
	PyObject *obTemplate = NULL;
	int idCaption = 0;
	if (!PyArg_ParseTuple(args,"O|i", 
	          &obTemplate, // @pyparm int/string|resource||The resource ID to use for the page.
	          &idCaption)) // @pyparm int|caption|0|The ID if the string resource to use for the caption.
		return NULL;
	CPythonPropertyPage *pPP;
	if (PyInt_Check(obTemplate)) {
		int id = (int)PyInt_AsLong(obTemplate);
		if (!PropSheetCheckForPageCreate(id))
			return NULL;
		pPP = new CPythonPropertyPage(id, idCaption);
	}
	else if (PyString_Check(obTemplate)) {
		char *value = PyString_AsString(obTemplate);
		if (!PropSheetCheckForPageCreate(value))
			return NULL;
		pPP = new CPythonPropertyPage(value, idCaption);
	}
	else
		RETURN_TYPE_ERR("parameter 1 must be an integer or string object");
	PyCPropertyPage *ret = (PyCPropertyPage *)ui_assoc_object::make( PyCPropertyPage::type, pPP);
	return ret;
}

// @pymethod <o PyCPropertyPage>|win32ui|CreatePropertyPageIndirect|Creates a property page object from a template.
PyObject *PyCPropertyPage::createIndirect(PyObject *, PyObject *args)
{
	PyObject *obTemplate = NULL;
	int idCaption = 0;
	// @pyparm list|resourceList||A list of [<o Dialog Header Tuple>, <o Dialog Item Tuple>, ...], which describe the page to be created.
	// @pyparm int|caption|0|The ID if the string resource to use for the caption.
	if (!PyArg_ParseTuple(args, "O|i", &obTemplate, &idCaption))
		return NULL;

	HGLOBAL h = MakeResourceFromDlgList(obTemplate);
	if (h == NULL)
		return NULL;

	CPythonPropertyPage *pPP = new CPythonPropertyPage(IDD_DUMMYPROPPAGE, idCaption);
	if (!pPP->SetTemplate(h))
		return NULL;
	PyCPropertyPage *ret = (PyCPropertyPage *)ui_assoc_object::make( PyCPropertyPage::type, pPP);
	return ret;
}

// @pymethod |PyCPropertyPage|CancelToClose|Changes the Cancel button to Close.
PyObject *ui_proppage_cancel_to_close( PyObject *self, PyObject *args )
{
	CPythonPropertyPage *pPP;
	if (!(pPP=GetPropPage(self)))
		return NULL;
	CHECK_NO_ARGS(args);
	pPP->CancelToClose();
	RETURN_NONE;
}
// @pymethod |PyCPropertyPage|SetModified|Sets the modified flag.
PyObject *ui_proppage_set_modified( PyObject *self, PyObject *args )
{
	CPythonPropertyPage *pPP;
	if (!(pPP=GetPropPage(self)))
		return NULL;
	BOOL bChanged = TRUE;
	if (!PyArg_ParseTuple( args, "|i", &bChanged))
		return NULL;
	pPP->SetModified(bChanged);
	RETURN_NONE;
}
// @pymethod |PyCPropertyPage|OnOK|Calls the default MFC OnOK handler.
PyObject *ui_proppage_on_ok( PyObject *self, PyObject *args )
{
	CPythonPropertyPage *pPP;
	if (!(pPP=GetPropPage(self)))
		return NULL;
	CHECK_NO_ARGS2(args, OnOK );
	// @xref <vm PyCPropertyPage.OnOK>
	pPP->CPropertyPage::OnOK();
	RETURN_NONE;
}
// @pymethod |PyCPropertyPage|OnCancel|Calls the default MFC OnCancel handler.
PyObject *ui_proppage_on_cancel( PyObject *self, PyObject *args )
{
	CPythonPropertyPage *pPP;
	if (!(pPP=GetPropPage(self)))
		return NULL;
	CHECK_NO_ARGS2(args, OnCancel );
	// @xref <vm PyCPropertyPage.OnCancel>
	pPP->CPropertyPage::OnCancel();
	RETURN_NONE;
}

// @pymethod |PyCPropertyPage|OnSetActive|Calls the default MFC OnSetActive handler.
PyObject *ui_proppage_on_set_active( PyObject *self, PyObject *args )
{
	CPythonPropertyPage *pPP;
	if (!(pPP=GetPropPage(self)))
		return NULL;
	CHECK_NO_ARGS2(args, OnSetActive );
	// @xref <vm PyCPropertyPage.OnSetActive>
	return PyInt_FromLong(pPP->CPropertyPage::OnSetActive());
}
// @pymethod |PyCPropertyPage|OnKillActive|Calls the default MFC OnKillActive handler.
PyObject *ui_proppage_on_kill_active( PyObject *self, PyObject *args )
{
	CPythonPropertyPage *pPP;
	if (!(pPP=GetPropPage(self)))
		return NULL;
	CHECK_NO_ARGS2(args, OnKillActive );
	// @xref <vm PyCPropertyPage.OnKillActive>
	return PyInt_FromLong(pPP->CPropertyPage::OnKillActive());
}


// @object PyCPropertyPage|A class which encapsulates an MFC CPropertyPage object.  Derived from a <o PyCDialog> object.
static struct PyMethodDef ui_proppage_methods[] = {
	{"CancelToClose",	ui_proppage_cancel_to_close,		1}, // @pymeth CancelToClose|Changes the Cancel button to Close.
	{"OnCancel",		ui_proppage_on_cancel,		1}, // @pymeth OnCancel|Calls the default MFC OnCancel handler.
	{"OnOK",			ui_proppage_on_ok,			1}, // @pymeth OnOK|Calls the default MFC OnOK handler.
	{"OnSetActive",		ui_proppage_on_set_active,			1}, // @pymeth OnOK|Calls the default MFC OnSetActive handler.
	{"OnKillActive",	ui_proppage_on_kill_active,			1}, // @pymeth OnOK|Calls the default MFC OnKillActive handler.
	{"SetModified",		ui_proppage_set_modified,		1}, // @pymeth SetModified|Sets the modified flag (for the Apply button)
	{NULL,			NULL}
};
// derived from dialog.
ui_type_CObject PyCPropertyPage::type("PyCPropertyPage", 
										 &PyCDialog::type, 
										 RUNTIME_CLASS(CPropertyPage), 
										 sizeof(PyCPropertyPage), 
										 ui_proppage_methods, 
										 GET_PY_CTOR(PyCPropertyPage));

/////////////////////////////////////////////////////////////////////
//
// Tab Control Object
//
// inherited from window
//
///////////////////////////////////////
ui_tabctrl_object::ui_tabctrl_object()
{
}

ui_tabctrl_object::~ui_tabctrl_object()
{
}

// @pymethod int|PyCTabCtrl|SetCurSel|Sets the current selection of a tab control.
PyObject *ui_tabctrl_set_cur_sel( PyObject *self, PyObject *args )
{
	CTabCtrl *pTab;
	if (!(pTab=PyGetTabCtrlWithWnd(self)))
		return NULL;
	int tab;
	// @pyparm int|index||The index of the tab to set current.
	if (!PyArg_ParseTuple( args, "i", &tab))
		return NULL;
	int rc = pTab->SetCurSel(tab);
	if (rc==-1)
		RETURN_ERR("SetCurSel failed");
	return Py_BuildValue("i", rc);
	// @rdesc The zero-based index of the previously selected item.
}
// @pymethod int|PyCTabCtrl|GetCurSel|Gets the current selection of a tab control.
PyObject *ui_tabctrl_get_cur_sel( PyObject *self, PyObject *args )
{
	CHECK_NO_ARGS(args);
	CTabCtrl *pTab;
	if ((pTab=PyGetTabCtrl(self))==NULL)
		return NULL;
	return Py_BuildValue("i", pTab->GetCurSel());
	// @rdesc The zero-based index of the currently selected item, or -1 if no selection.

}
// @pymethod int|PyCTabCtrl|GetItemCountl|Returns the number of tabs in the control.
PyObject *ui_tabctrl_get_item_count( PyObject *self, PyObject *args )
{
	CHECK_NO_ARGS(args);
	CTabCtrl *pTab;
	if ((pTab=PyGetTabCtrl(self))==NULL)
		return NULL;
	return Py_BuildValue("i", pTab->GetItemCount());
}

// @object PyCTabCtrl|A class which encapsulates an MFC CTabCtrl object.  Derived from a <o PyCWnd> object.
static struct PyMethodDef ui_tabctrl_methods[] = {
/*	{"AddTab",           ui_tabctrl_add_tab,            1},
	{"NextTab",          ui_tabctrl_next_tab,           1},
	{"RemoveTab",        ui_tabctrl_remove_tab,         1},*/
	{"GetCurSel",        ui_tabctrl_get_cur_sel,        1}, // @pymeth GetCurSel|Gets the current selection of a tab control.
	{"GetItemCount",     ui_tabctrl_get_item_count,     1}, // @pymeth GetItemCountl|Returns the number of tabs in the control.
	{"SetCurSel",        ui_tabctrl_set_cur_sel,        1}, // @pymeth SetCurSel|Sets the current selection of a tab control.
	{NULL,			NULL}
};

ui_type_CObject ui_tabctrl_object::type("PyCTabCtrl", 
										&PyCWnd::type, 
										RUNTIME_CLASS(CTabCtrl), 
										sizeof(ui_tabctrl_object), 
										ui_tabctrl_methods, 
										GET_PY_CTOR(ui_tabctrl_object));
