/*

	win32 hierlist data type

	Created July 1994, Mark Hammond (MHammond@skippinet.com.au)

Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc

*/
#include "stdafx.h"

#include "win32win.h"
#include "win32hl.h"

#include "hierlist.h"
#include "hierlvlb.h"
#include "hierlslb.h"

#define HLB_STD

#ifdef HLB_STD
#define LB_CLASS CListBoxHier
#define LB_CONTROLLER_CLASS CHierControlCListBox
#else
#define LB_CLASS CVListHier
#define LB_CONTROLLER_CLASS CHierControlCVListBox
#endif

class CPythonHierControl;
/////////////////////////

// define my hierlist item class.
class CPythonHierListItem : public CHierListItem
{
	DECLARE_DYNAMIC(CPythonHierListItem)
public:
	CPythonHierListItem(CPythonHierListItem *pParent, PyObject *tuple, CPythonHierControl *pMyController);
	virtual ~CPythonHierListItem();
	virtual unsigned GetNumChildren();
	virtual CHierListItem *GetChild(unsigned);
	virtual CHierListItem *GetParent();
	virtual CString GetText();
	virtual int GetBitmapRow();
	virtual int GetBitmapColumn();
	virtual BOOL IsExpandable() ;
	virtual void ClosedChildren(unsigned kids);

	virtual BOOL GetSubList() ;
	void KillCachedSubList();

	PyObject *myobject;
	ui_hierlist_item *pUIHLI;
	CPythonHierControl *pController;
	CPythonHierListItem *pParent;
	CPythonHierListItem **ppChildren;
	PyObject *pSubList;
	unsigned numChildren;
};

// define my hierlist controller class.
class CPythonHierControl : public LB_CONTROLLER_CLASS
{
	friend class CPythonHierListItem;
	DECLARE_DYNAMIC(CPythonHierControl);
public:
	CPythonHierControl(LB_CLASS *pLB, BOOL bUseHLI);
	~CPythonHierControl();
	PyObject *PyHierInit(CWnd *pParent, UINT listBoxID, UINT bitmapID, CSize bmpSize, BOOL bUseInd);
	void PyHierTerm(void);
	PyObject *PyGetSel(void);
	PyObject *PyGetSelItem(void);
	PyObject *PyIsSelOpen(void);
	virtual VOID AcceptRoot( CHierListItem *pRoot);
	virtual void PerformOpenDocument( CHierListItem *pItem );
	virtual void PerformItemSelected( CHierListItem *pItem );
	virtual void TakeDefaultAction( void );
	void DoTakeDefaultAction( void ) { CHierListControl::TakeDefaultAction(); }

	BOOL UsePythonHLI() {return bUsePythonHLI;}
protected:
	void DeleteVirtualHelpers();
private:
	LB_CLASS 		*pListBox;
	CVirtualHelper	*virtGetBitmapRow;
	CVirtualHelper	*virtGetBitmapColumn;
	CVirtualHelper	*virtGetText;
	CVirtualHelper	*virtIsExpandable;
	BOOL bUsePythonHLI;
};

//////////////////////////////////////////////////////////////////////////
//
// HierListItem
//
IMPLEMENT_DYNAMIC(CPythonHierListItem, CHierListItem)

CPythonHierListItem::CPythonHierListItem(CPythonHierListItem *parent, PyObject *initTuple, CPythonHierControl *pMyController)
{
	myobject = initTuple;
	pParent = parent;
	pSubList = NULL;
	ppChildren = NULL;
	numChildren = 0;
	ASSERT(myobject);
	Py_INCREF(myobject);
	pController = pMyController;
	if (pController->UsePythonHLI())
		pUIHLI = (ui_hierlist_item *)ui_assoc_object::make( ui_hierlist_item::type, this );
	else
		pUIHLI = NULL;
}
CPythonHierListItem::~CPythonHierListItem()
{
	DODECREF(myobject);
	if (pUIHLI) {
		pUIHLI->KillAssoc();
		DODECREF(pUIHLI);
	}

}

inline CHierListItem *CPythonHierListItem::GetParent()
{
	return pParent;
}

unsigned CPythonHierListItem::GetNumChildren()
{
	// must ensure GetSubList has been called.
	GetSubList();
	return numChildren;
}

// This is optimised to only touch children when requested.  I allocate
// an array big enough for all children.  Then, when a specific child
// is asked for, this array is checked, and if NULL, the array element
// is filled and returned.
CHierListItem *CPythonHierListItem::GetChild(unsigned uChild)
{
	if (!GetSubList())
		return NULL;
	if (ppChildren==NULL) {
		ppChildren = new CPythonHierListItem *[numChildren];
		memset(ppChildren, 0, sizeof(CPythonHierListItem *)*numChildren);
	}

	if (uChild < 0 || uChild >= numChildren)
		return NULL;
	if (ppChildren[uChild]==NULL)
		ppChildren[uChild] = new CPythonHierListItem( this, PyList_GetItem(pSubList, uChild ), pController );
	return ppChildren[uChild];
}
void CPythonHierListItem::ClosedChildren(unsigned kids)
{
	ASSERT(ppChildren==NULL || kids==numChildren);
	KillCachedSubList();
}
void CPythonHierListItem::KillCachedSubList()
{
	if (ppChildren==NULL)
		return;
	for (unsigned i=0;i<numChildren;i++)
		delete ppChildren[i];
	delete [] ppChildren;
	ppChildren = NULL;
	numChildren = 0;
	XDODECREF(pSubList);
	pSubList = NULL;
}
CString CPythonHierListItem::GetText()
{
	PyObject *objectUse = myobject;
	if (!pController->virtGetText->call( myobject, pUIHLI ) || !pController->virtGetText->retval(objectUse)) {
		// simple list (from Pythons POV, that is!)
		if (PyTuple_Check(myobject) && PyTuple_Size(myobject)>0)
			objectUse = PyTuple_GetItem(myobject,0);
	}
	return GetReprText(objectUse);
}
int CPythonHierListItem::GetBitmapRow() 
{
	int ret;
	if (!pController->virtGetBitmapRow->call( myobject, pUIHLI ) || !pController->virtGetBitmapRow->retval(ret))
		return 0;
	return ret;
}
int CPythonHierListItem::GetBitmapColumn()
{
	int nColumn;
	if (!pController->virtGetBitmapColumn->call( myobject, pUIHLI ) || 
		pController->virtGetBitmapColumn->retnone() ||
		!pController->virtGetBitmapColumn->retval(nColumn)) {
		if ( IsExpandable() ) {
			// Am I open ?
			if ( IsOpen() ) {
				nColumn = 2;
			} else {
				nColumn = 0;
			}
		} else {
			nColumn = 4;
		}
	}
	return nColumn;
}

// returns pointer to a python PyObject.
BOOL CPythonHierListItem::GetSubList()
{
	if (pSubList)
		return TRUE;
	CVirtualHelper virtGetSubList ("GetSubList", pController->pListBox);

	if (!virtGetSubList.call( myobject, pUIHLI ) || !virtGetSubList.retval(pSubList)) {
		// build the list manually.
		if (PyTuple_Check(myobject) && PyTuple_Size(myobject)>1)
			pSubList = PyTuple_GetItem(myobject,1);
	}
	if (pSubList==NULL || !PyList_Check(pSubList)) {
		pSubList = NULL;
		return FALSE;
	}
	Py_INCREF(pSubList);
	numChildren = PyList_Size(pSubList);
	return TRUE;
}
BOOL CPythonHierListItem::IsExpandable()
{
	if (ppChildren)
		return TRUE;
	BOOL ret;
	if (!pController->virtIsExpandable->call( myobject, pUIHLI ) || !pController->virtIsExpandable->retval(ret))
		return GetSubList();
	else
		return ret;
}

//////////////////////////////////////////////////////////////////////////
//
// CPythonHierControl
//
IMPLEMENT_DYNAMIC(CPythonHierControl, LB_CONTROLLER_CLASS);
CPythonHierControl::CPythonHierControl(LB_CLASS *pLB, BOOL bUseHLI)
{
	virtGetText = virtIsExpandable = virtGetBitmapRow = virtGetBitmapColumn = NULL;
	ASSERT(pLB);
	pListBox = pLB;
	bUsePythonHLI = bUseHLI;
}
CPythonHierControl::~CPythonHierControl()
{
	PyHierTerm();	// theoretically not needed, as PyHierTerm should 
					// be called at window close time.
}
void CPythonHierControl::DeleteVirtualHelpers()
{
	delete virtGetText;
	virtGetText = NULL;
	delete virtIsExpandable;
	virtIsExpandable = NULL;
	delete virtGetBitmapRow;
	virtGetBitmapRow = NULL;
	delete virtGetBitmapColumn;
	virtGetBitmapColumn = NULL;
}
PyObject *CPythonHierControl::PyHierInit(CWnd *pParent, UINT listBoxID, UINT bitmapID, CSize bmpSize, BOOL bUseInd)
{
	HierInit(pListBox, bitmapID, bmpSize.cx, bmpSize.cy, bUseInd );
	if (listBoxID==0)	{	// no listbox specified - create one filling the client.
		DWORD dwStyle = WS_VISIBLE | VLBS_USEDATAVALUES;	// virtual
		pListBox->Create( dwStyle, CFrameWnd::rectDefault, pParent, AFX_IDW_PANE_FIRST );
	} else {
		if (CWnd::FromHandlePermanent(pListBox->m_hWnd) != NULL)
			RETURN_ERR("There must not be a Python object already assocated with the ListBox");
		pListBox->SubclassDlgItem( listBoxID, pParent );
	}
	AcceptRoot(NULL);
	RETURN_NONE;
}
void CPythonHierControl::PyHierTerm(void)
{
	HierTerm();
	// could set the win_object bManualDelete flag, but seeing as I need to more.
	DeleteVirtualHelpers();
	delete (CPythonHierListItem *)m_pRootItem;
	m_pRootItem = NULL;
	delete pListBox;
	pListBox = NULL;
}

VOID CPythonHierControl::AcceptRoot( CHierListItem *pRoot)
{
	DeleteVirtualHelpers();
	void *assoc = pListBox;
	if (assoc==NULL)
		return;
	virtGetText = new CVirtualHelper("GetText", assoc);
	virtIsExpandable = new CVirtualHelper("IsExpandable", assoc);
	virtGetBitmapRow = new CVirtualHelper("GetBitmapRow", assoc);
	virtGetBitmapColumn = new CVirtualHelper("GetBitmapColumn", assoc);
	CHierListItem *oldRoot = m_pRootItem;
	CHierListControl::AcceptRoot(pRoot);
	delete oldRoot;
}

void CPythonHierControl::PerformOpenDocument( CHierListItem *pItem )
{
	CPythonHierListItem *pMy = (CPythonHierListItem *)pItem;
	CVirtualHelper virtPerformOpenDocument("PerformOpenDocument", pListBox);

	virtPerformOpenDocument.call(pMy->myobject, pMy->pUIHLI);
}
void CPythonHierControl::PerformItemSelected( CHierListItem *pItem )
{
	CPythonHierListItem *pMy = (CPythonHierListItem *)pItem;
	CVirtualHelper virtPerformItemSelected("PerformItemSelected", pListBox);
	virtPerformItemSelected.call(pMy->myobject, pMy->pUIHLI);
}
void CPythonHierControl::TakeDefaultAction()
{
	CVirtualHelper virtTakeDefaultAction("TakeDefaultAction", pListBox);
	if (!virtTakeDefaultAction.call())
		DoTakeDefaultAction();
	else {
		if (virtTakeDefaultAction.retnone())
			DoTakeDefaultAction();
	}
}
PyObject *CPythonHierControl::PyGetSel()
{
	CPythonHierListItem *pItem = (CPythonHierListItem *)ItemFromData(GetSelData());
	PyObject *ret;
	if (pItem==NULL)
		ret = Py_None;
	else
		ret = pItem->myobject;
	Py_INCREF(ret);
	return ret;
}
PyObject *CPythonHierControl::PyGetSelItem()
{
	CPythonHierListItem *pItem = (CPythonHierListItem *)ItemFromData(GetSelData());
	PyObject *ret;
	if (pItem==NULL)
		ret = Py_None;
	else
		ret = pItem->pUIHLI;
	Py_INCREF(ret);
	return ret;
}

PyObject *CPythonHierControl::PyIsSelOpen()
{
	CPythonHierListItem *pItem = (CPythonHierListItem *)ItemFromData(GetSelData());
	if (pItem==NULL || !pItem->IsOpen())
		RETURN_NONE;
	else
		return Py_BuildValue("i", 1);
}


/////////////////////////////////////////////////////////////////////
//
// hier_list PyObject
//
ui_hierlist_object::ui_hierlist_object()
{
}
ui_hierlist_object::~ui_hierlist_object()
{
}
// @pymethod |win32ui|CreateHierList|Creates a <o PyCHierList> object.
PyObject *ui_hierlist_object::create(PyObject *self, PyObject *args)
{
	char *title = NULL;
	BOOL bUseHLI = FALSE;
	// @pyparm int|bUseItems|0|Indicates if the HierList uses HierListItems or not.
	if (!PyArg_ParseTuple(args, "|i", &bUseHLI ))
		return NULL;
	
	LB_CLASS *pListBox = new LB_CLASS;	// ready to be attached to a window, or have one created.
	CPythonHierControl *pControl = new CPythonHierControl(pListBox, bUseHLI);
	pListBox->AcceptController(pControl);

	return ui_assoc_object::make( ui_hierlist_object::type, pListBox);
}
void ui_hierlist_object::DoKillAssoc( BOOL bDestructing /*= FALSE*/ )
{
	PyCWnd::DoKillAssoc(bDestructing);
	// delete any listbox objects I created.
	CPythonHierControl *hl = GetListObject();
	if (hl) {
		delete hl;
		hl = NULL;
	}
}
/////////////////////////////////////////////////////////////////////
//
// hier_list methods
//


void *ui_hierlist_object::GetGoodCppObject(ui_type *type_check) const
{
	LB_CLASS *pList = (LB_CLASS *)ui_assoc_object::GetGoodCppObject(&type);
	if (pList==NULL)
		return NULL;
	ASSERT_VALID(pList);
#ifdef HLB_STD
	if (!pList->IsKindOf(RUNTIME_CLASS(CListBoxHier))) {
#else
	if (!pList->IsKindOf(RUNTIME_CLASS(CVListHier))) {
#endif
		TRACE("GetListObject fails due to RTTI - got %s\n", pList->GetRuntimeClass()->m_lpszClassName);
		RETURN_ERR("win32ui: Internal error - C++ RTTI failed");
	}
	return pList;
}

/*static*/CPythonHierControl *ui_hierlist_object::GetListObject(PyObject *self)
{
	return ((ui_hierlist_object *)self)->GetListObject();
}

CPythonHierControl *ui_hierlist_object::GetListObject()
{
	PyObject *self = this;
	LB_CLASS *pList = (LB_CLASS *)ui_assoc_object::GetGoodCppObject(self, &type);
	if (pList==NULL) return NULL;
	CPythonHierControl *pOb = (CPythonHierControl *)pList->GetController();
	return pOb;
}

PyObject *ui_hierlist_object_accept_root( PyObject *self, PyObject *args )
{
	CPythonHierControl *hl = ui_hierlist_object::GetListObject(self);
	if (!hl)
		return NULL;
	PyObject *ob;
	if (!PyArg_ParseTuple(args,"O",&ob))
		return NULL;
	CPythonHierListItem *item = new CPythonHierListItem( NULL, ob, hl );
	hl->AcceptRoot(item);
	// must accept root first, else Python "virtuals" wont be set up.
	// call it a warning, as there is nothing we can do about it now
	if (!item->IsExpandable()) {
		PyErr_SetString(PyExc_ValueError,"Warning: the root is not expandable");
		return NULL;
	}
	RETURN_NONE;
}
PyObject *ui_hierlist_object_hier_init( PyObject *self, PyObject *args )
{
	CPythonHierControl *hl = ui_hierlist_object::GetListObject(self);
	if (!hl)
		return NULL;
	UINT childID, bitmapID;
	BOOL bUseInd=TRUE;
	CSize size;
	PyObject *obParent;
	if (!PyArg_ParseTuple(args,"Oii(ii)|i",&obParent, &childID, &bitmapID, &size.cx, &size.cy, &bUseInd))
		return NULL;
	if (!ui_base_class::is_uiobject(obParent, &PyCWnd::type))
		RETURN_TYPE_ERR("O param must be a PyCWnd object");
	CWnd *pParent = PyCWnd::GetPythonGenericWnd(obParent);
	if (!pParent)
		return NULL;

	return hl->PyHierInit( pParent, childID, bitmapID, size, bUseInd );
}

/*PyObject *ui_hierlist_object_ccc( PyObject *self, PyObject *args )
{
	CPythonHierControl *hl = ui_hierlist_object::GetListObject(self);
	if (!hl)
		return NULL;
	PyObject *ob = NULL;
	if (!PyArg_ParseTuple(args,"|O",&ob))
		return NULL;
	// ignoring this param for now!
//	hl->PyCheckChangedChildren(ob);
	hl->CheckChanged
	RETURN_NONE;
}*/
PyObject *ui_hierlist_object_close_sel( PyObject *self, PyObject *args )
{
	CPythonHierControl *hl = ui_hierlist_object::GetListObject(self);
	if (!hl)
		return NULL;
	CHECK_NO_ARGS(args);
	hl->CloseSelection();
	RETURN_NONE;
}
PyObject *ui_hierlist_object_expand_sel( PyObject *self, PyObject *args )
{
	CPythonHierControl *hl = ui_hierlist_object::GetListObject(self);
	if (!hl)
		return NULL;
	int bDeep = FALSE;
	if (!PyArg_ParseTuple(args,"|i",&bDeep))
		return NULL;
	// ignoring this param for now!
	hl->ExpandSelection(bDeep);
	RETURN_NONE;
}
PyObject *ui_hierlist_object_take_def_action( PyObject *self, PyObject *args )
{
	CPythonHierControl *hl = ui_hierlist_object::GetListObject(self);
	if (!hl)
		return NULL;
	CHECK_NO_ARGS(args);
	// must skip my virtual handler, as this is likely called from it.
	hl->DoTakeDefaultAction();
	RETURN_NONE;
}
PyObject *ui_hierlist_object_get_sel( PyObject *self, PyObject *args )
{
	CPythonHierControl *hl = ui_hierlist_object::GetListObject(self);
	if (!hl)
		return NULL;
	CHECK_NO_ARGS(args);
	// must skip my virtual handler, as this is likely called from it.
	return hl->PyGetSel();
}
PyObject *ui_hierlist_object_get_sel_item( PyObject *self, PyObject *args )
{
	CPythonHierControl *hl = ui_hierlist_object::GetListObject(self);
	if (!hl)
		return NULL;
	CHECK_NO_ARGS(args);
	// must skip my virtual handler, as this is likely called from it.
	return hl->PyGetSelItem();
}

PyObject *ui_hierlist_object_is_sel_open( PyObject *self, PyObject *args )
{
	CPythonHierControl *hl = ui_hierlist_object::GetListObject(self);
	if (!hl)
		return NULL;
	CHECK_NO_ARGS(args);
	// must skip my virtual handler, as this is likely called from it.
	return hl->PyIsSelOpen();
}

static struct PyMethodDef ui_hierlist_methods[] = {
	{"AcceptRoot",				ui_hierlist_object_accept_root,	1},
	{"HierInit",				ui_hierlist_object_hier_init,	1},
//	{"CheckChangedChildren",	ui_hierlist_object_ccc,		1},
	{"CloseSelection",			ui_hierlist_object_close_sel,	1},
	{"ExpandSelection",			ui_hierlist_object_expand_sel,	1},
	{"DoTakeDefaultAction",		ui_hierlist_object_take_def_action,	1},
	{"GetSel",					ui_hierlist_object_get_sel,	1},
	{"GetSelItem",				ui_hierlist_object_get_sel_item,	1},
	{"IsSelOpen",				ui_hierlist_object_is_sel_open,	1},
	{NULL,			NULL}		/* sentinel */
};

ui_type_CObject ui_hierlist_object::type("hier list", 
										 &PyCWnd::type, 
										 RUNTIME_CLASS(CWnd), 
										 sizeof(ui_hierlist_object), 
										 ui_hierlist_methods, 
										 GET_PY_CTOR(ui_hierlist_object));

void *ui_hierlist_item::GetGoodCppObject(ui_type *ui_type_check) const
{
	CPythonHierListItem *pHLI = (CPythonHierListItem *)ui_assoc_object::GetGoodCppObject(ui_type_check);
	if (pHLI==NULL)
    	RETURN_ERR("The HierListItem has been destroyed.");
	ASSERT_VALID(pHLI);	// must be valid if not NULL
	return pHLI;
}
/*static*/ CPythonHierListItem *ui_hierlist_item::GetHLI(PyObject *self)
{
	return (CPythonHierListItem *) ui_assoc_object::GetGoodCppObject(self, &type);
}

PyObject *
ui_hierlist_item_ccc(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CPythonHierListItem *pHLI = ui_hierlist_item::GetHLI(self);
	if (!pHLI)
		return NULL;
	DWORD dwData = pHLI->pController->LocateItemsData(pHLI);
	if (dwData==-1)
		RETURN_ERR("The item can not be located in the list");
	pHLI->pController->CheckChangedChildren(dwData);
	RETURN_NONE;
}
PyObject *
ui_hierlist_item_expand(PyObject *self, PyObject *args)
{
	BOOL bAll = FALSE;
	if (!PyArg_ParseTuple(args, "|i", &bAll ))
		return NULL;
	CPythonHierListItem *pHLI = ui_hierlist_item::GetHLI(self);
	if (!pHLI)
		return NULL;
	DWORD dwData = pHLI->pController->LocateItemsData(pHLI);
	if (dwData==-1)
		RETURN_ERR("The item can not be located in the list");
	pHLI->pController->ExpandItem(dwData, bAll);
	RETURN_NONE;
}
PyObject *
ui_hierlist_item_close(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CPythonHierListItem *pHLI = ui_hierlist_item::GetHLI(self);
	if (!pHLI)
		return NULL;
	DWORD dwData = pHLI->pController->LocateItemsData(pHLI);
	if (dwData==-1)
		RETURN_ERR("The item can not be located in the list");

	pHLI->pController->CloseItem(dwData);
	RETURN_NONE;
}


PyObject *
ui_hierlist_item_get_python_object(PyObject *self, PyObject *args)
{
	CPythonHierListItem *pHLI = ui_hierlist_item::GetHLI(self);
	if (!pHLI)
		return NULL;
	DOINCREF(pHLI->myobject);
	return pHLI->myobject;
}
PyObject *
ui_hierlist_item_get_parent(PyObject *self, PyObject *args)
{
	CPythonHierListItem *pHLI = ui_hierlist_item::GetHLI(self);
	if (!pHLI)
		return NULL;
	pHLI = pHLI->pParent;
	if (!pHLI)
		RETURN_NONE;
	DOINCREF(pHLI->pUIHLI);
	return pHLI->pUIHLI;
}
PyObject *
ui_hierlist_item_is_open(PyObject *self, PyObject *args)
{
	CPythonHierListItem *pHLI = ui_hierlist_item::GetHLI(self);
	if (!pHLI)
		return NULL;
	return Py_BuildValue("i", pHLI->IsOpen() );
}

static struct PyMethodDef ui_hierlist_item_methods[] = {
	{"Close",					ui_hierlist_item_close,	1},
	{"Expand",					ui_hierlist_item_expand,	1},
	{"GetPythonObject",			ui_hierlist_item_get_python_object,	1},
    {"GetParent",				ui_hierlist_item_get_parent,	1},
	{"IsOpen",					ui_hierlist_item_is_open,		1},
	{"CheckChangedChildren",	ui_hierlist_item_ccc,		1},
	{NULL,			NULL}		/* sentinel */
};
ui_type ui_hierlist_item::type("hier list item", 
							   &ui_assoc_object::type, 
							   sizeof(ui_hierlist_item), 
							   ui_hierlist_item_methods, 
							   GET_PY_CTOR(ui_hierlist_item));
