/* win32ctllist : implementation file

	List control object.  

	Created Feb 1997, Mark Hammond (MHammond@skippinet.com.au)

Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc

*/
#include "stdafx.h"

#include "win32win.h"
#include "win32dc.h"
#include "win32control.h"
#include "win32ctrltree.h"
#include "win32ImageList.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

const char *szErrTreeRequiresWindow = "The tree must have a window object for this operation";

PyCTreeCtrl::PyCTreeCtrl()
{
}
PyCTreeCtrl::~PyCTreeCtrl()
{
}
CTreeCtrl *GetTreeCtrl(PyObject *self)
{
	extern CTreeView *GetTreeViewPtr(PyObject *self);
	CTreeCtrl *rc;

	if (ui_base_class::is_uiobject(self, &PyCTreeView::type)) {
		CTreeView *pView = GetTreeViewPtr(self);
		if (pView)
			rc = &(pView->GetTreeCtrl());
		else
			rc = NULL;
	} else
		rc = (CTreeCtrl *)PyCWnd::GetPythonGenericWnd(self, &PyCTreeCtrl::type);
	if (rc && !::IsWindow(rc->m_hWnd))
		RETURN_ERR((char *)szErrTreeRequiresWindow);
	return rc;
}

#define MAKE_GET_INT_METH(fnname, mfcName) \
PyObject *fnname( PyObject *self, PyObject *args ) { \
	CHECK_NO_ARGS2(args,mfcName); \
	CTreeCtrl *pList = GetTreeCtrl(self); \
	if (!pList) return NULL; \
	int ret = pList->mfcName(); \
	return Py_BuildValue("i",ret); \
}

#define MAKE_GET_ITEM_INT_METH(fnname, mfcName) \
PyObject *fnname( PyObject *self, PyObject *args ) { \
	HTREEITEM htree; \
	if (!PyArg_ParseTuple( args, "i:" #mfcName, &htree)) \
		return NULL; \
	CTreeCtrl *pList = GetTreeCtrl(self); \
	if (!pList) return NULL; \
	int ret = pList->mfcName(htree); \
	return Py_BuildValue("i",ret); \
}

#define MAKE_GET_ITEM_ITEM_METH(fnname, mfcName) \
PyObject *fnname( PyObject *self, PyObject *args ) { \
	HTREEITEM htree; \
	if (!PyArg_ParseTuple( args, "i:" #mfcName, &htree)) \
		return NULL; \
	CTreeCtrl *pList = GetTreeCtrl(self); \
	if (!pList) return NULL; \
	HTREEITEM item = pList->mfcName(htree); \
	if (item==NULL) \
		RETURN_ERR(#mfcName " failed"); \
	return Py_BuildValue("i",item); \
}

#define MAKE_GET_ITEM_ITEM_INT_METH(fnname, mfcName) \
PyObject *fnname( PyObject *self, PyObject *args ) { \
	HTREEITEM htree; int code;\
	if (!PyArg_ParseTuple( args, "ii:" #mfcName, &htree, &code)) \
		return NULL; \
	CTreeCtrl *pList = GetTreeCtrl(self); \
	if (!pList) return NULL; \
	HTREEITEM item = pList->mfcName(htree, code); \
	if (item==NULL) \
		RETURN_ERR(#mfcName " failed"); \
	return Py_BuildValue("i",item); \
}

#define MAKE_GET_ITEM_VOID_METH(fnname, mfcName) \
PyObject *fnname( PyObject *self, PyObject *args ) { \
	if (!PyArg_ParseTuple( args, ":" #mfcName)) \
		return NULL; \
	CTreeCtrl *pList = GetTreeCtrl(self); \
	if (!pList) return NULL; \
	HTREEITEM item = pList->mfcName(); \
	if (item==NULL) \
		RETURN_ERR(#mfcName " failed"); \
	return Py_BuildValue("i",item); \
}

#define MAKE_SET_ITEMS_INTS_METH(fnname, mfcName) \
PyObject *fnname( PyObject *self, PyObject *args ) { \
	HTREEITEM item; int i1, i2;\
	if (!PyArg_ParseTuple( args, "iii:" #mfcName, &item, &i1, &i2)) \
		return NULL; \
	CTreeCtrl *pList = GetTreeCtrl(self); \
	if (!pList) return NULL; \
	if (!pList->mfcName(item, i1, i2)) \
		RETURN_ERR(#mfcName " failed"); \
	RETURN_NONE; \
}

#define MAKE_BOOL_ITEM_ACTION(fnname, mfcName) \
	PyObject *fnname( PyObject *self, PyObject *args ) { \
	HTREEITEM item; \
	if (!PyArg_ParseTuple( args, "i:" #mfcName, &item)) \
		return NULL; \
	CTreeCtrl *pList = GetTreeCtrl(self); \
	if (!pList) return NULL; \
	if (!pList->mfcName(item)) \
		RETURN_ERR(#mfcName " failed"); \
	RETURN_NONE; \
}

#define MAKE_BOOL_ITEM_INT_ACTION(fnname, mfcName) \
PyObject *fnname( PyObject *self, PyObject *args ) { \
	HTREEITEM item; int code;\
	if (!PyArg_ParseTuple( args, "ii:" #mfcName, &item, &code)) \
		return NULL; \
	CTreeCtrl *pList = GetTreeCtrl(self); \
	if (!pList) return NULL; \
	if (!pList->mfcName(item, code)) \
		RETURN_ERR(#mfcName " failed"); \
	RETURN_NONE;  \
}

#define MAKE_SETBOOL_INT_METH(fnname, mfcName) \
PyObject *fnname( PyObject *self, PyObject *args ) { \
	CTreeCtrl *pList = pList=GetTreeCtrl(self); \
	if (!pList) return NULL; \
	int val; \
	if (!PyArg_ParseTuple( args, "i:" #mfcName, &val)) \
		return NULL; \
	if (!pList->mfcName(val)) \
		RETURN_ERR(#mfcName "failed"); \
	RETURN_NONE; \
}
#define MAKE_SETVOID_INT_METH(fnname, mfcName) \
PyObject *fnname( PyObject *self, PyObject *args ) { \
	CTreeCtrl *pList = pList=GetTreeCtrl(self); \
	if (!pList) return NULL; \
	int val; \
	if (!PyArg_ParseTuple( args, "i:" #mfcName, &val)) \
		return NULL; \
	pList->mfcName(val); \
	RETURN_NONE; \
}

// @pymethod int|PyCTreeCtrl|GetCount|Retrieves the number of tree items associated with a tree view control.
MAKE_GET_INT_METH(PyCTreeCtrl_GetCount, GetCount )

// @pymethod int|PyCTreeCtrl|GetVisibleCount|Retrieves the number of visible tree items associated with a tree view control.
MAKE_GET_INT_METH(PyCTreeCtrl_GetVisibleCount, GetVisibleCount )

// @pymethod int|PyCTreeCtrl|GetIndent|Retrieves the offset (in pixels) of a tree view item from its parent.
MAKE_GET_INT_METH(PyCTreeCtrl_GetIndent, GetIndent )

// @pymethod |PyCTreeCtrl|SetIndent|Sets the offset (in pixels) of a tree view item from its parent.
// @pyparm int|indent||The new indent.
MAKE_SETVOID_INT_METH(PyCTreeCtrl_SetIndent, SetIndent)

// @pymethod HTREEITEM|PyCTreeCtrl|GetNextItem|Retrieves the next item.
// @pyparm HTREEITEM|item||The specified item
MAKE_GET_ITEM_ITEM_INT_METH(PyCTreeCtrl_GetNextItem, GetNextItem )

// @pymethod HTREEITEM|PyCTreeCtrl|GetChildItem|Retrieves the first child item.
// @pyparm HTREEITEM|item||The specified item
MAKE_GET_ITEM_ITEM_METH(PyCTreeCtrl_GetChildItem, GetChildItem )

// @pymethod |PyCTreeCtrl|SetItemImage|Sets the index of an items images.
// @pyparm HTREEITEM|item||The specified item
// @pyparm int|iImage||The offset of the image.
// @pyparm int|iSelectedImage||The offset of the selected image.
MAKE_SET_ITEMS_INTS_METH(PyCTreeCtrl_SetItemImage, SetItemImage )

// @pymethod |PyCTreeCtrl|SetItemState|Sets the state of item.
// @pyparm HTREEITEM|item||The specified item
// @pyparm int|state||The new state
// @pyparm int|stateMask||The mask for the new state
MAKE_SET_ITEMS_INTS_METH(PyCTreeCtrl_SetItemState, SetItemState )

// @pymethod int|PyCTreeCtrl|ItemHasChildren|Returns nonzero if the specified item has child items.
// @pyparm HTREEITEM|item||The specified item
MAKE_GET_ITEM_INT_METH(PyCTreeCtrl_ItemHasChildren, ItemHasChildren )

// @pymethod HTREEITEM|PyCTreeCtrl|GetNextSiblingItem|Retrieves the next sibling of the specified tree view item.
// @pyparm HTREEITEM|item||The specified item
MAKE_GET_ITEM_ITEM_METH(PyCTreeCtrl_GetNextSiblingItem, GetNextSiblingItem )

// @pymethod HTREEITEM|PyCTreeCtrl|GetPrevSiblingItem|Retrieves the previous sibling of the specified tree view item.
// @pyparm HTREEITEM|item||The specified item
MAKE_GET_ITEM_ITEM_METH(PyCTreeCtrl_GetPrevSiblingItem, GetPrevSiblingItem )

// @pymethod HTREEITEM|PyCTreeCtrl|GetParentItem|Retrieves the parent item of the specified tree view item.
// @pyparm HTREEITEM|item||The specified item
MAKE_GET_ITEM_ITEM_METH(PyCTreeCtrl_GetParentItem, GetParentItem )

// @pymethod HTREEITEM|PyCTreeCtrl|GetFirstVisibleItem|Retrieves the first visible item of the tree view control.
MAKE_GET_ITEM_VOID_METH(PyCTreeCtrl_GetFirstVisibleItem, GetFirstVisibleItem )

// @pymethod HTREEITEM|PyCTreeCtrl|GetNextVisibleItem|Retrieves the next visible item of the specified tree view item.
// @pyparm HTREEITEM|item||The specified item
MAKE_GET_ITEM_ITEM_METH(PyCTreeCtrl_GetNextVisibleItem, GetNextVisibleItem )

// @pymethod HTREEITEM|PyCTreeCtrl|GetPrevVisibleItem|Retrieves the previous visible item of the specified tree view item.
// @pyparm HTREEITEM|item||The specified item
MAKE_GET_ITEM_ITEM_METH(PyCTreeCtrl_GetPrevVisibleItem, GetPrevVisibleItem )

// @pymethod HTREEITEM|PyCTreeCtrl|GetSelectedItem|Retrieves the currently selected tree view item.
MAKE_GET_ITEM_VOID_METH(PyCTreeCtrl_GetSelectedItem, GetSelectedItem )

// @pymethod HTREEITEM|PyCTreeCtrl|GetDropHilightItem|Retrieves the target of a drag-and-drop operation.
MAKE_GET_ITEM_VOID_METH(PyCTreeCtrl_GetDropHilightItem, GetDropHilightItem )

// @pymethod HTREEITEM|PyCTreeCtrl|GetRootItem|Retrieves the root of the specified tree view item.
MAKE_GET_ITEM_VOID_METH(PyCTreeCtrl_GetRootItem, GetRootItem )


// @pymethod |PyCTreeCtrl|DeleteItem|Deletes the specified item.
// @pyparm HTREEITEM|item||The specified item
MAKE_BOOL_ITEM_ACTION(PyCTreeCtrl_DeleteItem, DeleteItem )

// @pymethod |PyCTreeCtrl|SelectItem|Selects a specified tree view item.
// @pyparm HTREEITEM|item||The specified item
MAKE_BOOL_ITEM_ACTION(PyCTreeCtrl_SelectItem, SelectItem )

// @pymethod |PyCTreeCtrl|SelectDropTarget|Redraws the tree item as the target of a drag-and-drop operation.
// @pyparm HTREEITEM|item||The specified item
MAKE_BOOL_ITEM_ACTION(PyCTreeCtrl_SelectDropTarget, SelectDropTarget )

// @pymethod |PyCTreeCtrl|SelectSetFirstVisible|Selects a specified tree view item as the first visible item.
// @pyparm HTREEITEM|item||The specified item
MAKE_BOOL_ITEM_ACTION(PyCTreeCtrl_SelectSetFirstVisible, SelectSetFirstVisible )

// @pymethod |PyCTreeCtrl|SortChildren|Sorts the children of a given parent item.
// @pyparm HTREEITEM|item||The specified parent item
MAKE_BOOL_ITEM_ACTION(PyCTreeCtrl_SortChildren, SortChildren )

// @pymethod |PyCTreeCtrl|Expand|Expands, or collapses, the child items of the specified tree view item.
// @pyparm HTREEITEM|item||The specified item
// @pyparm int|code||The action to take
MAKE_BOOL_ITEM_INT_ACTION(PyCTreeCtrl_Expand, Expand )

// @pymethod |PyCTreeCtrl|Select|Selects, scrolls into view, or redraws a specified tree view item.
// @pyparm HTREEITEM|item||The specified item
// @pyparm int|code||The action to take
MAKE_BOOL_ITEM_INT_ACTION(PyCTreeCtrl_Select, Select )

// @pymethod (int,int)|PyCTreeCtrl|GetItemImage|Retrieves the index of an items images.
// @pyparm HTREEITEM|item||The specified item
PyObject *PyCTreeCtrl_GetItemImage( PyObject *self, PyObject *args ) 
{
	HTREEITEM item;
	if (!PyArg_ParseTuple( args, "i:GetItemImage", &item))
		return NULL;
	CTreeCtrl *pList = GetTreeCtrl(self);
	if (!pList) return NULL;
	int res1, res2;
	if (!pList->GetItemImage(item, res1, res2))
	if (item==NULL)
		RETURN_ERR("GetItemImage failed");
	return Py_BuildValue("ii",res1, res2);
}

// @pymethod (int,int)|PyCTreeCtrl|GetItemState|Retrieves the state and mask of an item.
// @pyparm HTREEITEM|item||The specified item
// @pyparm int|stateMask||The mask for the result.
PyObject *PyCTreeCtrl_GetItemState( PyObject *self, PyObject *args ) 
{
	HTREEITEM item; UINT stateMask;
	if (!PyArg_ParseTuple( args, "ii:GetItemState", &item, &stateMask))
		return NULL;
	CTreeCtrl *pList = GetTreeCtrl(self);
	if (!pList) return NULL;
	return PyInt_FromLong(pList->GetItemState(item, stateMask));
}

// @pymethod <o PyCImageList>|PyCTreeCtrl|GetImageList|Retrieves the current image list.
PyObject *PyCTreeCtrl_GetImageList( PyObject *self, PyObject *args )
{
	CTreeCtrl *pList;
	if (!(pList=GetTreeCtrl(self)))
		return NULL;
	int nList;
	// @pyparm int|nImageList||Value specifying which image list to retrieve. It can be one of:
	// <nl>-	commctrl.LVSIL_NORMAL   Image list with large icons.
	// <nl>-	commctrl.LVSIL_SMALL   Image list with small icons.
	// <nl>-	commctrl.LVSIL_STATE   Image list with state images.
	if (!PyArg_ParseTuple(args, "i:GetImageList", &nList))
		return NULL;
	CImageList *ret = pList->GetImageList(nList);
	if (ret==NULL)
		RETURN_ERR("There is no image list available");
	return ui_assoc_object::make( PyCImageList::type, ret)->GetGoodRet();
}


// @pymethod int|PyCTreeCtrl|InsertItem|Inserts an item into the list.
PyObject *PyCTreeCtrl_InsertItem( PyObject *self, PyObject *args )
{
	CTreeCtrl *pList;
	HTREEITEM ret = NULL;
	UINT mask;
	int image, selImage, state, stateMask;
	LPARAM lParam;
	HTREEITEM hParent, hInsertAfter;
	char *text;
	if (!(pList=GetTreeCtrl(self)))
		return NULL;

	if (PyArg_ParseTuple(args, "iziiiiOii:InsertItem", 
		                 &mask, // @pyparmalt1 int|mask||Integer specifying which attributes to set
						 &text, // @pyparmalt1 string|text||The text of the item.
						 &image, // @pyparmalt1 int|image||The index of the image to use.
						 &selImage, // @pyparmalt1 int|selectedImage||The index of the items selected image.
						 &state, // @pyparmalt1 int|state||The initial state of the item.
						 &stateMask, // @pyparmalt1 int|stateMask||Specifies which bits of the state are valid.
						 &lParam, // @pyparmalt1 object|lParam||A user defined object for the item.
						 &hParent, // @pyparmalt1 HTREEITEM|parent||The parent of the item.
						 &hInsertAfter)) // @pyparmalt1 HTREEITEM|parent||The parent of the item.
		ret = pList->InsertItem(mask, text, image, selImage, state, stateMask, lParam, hParent, hInsertAfter);
	else {
		PyErr_Clear();
		hParent = TVI_ROOT;
		hInsertAfter = TVI_LAST;
		if (PyArg_ParseTuple(args, "sii|ii:InsertItem", 
		                 &text, // @pyparmalt2 string|text||The text for the item.
						 &image, // @pyparmalt2 int|image||The index of the image to use.
						 &selImage, // @pyparmalt2 int|selectedImage||The index of the items selected image.
						 &hParent, // @pyparmalt2 HTREEITEM|parent|commctrl.TVI_ROOT|The parent of the item.
						 &hInsertAfter)) // @pyparmalt2 HTREEITEM|insertAfter|commctrl.TVI_LAST|The item to insert the new item after, or TVI_FIRST, TVI_LAST or TVI_SORT
			ret = pList->InsertItem(text, image, selImage, hParent, hInsertAfter);
		else {
			PyErr_Clear();
			hParent = TVI_ROOT;
			hInsertAfter = TVI_LAST;
			if (PyArg_ParseTuple(args, "s|ii:InsertItem", 
				         &text, // @pyparmalt3 string|text||The text for the item.
						 &hParent, // @pyparmalt3 HTREEITEM|parent|commctrl.TVI_ROOT|The parent of the item.
						 &hInsertAfter)) // @pyparmalt3 HTREEITEM|parent|commctrl.TVI_LAST|The parent of the item.
				ret = pList->InsertItem(text, hParent, hInsertAfter);
			else {
				PyObject *obTVItem;
				TV_INSERTSTRUCT tvItem;
				PyErr_Clear();
				if (PyArg_ParseTuple(args, "iiO:InsertItem",
								 &tvItem.hParent, // @pyparm HTREEITEM|hParent||The parent item.  If commctrl.TVI_ROOT or 0, it is added to the root.
								 &tvItem.hInsertAfter, // @pyparm HTREEITEM|hInsertAfter||The item to insert after.  Can be an item or TVI_FIRST, TVI_LAST or TVI_SORT
								 &obTVItem)) { // @pyparm <om PyCTreeCtrl.TV_ITEM tuple>|item||A tuple describing the new item.

					if (!ParseTV_ITEMTuple(obTVItem, &tvItem.item))
						return NULL;
					ret = pList->InsertItem(&tvItem);
				} else {
					PyErr_Clear();
					RETURN_ERR("InsertItem could not parse the params.");
				}
			}
		}
	}
	if (ret==NULL)
		RETURN_ERR("InsertItem failed");
	return Py_BuildValue("i",ret);
}

// @pymethod int|PyCTreeCtrl|SetItem|Sets some of all of an items attributes.
PyObject *PyCTreeCtrl_SetItem( PyObject *self, PyObject *args )
{
	CTreeCtrl *pList;
	PyObject *obTVItem;
	if (!(pList=GetTreeCtrl(self)))
		return NULL;
	if (!PyArg_ParseTuple(args, "O:SetItem",
		                 &obTVItem)) // @pyparm <om PyCTreeCtrl.TV_ITEM tuple>|item||A tuple describing the new item.
		return NULL;
	TV_ITEM tvItem;
	if (!ParseTV_ITEMTuple(obTVItem, &tvItem))
		return NULL;
	if (!pList->SetItem(&tvItem))
		RETURN_ERR("SetItem failed");
	RETURN_NONE;
}

// @pymethod int|PyCTreeCtrl|SetImageList|Assigns an image list to a list view control.
PyObject *PyCTreeCtrl_SetImageList( PyObject *self, PyObject *args )
{
	CTreeCtrl *pList;
	PyObject *obList;
	int imageType;
	if (!(pList=GetTreeCtrl(self)))
		return NULL;
	if (!PyArg_ParseTuple(args, "Oi:SetImageList", 
		                  &obList, // @pyparm <o PyCImageList>|imageList||The Image List to use.
						  &imageType )) // @pyparm int|imageType||Type of image list. It can be one of (COMMCTRL.) LVSIL_NORMAL, LVSIL_SMALL or LVSIL_STATE
		return NULL;
	CImageList *pImageList = PyCImageList::GetImageList(obList);
	if (pImageList==NULL) return NULL;
	CImageList *pOldList = pList->SetImageList( pImageList, imageType );
	if (pOldList==NULL)
		RETURN_NONE;
	return ui_assoc_object::make( PyCImageList::type, pOldList )->GetGoodRet();
}

// @pymethod <om PyCTreeCtrl.TV_ITEM tuple>|PyCTreeCtrl|GetItem|Retrieves the details of an items attributes.
PyObject *PyCTreeCtrl_GetItem( PyObject *self, PyObject *args )
{
	HTREEITEM item;
	UINT mask = TVIF_CHILDREN | TVIF_HANDLE | TVIF_IMAGE | TVIF_PARAM | TVIF_SELECTEDIMAGE | TVIF_STATE | TVIF_TEXT; 

	if (!PyArg_ParseTuple( args, "i|i:GetItem", 
	                   &item, // @pyparm HTREEITEM|item||The item whose attributes are to be retrieved.
					   &mask)) // @pyparm int|mask|(all flags set)|The requested attributes.
		return NULL;

	CTreeCtrl *pList = GetTreeCtrl(self);
	if (!pList) return NULL;
	char textBuf[256];
	TV_ITEM tvItem;
	tvItem.hItem = item;
	tvItem.pszText = textBuf;
	tvItem.cchTextMax = sizeof(textBuf);
	tvItem.mask = TVIF_CHILDREN | TVIF_HANDLE | TVIF_IMAGE | TVIF_PARAM | TVIF_SELECTEDIMAGE | TVIF_STATE | TVIF_TEXT; 
	if (!pList->GetItem( &tvItem) )
		RETURN_ERR("GetItem failed");
	return MakeTV_ITEMTuple(&tvItem);
}

// @pymethod int|PyCTreeCtrl|GetItemText|Retrieves the text of a list view item or subitem.
PyObject *PyCTreeCtrl_GetItemText( PyObject *self, PyObject *args )
{
	HTREEITEM item;
	if (!PyArg_ParseTuple( args, "i:GetItemText", 
	                   &item)) // @pyparm HTREEITEM|item||The item whose text is to be retrieved.
		return NULL;
	CTreeCtrl *pList = GetTreeCtrl(self);
	if (!pList) return NULL;
	CString csText = pList->GetItemText(item);
	return PyString_FromString((char *)(const char *)csText);
}

// @pymethod int|PyCTreeCtrl|SetItemText|Changes the text of a list view item or subitem.
PyObject *PyCTreeCtrl_SetItemText( PyObject *self, PyObject *args )
{
	CTreeCtrl *pList = pList=GetTreeCtrl(self);
	if (!pList) return NULL;
	HTREEITEM item;
	char *text;
	if (!PyArg_ParseTuple( args, "is:SetItemText", 
	                   &item, // @pyparm HTREEITEM|item||The item whose text is to be retrieved.
					   &text)) // @pyparm string|text||String that contains the new item text.

		return NULL;
	if (!pList->SetItemText(item, text))
		RETURN_ERR("SetItemText failed");
	RETURN_NONE;
}

// @pymethod object|PyCTreeCtrl|GetItemData|Retrieves the application-specific value associated with an item.
PyObject *PyCTreeCtrl_GetItemData( PyObject *self, PyObject *args )
{
	HTREEITEM item;
	if (!PyArg_ParseTuple( args, "i:GetItemData", 
	                   &item)) // @pyparm HTREEITEM|item||The index of the item whose data is to be retrieved.

		return NULL;
	CTreeCtrl *pList = GetTreeCtrl(self);
	if (!pList) return NULL;
	PyObject *ret = PyWin_GetPythonObjectFromLong(pList->GetItemData(item));
	// inc ref count for return value.
	Py_INCREF(ret);
	return ret;
}

// @pymethod int|PyCTreeCtrl|SetItemData|Sets the items application-specific value.
PyObject *PyCTreeCtrl_SetItemData( PyObject *self, PyObject *args )
{
	CTreeCtrl *pList = pList=GetTreeCtrl(self);
	if (!pList) return NULL;
	HTREEITEM item;
	PyObject *data;
	if (!PyArg_ParseTuple( args, "iO:SetItemData", 
		                   &item, // @pyparm HTREEITEM|item||The item whose Data is to be set.
						   &data)) // @pyparm object|Data||New value for the data.
		return NULL;
	if (data==Py_None) data = NULL;
	if (!pList->SetItemData(item, (DWORD)data))
		RETURN_ERR("SetItemData failed");
	// @comm Note that a reference count is not added to the object.  This it is your
	// responsibility to make sure the object remains alive while in the list.
	RETURN_NONE;
}

// @pymethod object|PyCTreeCtrl|DeleteAllItems|Deletes all items in the control
PyObject *PyCTreeCtrl_DeleteAllItems( PyObject *self, PyObject *args )
{
	if (!PyArg_ParseTuple( args, ":DeleteAllItems"))
		return NULL;
	CTreeCtrl *pList = GetTreeCtrl(self);
	if (!pList) return NULL;
	if (!pList->DeleteAllItems())
		RETURN_ERR("DeleteAllItems failed");
	RETURN_NONE;
}

// @pymethod (int, int, int, int)|PyCTreeCtrl|GetItemRect|Retrieves the bounding rectangle of a tree view item.
PyObject *PyCTreeCtrl_GetItemRect( PyObject *self, PyObject *args )
{
	CTreeCtrl *pList = pList=GetTreeCtrl(self);
	if (!pList) return NULL;
	HTREEITEM item;
	RECT rect;
	BOOL bTextOnly;
	if (!PyArg_ParseTuple( args, "ii:GetItemRect", 
		                   &item, // @pyparm HTREEITEM|item||The item whose Data is to be set.
						   &bTextOnly)) // @pyparm int|bTextOnly||f this parameter is nonzero, the bounding rectangle includes only the text of the item. Otherwise it includes the entire line that the item occupies in the tree view control.
		return NULL;
	if (!pList->GetItemRect(item, &rect, bTextOnly))
		RETURN_ERR("GetItemRect failed");
	return Py_BuildValue("(iiii)",rect.left, rect.top, rect.right, rect.bottom);
}

// @pymethod <o PyCEdit>|PyCTreeCtrl|GetEditControl|Retrieves the handle of the edit control used to edit the specified tree view item.
PyObject *PyCTreeCtrl_GetEditControl( PyObject *self, PyObject *args )
{
	CTreeCtrl *pList = pList=GetTreeCtrl(self);
	if (!pList) return NULL;
	if (!PyArg_ParseTuple( args, ":GetEditControl"))
		return NULL;
	CEdit *pEdit = pList->GetEditControl();
	if (pEdit==NULL)
		RETURN_ERR("GetEditControl failed");
	return ui_assoc_object::make(UITypeFromCObject(pEdit), pEdit)->GetGoodRet();
}

// @pymethod <o PyCEdit>|PyCTreeCtrl|EditLabel|Edits a specified tree view item in-place.
PyObject *PyCTreeCtrl_EditLabel( PyObject *self, PyObject *args )
{
	CTreeCtrl *pList = pList=GetTreeCtrl(self);
	if (!pList) return NULL;
	HTREEITEM item;
	// @pyparm HTREEITEM|item||The item to edit.
	if (!PyArg_ParseTuple( args, "i:EditLabel", &item))
		return NULL;
	CEdit *pEdit = pList->EditLabel(item);
	if (pEdit==NULL)
		RETURN_ERR("EditLabel failed");
	return ui_assoc_object::make(UITypeFromCObject(pEdit), pEdit)->GetGoodRet();
}

// @pymethod int|PyCTreeCtrl|EnsureVisible|Ensures that a tree view item is visible in its tree view control.
PyObject *PyCTreeCtrl_EnsureVisible( PyObject *self, PyObject *args )
{
	CTreeCtrl *pList = pList=GetTreeCtrl(self);
	if (!pList) return NULL;
	HTREEITEM item;
	// @pyparm HTREEITEM|item||The item to edit.
	if (!PyArg_ParseTuple( args, "i:EnsureVisible", &item))
		return NULL;
	if (!pList->EnsureVisible(item))
		RETURN_ERR("EnsureVisible failed");
	RETURN_NONE;
}

// @pymethod <o PyCImageList>|PyCTreeCtrl|CreateDragImage|Creates a dragging bitmap for the specified tree view item.
PyObject *PyCTreeCtrl_CreateDragImage( PyObject *self, PyObject *args )
{
	CTreeCtrl *pList = pList=GetTreeCtrl(self);
	if (!pList) return NULL;
	HTREEITEM item;
	// @pyparm HTREEITEM|item||The item to edit.
	if (!PyArg_ParseTuple( args, "i:CreateDragImage", &item))
		return NULL;
	CImageList *pIL = pList->CreateDragImage(item);
	if (pIL==NULL)
		RETURN_ERR("CreateDragImage failed");
	return ui_assoc_object::make(PyCImageList::type, pIL)->GetGoodRet();
}


// @object PyCTreeCtrl|A class which encapsulates an MFC CTreeCtrl object.  Derived from a <o PyCWnd> object.
static struct PyMethodDef PyCTreeCtrl_methods[] = {
	// Same order as MFC doco.
	{"GetCount",        PyCTreeCtrl_GetCount,  1}, // @pymeth GetCount|Retrieves the number of tree items associated with a tree view control.
	{"GetIndent",      PyCTreeCtrl_GetIndent,  1}, // @pymeth GetIndent|Retrieves the offset (in pixels) of a tree view item from its parent.
	{"SetIndent",      PyCTreeCtrl_SetIndent,  1}, // @pymeth SetIndent|Sets the offset (in pixels) of a tree view item from its parent.
	{"GetImageList",   PyCTreeCtrl_GetImageList,  1}, // @pymeth GetImageList|Retrieves the current image list.
	{"SetImageList",   PyCTreeCtrl_SetImageList, 1}, // @pymeth SetImageList|Assigns an image list to a list view control.
	{"GetNextItem",    PyCTreeCtrl_GetNextItem,  1}, // @pymeth GetNextItem|Retrieves the next item.
	{"ItemHasChildren",PyCTreeCtrl_ItemHasChildren,  1}, // @pymeth ItemHasChildren|Returns nonzero if the specified item has child items.
	{"GetChildItem",   PyCTreeCtrl_GetChildItem,  1}, // @pymeth GetChildItem|Retrieves the child item of the specified tree view item.
	{"GetNextSiblingItem",PyCTreeCtrl_GetNextSiblingItem,  1}, // @pymeth GetNextSiblingItem|Retrieves the next sibling of the specified tree view item.
	{"GetPrevSiblingItem",PyCTreeCtrl_GetPrevSiblingItem,  1}, // @pymeth GetPrevSiblingItem|Retrieves the previous sibling of the specified tree view item.
	{"GetParentItem",PyCTreeCtrl_GetParentItem,  1}, // @pymeth GetParentItem|Retrieves the parent item of the specified tree view item.
	{"GetFirstVisibleItem",PyCTreeCtrl_GetFirstVisibleItem,  1}, // @pymeth GetFirstVisibleItem|Retrieves the first visible item of the specified tree view item.
	{"GetNextVisibleItem",PyCTreeCtrl_GetNextVisibleItem,  1}, // @pymeth GetNextVisibleItem|Retrieves the next visible item of the specified tree view item.
	{"GetPrevVisibleItem",PyCTreeCtrl_GetPrevVisibleItem,  1}, // @pymeth GetNextVisibleItem|Retrieves the previous visible item of the specified tree view item.
	{"GetSelectedItem",PyCTreeCtrl_GetSelectedItem,  1}, // @pymeth GetSelectedItem|Retrieves the currently selected tree view item.
	{"GetDropHilightItem",PyCTreeCtrl_GetDropHilightItem,  1}, // @pymeth GetDropHilightItem|Retrieves the target of a drag-and-drop operation.
	{"GetRootItem",       PyCTreeCtrl_GetRootItem,  1}, // @pymeth GetRootItem|Retrieves the root of the specified tree view item.
	{"GetItem",           PyCTreeCtrl_GetItem,  1}, // @pymeth GetItem|Retrieves the details of an items attributes.
	{"SetItem",           PyCTreeCtrl_SetItem, 1}, // @pymeth SetItem|Sets some of all of an items attributes.
	{"GetItemState",      PyCTreeCtrl_GetItemState,  1}, // @pymeth GetItemState|Retrieves the state of an item.
	{"SetItemState",      PyCTreeCtrl_SetItemState, 1}, // @pymeth SetItemState|Sets the state of an item.
	{"GetItemImage",      PyCTreeCtrl_GetItemImage,  1}, // @pymeth GetItemState|Retrieves the index of an items image.
	{"SetItemImage",      PyCTreeCtrl_SetItemImage, 1}, // @pymeth SetItemState|Sets the state of an item.
	{"SetItemText",    PyCTreeCtrl_SetItemText, 1}, // @pymeth SetItemText|Changes the text of a list view item or subitem.
	{"GetItemText",    PyCTreeCtrl_GetItemText, 1}, // @pymeth GetItemText|Retrieves the text of a list view item or subitem.
	{"GetItemData",      PyCTreeCtrl_GetItemData,  1}, // @pymeth GetItemState|Retrieves the object associated with an item.
	{"SetItemData",      PyCTreeCtrl_SetItemData, 1}, // @pymeth SetItemState|Sets the object associated with an item.
	{"GetItemRect",      PyCTreeCtrl_GetItemRect, 1}, // @pymeth GetItemRect|Retrieves the bounding rectangle of a tree view item.
	{"GetEditControl",   PyCTreeCtrl_GetEditControl, 1}, // @pymeth GetEditControl|Retrieves the handle of the edit control used to edit the specified tree view item.
	{"GetVisibleCount",   PyCTreeCtrl_GetVisibleCount, 1}, // @pymeth GetVisibleCount|Retrieves the number of visible tree items associated with a tree view control.

	{"InsertItem",     PyCTreeCtrl_InsertItem,  1}, // @pymeth InsertItem|Inserts an item into the list.
	{"DeleteItem",     PyCTreeCtrl_DeleteItem,  1}, // @pymeth DeleteItem|Deletes an item from the list.
	{"DeleteAllItems", PyCTreeCtrl_DeleteAllItems,  1}, // @pymeth DeleteAllItems|Deletes all items from the list.
	{"Expand",         PyCTreeCtrl_Expand,  1}, // @pymeth Expand|Expands, or collapses, the child items of the specified tree view item.
	{"Select",         PyCTreeCtrl_Select,  1}, // @pymeth Select|Selects, scrolls into view, or redraws a specified tree view item.
	{"SelectItem",     PyCTreeCtrl_SelectItem,  1}, // @pymeth SelectItem|Selects a specified tree view item.
	{"SelectDropTarget",PyCTreeCtrl_SelectDropTarget,  1}, // @pymeth SelectDropTarget|Redraws the tree item as the target of a drag-and-drop operation.
	{"SelectSetFirstVisible",PyCTreeCtrl_SelectSetFirstVisible,  1}, // @pymeth SelectSetFirstVisible|Selects a specified tree view item as the first visible item.
	{"EditLabel",      PyCTreeCtrl_EditLabel,  1}, // @pymeth EditLabel|Edits a specified tree view item in-place.
	{"CreateDragImage",PyCTreeCtrl_CreateDragImage,  1}, // @pymeth CreateDragImage|Creates a dragging bitmap for the specified tree view item.
	{"SortChildren",   PyCTreeCtrl_SortChildren,  1}, // @pymeth SortChildren|Sorts the children of a given parent item.
	{"EnsureVisible",  PyCTreeCtrl_EnsureVisible,  1}, // @pymeth EnsureVisible|Ensures that a tree view item is visible in its tree view control.
	{NULL,			NULL}
};
// @comm Sam Rushing has found the following tidbits:
// You can implement dynamic collapsing and expanding of events for large
// collections yourself - see KB Q130697<nl>
// The MFC docs tell you to use TVE_COLLAPSERESET in order to
// throw away the child items when collapsing a node.  They neglect to
// tell you a very important tidbit: that you need to combine the flag
// with TVE_COLLAPSE.  This is pointed out in the docs for
// TreeView_Expand(), but not in those for CTreeCtrl::Expand.

ui_type_CObject PyCTreeCtrl::type("PyCTreeCtrl", 
									 &PyCWnd::type, 
									 RUNTIME_CLASS(CTreeCtrl), 
									 sizeof(PyCTreeCtrl), 
									 PyCTreeCtrl_methods, 
									 GET_PY_CTOR(PyCTreeCtrl));
