// Pythonview.cpp - implementation of a CEditView, especially set up for
// python support.
//
// Note that this source file contains embedded documentation.
// This documentation consists of marked up text inside the
// C comments, and is prefixed with an '@' symbol.  The source
// files are processed by a tool called "autoduck" which
// generates Windows .hlp files.
// @doc

#include "stdafx.h"
#include "pythonwin.h"
#include "pythonwnd.h"
#include "pythonview.h"
#include "win32ui.h"
#include "win32dc.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CPythonView
#undef new
template <class P>
CPythonViewTemp<P>::CPythonViewTemp()
{
}

template <class P>
CPythonViewTemp<P>::CPythonViewTemp(UINT templ) :
	P(templ)
{
}

template <class P>
CPythonViewTemp<P>::CPythonViewTemp(LPCTSTR templ) :
	P(templ)
{
}

template <class P>
CPythonViewTemp<P>::~CPythonViewTemp()
{
	Python_delete_assoc(this);
}

template <class P>
BOOL CPythonViewTemp<P>::OnCmdMsg(UINT nID, int nCode,
		       void* pExtra, AFX_CMDHANDLERINFO*pHandlerInfo)
{
  // yield to Python first
  if (Python_OnCmdMsg (this, nID, nCode, pExtra, pHandlerInfo))
    return TRUE;
  else
    return P::OnCmdMsg (nID, nCode, pExtra, pHandlerInfo);
}

template <class P>
void CPythonViewTemp<P>::OnUpdate (CView* pSender, LPARAM lHint, CObject* pHint)
{
	// @pyvirtual |PyCView|OnUpdate|Called by the framework when a view needs updating.
	// @comm Typically you should not perform any drawing directly from OnUpdate. 
	// Instead, determine the rectangle describing, in device coordinates, the 
	// area that requires updating; pass this rectangle to <om PyCWnd.InvalidateRect>. 
	// You can then paint the update next <om PyCView.OnDraw>
	// @xref <om PyCView.OnUpdate>
	CVirtualHelper helper ("OnUpdate", this);
	if (!helper.HaveHandler())
		P::OnUpdate (pSender, lHint, pHint);
	else
		helper.call(pSender, (PyObject *)lHint);
}

template <class P>
BOOL CPythonViewTemp<P>::OnNotify (WPARAM wParam, LPARAM lParam, LRESULT *pResult)
{
  // yield to Python first
  if (Python_OnNotify (this, wParam, lParam, pResult))
    return TRUE;
  else
    return P::OnNotify (wParam, lParam, pResult);
}

template <class P>
LRESULT CPythonViewTemp<P>::WindowProc(UINT message, WPARAM wParam, LPARAM lParam)
{
	// @pyvirtual int|PyCView|WindowProc|Default message handler.
	LRESULT res;
	CVirtualHelper helper( "WindowProc", this );
	if (!helper.HaveHandler() || !helper.call(message, wParam, lParam) || !helper.retval(res))
		return P::WindowProc(message, wParam, lParam);
	return res;
}

template <class P>
void CPythonViewTemp<P>::OnActivateView( BOOL bActivate, CView* pActivateView, CView* pDeactiveView )
{
	// @pyvirtual |PyCView|OnActivateView|Called by the framework when a view is activated or deactivated.
	// @xref <om PyCView.OnActivateView>
	CVirtualHelper helper ("OnActivateView", this);
	if (!helper.HaveHandler())
		P::OnActivateView(bActivate, pActivateView, pDeactiveView );
	else {
		// @pyparm int|bActivate||Indicates whether the view is being activated or deactivated.
		// @pyparm <o PyCWnd>|activateView||The view object that is being activated.
		// @pyparm <o PyCWnd>|DeactivateView||The view object that is being deactivated.
		helper.call (bActivate, pActivateView, pDeactiveView);

		// @comm If a handler exists, the base MFC implementation is not called.
		// <nl>The activateView and deactiveView parameters are the same objects if the 
		// application's main frame window is activated with no change in the 
		// active viewfor example, if the focus is being transferred from 
		// another application to this one, rather than from one view to 
		// another within the application.
		// This allows a view to re-realize its palette, if needed.
	}
}

template <class P>
BOOL CPythonViewTemp<P>::OnPreparePrinting (CPrintInfo *pInfo)
{
  // default preparation
  return DoPreparePrinting (pInfo);
}

template <class P>
void CPythonViewTemp<P>::OnPrepareDC (CDC *pDC, CPrintInfo *pInfo)
{
	// @pyvirtual |PyCScrollView|OnPrepareDC|Called to prepare the device context for a view.
	// @xref <om PyCView.OnPrepareDC>
	CVirtualHelper helper ("OnPrepareDC", this);
	if (!helper.HaveHandler()) {
		P::OnPrepareDC (pDC, pInfo);
		return;
	}
	P::OnPrepareDC (pDC, pInfo);
	helper.call (pDC, pInfo);
	// @pyparm <o PyCDC>|dc||The DC object.
}

template <class P>
void CPythonViewTemp<P>::OnInitialUpdate()
{
	// @pyvirtual tuple|PyCView|OnInitialUpdate|Called before the first update for a view.
	// @comm The MFC base class is called only if no handler exists.
	// @xref <om PyCView.OnInitialUpdate>
	CVirtualHelper helper ("OnInitialUpdate", this);
	if (helper.HaveHandler())
		helper.call();
	else
		P::OnInitialUpdate();
	// pick arbitary size.  python view can change later
	// and only do it if it hasn't already been done
//	if (m_nMapMode == 0)
//		SetScrollSizes (MM_TEXT, CSize (100,100));
}

template <class P>
void CPythonViewTemp<P>::OnDraw( CDC *pDC )
{
	// @pyvirtual |PyCView|OnDraw|Called when the view should be drawn.
	CVirtualHelper helper( "OnDraw", this );
	// @pyparm <o PyCDC>|dc||The DC object.
	helper.call(pDC);
}

template <class P>
BOOL CPythonViewTemp<P>::PreCreateWindow(CREATESTRUCT &cs)
{
	// @pyvirtual BOOL|PyCScrollView|PreCreateWindow|Called by the framework before the creation of the Windows window attached to this View object.
	// @pyparm tuple|CREATESTRUCT||A tuple describing a CREATESTRUCT structure.
	// @xref <om PyCView.OnInitialUpdate>
	CVirtualHelper helper("PreCreateWindow", this);
	if (helper.HaveHandler()) {
		if (!helper.call(&cs) || !helper.retval(cs))
			return FALSE;
		return TRUE;
	} else {
		return P::PreCreateWindow(cs);
	}
}


IMPLEMENT_DYNCREATE(CPythonView, CScrollView)

CPythonView::CPythonView()
{
}

CPythonView::~CPythonView()
{
}

BOOL CPythonView::OnPreparePrinting (CPrintInfo *pInfo)
{
  // default preparation
  return DoPreparePrinting (pInfo);
}

void CPythonView::OnPrepareDC (CDC *pDC, CPrintInfo *pInfo)
{
	// @pyvirtual |PyCView|OnPrepareDC|Called to prepare the device context for a view.
	// @xref <om PyCView.OnPrepareDC>
	CVirtualHelper helper ("OnPrepareDC", this);
	helper.call (pDC, pInfo);
	if (m_nMapMode == 0) {
		// base class will ASSERT
		Python_set_error("Must call SetScrollSizes() or SetScaleToFitSize() before painting scroll view.");
		return;
	}
	CScrollView::OnPrepareDC (pDC, pInfo);
	// @pyparm <o PyCDC>|dc||The DC object.
}


BEGIN_MESSAGE_MAP(CPythonView, CScrollView)
	//{{AFX_MSG_MAP(CPythonView)
	ON_WM_PALETTECHANGED()
	ON_WM_PALETTEISCHANGING()
	ON_WM_WININICHANGE()
	ON_WM_SYSCOLORCHANGE()
	ON_WM_ERASEBKGND()
	ON_WM_GETMINMAXINFO()
	ON_WM_QUERYNEWPALETTE()
	//}}AFX_MSG_MAP
	ON_COMMAND(ID_FILE_PRINT, CView::OnFilePrint)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CPythonView message handlers
BOOL
CPythonView::SetDynamicScrollBars (BOOL dynamic)
{
  BOOL old = m_bInsideUpdate;

  // Prevent MFC from hiding/showing scrollbars by setting recursive
  // protection variable.
  if (dynamic)
    m_bInsideUpdate = FALSE;
  else
    m_bInsideUpdate = TRUE;

  return (old);
}

void CPythonView::OnPaletteChanged(CWnd* pFocusWnd) 
{
	CScrollView::OnPaletteChanged(pFocusWnd);
	PyWnd_OnPaletteChanged(this, pFocusWnd);
}

void CPythonView::OnPaletteIsChanging(CWnd* pRealizeWnd) 
{
	CScrollView::OnPaletteIsChanging(pRealizeWnd);
	PyWnd_OnPaletteIsChanging(this, pRealizeWnd);
}


void CPythonView::OnWinIniChange(LPCTSTR lpszSection) 
{
	CScrollView::OnWinIniChange(lpszSection);
	PyWnd_OnWinIniChange(this, lpszSection);
}

void CPythonView::OnSysColorChange() 
{
	CScrollView::OnSysColorChange();
	PyWnd_OnSysColorChange(this);
}

BOOL CPythonView::OnQueryNewPalette() 
{
	// @pyvirtual BOOL|PyCScrollView|OnQueryNewPalette|Informs the window it is about to receive input focus, and shoudl realize its logical palette.
	// @comm The base class method must be called manually via <om PyCScrollView.OnQueryNewPalette>
	// @xref <om PyCScrollView.OnQueryNewPalette>
	BOOL ret;
	CVirtualHelper helper( "OnQueryNewPalette", this );
	if (helper.call() && helper.retval(ret))
		return ret;
	return FALSE;
}

//////////////////////////////////////////////////////////////////////////////////
//
// CPythonListView
//
IMPLEMENT_DYNCREATE(CPythonListView, CListView)

CPythonListView::CPythonListView()
{
}
CPythonListView::~CPythonListView()
{
}

void CPythonListView::DrawItem( LPDRAWITEMSTRUCT lpDIS )
{
	CVirtualHelper helper("DrawItem", this);
	PyObject *obData = PyWin_GetPythonObjectFromLong(lpDIS->itemData);
	if (obData==NULL) {
		gui_print_error();
		PyErr_SetString(ui_module_error, "DrawItem could not convert the Python object");
		gui_print_error();
		obData = Py_None;
	}

	// Get the MFC device context
	CDC *pDC = CDC::FromHandle(lpDIS->hDC);
	PyObject *obDC = ui_dc_object::make(ui_dc_object::type, pDC);
	if (obDC==NULL) {
		gui_print_error();
		PyErr_SetString(ui_module_error, "DrawItem could not convert the DC object");
		gui_print_error();
		obDC = Py_None;
	}

	PyObject *args = Py_BuildValue("iiiiiiO(iiii)O",
			lpDIS->CtlType, lpDIS->CtlID, lpDIS->itemID,
			lpDIS->itemAction, lpDIS->itemState, lpDIS->hwndItem,
			obDC, 
			lpDIS->rcItem.left, lpDIS->rcItem.top, lpDIS->rcItem.right, lpDIS->rcItem.bottom,
			obData);
	ASSERT(args);
	if (!args) {
		gui_print_error();
		PyErr_SetString(ui_module_error, "DrawItem could not convert args - handler not called.");
		return; // not too much we can do
	}
	// make the call.
	helper.call_args(args);
	// Cleanup.
	Py_DECREF(args);
	// The DC is no longer valid.
	Python_delete_assoc(pDC);
}

BEGIN_MESSAGE_MAP(CPythonListView, CListView)
	//{{AFX_MSG_MAP(CPythonListView)
	ON_WM_PALETTECHANGED()
	ON_WM_PALETTEISCHANGING()
	ON_WM_WININICHANGE()
	ON_WM_SYSCOLORCHANGE()
	ON_WM_ERASEBKGND()
	ON_WM_GETMINMAXINFO()
	ON_WM_QUERYNEWPALETTE()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CPythonListView message handlers
void CPythonListView::OnPaletteChanged(CWnd* pFocusWnd) 
{
	CListView::OnPaletteChanged(pFocusWnd);
	PyWnd_OnPaletteChanged(this, pFocusWnd);
}

void CPythonListView::OnPaletteIsChanging(CWnd* pRealizeWnd) 
{
	CListView::OnPaletteIsChanging(pRealizeWnd);
	PyWnd_OnPaletteIsChanging(this, pRealizeWnd);
}


void CPythonListView::OnWinIniChange(LPCTSTR lpszSection) 
{
	CListView::OnWinIniChange(lpszSection);
	PyWnd_OnWinIniChange(this, lpszSection);
}

void CPythonListView::OnSysColorChange() 
{
	CListView::OnSysColorChange();
	PyWnd_OnSysColorChange(this);
}

BOOL CPythonListView::OnQueryNewPalette() 
{
	// @pyvirtual BOOL|PyCListView|OnQueryNewPalette|Informs the window it is about to receive input focus, and shoudl realize its logical palette.
	// @comm The base class method must be called manually via <om PyCView.OnQueryNewPalette>
	// @xref <om PyCListView.OnQueryNewPalette>
	BOOL ret;
	CVirtualHelper helper( "OnQueryNewPalette", this );
	if (helper.call() && helper.retval(ret))
		return ret;
	return FALSE;
}

//////////////////////////////////////////////////////////////////////////////////
//
// CPythonTreeView
//
IMPLEMENT_DYNCREATE(CPythonTreeView, CTreeView)

CPythonTreeView::CPythonTreeView()
{
}
CPythonTreeView::~CPythonTreeView()
{
}

void CPythonTreeView::DrawItem( LPDRAWITEMSTRUCT lpDIS )
{
	CVirtualHelper helper("DrawItem", this);
	PyObject *obData = PyWin_GetPythonObjectFromLong(lpDIS->itemData);
	if (obData==NULL) {
		gui_print_error();
		PyErr_SetString(ui_module_error, "DrawItem could not convert the Python object");
		gui_print_error();
		obData = Py_None;
	}

	// Get the MFC device context
	CDC *pDC = CDC::FromHandle(lpDIS->hDC);
	PyObject *obDC = ui_dc_object::make(ui_dc_object::type, pDC);
	if (obDC==NULL) {
		gui_print_error();
		PyErr_SetString(ui_module_error, "DrawItem could not convert the DC object");
		gui_print_error();
		obDC = Py_None;
	}

	PyObject *args = Py_BuildValue("iiiiiiO(iiii)O",
			lpDIS->CtlType, lpDIS->CtlID, lpDIS->itemID,
			lpDIS->itemAction, lpDIS->itemState, lpDIS->hwndItem,
			obDC, 
			lpDIS->rcItem.left, lpDIS->rcItem.top, lpDIS->rcItem.right, lpDIS->rcItem.bottom,
			obData);
	ASSERT(args);
	if (!args) {
		gui_print_error();
		PyErr_SetString(ui_module_error, "DrawItem could not convert args - handler not called.");
		return; // not too much we can do
	}
	// make the call.
	helper.call_args(args);
	// Cleanup.
	Py_DECREF(args);
	// The DC is no longer valid.
	Python_delete_assoc(pDC);
}

BEGIN_MESSAGE_MAP(CPythonTreeView, CTreeView)
	//{{AFX_MSG_MAP(CPythonTreeView)
	ON_WM_PALETTECHANGED()
	ON_WM_PALETTEISCHANGING()
	ON_WM_WININICHANGE()
	ON_WM_SYSCOLORCHANGE()
	ON_WM_ERASEBKGND()
	ON_WM_GETMINMAXINFO()
	ON_WM_QUERYNEWPALETTE()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CPythonTreeView message handlers
void CPythonTreeView::OnPaletteChanged(CWnd* pFocusWnd) 
{
	CTreeView::OnPaletteChanged(pFocusWnd);
	PyWnd_OnPaletteChanged(this, pFocusWnd);
}

void CPythonTreeView::OnPaletteIsChanging(CWnd* pRealizeWnd) 
{
	CTreeView::OnPaletteIsChanging(pRealizeWnd);
	PyWnd_OnPaletteIsChanging(this, pRealizeWnd);
}


void CPythonTreeView::OnWinIniChange(LPCTSTR lpszSection) 
{
	CTreeView::OnWinIniChange(lpszSection);
	PyWnd_OnWinIniChange(this, lpszSection);
}

void CPythonTreeView::OnSysColorChange() 
{
	CTreeView::OnSysColorChange();
	PyWnd_OnSysColorChange(this);
}

BOOL CPythonTreeView::OnQueryNewPalette() 
{
	// @pyvirtual BOOL|PyCTreeView|OnQueryNewPalette|Informs the window it is about to receive input focus, and shoudl realize its logical palette.
	// @comm The base class method must be called manually via <om PyCView.OnQueryNewPalette>
	// @xref <om PyCTreeView.OnQueryNewPalette>
	BOOL ret;
	CVirtualHelper helper( "OnQueryNewPalette", this );
	if (helper.call() && helper.retval(ret))
		return ret;
	return FALSE;
}

//////////////////////////////////////////////////////////////////////////////////
//
// CPythonFormView
//
IMPLEMENT_DYNAMIC(CPythonFormView, CFormView)

CPythonFormView::CPythonFormView(UINT templ) :
	CPythonViewTemp<CFormView>(templ)
{
}
	CPythonFormView::CPythonFormView(LPCTSTR templ) :
	CPythonViewTemp<CFormView>(templ)
{
}

BEGIN_MESSAGE_MAP(CPythonFormView, CFormView)
	//{{AFX_MSG_MAP(CPythonListView)
	ON_WM_PALETTECHANGED()
	ON_WM_PALETTEISCHANGING()
	ON_WM_WININICHANGE()
	ON_WM_SYSCOLORCHANGE()
	ON_WM_ERASEBKGND()
	ON_WM_GETMINMAXINFO()
	ON_WM_QUERYNEWPALETTE()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CPythonFormView message handlers
void CPythonFormView::OnPaletteChanged(CWnd* pFocusWnd) 
{
	CFormView::OnPaletteChanged(pFocusWnd);
	PyWnd_OnPaletteChanged(this, pFocusWnd);
}

void CPythonFormView::OnPaletteIsChanging(CWnd* pRealizeWnd) 
{
	CFormView::OnPaletteIsChanging(pRealizeWnd);
	PyWnd_OnPaletteIsChanging(this, pRealizeWnd);
}

void CPythonFormView::OnWinIniChange(LPCTSTR lpszSection) 
{
	CFormView::OnWinIniChange(lpszSection);
	PyWnd_OnWinIniChange(this, lpszSection);
}

void CPythonFormView::OnSysColorChange() 
{
	CFormView::OnSysColorChange();
	PyWnd_OnSysColorChange(this);
}

BOOL CPythonFormView::OnQueryNewPalette() 
{
	// @pyvirtual BOOL|PyCFormView|OnQueryNewPalette|Informs the window it is about to receive input focus, and shoudl realize its logical palette.
	// @comm The base class method must be called manually via <om PyCView.OnQueryNewPalette>
	// @xref <om PyCFormView.OnQueryNewPalette>
	BOOL ret;
	CVirtualHelper helper( "OnQueryNewPalette", this );
	if (helper.call() && helper.retval(ret))
		return ret;
	return FALSE;
}
