/*

	win32 window data type

	Created July 1994, Mark Hammond (MHammond@skippinet.com.au)

Note that this source file contains embedded documentation.
This documentation consists of marked up text inside the
C comments, and is prefixed with an '@' symbol.  The source
files are processed by a tool called "autoduck" which
generates Windows .hlp files.
@doc

*/
#include "stdafx.h"

#include "win32win.h"
#include "win32doc.h"
#include "win32dc.h"
#include "win32control.h"
#include "win32toolbar.h"
#include "win32menu.h"
#include "win32gdi.h"
#include "win32font.h"
#include "win32cmdui.h"

static char *szErrMsgBadHandle = "The window handle does not specify a valid window";

// Beat protected members!
class WndHack : public CWnd {
public:
	LRESULT DefWindowProc(UINT message, WPARAM wParam, LPARAM lParam)
	{ return CWnd::DefWindowProc(message, wParam, lParam);}
	BOOL OnWndMsg(UINT message, WPARAM wParam, LPARAM lParam, LRESULT* pResult)
	{ return CWnd::OnWndMsg(message, wParam, lParam, pResult);}
};


BOOL Python_check_message(const MSG *msg)	// TRUE if fully processed.
{
	ui_assoc_object *pObj;
	PyObject *method;
	CWnd *pWnd = CWnd::FromHandlePermanent(msg->hwnd);
	if (pWnd &&
		(pObj=ui_assoc_object::GetPyObject(pWnd)) && 
		pObj->is_uiobject( &PyCWnd::type ) &&
		((PyCWnd *)pObj)->pMessageHookList && 
		((PyCWnd *)pObj)->pMessageHookList->Lookup(msg->message,(void *&)method)) {

#ifdef TRACE_CALLBACKS
		TRACE("Message callback: message %04X, object %s (hwnd %p) (%p)\n",msg->message, (const char *)GetReprText(pObj), pWnd, pWnd->GetSafeHwnd());
#endif
		// Our Python convention is TRUE means "pass it on"
		return Python_callback(method, msg)==0;
	}
	return FALSE;	// dont want it.
}

BOOL Python_check_key_message(const MSG *msg)
{
	ui_assoc_object *pObj;
	BOOL bPassOn = TRUE;
	CWnd *pWnd = msg->hwnd ? CWnd::FromHandlePermanent(msg->hwnd) : NULL;
	if (pWnd && (pObj=ui_assoc_object::GetPyObject(pWnd)) &&
		 pObj->is_uiobject( &PyCWnd::type ))
		bPassOn = ((PyCWnd*)pObj)->check_key_stroke(msg->wParam);
	return !bPassOn;
}

// WARNING - the return ptr may be temporary.
CWnd *GetWndPtrFromParam(PyObject *ob, ui_type_CObject &type)
{
	if (PyInt_Check(ob)) {
		HWND hwnd = (HWND)PyInt_AsLong(ob);
		if (!IsWindow(hwnd))
			RETURN_ERR(szErrMsgBadHandle);
		CWnd *ret = CWnd::FromHandle(hwnd);
		if (ret==NULL)
			RETURN_ERR("The handle could not be converted to a window (CWnd::FromHandle() failed!)");
		return ret;
	} else if (ui_base_class::is_uiobject(ob, &type)) {
		return (CWnd *)PyCWnd::GetPythonGenericWnd(ob, &type);
	} else {
		char buf[128];
		wsprintf(buf,"Argument must be a %s object, or integer containing a HWND", type.tp_name);
		RETURN_ERR(buf);
	}
}

CWnd *GetWndPtr(PyObject *self)
{
	return (CWnd *)PyCWnd::GetPythonGenericWnd(self);
}
CWnd *GetWndPtrGoodHWnd(PyObject *self)
{
	CWnd *ret = PyCWnd::GetPythonGenericWnd(self);
	if (ret && ret->m_hWnd && !::IsWindow(ret->m_hWnd)) {
		RETURN_ERR(szErrMsgBadHandle);
	}
	return ret;
}

CFrameWnd *GetFramePtr(PyObject *self)
{
	return (CFrameWnd *)PyCWnd::GetPythonGenericWnd(self, &PyCFrameWnd::type);
}
CPythonFrame *GetPythonFrame(PyObject *self)
{
	return (CPythonFrame *)PyCWnd::GetPythonGenericWnd(self, &PyCFrameWnd::type);
}
CMDIFrameWnd *GetMDIFrame(PyObject *self)
{
	return (CMDIFrameWnd *)PyCWnd::GetPythonGenericWnd(self, &PyCMDIFrameWnd::type);
}

// @pymethod |PyCWnd|SCROLLINFO tuple|Describes a SCROLLINFO tuple
// @pyparm int|addnMask||Additional mask information.  Python automatically fills the mask for valid items, so currently the only valid values are zero, and win32con.SIF_DISABLENOSCROLL.
// @pyparm int|min||The minimum scrolling position.  Both min and max, or neither, must be provided.
// @pyparm int|max||The maximum scrolling position.  Both min and max, or neither, must be provided.
// @pyparm int|page||Specifies the page size. A scroll bar uses this value to determine the appropriate size of the proportional scroll box.
// @pyparm int|pos||Specifies the position of the scroll box.
// @pyparm int|trackPos||Specifies the immediate position of a scroll box that the user 
// is dragging. An application can retrieve this value while processing 
// the SB_THUMBTRACK notification message. An application cannot set 
// the immediate scroll position; the <om PyCWnd.SetScrollInfo> function ignores 
// this member.
// @comm When passed to Python, will always be a tuple of size 6, and items may be None if not available.
// @comm When passed from Python, it must have the addn mask attribute, but all other items may be None, or not exist.
// <nl>userob is any Python object at all, but no reference count is kept, so you must ensure the object remains referenced throught the lists life.
BOOL ParseSCROLLINFOTuple( PyObject *args, SCROLLINFO *pInfo)
{
	PyObject *ob;
	int len = PyTuple_Size(args);
	if (len<1 || len > 5) {
		PyErr_SetString(PyExc_TypeError, "SCROLLINFO tuple has invalid size");
		return FALSE;
	}
	PyErr_Clear(); // clear any errors, so I can detect my own.
	// 0 - mask.
	if ((ob=PyTuple_GetItem(args, 0))==NULL)
		return FALSE;
	pInfo->fMask = (UINT)PyInt_AsLong(ob);
	// 1/2 - nMin/nMax
	if (len==2) {
		PyErr_SetString(PyExc_TypeError, "SCROLLINFO - Both min and max, or neither, must be provided.");
		return FALSE;
	}
	if (len<3) return TRUE;
	if ((ob=PyTuple_GetItem(args, 1))==NULL)
		return FALSE;
	if (ob != Py_None) {
		pInfo->fMask |= SIF_RANGE;
		pInfo->nMin = PyInt_AsLong(ob);
		if ((ob=PyTuple_GetItem(args, 2))==NULL)
			return FALSE;
		pInfo->nMax = PyInt_AsLong(ob);
	}
	// 3 == nPage.
	if (len<4) return TRUE;
	if ((ob=PyTuple_GetItem(args, 3))==NULL)
		return FALSE;
	if (ob != Py_None) {
		pInfo->fMask |=SIF_PAGE;
		pInfo->nPage = PyInt_AsLong(ob);
	}
	// 4 == nPos
	if (len<5) return TRUE;
	if ((ob=PyTuple_GetItem(args, 4))==NULL)
		return FALSE;
	if (ob != Py_None) {
		pInfo->fMask |=SIF_POS;
		pInfo->nPos = PyInt_AsLong(ob);
	}
	// 5 == trackpos
	if (len<6) return TRUE;
	if ((ob=PyTuple_GetItem(args, 5))==NULL)
		return FALSE;
	if (ob != Py_None) {
		pInfo->nTrackPos = PyInt_AsLong(ob);
	}
	return TRUE;
}

PyObject *MakeSCROLLINFOTuple(SCROLLINFO *pInfo)
{
	PyObject *ret = PyTuple_New(6);
	if (ret==NULL) return NULL;
	PyTuple_SET_ITEM(ret, 0, PyInt_FromLong(0));
	if (pInfo->fMask & SIF_RANGE) {
		PyTuple_SET_ITEM(ret, 1, PyInt_FromLong(pInfo->nMin));
		PyTuple_SET_ITEM(ret, 2, PyInt_FromLong(pInfo->nMax));
	} else {
		Py_INCREF(Py_None);
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 1, Py_None);
		PyTuple_SET_ITEM(ret, 2, Py_None);
	}
	if (pInfo->fMask & SIF_PAGE) {
		PyTuple_SET_ITEM(ret, 3, PyInt_FromLong(pInfo->nPage));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 3, Py_None);
	}
	if (pInfo->fMask & SIF_POS) {
		PyTuple_SET_ITEM(ret, 4, PyInt_FromLong(pInfo->nPos));
	} else {
		Py_INCREF(Py_None);
		PyTuple_SET_ITEM(ret, 4, Py_None);
	}
	PyTuple_SET_ITEM(ret, 5, PyInt_FromLong(pInfo->nTrackPos));
	return ret;
}

/////////////////////////////////////////////////////////////////////
//
// win32ui methods that deal with windows.
//
//////////////////////////////////////////////////////////////////////

// @pymethod <o PyCWnd>|win32ui|CreateWindowFromHandle|Creates a <o PyCWnd> from an integer containing a HWND
PyObject *
PyCWnd::CreateWindowFromHandle(PyObject *self, PyObject *args)
{
	int hwnd;
	if (!PyArg_ParseTuple(args, "i:CreateWindowFromHandle",
		    &hwnd)) // @pyparm int|hwnd||The window handle.
		return NULL;
	CWnd *pWnd = CWnd::FromHandle((HWND)hwnd);
	if (pWnd==NULL)
		RETURN_ERR("The window handle is invalid.");
	return PyCWnd::make( UITypeFromCObject(pWnd), pWnd)->GetGoodRet();
}

// @pymethod <o PyCWnd>|win32ui|CreateControl|Creates an OLE control.
PyObject *
PyCWnd::CreateControl(PyObject *self, PyObject *args)
{
	USES_CONVERSION;
	PyObject *parent = Py_None;
	int id;
	int style;
	CRect rect(0,0,0,0);
	PyObject *obPersist = Py_None;
	int bStorage = FALSE;
	const char *szClass, *szWndName, *szLicKey = NULL;
	if (!PyArg_ParseTuple(args, "szi(iiii)Oi|Oiz", 
	          &szClass,   // @pyparm string|classId||The class ID for the window.
	          &szWndName, // @pyparm string|windowName||The title for the window.
	          &style,     // @pyparm int|style||The style for the control.
			  // @pyparm (left, top, right, bottom)|rect||The default position of the window.
	          &rect.left, &rect.top, &rect.right, &rect.bottom,
	          &parent,    // @pyparm <o PyCWnd>|parent||The parent window
	          &id,        // @pyparm int|id||The child ID for the view
			  &obPersist, // @pyparm object|obPersist|None|Place holder for future support.
			  &bStorage,  // @pyparm int|bStorage|FALSE|Not used.
			  &szLicKey ))// @pyparm string|licKey|None|The licence key for the control.
		return NULL;

	CLSID clsid;
	HRESULT hr = AfxGetClassIDFromString(szClass, &clsid);
	if (FAILED(hr))
		RETURN_ERR("The CLSID is invalid");

	CWnd *pWnd = new CWnd;
	if (!ui_base_class::is_uiobject(parent, &PyCWnd::type))
		RETURN_TYPE_ERR("Argument must be a PyCWnd");
	CWnd *pWndParent = GetWndPtr( parent );
	if (pWnd==NULL || pWndParent==NULL)
		return NULL;
	if (!pWnd->CreateControl(clsid, szWndName, style, rect, pWndParent, id, NULL, bStorage, T2OLE(szLicKey)))
		RETURN_ERR("CreateControl failed");
	return PyCWnd::make( UITypeFromCObject(pWnd), pWnd)->GetGoodRet();
}


// @pymethod <o PyCWnd>|win32ui|GetActiveWindow|Retrieves the active window.
PyObject *
PyCWnd::GetActiveWindow(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS2(args, GetActiveWindow);
	CWnd *pWnd = CWnd::GetActiveWindow();
	if (pWnd==NULL)
		RETURN_ERR("No window is active.");
	return PyCWnd::make( UITypeFromCObject(pWnd), pWnd)->GetGoodRet();
}

// @pymethod <o PyCWnd>|win32ui|FindWindow|Searches for the specified top-level window
PyObject *
PyCWnd::FindWindow(PyObject *self, PyObject *args)
{
	char *szClassName;
	char *szWndName;
	if (!PyArg_ParseTuple(args, "zz:FindWindow",
		    &szClassName, // @pyparm string|className||The window class name to find, else None
		    &szWndName)) // @pyparm string|windowName||The window name (ie, title) to find, else None
		return NULL;
	CWnd *pWnd = CWnd::FindWindow( szClassName, szWndName );
	if (pWnd==NULL)
		RETURN_ERR("No window can be found.");
	return PyCWnd::make( UITypeFromCObject(pWnd), pWnd)->GetGoodRet();
}

// @pymethod <o PyCWnd>|win32ui|FindWindowEx|Searches for the specified top-level or child window
PyObject *
PyCWnd::FindWindowEx(PyObject *self, PyObject *args)
{
	char *szClassName;
	char *szWndName;
	PyObject *obParent;
	PyObject *obChildAfter;
	if (!PyArg_ParseTuple(args, "OOzz:FindWindowEx",
			&obParent, // @pyparm <o PyCWnd>|parentWindow||The parent whose children will be searched.  If None, the desktops window will be used.
			&obChildAfter, // @pyparm <o PyCWnd>|childAfter||The search begins with the next window in the Z order.  If None, all children are searched.
		    &szClassName, // @pyparm string|className||The window class name to find, else None
		    &szWndName)) // @pyparm string|windowName||The window name (ie, title) to find, else None
		return NULL;
	CWnd *pParent = NULL;
	if (obParent != Py_None)
		if ((pParent=GetWndPtrFromParam(obParent, PyCWnd::type))==NULL)
			return NULL;
	CWnd *pChildAfter = NULL;
	if (obChildAfter != Py_None)
		if ((pChildAfter=GetWndPtrFromParam(obChildAfter, PyCWnd::type))==NULL)
			return NULL;
	HWND hwnd = ::FindWindowEx(pParent->GetSafeHwnd(), pChildAfter->GetSafeHwnd(),
				szClassName, szWndName);
	if (hwnd==NULL)
		RETURN_ERR("No window can be found.");
	return PyCWnd::make( PyCWnd::type, NULL, hwnd)->GetGoodRet();
}

/////////////////////////////////////////////////////////////////////
//
// Window object
//
//////////////////////////////////////////////////////////////////////
PyCWnd::PyCWnd()
{
	pMessageHookList=NULL;
	pKeyHookList = NULL;
	obKeyStrokeHandler = NULL;
	bDidSubclass = FALSE;
}
PyCWnd::~PyCWnd()
{
	DoKillAssoc(TRUE);
}

BOOL PyCWnd::check_key_stroke(UINT ch)
{
	PyObject *pythonObject;
	BOOL bCallBase = TRUE;
	if (obKeyStrokeHandler!= NULL)
		bCallBase = Python_callback(obKeyStrokeHandler, (int)ch);

	if (bCallBase && pKeyHookList && pKeyHookList->Lookup((WORD)ch, (void *&)pythonObject))
		bCallBase = Python_callback(pythonObject,(int)ch);
	return bCallBase;
}

CWnd *PyCWnd::GetPythonGenericWnd(PyObject *self, ui_type_CObject *pType)
{
	return (CWnd *)ui_assoc_object::GetGoodCppObject( self, pType );
}

void PyCWnd::DoKillAssoc( BOOL bDestructing /*= FALSE*/ )
{
	free_hook_list(this,&pMessageHookList);
	free_hook_list(this,&pKeyHookList);
	Py_XDECREF(obKeyStrokeHandler);
	obKeyStrokeHandler = NULL;
	PyCCmdTarget::DoKillAssoc(bDestructing);
	if (bManualDelete || bDidSubclass) {
		CWnd *pWnd = GetWndPtr(this);	// get pointer before killing it.
		if (bDidSubclass) {
//			pWnd->Detach();
			pWnd->UnsubclassWindow();
			bDidSubclass = FALSE;
		}
// DONT detach - bDidSubclass is only logic needed.
//		if (pWnd->GetSafeHwnd()) {
//			TRACE("Warning - DoKillAssoc detaching from existing window\n");
//			pWnd->Detach();
//		}
		if (bManualDelete) {
			delete pWnd;
			bManualDelete = FALSE;
		}
	}
}
/*static*/ PyCWnd *PyCWnd::make( ui_type_CObject &makeType, CWnd *pSearch, HWND wnd /*=NULL*/ )
{
	BOOL bManualDelete = FALSE;
	BOOL bDidSubclass = FALSE;
	ASSERT(pSearch || wnd);
	if (pSearch)
		wnd = pSearch->GetSafeHwnd();
	// must have a permanent object for this window.
	BOOL bMadeNew = FALSE;
	pSearch = CWnd::FromHandlePermanent(wnd);
	if (pSearch==NULL) {
		if (!IsWindow(wnd))
			RETURN_ERR("The window can not be created as it has an invalid handle");
		CWnd *pWnd = new CWnd();	// this will except, rather than return NULL!
//		pWnd->Attach(wnd);
		pWnd->SubclassWindow(wnd);
		pSearch = pWnd;	// this is now the object we use, and will get in the future from GetWindow()
		bManualDelete = bDidSubclass = bMadeNew = TRUE;
	}
	PyCWnd *obj = (PyCWnd *)ui_assoc_object::make( makeType, pSearch );
	if (obj && bMadeNew) {
		obj->bManualDelete = bManualDelete;
		obj->bDidSubclass = bDidSubclass;
	}
	return obj;
}
///////////////////////////////////
// Python methods
//

// @pymethod |PyCWnd|ActivateFrame|Searches upwards for a parent window which has
// a frame, and activates it.
static PyObject *
ui_window_activate_frame(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	int cmdShow = SW_SHOW;
	if (!PyArg_ParseTuple(args,"|i:ActivateFrame", &cmdShow)) // @pyparm int|cmdShow|SW_SHOW|
	// The param passed to <mf CFrameWnd::ShowWindow>.  See also <om PyCWnd.ShowWindow>.
		return NULL;
	while (pWnd && !pWnd->IsKindOf(RUNTIME_CLASS(CFrameWnd)))
		pWnd = pWnd->GetParent();
	if (!pWnd)
		RETURN_ERR("The specified window does not have a parent frame window");
	GUI_BGN_SAVE;
	((CFrameWnd *)pWnd)->ActivateFrame(cmdShow); // @pyseemfc CFrameWnd|ActivateFrame
	
	GUI_END_SAVE;
	RETURN_NONE;
}

// @pymethod |PyCWnd|BringWindowToTop|Brings the window to the top of a stack of overlapping windows.
static PyObject *
ui_window_bring_window_to_top(PyObject *self, PyObject *args)
{
	// @comm This method activates pop-up, top-level, and MDI child windows. 
	// The BringWindowToTop member function should be used to uncover any window that is partially or 
	// completely obscured by any overlapping windows.<nl>
	// Calling this method is similar to calling the <om PyCWnd.SetWindowPos> method to 
	// change a window's position in the Z order. The BringWindowToTop method 
	// does not change the window style to make it a top-level window of the desktop.
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	pWnd->BringWindowToTop(); // @pyseemfc CWnd|BringWindowToTop
	RETURN_NONE;
}

// @pymethod (left, top, right, bottom)|PyCWnd|CalcWindowRect|
// Computes the size of the window rectangle based on the desired client
// rectangle size.  The resulting size can then be used as the initial
// size for the window object.
static PyObject *
ui_window_calc_window_rect(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	CRect rect;
	UINT nAdjustType = CWnd::adjustBorder;
	// @pyparm (left, top, right, bottom)|rect||The size to calculate from
	// @pyparm int|nAdjustType|adjustBorder|An enumerated type used for in-place editing. It can have the following values: CWnd::adjustBorder = 0, which 
	// means that scrollbar sizes are ignored in calculation; and CWnd::adjustOutside = 1, which means that they are added into the final
	// measurements of the rectangle.
	if (!PyArg_ParseTuple(args,"(iiii)|i:CalcWindowRect", &rect.left, &rect.top, &rect.right, &rect.bottom, &nAdjustType))
		return NULL;
	pWnd->CalcWindowRect( &rect, nAdjustType );  // @pyseemfc CWnd|CalcWindowRect
	return Py_BuildValue("(iiii)",rect.left, rect.top, rect.right, rect.bottom);
}
// @pymethod |PyCWnd|CheckRadioButton|Selects the specified radio button, and clears 
// all others in the group.
static PyObject *
ui_window_check_radio_button(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	int idFirst, idLast, idCheck;
	if (!PyArg_ParseTuple(args,"iii:CheckRadioButton", 
			&idFirst, // @pyparm int|idFirst||The identifier of the first radio button in the group.
			&idLast,  // @pyparm int|idLast||The identifier of the last radio button in the group.
			&idCheck))// @pyparm int|idCheck||The identifier of the radio button to be checked.
		return NULL;
	GUI_BGN_SAVE;
	pWnd->CheckRadioButton(idFirst, idLast, idCheck); // @pyseemfc CWnd|CheckRadioButton
	GUI_END_SAVE;
	RETURN_NONE;
}

// @pymethod int|PyCWnd|DefWindowProc|Calls the default message handler.
static PyObject *
ui_window_def_window_proc(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	int message, wparam, lparam;
	if (!PyArg_ParseTuple(args,"iii:DefWindowProc", 
			&message, // @pyparm int|message||The Windows message.
			&wparam,  // @pyparm int|idLast||The lParam for the message.
			&lparam))// @pyparm int|idCheck||The wParam for the message.
		return NULL;
	return Py_BuildValue("i", ((WndHack *)pWnd)->DefWindowProc(message, (WPARAM)wparam, (LPARAM)lparam)); // @pyseemfc CWnd|DefWindowProc
}


// @pymethod (left, top, right, bottom)|PyCWnd|ClientToScreen|Converts the client coordinates of a given point or rectangle on the display to screen coordinates.
// The new screen coordinates are relative to the upper-left corner of the system display. 
// This function assumes that the given point or rectangle is in client coordinates.
static PyObject *
ui_window_client_to_screen(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	CRect rect;
	// @pyparm (left, top, right, bottom)|rect||The client coordinates.
	if (!PyArg_ParseTuple(args,"(iiii):ClientToScreen", &rect.left, &rect.top, &rect.right, &rect.bottom))
		return NULL;
	// @pyseemfc CWnd|ClientToScreen
	pWnd->ClientToScreen( &rect );
	return Py_BuildValue("(iiii)",rect.left, rect.top, rect.right, rect.bottom);
}

// @pymethod |PyCWnd|DlgDirList|Fill a list box with a file or directory listing.
static PyObject *
ui_window_dlg_dir_list(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	char *defPath;
	int nIDListBox, nIDStaticPath, nFileType;
	if (!PyArg_ParseTuple(args,"siii:DlgDirList", 
			&defPath,        // @pyparm string|defPath||The file spec to fill the list box with
			&nIDListBox,     // @pyparm int|idListbox||The Id of the listbox control to fill.
			&nIDStaticPath,  // @pyparm int|idStaticPath||The Id of the static control used to display the current drive and directory. If idStaticPath is 0, it is assumed that no such control exists.
			&nFileType))	 // @pyparm int|fileType||Specifies the attributes of the files to be displayed. 
			                 // It can be any combination of DDL_READWRITE, DDL_READONLY, DDL_HIDDEN, DDL_SYSTEM, DDL_DIRECTORY, DDL_ARCHIVE, DDL_POSTMSGS, DDL_DRIVES or DDL_EXCLUSIVE
		return NULL;
	char pathBuf[MAX_PATH+1];
	strncpy(pathBuf, defPath, MAX_PATH);
	pathBuf[MAX_PATH] = '\0';
	int rc;
	GUI_BGN_SAVE;
	// @pyseemfc CWnd|DlgDirList
	rc = pWnd->DlgDirList( pathBuf, nIDListBox, nIDStaticPath, nFileType);
	GUI_END_SAVE;
	if (!rc)
		RETURN_ERR("DlgDirList failed");
	RETURN_NONE;
}
// @pymethod |PyCWnd|DlgDirListComboBox|Fill a combo with a file or directory listing.  See <om PyCWnd.DlgDirList> for details.
static PyObject *
ui_window_dlg_dir_list_combo(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	char *defPath;
	int nIDListBox, nIDStaticPath, nFileType;
	if (!PyArg_ParseTuple(args,"siii:DlgDirListComboBox", &defPath, &nIDListBox, &nIDStaticPath, &nFileType))
		return NULL;
	char pathBuf[MAX_PATH+1];
	strncpy(pathBuf, defPath, MAX_PATH);
	pathBuf[MAX_PATH] = '\0';
	int rc;
	GUI_BGN_SAVE;
	rc = pWnd->DlgDirListComboBox( pathBuf, nIDListBox, nIDStaticPath, nFileType);
	// @pyseemfc CWnd|DlgDirListComboBox

	GUI_END_SAVE;
	if (!rc)
		RETURN_ERR("DlgDirListComboBox failed");
	RETURN_NONE;
}
// @pymethod string|PyCWnd|DlgDirSelect|
// Retrieves the current selection from a list box. It assumes that the list box has been filled by the <om PyCWnd.DlgDirList> member function and that the selection is a drive letter, a file, or a directory name.
static PyObject *
ui_window_dlg_dir_select(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	int nIDListBox;
	// @pyparm int|idListbox||The Id of the listbox.
	if (!PyArg_ParseTuple(args,"i:DlgDirSelect", &nIDListBox))
		return NULL;
	int rc;
	char buf[MAX_PATH]
	GUI_BGN_SAVE;
	rc = pWnd->DlgDirSelect( buf, nIDListBox);
	// @pyseemfc CWnd|DlgDirSelect
	GUI_END_SAVE;
	if (!rc)
		RETURN_ERR("DlgDirSelect failed");
	return Py_BuildValue("s", buf);
}
// @pymethod string|PyCWnd|DlgDirSelectComboBox|
// Retrieves the current selection from the list box of a combo box. It assumes that the list box has been filled by the <om PyCWnd.DlgDirListComboBox> member function and that the selection is a drive letter, a file, or a directory name. 
static PyObject *
ui_window_dlg_dir_select_combo(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	int nIDListBox;
	// @pyparm int|idListbox||The Id of the combobox.
	if (!PyArg_ParseTuple(args,"i:DlgDirSelectComboBox", &nIDListBox))
		return NULL;
	int rc;
	char buf[MAX_PATH]
	GUI_BGN_SAVE;
	// @pyseemfc CWnd|DlgDirSelectComboBox
	rc = pWnd->DlgDirSelectComboBox( buf, nIDListBox);
	GUI_END_SAVE;
	if (!rc)
		RETURN_ERR("DlgDirSelectComboBox failed");
	return Py_BuildValue("s", buf);
}

// @pymethod |PyCWnd|DragAcceptFiles|Indicates that the window and children supports files dropped from file manager
static PyObject *
ui_window_drag_accept_files(PyObject *self, PyObject *args)
{
	BOOL accept = TRUE;
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	// @pyparm int|bAccept|1|A flag indicating if files are accepted.
	if (!PyArg_ParseTuple(args,"|i:DragAcceptFiles", &accept))
		return NULL;
	GUI_BGN_SAVE;
	pWnd->DragAcceptFiles(accept);
	// @pyseemfc CWnd|DragAcceptFiles

	GUI_END_SAVE;
	RETURN_NONE;
}
// @pymethod |PyCWnd|DestroyWindow|Destroy the window attached to the object.
static PyObject *
ui_window_destroy_window(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	BOOL rc;
	GUI_BGN_SAVE;
	rc=pWnd->DestroyWindow();
	GUI_END_SAVE;
	if (!rc)
		RETURN_ERR("DestroyWindow could not destroy the window");
	RETURN_NONE;
}

// @pymethod |PyCWnd|DrawMenuBar|Redraws the menu bar.  Can be called if the menu changes.
static PyObject *
ui_window_draw_menu_bar(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	pWnd->DrawMenuBar();
	RETURN_NONE;
}
// @pymethod |PyCWnd|EnableWindow|Enables or disables the window.  Typically used for dialog controls.
static PyObject *
ui_window_enable_window(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	BOOL bEnable = TRUE;
	// @pyparm int|bEnable|1|A flag indicating if the window is to be enabled or disabled.
	if (!PyArg_ParseTuple(args,"|i:EnableWindow", &bEnable))
		return NULL;
	int rc;
	GUI_BGN_SAVE;
	rc = pWnd->EnableWindow(bEnable);
	// @pyseemfc CWnd|EnableWindow
	GUI_END_SAVE;
	return Py_BuildValue("i", rc);
}

// @pymethod <o PyCWnd>|PyCWnd|GetActiveWindow|Retrieves the ActiveWindow, or None
PyObject *
PyCWnd::get_active_window(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	CHECK_NO_ARGS(args);
    CWnd *pRel = pWnd->GetActiveWindow();
	if (!pRel)
		RETURN_NONE;
	return PyCWnd::make( UITypeFromCObject(pRel), pRel)->GetGoodRet();
}

// @pymethod int|PyCWnd|GetCheckedRadioButton|Returns the ID of the checked radio button, or 0 if none is selected.
static PyObject *
ui_window_get_checked_rb(PyObject *self, PyObject *args)
{
	int idFirst, idLast;
	if (!PyArg_ParseTuple(args, "ii:GetCheckedRadioButton", 
	           &idFirst, // @pyparm int|idFirst||The Id of the first radio button in the group.
	           &idLast ))// @pyparm int|idLast||The Id of the last radio button in the group.
		return NULL;
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	// @pyseemfc CWnd|GetCheckedRadioButton
	return Py_BuildValue( "i", pWnd->GetCheckedRadioButton(idFirst, idLast));
}
// @pymethod (left, top, right, bottom)|PyCWnd|GetClientRect|Returns the client coordinates of the window.  left and top will be zero.
static PyObject *
ui_window_get_client_rect(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtr(self);
	CRect rect;
	if (!pWnd)
		return NULL;
	pWnd->GetClientRect( &rect );
	return Py_BuildValue("(iiii)",rect.left, rect.top, rect.right, rect.bottom);
}
// @pymethod <o PyCWnd>|PyCWnd|GetDlgItem|Returns a window object for the child window or control with the specified ID.
// The type of the return object will be as specific as possible, but will always
// be derived from an <o PyCWnd> object.
static PyObject *
ui_window_get_dlg_item(PyObject *self, PyObject *args)
{
	int id;
	if (!PyArg_ParseTuple(args, "i:GetDlgItem", &id )) // @pyparm int|idControl||The Id of the control to be retrieved.
		return NULL;
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	CWnd *pChild = pWnd->GetDlgItem(id);
	// @pyseemfc CWnd|GetDlgItem

	if (!pChild)
		RETURN_ERR("No dialog control with that ID");
	return PyCWnd::make( UITypeFromCObject(pChild), pChild )->GetGoodRet();
}

// @pymethod int|PyCWnd|GetDlgCtrlID|Returns the ID of this child window.
static PyObject *
ui_window_get_dlg_ctrl_id(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	// @pyseemfc CWnd|GetDlgCtrlId
	return Py_BuildValue("i", pWnd->GetDlgCtrlID());
}

// @pymethod <o PyCMenu>|PyCWnd|GetMenu|Returns the menu object for the window's menu.
static PyObject *
ui_window_get_menu(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	// @pyseemfc CWnd|GetMenu
	HMENU hMenu = ::GetMenu(pWnd->m_hWnd);
	if (hMenu==NULL)
		RETURN_ERR("The window has no menu");
	return ui_assoc_object::make(PyCMenu::type, hMenu)->GetGoodRet();
}
// @pymethod <o PyCWnd>|PyCWnd|GetParent|Returns the window's parent.
static PyObject *
ui_window_get_parent(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	// @pyseemfc CWnd|GetParent
	if (pWnd->m_hWnd==NULL)
		RETURN_ERR("The window object does not have a Windows window attached");
	CWnd *pParent = pWnd->GetParent();
	if (!pParent)
		RETURN_NONE;

	return PyCWnd::make( UITypeFromCObject(pParent), pParent )->GetGoodRet();
}
// @pymethod int|PyCWnd|GetSafeHwnd|Returns the HWnd of this window.
static PyObject *
ui_window_get_safe_hwnd(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	// @pyseemfc CWnd|GetSafeHwnd
	return Py_BuildValue("i", pWnd->GetSafeHwnd());
}

// @pymethod int|PyCWnd|GetScrollInfo|Returns information about a scroll bar
static PyObject *
ui_window_get_scroll_info (PyObject *self, PyObject *args)
{
	int nBar;
	UINT nMask = SIF_ALL;
	// @pyparm int|nBar||The scroll bar to examine.  Can be one of win32con.SB_BOTH, win32con.SB_VERT or win32con.SB_HORZ
	// @pyparm int|mask|SIF_ALL|The mask for attributes to retrieve.
	if (!PyArg_ParseTuple(args, "i|i:GetScrollInfo", &nBar, &nMask))
		return NULL;
	CWnd *pWnd = GetWndPtr(self);
	if (pWnd == NULL)
		return NULL;
	SCROLLINFO info;
	info.cbSize = sizeof(SCROLLINFO);
	info.fMask = nMask; // Is this necessary?
	if (!pWnd->GetScrollInfo(nBar, &info, nMask))
		RETURN_ERR("GetScrollInfo failed");
	return MakeSCROLLINFOTuple(&info);
}

// @pymethod int|PyCWnd|GetScrollPos|Retrieves the current position of the scroll box of a scroll bar. 
static PyObject *
ui_window_get_scroll_pos (PyObject *self, PyObject *args)
{
	int nBar;
	// @pyparm int|nBar||The scroll bar to examine.  Can be one of win32con.SB_VERT or win32con.SB_HORZ
	if (!PyArg_ParseTuple(args, "i:GetScrollPos", &nBar))
		return NULL;
	CWnd *pWnd = GetWndPtr(self);
	if (pWnd == NULL)
		return NULL;
	return PyInt_FromLong(pWnd->GetScrollPos(nBar));
	
}

// @pymethod <o PyCMenu>|PyCWnd|GetSystemMenu|Returns the menu object for the window's system menu.
static PyObject *
ui_window_get_system_menu(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtrGoodHWnd(self);
	if (!pWnd)
		return NULL;
	// @pyseemfc CWnd|GetSystemMenu
	HMENU hMenu = ::GetSystemMenu(pWnd->m_hWnd,FALSE);
	return ui_assoc_object::make( PyCMenu::type, hMenu)->GetGoodRet();
}
// @pymethod <o PyCWnd>|PyCWnd|GetTopWindow|Identifies the top-level child window in a linked list of child windows. 
PyObject *
PyCWnd::get_top_window(PyObject *self, PyObject *args)
{
	// @comm Searches for the top-level child window that belongs to this window. If this window has no children, this function returns None
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtrGoodHWnd(self);
	if (!pWnd)
		return NULL;
	CWnd *pRel = pWnd->GetTopWindow();
	if (!pRel)
		RETURN_NONE;
	// @pyseemfc CWnd|GetTopWindow
	return PyCWnd::make( UITypeFromCObject(pRel), pRel)->GetGoodRet();
	// @rdesc If no child windows exist, the value is None.
}

// @pymethod <o PyCWnd>|PyCWnd|GetWindow|Returns a window, with the specified relationship to this window.
PyObject *
PyCWnd::get_window(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtrGoodHWnd(self);
	if (!pWnd)
		return NULL;
	int type;
	// @pyparm int|type||
	// Specifies the relationship between the current and the returned window. It can take one of the following values:
	// GW_CHILD, GW_HWNDFIRST, GW_HWNDLAST, GW_HWNDNEXT, GW_HWNDPREV or GW_OWNER
	if (!PyArg_ParseTuple(args,"i:GetWindow",&type))
		return NULL;
	// @pyseemfc CWnd|GetWindow
    CWnd *pRel = pWnd->GetWindow(type);
	if (!pRel)
		RETURN_NONE;
	return PyCWnd::make( UITypeFromCObject(pRel), pRel)->GetGoodRet();
}

// @pymethod tuple|PyCWnd|GetWindowPlacement|Returns placement information about the current window.
static PyObject *
ui_window_get_window_placement(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtrGoodHWnd(self);
	if (!pWnd)
		return NULL;
	WINDOWPLACEMENT pment;
	pment.length=sizeof(pment);
	// @pyseemfc CWnd|GetWindowPlacement
	pWnd->GetWindowPlacement( &pment );
	return Py_BuildValue("(ii(ii)(ii)(iiii))",pment.flags, pment.showCmd,
										pment.ptMinPosition.x,pment.ptMinPosition.y,
										pment.ptMaxPosition.x,pment.ptMaxPosition.y,
										pment.rcNormalPosition.left, pment.rcNormalPosition.top,
										pment.rcNormalPosition.right, pment.rcNormalPosition.bottom);
}

// @pymethod (left, top, right, bottom)|PyCWnd|GetWindowRect|Returns the screen coordinates of the windows upper left corner
static PyObject *
ui_window_get_window_rect(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtrGoodHWnd(self);
	if (!pWnd)
		return NULL;
	CRect rect;
	pWnd->GetWindowRect( &rect );
	// @pyseemfc CWnd|GetWindowRect
	return Py_BuildValue("(iiii)",rect.left, rect.top, rect.right, rect.bottom);
}
// @pymethod string|PyCWnd|GetWindowText|Returns the windows text.
static PyObject *
ui_window_get_window_text(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtrGoodHWnd(self);
	if (!pWnd)
		return NULL;
	CString csText;
	// @pyseemfc CWnd|Py_BuildValue
	pWnd->GetWindowText( csText );
	return Py_BuildValue("s",(const char *)csText);
}
// @pymethod object|PyCWnd|HookKeyStroke|Hook a key stroke handler
static PyObject *
ui_window_hook_key_stroke(PyObject *self, PyObject *args)
{
	// @comm The handler object passed will be called as the application receives WM_CHAR message for the specified character code.
	// The handler will be called with 2 arguments<nl>
	// * The handler object (as per all hook functions)<nl>
	// * The keystroke being handled.<nl>
	// If the handler returns TRUE, then the keystroke will be passed on to the
	// default handler, otherwise the keystroke will be consumed.<nl>
	// Note: This handler will not be called if a <om PyCWnd.HookAllKeyStrokes> hook is in place.

	// @pyparm object|obHandler||The handler of the keystroke.  This must be a callable object.
	// @pyparm int|ch||The ID for the keystroke to be handled. 
	// This may be an ascii code, or a virtual key code.
	// @rdesc The return value is the previous handler, or None.
	PyCWnd *s = (PyCWnd *)self;
	return add_hook_list(s, args,&s->pKeyHookList);
}

// @pymethod |PyCWnd|HookAllKeyStrokes|Hook a key stroke handler for all key strokes.
static PyObject *
ui_window_hook_all_key_strokes(PyObject *self, PyObject *args)
{
	// @comm The handler object passed will be called as the application receives WM_CHAR messages.
	// The handler will be called with 2 arguments<nl>
	// * The handler object (as per all hook functions).<nl>
	// * The keystroke being handled.<nl>
	// If the handler returns TRUE, then the keystroke will be passed on to the
	// default handler, otherwise it will be consumed.<nl>
	// Note: This handler will prevent any <om PyCWnd.HookKeyStroke> hooks from being called.
	PyCWnd *s = (PyCWnd *)self;
	PyObject *obHandler;

	// @pyparm object|obHandler||The handler for the keystrokes.  This must be a callable object.
	if (!PyArg_ParseTuple(args,"O:HookAllKeyStrokes", &obHandler))
		return NULL;
	if (!PyCallable_Check(obHandler))
		RETURN_ERR("The parameter must be a callable object");
	Py_XDECREF(s->obKeyStrokeHandler);
	s->obKeyStrokeHandler = obHandler;
	Py_INCREF(s->obKeyStrokeHandler);
	RETURN_NONE;
}

// @pymethod object|PyCWnd|HookMessage|Hook a message notification handler
static PyObject *
ui_window_hook_message(PyObject *self, PyObject *args)
{
	// @comm The handler object passed will be called as the application receives messages with the specified ID.
	// Note that it is not possible for PythonWin to consume a message - it is always passed on to the default handler.
	// The handler will be called with 2 arguments<nl>
	// * The handler object (as per all hook functions).<nl>
	// * A tuple representing the message.<nl>
	// The message tuple is built with the following code:<nl>
	// Py_BuildValue("O(iiiii(ii))",meth,msg->hwnd,msg->message,msg->wParam,msg->lParam,msg->time,msg->pt.x,msg->pt.y);

	// @pyparm object|obHandler||The handler for the message notification.  This must be a callable object.
	// @pyparm int|message||The ID of the message to be handled. 
	// @rdesc The return value is the previous handler, or None.
	PyCWnd *s = (PyCWnd *)self;
	return add_hook_list(s, args,&s->pMessageHookList);
}
// @pymethod int|PyCWnd|IsChild|Determines if a given window is a child of this window.
PyObject *
ui_window_is_child(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	PyObject *ob;
	// @pyparm <o PyCWnd>|obWnd||The window to be checked
	if (!PyArg_ParseTuple(args, "O:IsChild", &ob ))
    	return NULL;
	CWnd *pTest = GetWndPtrFromParam(ob, PyCWnd::type);
	if (pTest==NULL)
		return NULL;
	// @pyseemfc CWnd|IsChild
	return Py_BuildValue("i", pWnd->IsChild(pTest) );
}

// @pymethod int|PyCWnd|IsDlgButtonChecked|Determines if a dialog button is checked.
PyObject *
ui_window_is_dlg_button_checked(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	int id;
	// @pyparm int|idCtl||The ID of the button to check.
	if (!PyArg_ParseTuple(args, "i:IsDlgButtonChecked", &id ))
		return NULL;
	// @pyseemfc CWnd|IsDlgButtonChecked
	return Py_BuildValue("i", pWnd->IsDlgButtonChecked(id) );
}
// @pymethod int|PyCWnd|IsIconic|Determines if the window is currently displayed as an icon.
static PyObject *
ui_window_is_iconic(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS2(args,IsIconic);
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	return PyInt_FromLong(pWnd->IsIconic() );
}
// @pymethod int|PyCWnd|IsZoomed|Determines if the window is currently maximised.
static PyObject *
ui_window_is_zoomed(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS2(args,IsZoomed);
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	return PyInt_FromLong(pWnd->IsZoomed() );
}

// @pymethod int|PyCWnd|IsWindowVisible|Determines if the window is currently visible.
static PyObject *
ui_window_is_window_visible(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS2(args,IsWindowVisible);
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd) return NULL;
	return PyInt_FromLong(pWnd->IsWindowVisible() );
}

// @pymethod int|PyCWnd|IsWindowEnabled|Determines if the window is currently enabled.
static PyObject *
ui_window_is_window_enabled(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS2(args,IsWindowEnabled);
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd) return NULL;
	return PyInt_FromLong(pWnd->IsWindowEnabled() );
}

// @pymethod |PyCWnd|MessageBox|Display a message box.
static PyObject *
ui_window_message_box(PyObject * self, PyObject * args)
{
  char *message;
  long style = MB_OK;
  const char *title = NULL;

	
  if (!PyArg_ParseTuple(args, "s|zl:MessageBox", 
            &message, // @pyparm string|message||The message to be displayed in the message box.
            &title,   // @pyparm string/None|title|None|The title for the message box.  If None, the applications title will be used.
            &style))  // @pyparm int|style|win32con.MB_OK|The style of the message box.
    return NULL;
  CWnd *pWnd = GetWndPtr(self);
  if (!pWnd)
    return NULL;
  int rc;
  GUI_BGN_SAVE;
  // @pyseemfc CWnd|MessageBox

  rc = pWnd->MessageBox(message, title, style);
  GUI_END_SAVE;
  return Py_BuildValue("i",rc);
  // @rdesc An integer identifying the button pressed to dismiss the dialog.
}

// @pymethod |PyCWnd|ModifyStyle|Modifies the style of a window.
static PyObject *
ui_window_modify_style(PyObject *self, PyObject *args)
{
	unsigned flags = 0;
	int add;
	int remove;
	if (!PyArg_ParseTuple(args, "ii|i:ModifyStyle", 
	          &remove,   // @pyparm int|remove||Specifies window styles to be removed during style modification.
  	          &add,// @pyparm int|add||Specifies window styles to be added during style modification.
  	          &flags)) // @pyparm int|flags|0|Flags to be passed to SetWindowPos, or zero if SetWindowPos should not be called. The default is zero. 
		return NULL;

	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	BOOL rc;
	GUI_BGN_SAVE;
	rc = pWnd->ModifyStyle(remove, add, flags );
	// @pyseemfc CWnd|ModifyStyle
	GUI_END_SAVE;
	if (!rc)
		RETURN_API_ERR("ModifyStyle");
	RETURN_NONE;
	// @comm If nFlags is nonzero, ModifyStyle calls the Windows API function ::SetWindowPos and redraws the window by combining nFlags with the following four preset flags:
	// <nl>* SWP_NOSIZE	Retains the current size.
	// <nl>* SWP_NOMOVE	Retains the current position.
	// <nl>* SWP_NOZORDER	Retains the current Z order.
	// <nl>* SWP_NOACTIVATE	Does not activate the window.
	// <nl>See also <om PyCWnd.ModifyStyleEx>
}

// @pymethod |PyCWnd|ModifyStyleEx|Modifies the extended style of a window.
static PyObject *
ui_window_modify_style_ex(PyObject *self, PyObject *args)
{
	unsigned flags = 0;
	int add;
	int remove;
	if (!PyArg_ParseTuple(args, "ii|i:ModifyStyleEx", 
	          &remove,   // @pyparm int|remove||Specifies extended window styles to be removed during style modification.
  	          &add,// @pyparm int|add||Specifies extended extended window styles to be added during style modification.
  	          &flags)) // @pyparm int|flags|0|Flags to be passed to SetWindowPos, or zero if SetWindowPos should not be called. The default is zero. 
		return NULL;

	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	BOOL rc;
	GUI_BGN_SAVE;
	rc = pWnd->ModifyStyleEx(remove, add, flags );
	// @pyseemfc CWnd|ModifyStyleEx
	GUI_END_SAVE;
	if (!rc)
		RETURN_API_ERR("ModifyStyle");
	RETURN_NONE;
	// @comm If nFlags is nonzero, ModifyStyleEx calls the Windows API function ::SetWindowPos and redraws the window by combining nFlags with the following four preset flags:
	// <nl>* SWP_NOSIZE	Retains the current size.
	// <nl>* SWP_NOMOVE	Retains the current position.
	// <nl>* SWP_NOZORDER	Retains the current Z order.
	// <nl>* SWP_NOACTIVATE	Does not activate the window.
	// <nl>See also <om PyCWnd.ModifyStyle>
}


// @pymethod |PyCWnd|MoveWindow|Move a window to a new location.
static PyObject *
ui_window_move_window(PyObject *self, PyObject *args)
{
	CRect rect;
	BOOL bRepaint= TRUE;
	if (!PyArg_ParseTuple(args, "(iiii)|i:MoveWindow", 
	          // @pyparm (left, top, right, bottom)|rect||The new location of the window, relative to the parent.
	          &rect.left, &rect.top, &rect.right, &rect.bottom, 
	          &bRepaint)) // @pyparm int|bRepaint|1|Indicates if the window should be repainted after the move.
		return NULL;

	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	GUI_BGN_SAVE;
	pWnd->MoveWindow(rect, bRepaint);
	// @pyseemfc CWnd|MoveWindow
	GUI_END_SAVE;
	RETURN_NONE;
}

// @pymethod (int,int)|PyCWnd|OnWndMsg|Calls the default MFC Window Message handler.
static PyObject *
ui_window_on_wnd_msg(PyObject *self, PyObject *args)
{
	LRESULT res;
	int msg, wParam, lParam;
	CRect rect;
	BOOL bRepaint= TRUE;
	if (!PyArg_ParseTuple(args, "iii:OnWndMsg", 
	          &msg, // @pyparm int|msg||The message
			  (int *)&wParam, // @pyparm int|wParam||The wParam for the message
			  (int *)&lParam)) // @pyparm int|lParam||The lParam for the message
		return NULL;

	WndHack *pWnd = (WndHack *)GetWndPtr(self);
	if (!pWnd)
		return NULL;
	BOOL rc = pWnd->WndHack::OnWndMsg(msg, wParam, lParam, &res );
	// @pyseemfc CWnd|OnWndMsg
	// @rdesc The return value is a tuple of (int, int), being the
	// return value from the MFC function call, and the value of the
	// lResult param.  Please see the MFC documentation for more details.
	return Py_BuildValue("ii", rc, (int)res);
	RETURN_NONE;
}

// @pymethod |PyCWnd|PostMessage|Post a message to the window.
PyObject *
ui_window_post_message(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	UINT message;
	WPARAM wParam=0;
	LPARAM lParam=0;
	if (!PyArg_ParseTuple(args, "i|ii:PostMessage", 
	          &message, // @pyparm int|idMessage||The ID of the message to post.
	          &wParam,  // @pyparm int|wParam||The wParam for the message
	          &lParam)) // @pyparm int|lParam||The lParam for the message
		return NULL;
	// @pyseemfc CWnd|PostMessage
	if (!pWnd->PostMessage(message, wParam, lParam))
		RETURN_API_ERR("CWnd::PostMessage");
	RETURN_NONE;
}
// @pymethod (left, top, right, bottom)|PyCWnd|ScreenToClient|Converts the screen coordinates of a given point or rectangle on the display to client coordinates. 
static PyObject *
ui_window_screen_to_client(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	CRect rect;
    // @pyparm (left, top, right, bottom)|rect||The coordinates to convert.

	if (!PyArg_ParseTuple(args,"(iiii):ScreenToClient", &rect.left, &rect.top, &rect.right, &rect.bottom))
		return NULL;
	// @pyseemfc CWnd|ScreenToClient
	pWnd->ScreenToClient( &rect );
	return Py_BuildValue("(iiii)",rect.left, rect.top, rect.right, rect.bottom);
}
// @pymethod <o PyCWnd>|PyCWnd|SetActiveWindow|Sets the window active.  Returns the previously active window, or None.
static PyObject *
ui_window_set_active_window(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;

	CHECK_NO_ARGS(args);
	CWnd *pRel;
	GUI_BGN_SAVE;
    pRel = pWnd->SetActiveWindow();
	GUI_END_SAVE;
	if (!pRel)
		RETURN_NONE;
	return PyCWnd::make( UITypeFromCObject(pRel), pRel)->GetGoodRet();
}
// @pymethod |PyCWnd|SetRedraw|Allows changes to be redrawn or to prevent changes from being redrawn. 
static PyObject *
ui_window_set_redraw(PyObject *self, PyObject *args)
{
	CWnd *pView = GetWndPtr(self);
	if (!pView)
		return NULL;
	BOOL bState = TRUE;
    // @pyparm int|bState|1|Specifies the state of the redraw flag.
	if (!PyArg_ParseTuple(args, "i:SetRedraw", &bState))
		return NULL;
	GUI_BGN_SAVE;
	// @pyseemfc CWnd|SetRedraw
	pView->SetRedraw(bState);
	GUI_END_SAVE;
	RETURN_NONE;
}

// @pymethod int|PyCWnd|SetScrollInfo|Set information about a scroll bar
static PyObject *
ui_window_set_scroll_info (PyObject *self, PyObject *args)
{
	int nBar;
	BOOL bRedraw = TRUE;
	PyObject *obInfo;
	// @pyparm int|nBar||The scroll bar to examine.  Can be one of win32con.SB_BOTH, win32con.SB_VERT or win32con.SB_HORZ
	// @pyparm int|redraw|1|A flag indicating if the scrollbar should be re-drawn.
	if (!PyArg_ParseTuple(args, "iO|i:SetScrollInfo", &nBar, &obInfo, &bRedraw))
		return NULL;
	SCROLLINFO info;
	info.cbSize = sizeof(SCROLLINFO);
	if (ParseSCROLLINFOTuple(obInfo, &info))
		return NULL;
	CWnd *pWnd = GetWndPtr(self);
	if (pWnd == NULL)
		return NULL;
	if (!pWnd->SetScrollInfo(nBar, &info, bRedraw))
		RETURN_ERR("SetScrollInfo failed");
	RETURN_NONE;
}

// @pymethod int|PyCWnd|SetScrollPos|Sets the current position of the scroll box of a scroll bar. 
static PyObject *
ui_window_set_scroll_pos (PyObject *self, PyObject *args)
{
	int nBar;
	BOOL bRedraw = TRUE;
	int nPos;
	// @pyparm int|nBar||The scroll bar to set.  Can be one of win32con.SB_VERT or win32con.SB_HORZ
	// @pyparm int|nPos||The new position
	// @pyparm int|redraw|1|A flag indicating if the scrollbar should be redrawn.
	if (!PyArg_ParseTuple(args, "ii|i:SetScrollPos", &nBar, &nPos, &bRedraw))
		return NULL;
	CWnd *pWnd = GetWndPtr(self);
	if (pWnd == NULL)
		return NULL;
	return PyInt_FromLong(pWnd->SetScrollPos(nBar, nPos, bRedraw));
}



// @pymethod |PyCWnd|SetWindowPlacement|Sets the windows placement
static PyObject *
ui_window_set_window_placement(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	WINDOWPLACEMENT pment;
	pment.length=sizeof(pment);
    // @pyparm (tuple)|placement||A tuple representing the WINDOWPLACEMENT structure.
	if (!PyArg_ParseTuple(args,"ii(ii)(ii)(iiii):SetWindowPlacement",&pment.flags, &pment.showCmd,
										&pment.ptMinPosition.x,&pment.ptMinPosition.y,
										&pment.ptMaxPosition.x,&pment.ptMaxPosition.y,
										&pment.rcNormalPosition.left, &pment.rcNormalPosition.top,
										&pment.rcNormalPosition.right, &pment.rcNormalPosition.bottom))
		return NULL;
	int rc;
	GUI_BGN_SAVE;
	// @pyseemfc CWnd|SetWindowPlacement
	rc=pWnd->SetWindowPlacement( &pment );
	GUI_END_SAVE;
	if (!rc)
		RETURN_API_ERR("CWnd::SetWindowPlacement");
	RETURN_NONE;
}
// @pymethod |PyCWnd|SetWindowPos|Sets the windows position information
static PyObject *
ui_window_set_window_pos(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;

	HWND insertAfter;
	int x,y,cx,cy;
	int flags;
    // @pyparm int|hWndInsertAfter||A hwnd, else one of the win32con.HWND_* constants.
	// @pyparm (x,y,cx,cy)|position||The new position of the window.
	// @pyparm int|flags||Window positioning flags.
	if (!PyArg_ParseTuple(args,"i(iiii)i:SetWindowPos",
		        (int *)(&insertAfter), &x, &y, &cx, &cy, &flags ))
		return NULL;
	GUI_BGN_SAVE;
	// @pyseemfc CWnd|SetWindowPos
	::SetWindowPos( pWnd->GetSafeHwnd(), insertAfter, x, y, cx, cy, flags)
	GUI_END_SAVE;
	RETURN_NONE;
}

// @pymethod |PyCWnd|SetWindowText|Sets the window's text.
static PyObject *
ui_window_set_window_text(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	char *msg;
	// @pyparm string|text||The windows text.
	if (!PyArg_ParseTuple(args, "s:SetWindowText", &msg))
		return NULL;
	// @pyseemfc CWnd|SetWindowText
	pWnd->SetWindowText(msg);
	RETURN_NONE;
}

// @pymethod |PyCWnd|SendMessage|Send a message to the window.
PyObject *
ui_window_send_message(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	int message;
	int wParam=0;
	int lParam=0;
	if (!PyArg_ParseTuple(args, "i|ii:SendMessage",
		      &message, // @pyparm int|idMessage||The ID of the message to send.
	          &wParam,  // @pyparm int|wParam||The wParam for the message
	          &lParam)) // @pyparm int|lParam||The lParam for the message

		return NULL;
	int rc;
	GUI_BGN_SAVE;
	// @pyseemfc CWnd|SendMessage
	rc = pWnd->SendMessage(message, wParam, lParam);
	GUI_END_SAVE;
	return Py_BuildValue("i",rc);
}
// @pymethod |PyCWnd|SendMessageToDescendants|Send a message to all descendant windows.
PyObject *
ui_window_send_message_to_desc(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	UINT message;
	WPARAM wParam=0;
	LPARAM lParam=0;
	BOOL bDeep = TRUE;
	if (!PyArg_ParseTuple(args, "i|iii:SendMessageToDescendants", 
		      &message, // @pyparm int|idMessage||The ID of the message to send.
	          &wParam,  // @pyparm int|wParam||The wParam for the message
	          &lParam,  // @pyparm int|lParam||The lParam for the message
	          &bDeep))  // @pyparm int|bDeep|1|Indicates if the message should be recursively sent to all children
		return NULL;
	GUI_BGN_SAVE;
	// @pyseemfc CWnd|SendMessageToDescendants
	pWnd->SendMessageToDescendants(message, wParam, lParam, bDeep);
	GUI_END_SAVE;
	RETURN_NONE;
}

// @pymethod |PyCWnd|ShowScrollBar|Shows or hides a scroll bar.
// An application should not call ShowScrollBar to hide a scroll bar while processing a scroll-bar notification message.
static PyObject *
ui_window_show_scrollbar(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	int bar;
	BOOL bShow = TRUE;
	if (!PyArg_ParseTuple(args, "i|i:ShowScrollBar", 
	          &bar, // @pyparm int|nBar||Specifies whether the scroll bar is a control or part of a window's nonclient area.
			        // If it is part of the nonclient area, nBar also indicates whether the scroll bar is positioned horizontally, vertically, or both.
			        // It must be one of win32con.SB_BOTH, win32con.SB_HORZ or win32con.SB_VERT.
	          &bShow))// @pyparm int|bShow|1|Indicates if the scroll bar should be shown or hidden.

		return NULL;
	GUI_BGN_SAVE;
	// @pyseemfc CWnd|ShowScrollBar
	pWnd->ShowScrollBar(bar, bShow);
	GUI_END_SAVE;
	RETURN_NONE;
}

// @pymethod int|PyCWnd|ShowWindow|Sets the visibility state of the window.
PyObject *
ui_window_show_window(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	int style = SW_SHOWNORMAL;
	// @pyparm int|style|win32con.SW_SHOWNORMAL|Specifies how the window is to be shown.
	// It must be one of win32con.SW_HIDE, win32con.SW_MINIMIZE, win32con.SW_RESTORE, win32con.SW_SHOW, win32con.SW_SHOWMAXIMIZED
	// win32con.SW_SHOWMINIMIZED, win32con.SW_SHOWMINNOACTIVE, win32con.SW_SHOWNA, win32con.SW_SHOWNOACTIVATE,  or win32con.SW_SHOWNORMAL
	if (!PyArg_ParseTuple(args, "|i:ShowWindow", &style))
		return NULL;
	int rc;
	GUI_BGN_SAVE;
	rc = pWnd->ShowWindow(style);
	// @pyseemfc CWnd|ShowWindow
	GUI_END_SAVE;
	return Py_BuildValue("i",rc);
	// @rdesc Returns TRUE is the window was previously visible.
}
// @pymethod int|PyCWnd|UpdateData|Initialises data in a dialog box, or to retrieves and validates dialog data.
// Returns nonzero if the operation is successful; otherwise 0. If bSaveAndValidate is TRUE, then a return value of nonzero means that the data is successfully validated.
static PyObject *
ui_window_update_data(PyObject *self, PyObject *args)
{
	int bSAV = TRUE;
	// @pyparm int|bSaveAndValidate|1|Flag that indicates whether dialog box is being initialized (FALSE) or data is being retrieved (TRUE).
	if (!PyArg_ParseTuple(args, "|i:UpdateData", &bSAV))
		return NULL;
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	BOOL rc = FALSE;
	if (pWnd->m_hWnd==NULL)
		RETURN_ERR("There is no window for the dialog object - the data could not be updated.");
	if (!::IsWindow(pWnd->m_hWnd))
		RETURN_ERR("The dialog object has an invalid window");
	GUI_BGN_SAVE;
	// @pyseemfc CWnd|UpdateData
	rc = pWnd->UpdateData(bSAV);
	GUI_END_SAVE;
	return Py_BuildValue("i", rc );
}
// @pymethod |PyCWnd|UpdateWindow|Updates a window.  This forces a paint message to be sent to the window, if any part of the window is marked as invalid.
static PyObject *
ui_window_update_window(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	GUI_BGN_SAVE;
	pWnd->UpdateWindow();
	GUI_END_SAVE;
	RETURN_NONE;
}

// @pymethod |PyCWnd|InvalidateRect|Invalidates an area of a window.
static PyObject *
ui_window_invalidate_rect(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr(self);
	BOOL erase=TRUE;

	if (!pWnd)
		return NULL;
	CRect rect (CFrameWnd::rectDefault), *r;
	// @pyparm (left, top, right, bottom)|rect|(0,0,0,0)|Rectangle to be
	// updated.  If default param is used, the entire window is invalidated.
	// @pyparm int|bErase|1|Specifies whether the background within the update region is to be erased.
	if (!PyArg_ParseTuple (args,
	                      "|(iiii)i:InvalidateRect",
	                      &rect.left, &rect.top,
	                      &rect.right, &rect.bottom,
	                      &erase)) {
		return NULL;
	}
	if (rect==CFrameWnd::rectDefault)
		r = NULL;
	else
		r = &rect;
	GUI_BGN_SAVE;
	pWnd->InvalidateRect (r, erase);
	// @pyseemfc CWnd|InvalidateRect
	GUI_END_SAVE;
	RETURN_NONE;
}

// @pymethod <o PyCDC>|PyCWnd|GetDC|Gets the windows current DC object.
static PyObject *
ui_window_get_dc (PyObject *self, PyObject *args)
{
  CHECK_NO_ARGS(args);
  CWnd *pWnd = (CWnd *) GetWndPtr (self);
  if (pWnd == NULL)
    return NULL;

  // create MFC device context
  CDC *pDC = pWnd->GetDC();
  if (pDC==NULL)
    RETURN_ERR ("Could not get the DC for the window.");


  // create Python device context
  ui_dc_object *dc =
    (ui_dc_object *) ui_assoc_object::make (ui_dc_object::type, pDC)->GetGoodRet();
    return dc;
}

// @pymethod |PyCWnd|SetCapture|Causes all subsequent mouse input to be sent to the window object regardless of the position of the cursor. 
static PyObject *
ui_window_set_capture (PyObject *self, PyObject *args)
{
  CHECK_NO_ARGS(args);
  CWnd *pWnd = GetWndPtr (self);
  if (pWnd == NULL)
	return NULL;

  pWnd->SetCapture();
  RETURN_NONE;
}
// @pymethod |PyCWnd|SetFocus|Claims the input focus.  The object that previously had the focus loses it.
static PyObject *
ui_window_set_focus (PyObject *self, PyObject *args)
{
  CHECK_NO_ARGS(args);
  CWnd *pWnd = GetWndPtr (self);
  if (pWnd == NULL)
	return NULL;
  GUI_BGN_SAVE;
  pWnd->SetFocus();
  GUI_END_SAVE;
  RETURN_NONE;
}

// @pymethod |PyCWnd|SetFont|Sets the windows current font to the specified font.
static PyObject *
ui_window_set_font (PyObject * self, PyObject *args)
{
	PyObject * pfont;
	int redraw=1;
	CWnd *pWnd = GetWndPtr (self);
	if (pWnd == NULL) {
		return NULL;
	// @pyparm <o PyCFont>|font||The new font to use.
	// @pyparm int|bRedraw|1|If TRUE, redraw the window.
	} else if (!PyArg_ParseTuple (args, "O|i", &pfont, &redraw)) {
		return NULL;
	} else if (!ui_base_class::is_uiobject (pfont, &PyCFont::type)) {
		RETURN_ERR ("First argument must be a font object.");
	} else {
		pWnd->SetFont (((PyCFont *)pfont)->GetFont(), redraw);
	RETURN_NONE;
	}
}


// @pymethod |PyCWnd|SetForegroundWindow|Puts the window into the foreground and activates the window.
static PyObject *
ui_window_set_foreground_window(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS(args);
	CWnd *pWnd = GetWndPtr(self);
	if (!pWnd)
		return NULL;
	BOOL rc;
	GUI_BGN_SAVE;
	rc=pWnd->SetForegroundWindow();
	GUI_END_SAVE;
	if (!rc)
		RETURN_API_ERR("SetForegroundWindow");
	RETURN_NONE;
}

// @pymethod |PyCWnd|SetMenu|Sets the menu for a window.
PyObject *
ui_window_set_menu(PyObject *self, PyObject *args)
{
	CWnd *pWnd = GetWndPtr (self);
	if (!pWnd)
		return NULL;
	PyObject *menuObject;
	//@pyparm PyCMenu|menuObj||The menu object to set, or None to remove the window.
	if (!PyArg_ParseTuple(args, "O:SetMenu", &menuObject))
		return NULL;
	CMenu *pMenu = NULL;
	if (menuObject!=Py_None) {
		if (!ui_base_class::is_uiobject(menuObject, &PyCMenu::type))
			RETURN_TYPE_ERR("passed object must None or be a PyCMenu");
		HMENU hMenu = PyCMenu::GetMenu(menuObject);
		if (hMenu==NULL)
			return NULL;
		pMenu=CMenu::FromHandle(hMenu);
		if (pMenu==NULL)
			RETURN_TYPE_ERR("The menu object is invalid");
	}
	if (!pWnd->SetMenu(pMenu))
		RETURN_API_ERR("CWnd::SetMenu");
	RETURN_NONE;
}

// @pymethod |PyCWnd|ReleaseCapture|Releases the mouse capture for this window.  See <om PyCWnd.SetCapture>.
static PyObject *
ui_window_release_capture (PyObject *self, PyObject *args)
{
  CHECK_NO_ARGS(args);
  CWnd	*view = GetWndPtr (self);
  if (view == NULL)
	return NULL;

  if (view->GetCapture() == view)
	ReleaseCapture();
  RETURN_NONE;
}

// @pymethod |PyCWnd|ReleaseDC|Releases a device context, freeing it for use by other applications. 
static PyObject *
ui_window_release_dc (PyObject *self, PyObject *args)
{
	PyObject *obDC;
	CWnd *view = GetWndPtr (self);
	if (view == NULL)
		return NULL;

	// @pyparm <o PyCDC>|dc||The DC to be released.
	if (!PyArg_ParseTuple(args, "O:ReleaseDC", &obDC))
		return NULL;

	// Get the MFC device context
	CDC *pDC;
	if (!(pDC=ui_dc_object::GetDC(obDC)))
		return NULL;
	
	if (!view->ReleaseDC(pDC))
		RETURN_ERR("ReleaseDC failed");
	RETURN_NONE;
}

// @pymethod int|PyCWnd|MouseCaptured|Returns 1 if the window has the mouse capture, else 0
static PyObject *
ui_window_mouse_captured (PyObject *self, PyObject *args)
{
  CHECK_NO_ARGS(args);
  CWnd *view = GetWndPtr (self);
  if (view == NULL)
	return NULL;

  // Returns true if this win has the mouse captured
  return Py_BuildValue ("i", view->GetCapture() == view);
}

// @pymethod int|PyCWnd|UpdateDialogControls|Updates the state of dialog buttons and other controls in a dialog box or window that uses the <om PyCCmdUI.HookCommandUpdate> callback mechanism.
static PyObject *
ui_window_udc (PyObject *self, PyObject *args)
{
	PyObject *obTarget;
	BOOL bDisable;
	// @pyparm <o PyCCmdTarget>|pTarget||The main frame window of the application, and is used for routing update messages.
	// @pyparm int|disableIfNoHandler||Flag that indicates whether a control that has no update handler should be automatically displayed as disabled.
	if (!PyArg_ParseTuple(args, "Oi:UpdateDialogControls", &obTarget, &bDisable))
		return NULL;

	extern CCmdTarget *GetCCmdTargetPtr(PyObject *self);
	CCmdTarget *pTarget;
	if (!(pTarget=GetCCmdTargetPtr(obTarget)))
		return NULL;
	CWnd *view = GetWndPtr (self);
	if (view == NULL)
		return NULL;
	view->UpdateDialogControls(pTarget, bDisable);
	RETURN_NONE;
}

// @pymethod |PyCWnd|ShowCaret|Shows the caret
PyObject *
ui_window_show_caret(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS2(args, ShowCaret);
	CWnd *pWnd = GetWndPtr (self);
	if (pWnd == NULL)
		return NULL;
	pWnd->ShowCaret();
	RETURN_NONE;
	// @comm See also <om PyCWnd.HideCaret>
}

// @pymethod |PyCWnd|HideCaret|Hides the caret
PyObject *
ui_window_hide_caret(PyObject *self, PyObject *args)
{
	CHECK_NO_ARGS2(args, HideCaret);
	CWnd *pWnd = GetWndPtr (self);
	if (pWnd == NULL)
		return NULL;
	pWnd->HideCaret();
	RETURN_NONE;
	// @comm See also <om PyCWnd.ShowCaret>
}


///////////////////////////////////////
//
// Window Methods
//
///////////////////////////////////////
// @object PyCWnd|A base window class.  Encapsulates an MFC <c CWnd> class
static struct PyMethodDef PyCWnd_methods[] = {
	// Window Methods
	{"ActivateFrame",		ui_window_activate_frame,		1}, // @pymeth ActivateFrame|Searches upwards for a parent window which has a frame, and activates it.
	{"BringWindowToTop",	ui_window_bring_window_to_top,	1}, // @pymeth BringWindowToTop|Brings the window to the top of a stack of overlapping windows.
	{"CalcWindowRect",		ui_window_calc_window_rect,		1}, // @pymeth CalcWindowRect|Computes the size of the window rectangle based on the desired client rectangle size.
	{"CheckRadioButton",	ui_window_check_radio_button,	1}, // @pymeth CheckRadioButton|Selects a specified radio button
	{"ClientToScreen",		ui_window_client_to_screen,		1}, // @pymeth ClientToScreen|Convert coordinates from Client to Screen
	{"DefWindowProc",		ui_window_def_window_proc,		1}, // @pymeth DefWindowProc|Calls the default message handler.
	{"DestroyWindow",		ui_window_destroy_window,		1}, // @pymeth DestroyWindow|Destroys the window attached to the object.
	{"DlgDirList",			ui_window_dlg_dir_list,			1},	// @pymeth DlgDirList|Fill a listbox control with a file specification.
	{"DlgDirListComboBox",	ui_window_dlg_dir_list_combo,	1},	// @pymeth DlgDirListComboBox|Fill a combobox control with a file specification.
	{"DlgDirSelect",		ui_window_dlg_dir_select,		1}, // @pymeth DlgDirSelect|Retrieves the current selection from a list box.
	{"DlgDirSelectComboBox",ui_window_dlg_dir_select_combo,	1}, // @pymeth DlgDirSelectComboBox|Retrieves the current selection from a combo box.
	{"DragAcceptFiles",		ui_window_drag_accept_files,	1}, // @pymeth DragAcceptFiles|Indicate the window can accept files dragges from file manager.
	{"DrawMenuBar",         ui_window_draw_menu_bar,        1}, // @pymeth DrawMenuBar|Redraw the windows menu bar.
	{"EnableWindow",		ui_window_enable_window,		1}, // @pymeth EnableWindow|Enable or disable the window.
	{"GetActiveWindow",		PyCWnd::get_active_window,	1}, // @pymeth GetActiveWindow|Get the active window.
	{"GetCheckedRadioButton",ui_window_get_checked_rb,	1}, // @pymeth GetCheckedRadioButton|Get the ID of the checked a radio button in a group.
	{"GetClientRect",		ui_window_get_client_rect,		1}, // @pymeth GetClientRect|Gets the client rectangle for thewindow.
	{"GetDC",				ui_window_get_dc,				1}, // @pymeth GetDC|Gets the window's current device context.
	{"GetDlgCtrlID",		ui_window_get_dlg_ctrl_id,		1}, // @pymeth GetDlgCtrlID|Get the current window's control id.
	{"GetDlgItem",			ui_window_get_dlg_item,			1}, // @pymeth GetDlgItem|Get a child control by Id
	{"GetMenu",				ui_window_get_menu,				1}, // @pymeth GetMenu|Get the current menu for a window.
	{"GetParent",			ui_window_get_parent,			1}, // @pymeth GetParent|Get the parent window.
	{"GetSafeHwnd",         ui_window_get_safe_hwnd,        1}, // @pymeth GetSafeHwnd|Returns the HWnd of this window.
	{"GetScrollInfo",	    ui_window_get_scroll_info,		1}, // @pymeth GetScrollInfo|Retrieve information about a scroll bar
	{"GetScrollPos",        ui_window_get_scroll_pos,       1}, // @pymeth GetScrollPos|Retrieves the current position of the scroll box of a scroll bar.
	{"GetSystemMenu",		ui_window_get_system_menu,		1}, // @pymeth GetSystemMenu|Get the system menu for the window.
	{"GetTopWindow",		PyCWnd::get_top_window,1},// @pymeth GetTopWindow|Get the top level window attached to this window.
	{"GetWindow",			PyCWnd::get_window,	1}, // @pymeth GetWindow|Get a specified window (eg, parent, child, etc).
	{"GetWindowPlacement",	ui_window_get_window_placement,	1}, // @pymeth GetWindowPlacement|Gets the window's current placement information.
	{"GetWindowRect",		ui_window_get_window_rect,		1}, // @pymeth GetWindowRect|Get the windows rectangle.
	{"GetWindowText",		ui_window_get_window_text,		1}, // @pymeth GetWindowText|Get the window's current text.
	{"HideCaret",		    ui_window_hide_caret,		1}, // @pymeth HideCaret|Hides the caret
	{"HookAllKeyStrokes",   ui_window_hook_all_key_strokes, 1}, // @pymeth HookAllKeyStrokes|Hook a handler for all keystroke messages.
	{"HookKeyStroke", 		ui_window_hook_key_stroke,		1}, // @pymeth HookKeyStroke|Hook a keystroke handler.
	{"HookMessage",			ui_window_hook_message,			1}, // @pymeth HookMessage|Hook a message notification handler.
	{"InvalidateRect",		ui_window_invalidate_rect,		1},	// @pymeth InvalidateRect|Invalidate a specified rectangle in a window.
	{"IsChild",				ui_window_is_child,				1}, // @pymeth IsChild|Indicates if a window is a child.
	{"IsDlgButtonChecked",	ui_window_is_dlg_button_checked,1}, // @pymeth IsDlgButtonChecked|Indicates if a dialog botton is checked.
	{"IsIconic",			ui_window_is_iconic,			1}, // @pymeth IsIconic|Indicates if the window is currently minimised.
	{"IsZoomed",			ui_window_is_zoomed,			1}, // @pymeth IsZoomed|Indicates if the window is currently maximised.
	{"IsWindowVisible",		ui_window_is_window_visible,	1}, // @pymeth IsWindowVisible|Determines if the window is currently visible.
	{"IsWindowEnabled",		ui_window_is_window_enabled,	1}, // @pymeth IsWindowVisible|Determines if the window is currently enabled.
	{"MouseCaptured",		ui_window_mouse_captured,		1}, // @pymeth MouseCaptured|Indicates if the window currently has the mouse captured.
	{"MessageBox",			ui_window_message_box,			1}, // @pymeth MessageBox|Displays a message box.
	{"ModifyStyle",         ui_window_modify_style,         1}, // @pymeth ModifyStyle|Modifies the style of a window.
	{"ModifyStyleEx",       ui_window_modify_style_ex,      1}, // @pymeth ModifyStyleEx|Modifies the style of a window.
	{"MoveWindow",			ui_window_move_window,			1}, // @pymeth MoveWindow|Moves the window to a new location.
	{"OnWndMsg",            ui_window_on_wnd_msg,           1}, // @pymeth OnWndMsg|Calls the default MFC Window Message handler.
	{"ReleaseCapture",		ui_window_release_capture,		1}, // @pymeth ReleaseCapture|Releases the mouse capture for the window.
	{"ReleaseDC",			ui_window_release_dc,		    1}, // @pymeth ReleaseDC|Releases a device context, freeing it for use by other applications.
	{"PostMessage",			ui_window_post_message,			1}, // @pymeth PostMessage|Post a message to the window.
	{"SendMessageToDescendants",ui_window_send_message_to_desc,	1}, // @pymeth SendMessageToDescendants|Send a message to a window's children.
	{"SendMessage",			ui_window_send_message,			1}, // @pymeth SendMessage|Send a message to the window.
	{"SetActiveWindow",		ui_window_set_active_window,	1}, // @pymeth SetActiveWindow|Sets the window active.
	{"SetForegroundWindow",	ui_window_set_foreground_window,1}, // @pymeth SetForegroundWindow|Puts the window into the foreground and activates the window.
	{"SetWindowPos",        ui_window_set_window_pos,       1}, // @pymeth SetWindowPos|Sets the windows position information.
	{"ScreenToClient",		ui_window_screen_to_client,		1}, // @pymeth ScreenToClient|Converts from screen coordinates to client coordinates.
	{"SetCapture",			ui_window_set_capture,			1}, // @pymeth SetCapture|Captures the mouse input for thw window.
	{"SetFocus",			ui_window_set_focus,			1}, // @pymeth SetFocus|Sets focus to the window.
	{"SetFont",				ui_window_set_font,				1}, // @pymeth SetFont|Sets the windows current font to the specified font.
	{"SetMenu",				ui_window_set_menu,				1}, // @pymeth SetMenu|Sets the menu for a window.
	{"SetRedraw",			ui_window_set_redraw,			1}, // @pymeth SetRedraw|Sets the redraw flag for the window.
	{"SetScrollPos",	    ui_window_set_scroll_pos,		1}, // @pymeth SetScrollPos|Sets the current position of the scroll box of a scroll bar. 
	{"SetScrollInfo",	    ui_window_set_scroll_info,		1}, // @pymeth SetScrollInfo|Set information about a scroll bar
	{"SetWindowPlacement",	ui_window_set_window_placement,	1}, // @pymeth SetWindowPlacement|Sets the window's placement options.
	{"SetWindowText",		ui_window_set_window_text,		1}, // @pymeth SetWindowText|Sets the window's text.
	{"ShowCaret",		    ui_window_show_caret,		    1}, // @pymeth ShowCaret|Shows the caret
	{"ShowScrollBar",		ui_window_show_scrollbar,		1}, // @pymeth ShowScrollBar|Shows/Hides the window's scroll bars.
	{"ShowWindow",			ui_window_show_window,			1}, // @pymeth ShowWindow|Shows the window.
	{"UpdateData",			ui_window_update_data,			1}, // @pymeth UpdateData|Updates a windows dialog data.
	{"UpdateDialogControls",ui_window_udc,			        1}, // @pymeth UpdateDialogControls|Updates the state of dialog buttons and other controls in a dialog box or window that uses the <om PyCCmdUI.HookCommandUpdate> callback mechanism.
	{"UpdateWindow",		ui_window_update_window,		1}, // @pymeth UpdateWindow|Updates a window.
	{NULL,			NULL}
};

CString PyCWnd::repr()
{
	CString csRet;
	char *buf = csRet.GetBuffer(40);
	int numMsg = pMessageHookList ? pMessageHookList->GetCount() : 0;
	int numKey = pKeyHookList ? pKeyHookList->GetCount() : 0;
	char *hookStr = obKeyStrokeHandler ? " (AllKeys Hook Active)" : "";
	sprintf(buf, ", mh=%d, kh=%d%s", numMsg, numKey, hookStr);
	csRet.ReleaseBuffer(-1);
	return PyCCmdTarget::repr() + csRet;
}

ui_type_CObject PyCWnd::type("PyCWnd", 
							 &PyCCmdTarget::type, 
							 RUNTIME_CLASS(CWnd), 
							 sizeof(PyCWnd), 
							 PyCWnd_methods, 
							 GET_PY_CTOR(PyCWnd) );


/////////////////////////////////////////////////////////////////////
//
// Frame Window objects
//
// MDIFrameWindow is the application frame, MDIChildWindow is the child frame.
//
//////////////////////////////////////////////////////////////////////
// @pymethod <c PyCDocument>|PyCFrameWnd|GetActiveDocument|Gets the currently active document, else None
static PyObject *
ui_frame_get_active_document(PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	CHECK_NO_ARGS(args);
    CDocument *pDoc = pFrame->GetActiveDocument(); // @pyseemfc CFrameWnd|GetActiveDocument
	if (!pDoc)
		RETURN_NONE;
	return ui_assoc_object::make( PyCDocument::type, pDoc)->GetGoodRet();
}
// @pymethod |PyCFrameWnd|LoadAccelTable|Loads an accelerator table.
static PyObject *
PyCFrameWnd_LoadAccelTable(PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	PyObject *obID;
	if (!PyArg_ParseTuple(args, "O", &obID))
		return NULL;
	char *res;
	if (PyInt_Check(obID))
		res = MAKEINTRESOURCE(PyInt_AsLong(obID));
	else if (PyString_Check(obID))
		res = PyString_AsString(obID);
	else
		RETURN_TYPE_ERR("The param must be an integer or string");
	if (!pFrame->LoadAccelTable(res))
		RETURN_ERR("LoadAccelTable failed");
	RETURN_NONE;
}
// @pymethod |PyCFrameWnd|LoadFrame|Loads a Windows frame window and associated resources
static PyObject *
ui_frame_load_frame(PyObject *self, PyObject *args)
{
	int idResource = IDR_PYTHONTYPE; 
	long style = -1;
	CPythonFrame *pFrame = GetPythonFrame(self);
	BOOL bMakeVisible = TRUE;
	if (!pFrame)
		return NULL;
	if (pFrame->m_hWnd!=NULL)
		RETURN_ERR("The frame already has a window");
	PythonCreateContext cc;
	PyObject *wndParent = Py_None;
	PyObject *contextObject = Py_None;
	if (!PyArg_ParseTuple(args, "|ilOO:LoadFrame", 
	                      &idResource, // @pyparm int|idResource|IDR_PYTHONTYPE|The Id of the resources (menu, icon, etc) for this window
	                      &style,	   // @pyparm long|style|-1|The window style.  Note -1 implies win32con.WS_OVERLAPPEDWINDOW\|win32con.FWS_ADDTOTITLE
						  &wndParent,  // @pyparm <o PyCWnd>|wndParent|None|The parent of the window, or None.
						  &contextObject)) // @pyparm object|context|None|An object passed to the OnCreateClient for the frame,
		return NULL;
	if (style==-1) style=WS_OVERLAPPEDWINDOW | FWS_ADDTOTITLE;   // default frame styles;
	CWnd *pParent = NULL;
	if (wndParent!=Py_None) {
		pParent = GetWndPtrFromParam( wndParent, PyCWnd::type );
		if (pParent==NULL)
			return NULL;
	}
	cc.SetPythonObject(contextObject);
//	cc. = idResource
	// OnCreateClient will be called during this!
	CProtectedWinApp *pApp = GetProtectedApp();
	if (!pApp) return NULL;
	BOOL ok;
	CWnd *pMain = pApp->GetMainFrame();
	if (pMain==NULL)
		RETURN_ERR("There is no main application frame - an MDI child can not be created.");
	if (!pMain->IsKindOf(RUNTIME_CLASS(CMDIFrameWnd)))
		RETURN_ERR("There is no MDI Frame Window available - an MDI child can not be created.");
	
	GUI_BGN_SAVE;
	ok = pFrame->LoadFrame(idResource, style, pParent, &cc); // @pyseemfc CFrameWnd|LoadFrame
	GUI_END_SAVE;
	if (!ok) {
		RETURN_ERR("LoadFrame failed\n");
		// frame will be deleted in PostNcDestroy cleanup
	}
	RETURN_NONE;
}

// @pymethod |PyCFrameWnd|RecalcLayout|Called by the framework when the standard control bars are toggled on or off or when the frame window is resized.
static PyObject *
ui_frame_recalc_layout(PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	BOOL bNotify = TRUE;
	if (!PyArg_ParseTuple(args,"|i:RecalcLayout", 
		&bNotify)) // @pyparm int|bNotify|1|Notify flag
		return NULL;
	pFrame->RecalcLayout(bNotify); // @pyseemfc CFrameWnd|RecalcLayout
	RETURN_NONE;
}
// @pymethod |PyCFrameWnd|EnableDocking|Enable dockable control bars in a frame window
PyObject *PyCFrameWnd_EnableDocking( PyObject *self, PyObject *args ) 
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	int style;
	// @pyparm int|style||Specifies which sides of the frame window can serve as docking sites for control bars.
	if (!PyArg_ParseTuple( args, "i:EnableDocking", &style))
		return NULL;
	pFrame->EnableDocking(style);
	RETURN_NONE;
	// @comm By default, control bars will be docked to a side of the frame window in the following order: top, bottom, left, right.
}

// @pymethod |PyCFrameWnd|DockControlBar|Docks a control bar.
static PyObject *
PyCFrameWnd_DockControlBar(PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	PyObject *ob;
	int docBarId = 0;
	CRect rect(0,0,0,0);
	if (!PyArg_ParseTuple(args,"O|i(iiii):DockControlBar", 
		       &ob, // @pyparm <o PyCControlBar>|controlBar||The control bar to dock.
			   &docBarId, // @pyparm int|dockBarId|0|Determines which sides of the frame window to consider for docking.
			   // @pyparm left, top, right, bottom|int, int, int, int|0,0,0,0|Determines, in screen coordinates, where the control bar will be docked in the nonclient area of the destination frame window.
			   &rect.left, &rect.top, &rect.right, &rect.bottom))
		return NULL;
	CControlBar *pControlBar = PyCControlBar::GetControlBar(ob);
	if (pControlBar==NULL)
		return NULL;
	CRect *pRect = (rect.left==rect.right==0) ? NULL : &rect;
	GUI_BGN_SAVE;
	PyObject *rc;
	__try {
		pFrame->DockControlBar(pControlBar, docBarId, pRect); // @pyseemfc CFrameWnd|DockControlBar
		rc = Py_None;
		Py_INCREF(Py_None);
	}
	__except (EXCEPTION_EXECUTE_HANDLER) {
		PyErr_SetString(ui_module_error, "DockControlBar caused exception.");
		rc = NULL;
	}
	GUI_END_SAVE;
	return rc;
}

// @pymethod |PyCFrameWnd|FloatControlBar|Floats a control bar.
static PyObject *
PyCFrameWnd_FloatControlBar(PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	PyObject *ob;
	int style = CBRS_ALIGN_TOP;
	CPoint pt;
	if (!PyArg_ParseTuple(args,"O(ii)|i:FloatControlBar", 
		       &ob, // @pyparm <o PyCControlBar>|controlBar||The control bar to dock.
			   &pt.x, &pt.y, // @pyparm x,y|int, int||he location, in screen coordinates, where the top left corner of the control bar will be placed.
			   &style)) // @pyparm int|style|CBRS_ALIGN_TOP|Determines which sides of the frame window to consider for docking.
		return NULL;
	CControlBar *pControlBar = PyCControlBar::GetControlBar(ob);
	if (pControlBar==NULL)
		return NULL;
	GUI_BGN_SAVE;
	pFrame->FloatControlBar(pControlBar, pt, style); // @pyseemfc CFrameWnd|FloatControlBar
	GUI_END_SAVE;
	RETURN_NONE;
}
// @pymethod |PyCFrameWnd|ShowControlBar|Shows a control bar.
static PyObject *
PyCFrameWnd_ShowControlBar(PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	PyObject *ob;
	BOOL bShow, bDelay;
	if (!PyArg_ParseTuple(args,"Oii:ShowControlBar", 
		       &ob, // @pyparm <o PyCControlBar>|controlBar||The control bar to dock.
			   &bShow, // @pyparm int|bShow||Show or hide flag.
			   &bDelay)) // @pyparm int|bDelay||If TRUE, delay showing the control bar. If FALSE, show the control bar immediately.

		return NULL;
	CControlBar *pControlBar = PyCControlBar::GetControlBar(ob);
	if (pControlBar==NULL)
		return NULL;
	GUI_BGN_SAVE;
	pFrame->ShowControlBar(pControlBar,bShow, bDelay); // @pyseemfc CFrameWnd|ShowControlBar
	GUI_END_SAVE;
	RETURN_NONE;
}

// @pymethod |PyCFrameWnd|SaveBarState|Saves a control bars settings
static PyObject *
PyCFrameWnd_SaveBarState(PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	char *profileName;
	if (!PyArg_ParseTuple(args,"s:SaveBarState", 
		       &profileName)) // @pyparm string|profileName||Name of a section in the initialization file or a key in the Windows registry where state information is stored.
		return NULL;
	GUI_BGN_SAVE;
	pFrame->SaveBarState(profileName); // @pyseemfc CFrameWnd|SaveBarState
	GUI_END_SAVE;
	RETURN_NONE;
}

// @pymethod |PyCFrameWnd|LoadBarState|Loads a control bars settings
static PyObject *
PyCFrameWnd_LoadBarState(PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	char *profileName;
	if (!PyArg_ParseTuple(args,"s:LoadBarState", 
		       &profileName)) // @pyparm string|profileName||Name of a section in the initialization file or a key in the Windows registry where state information is stored.
		return NULL;
	GUI_BGN_SAVE;
	try {
		pFrame->LoadBarState(profileName); // @pyseemfc CFrameWnd|LoadBarState
	}
	catch (...) {
		GUI_END_SAVE;
		RETURN_ERR("LoadBarState failed (with win32 exception!)");
	}
	GUI_END_SAVE;
	RETURN_NONE;
}

// @pymethod |PyCFrameWnd|BeginModalState|Sets the frame window to modal.
static PyObject *
PyCFrameWnd_BeginModalState( PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	CHECK_NO_ARGS2(args, BeginModalState );
	pFrame->BeginModalState();
	RETURN_NONE;
}
// @pymethod |PyCFrameWnd|EndModalState|Ends the frame windows modal state. Enables all of the windows disabled by <om PyCFrameWnd.BeginModalState>.
static PyObject *
PyCFrameWnd_EndModalState( PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	CHECK_NO_ARGS2(args, EndModalState );
	pFrame->EndModalState();
	RETURN_NONE;
}
// @pymethod int|PyCFrameWnd|InModalState|Returns a value indicating whether or not a frame window is in a modal state.
static PyObject *
PyCFrameWnd_InModalState( PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	CHECK_NO_ARGS2(args, InModalState );
	return Py_BuildValue("i", pFrame->InModalState());
}


// @pymethod int|PyCFrameWnd|IsTracking|Determines if splitter bar is currently being moved.
static PyObject *
PyCFrameWnd_IsTracking( PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	CHECK_NO_ARGS2(args, EndModalState );
	return Py_BuildValue("i", pFrame->IsTracking());
}

// @pymethod string|PyCFrameWnd|GetMessageString|Retrieves message corresponding to a command ID.
static PyObject *
PyCFrameWnd_GetMessageString( PyObject *self, PyObject *args)
{
	int id;
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	if (!PyArg_ParseTuple(args, "i", &id )) // @pyparm int|id||The ID to be retrieved
		return NULL;
	CString csRet;
	// @xref <vm PyCMDIChildWnd.GetMessageString>
	pFrame->CFrameWnd::GetMessageString(id, csRet);
	return Py_BuildValue("s", (const char *)csRet);
}

// @pymethod <o PyCControlBar>|PyCFrameWnd|GetControlBar|Retrieves the specified control bar.
static PyObject *
PyCFrameWnd_GetControlBar( PyObject *self, PyObject *args)
{
	int id;
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	if (!PyArg_ParseTuple(args, "i", &id )) // @pyparm int|id||The ID of the toolbar to be retrieved
		return NULL;
	CControlBar *pRet = pFrame->GetControlBar(id);
	if (pRet==NULL)
		RETURN_ERR("There is no control bar with that ID");
	return ui_assoc_object::make(UITypeFromCObject(pRet), pRet)->GetGoodRet();
}
// @pymethod <o PyCWnd>|PyCFrameWnd|GetMessageBar|Retrieves the message bar for the frame.
static PyObject *
PyCFrameWnd_GetMessageBar( PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	CHECK_NO_ARGS2(args, GetMessageBar);
	CWnd *pRet = pFrame->GetMessageBar();
	if (pRet==NULL)
		RETURN_ERR("There is no message bar.");
	return ui_assoc_object::make(UITypeFromCObject(pRet), pRet)->GetGoodRet();
}


// @pymethod string|PyCFrameWnd|ShowOwnedWindows|Shows all windows that are descendants of the <o PyCFrameWnd> object.
static PyObject *
PyCFrameWnd_ShowOwnedWindows( PyObject *self, PyObject *args)
{
	BOOL bShow = TRUE;
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	if (!PyArg_ParseTuple(args, "|i", &bShow )) // @pyparm int|bShow|1|Flag
		return NULL;
	CString csRet;
	pFrame->CFrameWnd::ShowOwnedWindows(bShow);
	RETURN_NONE;
}

// @pymethod |PyCFrameWnd|SetActiveView|Sets the active view for a frame.
static PyObject *
PyCFrameWnd_SetActiveView(PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	PyObject *ob;
	BOOL bNotify = TRUE;
	if (!PyArg_ParseTuple(args,"O|i:SetActiveView", 
		       &ob, // @pyparm <o PyCView>|view||The view to set active.
			   &bNotify)) // @pyparm int|bNotify|1|Specifies whether the view is to be notified of activation. If TRUE, OnActivateView is called for the new view; if FALSE, it is not.
		return NULL;
	CView *pView = PyCView::GetViewPtr(ob);

	if (pView==NULL)
		return NULL;
	GUI_BGN_SAVE;
	pFrame->SetActiveView(pView, bNotify);
	GUI_END_SAVE;
	RETURN_NONE;
}

// @pymethod <o PyCView>|PyCFrameWnd|GetActiveView|Retrieves the active view.
static PyObject *
PyCFrameWnd_GetActiveView( PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	CHECK_NO_ARGS2(args, GetActiveView);
	CView *pRet = pFrame->GetActiveView();
	if (pRet==NULL)
		RETURN_ERR("There is no active view.");
	return ui_assoc_object::make(UITypeFromCObject(pRet), pRet)->GetGoodRet();
}

// @pymethod int|PyCFrameWnd|OnBarCheck|Changes the state of the specified controlbar.
static PyObject *
PyCFrameWnd_OnBarCheck( PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	int id;
	if (!PyArg_ParseTuple(args, "i:OnBarCheck", &id)) // @pyparm int|id||The control ID of the control bar.
		return NULL;
	return PyInt_FromLong(pFrame->OnBarCheck(id));
}

// @pymethod int|PyCFrameWnd|OnUpdateControlBarMenu|Checks the state of a menu item
static PyObject *
PyCFrameWnd_OnUpdateControlBarMenu( PyObject *self, PyObject *args)
{
	CFrameWnd *pFrame = GetFramePtr(self);
	if (!pFrame)
		return NULL;
	PyObject *obCU;
	if (!PyArg_ParseTuple(args, "O:OnUpdateControlBarMenu", &obCU)) // @pyparm <o PyCCmdUI>|cmdUI||A cmdui object
		return NULL;
	CCmdUI *pCU = PyCCmdUI::GetCCmdUIPtr(obCU);
	if (pCU==NULL)
		return NULL;
	pFrame->OnUpdateControlBarMenu(pCU);
	RETURN_NONE;
}

// @object PyCFrameWnd|A windows frame window.  Encapsulates an MFC <c CFrameWnd> class.  Derived from a <o PyCWnd> object.
static struct PyMethodDef PyCFrameWnd_methods[] = {
	{"BeginModalState",	    PyCFrameWnd_BeginModalState,	1}, // @pymeth BeginModalState|Sets the frame window to modal.
	{"EndModalState",	    PyCFrameWnd_EndModalState,	1}, // @pymeth EndModalState|Ends the frame windows modal state. Enables all of the windows disabled by <om PyCFrameWnd.BeginModalState>.
	{"DockControlBar",	    PyCFrameWnd_DockControlBar,	1}, // @pymeth DockControlBar|Docks a control bar.
	{"EnableDocking",	    PyCFrameWnd_EnableDocking,	1}, // @pymeth EnableDocking|Enable dockable control bars in a frame window
	{"FloatControlBar",	    PyCFrameWnd_FloatControlBar,	1}, // @pymeth FloatControlBar|Floats a control bar.
	{"GetActiveDocument",	ui_frame_get_active_document,	1}, // @pymeth GetActiveDocument|Returns the currently active document
	{"GetMessageString",	PyCFrameWnd_GetMessageString,	1}, // @pymeth GetMessageString|Retrieves message corresponding to a command ID.
	{"GetMessageBar",		PyCFrameWnd_GetMessageBar,	1}, // @pymeth GetMessageBar|Retrieves the message bar for the frame.
	{"IsTracking",			PyCFrameWnd_IsTracking,	1}, // @pymeth IsTracking|Determines if splitter bar is currently being moved.
	{"InModalState",		PyCFrameWnd_InModalState,	1}, // @pymeth InModalState|Returns a value indicating whether or not a frame window is in a modal state.
	{"LoadAccelTable",      PyCFrameWnd_LoadAccelTable, 1}, // @pymeth LoadAccelTable|Loads an accelerator table.
	{"LoadFrame",			ui_frame_load_frame,			1}, // @pymeth LoadFrame|Creates the MDI Window's frame
	{"LoadBarState",		PyCFrameWnd_LoadBarState,			1}, // @pymeth LoadBarState|Loads a control bars settings
	{"SaveBarState",		PyCFrameWnd_SaveBarState,			1}, // @pymeth SaveBarState|Saves a control bars settings
	{"ShowControlBar",		PyCFrameWnd_ShowControlBar,			1}, // @pymeth ShowControlBar|Shows a control bar.
	{"RecalcLayout",		ui_frame_recalc_layout,			1}, // @pymeth RecalcLayout|Called by the framework when the standard control bars are toggled on or off or when the frame window is resized.
	{"GetActiveView",		PyCFrameWnd_GetActiveView,	1}, // @pymeth GetActiveView|Retrieves the active view.
	{"OnBarCheck",		    PyCFrameWnd_OnBarCheck,	1}, // @pymeth OnBarCheck|Changes the state of the specified controlbar.
	{"OnUpdateControlBarMenu",PyCFrameWnd_OnUpdateControlBarMenu,	1}, // @pymeth OnUpdateControlBarMenu|Checks the state of a menu item
	{"SetActiveView",		PyCFrameWnd_SetActiveView,		1}, // @pymeth SetActiveView|Sets the active view for a frame.
	{NULL,			NULL}
};

ui_type_CObject PyCFrameWnd::type("PyCFrameWnd", 
								  &PyCWnd::type, 
								  RUNTIME_CLASS(CFrameWnd),
								  sizeof(PyCFrameWnd), 
								  PyCFrameWnd_methods, 
								  GET_PY_CTOR(PyCFrameWnd));

// @pymethod tuple|PyCMDIFrameWnd|PreCreateWindow|Calls the underlying MFC PreCreateWindow method.
PyObject *
ui_mdi_frame_window_pre_create_window(PyObject *self, PyObject *args)
{
	CMDIFrameWnd *pWnd = GetMDIFrame(self);
	if (!pWnd)
		return NULL;
	CREATESTRUCT cs;
	//@pyparm tuple|createStruct||A tuple representing a CREATESTRUCT structure.
	if (!CreateStructFromPyObject( &cs, args, "PreCreateWindow", TRUE))
		return NULL;

	// @xref <vm PyCMDIChildWnd.PreCreateWindow>
	if (!pWnd->CMDIFrameWnd::PreCreateWindow(cs))
		RETURN_ERR("CMDIFrameWnd::PreCreateWindow failed");
	return PyObjectFromCreateStruct(&cs);
}

// @object PyCMDIFrameWnd|A main application frame window.  Encapsulates an MFC <c CMDIFrameWnd> class
static struct PyMethodDef PyCMDIFrameWnd_methods[] = {
	{"PreCreateWindow", ui_mdi_frame_window_pre_create_window, 1}, // @pymeth PreCreateWindow|Calls the underlying MFC PreCreateWindow method.
	{NULL,			NULL}
};

ui_type_CObject PyCMDIFrameWnd::type("PyCMDIFrameWnd", 
									 &PyCFrameWnd::type, 
									 RUNTIME_CLASS(CMDIFrameWnd), 
									 sizeof(PyCMDIFrameWnd), 
									 PyCMDIFrameWnd_methods, 
									 GET_PY_CTOR(PyCMDIFrameWnd));

// @pymethod |PyCMDIChildWnd|SetMenu|Sets the menu for a window.
PyObject *
ui_mdi_child_window_set_menu(PyObject *self, PyObject *args)
{
	CPythonFrame *pWnd = GetPythonFrame(self);
	if (!pWnd)
		return NULL;
	int id;
	BOOL bRet;
	//@pyparm PyCMenu|menuObj||The menu object to set
    // @pyparmalt1 int|menuId||The menu ID of the menu
	if (!PyArg_ParseTuple(args, "i:SetMenu", &id)) {
		PyObject *menuObject;
		PyErr_Clear();
		if (!PyArg_ParseTuple(args, "O:SetMenu", &menuObject))
			RETURN_TYPE_ERR("bad argument.  Must be format 'O' or 'i'");
		if (!ui_base_class::is_uiobject(menuObject, &PyCMenu::type))
			RETURN_TYPE_ERR("passed object must be a PyCMenu");
		HMENU hMenu = PyCMenu::GetMenu(menuObject);
		if (hMenu==NULL)
			return NULL;
		bRet = pWnd->SetMenu(hMenu, menuObject);
	} else {
		bRet = pWnd->SetMenu(id);
	}
	if (!bRet)
		RETURN_API_ERR("CWnd::SetMenu");
	RETURN_NONE;
}

// @pymethod |PyCMDIChildWnd|ActivateFrame|Calls the underlying MFC ActivateFrame method.
PyObject *
ui_mdi_child_window_activate_frame(PyObject *self, PyObject *args)
{
	CPythonFrame *pWnd = GetPythonFrame(self);
	if (!pWnd)
		return NULL;
	int cmdShow = -1;
	//@pyparm int|cmdShow|-1|The status of the window.
	if (!PyArg_ParseTuple(args, "|i:ActivateFrame", &cmdShow))
		return NULL;
	// @xref <vm PyCMDIChildWnd.ActivateFrame>
	pWnd->CMDIChildWnd::ActivateFrame(cmdShow);
	RETURN_NONE;
}

// @pymethod tuple|PyCMDIChildWnd|PreCreateWindow|Calls the underlying MFC PreCreateWindow method.
PyObject *
ui_mdi_child_window_pre_create_window(PyObject *self, PyObject *args)
{
	CPythonFrame *pWnd = GetPythonFrame(self);
	if (!pWnd)
		return NULL;
	CREATESTRUCT cs;
	//@pyparm tuple|createStruct||A tuple representing a CREATESTRUCT structure.
	if (!CreateStructFromPyObject( &cs, args, "PreCreateWindow", TRUE))
		return NULL;

	// @xref <vm PyCMDIChildWnd.PreCreateWindow>
	if (!pWnd->CMDIChildWnd::PreCreateWindow(cs))
		RETURN_ERR("CMDIChildWnd::PreCreateWindow failed");
	return PyObjectFromCreateStruct(&cs);
}

// @object PyCMDIChildWnd|A windows frame window.  Encapsulates an MFC <c CMDIChildWindow> class
static struct PyMethodDef PyCMDIChildWnd_methods[] = {
	{"ActivateFrame", ui_mdi_child_window_activate_frame, 1}, // @pymeth ActivateFrame|Calls the underlying MFC ActivateFrame method.
	{"PreCreateWindow", ui_mdi_child_window_pre_create_window, 1}, // @pymeth PreCreateWindow|Calls the underlying MFC PreCreateWindow method.
	{"SetMenu",     ui_mdi_child_window_set_menu, 1},// @pymeth SetMenu|Set the menu for the frame window
	{NULL,			NULL}
};

ui_type_CObject PyCMDIChildWnd::type("PyCMDIChildWnd", 
									 &PyCFrameWnd::type, 
									 RUNTIME_CLASS(CMDIChildWnd), 
									 sizeof(PyCMDIChildWnd), 
									 PyCMDIChildWnd_methods, 
									 GET_PY_CTOR(PyCMDIChildWnd));
