#!/usr/bin/python
import string, sys
from xml.dom.ext.reader import Sax
from xml.dom.Node import Node

def GenFiles(fileName, program_name, display_class):
    dom = Sax.FromXmlFile(fileName, validate=0)

    header = CreateHeader(dom, program_name)
    classes = dom.getElementsByTagName('class')
    for klass in classes:
        if not display_class:
            GenClassFile(klass, header)
        else:
            print 'HTML%sElement' % klass.getAttribute('name')


def CreateHeader(dom, prog_name):
    result = ''
    header = dom.getElementsByTagName('header')
    if header:
        result = result + string.strip(header[0].childNodes[0].data)
        result = result + '\n\n'

    result = result + '### This is file generated by %s.\n' % prog_name
    result = result + '### DO NOT EDIT!\n\n'

    copyright = dom.getElementsByTagName('copyright')
    if copyright:
        result = result + '"""\n'
        result = result + string.strip(copyright[0].childNodes[0].data) + '\n'
        result = result + '"""\n\n'

    return result

# Helper function for indenting Python
def indent(count, text, tab=' '*4):
    return tab*count + text

# Get/Set routines for DOMString attributes
def stringGetAttr(name, value):
    return indent(2, 'return self.getAttribute("%s")\n\n' % name)

def stringSetAttr(name):
    return indent(2, 'self.setAttribute("%s", value)\n\n' % name)

# Routines for boolean attributes
def boolGetAttr(name, value):
    return indent(2, 'return self.hasAttribute("%s")\n\n' % name)

def boolSetAttr(name):
    result =          indent(2, 'if value:\n')
    result = result + indent(3,   'self.setAttribute("%s", None)\n' % name)
    result = result + indent(2, 'else:\n')
    result = result + indent(3,   'self.removeAttribute("%s")\n\n' % name)
    return result

# Routines for number attributes
def longGetAttr(name, value):
    result =          indent(2, 'value = self.getAttribute("%s")\n' % name)
    result = result + indent(2, 'if value:\n')
    result = result + indent(3,   'return int(value)\n')
    result = result + indent(2, 'return 0\n\n')
    return result

def longSetAttr(name):
    return indent(2, 'self.setAttribute("%s", str(value))\n\n' % name)

# Routines for value-list attributes
def listGetAttr(name, value):
    return indent(2, 'return string.capitalize(self.getAttribute("%s"))\n\n' % name)

# Routines for attributes mapped to Text nodes
def nodeGetAttr(dummy, value):
    result =          indent(2, 'if not self.firstChild:\n')
    result = result + indent(3,   'return ''\n')
    result = result + indent(2, 'if self.firstChild == self.lastChild:\n')
    result = result + indent(3,   'return self.firstChild.data\n')
    result = result + indent(2, 'self.normalize()\n')
    result = result + indent(2, 'text = filter(lambda x: x.nodeType == Node.TEXT_NODE, self.childNodes)\n')
    result = result + indent(2, 'return text[0].data\n\n')
    return result

def nodeSetAttr(dummy):
    result =          indent(2, 'text = None\n')
    result = result + indent(2, 'for node in self.childNodes:\n')
    result = result + indent(3,   'if not text and node.nodeType == Node.TEXT_NODE:\n')
    result = result + indent(4,     'text = node\n')
    result = result + indent(3,   'else:\n')
    result = result + indent(4,     'self.removeChild(node)\n')
    result = result + indent(2, 'if text:\n')
    result = result + indent(3,   'text.data = value\n')
    result = result + indent(2, 'else:\n')
    result = result + indent(3,   'text = self.ownerDocument.createTextNode(value)\n')
    result = result + indent(3,   'self.appendChild(text)\n\n')
    return result

#Routines for constant attributes
def constGetAttr(name, value):
    if not value:
        value = 'None'
    else:
        value = '"%s"' % value
    return indent(2, 'return %s\n\n' % value)

#Routines for form based classes
def formGetAttr(dummy,dummy2):
    result =          indent(2, 'parent = self.parentNode\n')
    result = result + indent(2, 'while parent:\n')
    result = result + indent(3,   'if parent.nodeName == "FORM":\n')
    result = result + indent(4,     'return parent\n')
    result = result + indent(3,   'parent = parent.parentNode\n')
    result = result + indent(2, 'return None\n\n')
    return result


g_valueTypeMap = {
    'bool' : (boolGetAttr, boolSetAttr),
    'long' : (longGetAttr, longSetAttr),
    'list' : (listGetAttr, stringSetAttr),
    'node' : (nodeGetAttr, nodeSetAttr),
    'string' : (stringGetAttr, stringSetAttr),
    'form' : (formGetAttr, None),
    'const' : (constGetAttr, None)
    }

def GenClassFile(klass, header):
    class_name = 'HTML%sElement' % klass.getAttribute('name')
    fileName = class_name + '.py'
    file = open(fileName, 'w')

    # General header stuff
    file.write(string.replace(header, '$FILE$', class_name))

    # TraceOut support
    file.write('from Ft.Lib import TraceOut\n')
    file.write('trace = TraceOut.TraceOut(["4DOM.HTML.%s"])\n\n' % class_name)

    # Import statements
    file.write('import string\n')
    file.write('from xml.dom.Node import Node\n')
    baseclass = klass.getElementsByTagName('baseclass')[0].getAttribute('name')
    base_name = string.split(baseclass, '.')[-1]
    file.write('from %s import %s\n' % (baseclass, base_name))
    file.write('\n')

    # Class declaration
    file.write('class %s(%s):\n\n' % (class_name, base_name))

    # Constructor
    file.write(indent(1, 'def __init__(self, ownerDocument, nodeName'))
    multiple = klass.getAttribute('multiple')
    tag_name = klass.getAttribute('tagname')
    if not multiple:
        if not tag_name:
            tag_name = string.upper(klass.getAttribute('name'))
        file.write('="%s"' % tag_name)
    file.write('):\n')
    file.write(indent(2, '%s.__init__(self, ownerDocument, nodeName)\n\n' % base_name))

    # Attributes
    file.write(indent(1, '### Attribute Methods ###\n\n'))
    attrs = klass.getElementsByTagName('attribute')
    read_attrs = []
    write_attrs = []
    for attr in attrs:
        dom_name = attr.getAttribute('name')
        value_type = attr.getAttribute('type')
        html_name = attr.getAttribute('htmlname')
        if not html_name:
            html_name = string.upper(dom_name)
        value = attr.getAttribute('value')    # for const value-type
        permissions = attr.getElementsByTagName('permissions')[0]
        readable = int(permissions.getAttribute('readable'))
        writeable = int(permissions.getAttribute('writeable'))

        if readable:
            file.write(indent(1, 'def _get_%s(self):\n' % dom_name))
            get_func = g_valueTypeMap[value_type][0]
            file.write(get_func(html_name, value))
            read_attrs.append(dom_name)

        if writeable:
            file.write(indent(1, 'def _set_%s(self, value):\n' % dom_name))
            set_func = g_valueTypeMap[value_type][1]
            try:
                file.write(set_func(html_name or value))
            except:
                raise "Set function '%s' in class %s, attribute %s" % (value_type, class_name, dom_name)
            write_attrs.append(dom_name)

    # Methods
    methods = klass.getElementsByTagName('method')
    if methods:
        file.write(indent(1, '### Methods ###\n\n'))
    for method in methods:
        method_name = method.getAttribute('name')
        params = method.getElementsByTagName('params')[0].childNodes
        param_list = []
        for param in params:
            arg = param.getAttribute('name')
            default = param.firstChild
            param_list.append((arg,default))
        file.write(indent(1, 'def %s(self' % method_name))
        for arg,default in param_list:
            file.write(', %s' % arg)
            if default:
                file.write('=%s' % default.data)
        file.write('):\n')

        # The function code
        code = method.getElementsByTagName('code')[0].firstChild
        if code:
            lines = string.split(string.strip(code.data), '\n')
            for line in lines:
                writeTab(file, 2, line)
        else:
            file.write(indent(2, 'pass\n'))
        file.write('\n')

    # Attribute access control
    file.write(indent(1, '### Attribute Access Mappings ###\n\n'))

    file.write(indent(1, '_readComputedAttrs = %s._readComputedAttrs.copy()\n' % base_name))
    if len(read_attrs):
        file.write(indent(1, '_readComputedAttrs.update({\n'))
        for attr in read_attrs[:-1]:
            file.write(indent(2, '"%s" : _get_%s,\n' % (attr, attr)))
        attr = read_attrs[-1]
        file.write(indent(2, '"%s" : _get_%s\n' % (attr, attr)))
        file.write(indent(2, '})\n\n'))

    file.write(indent(1, '_writeComputedAttrs = %s._writeComputedAttrs.copy()\n' % base_name))
    if len(write_attrs):
        file.write(indent(1, '_writeComputedAttrs.update({\n'))
        for attr in write_attrs[:-1]:
            file.write(indent(2, '"%s" : _set_%s,\n' % (attr, attr)))
        attr = write_attrs[-1]
        file.write(indent(2, '"%s" : _set_%s\n' % (attr, attr)))
        file.write(indent(2, '})\n\n'))

    file.write(indent(1, '_readOnlyAttrs = filter(lambda k,m=_writeComputedAttrs: not m.has_key(k),\n'))
    file.write(indent(1, '                 %s._readOnlyAttrs + _readComputedAttrs.keys())\n\n' % base_name))


if __name__ == '__main__':
    import getopt, os
    prog_name = os.path.split(sys.argv[0])[1]
    short_opts = 'hl'
    long_opts = [
        'help',
        'list'
        ]

    usage ="""Usage: %s [options] <XML input file>
Options:
  -h, --help        Print this message and exit
  -l, --list        List the classes in the input file
                    (doesn't generate files)
""" % (prog_name)

    command_line_error = 0
    bad_options = []

    finished = 0
    args = sys.argv[1:]
    while not finished:
        try:
            optlist, args = getopt.getopt(args, short_opts, long_opts)
        except getopt.error, data:
            bad_options.append(string.split(data)[1])
            args.remove(bad_options[-1])
            command_line_error = 1
        else:
            finished = 1

    display_usage = 0
    display_class = 0
    for op in optlist:
        if op[0] == '-h' or op[0] == '--help':
            display_usage = 1
        elif op[0] == '-l' or op[0] == '--list':
            display_class = 1

    spec_file = ''
    if args:
        spec_file = args[0]
        if not os.path.exists(spec_file):
            import errno
            print "%s: %s '%s'" %(prog_name, os.strerror(errno.ENOENT), spec_file)
            spec_file = ''
            command_line_error = 1
    else:
        command_line_error = 1

    if command_line_error or display_usage:
        for op in bad_options:
            print "%s: Unrecognized option '%s'" % (prog_name, op)
        if not spec_file:
            print "%s: No input files" % (prog_name)
        if display_usage:
            print usage
        sys.exit(command_line_error)

    GenFiles(spec_file, prog_name, display_class)

