########################################################################
#
# File Name:            Sax2.py
#
# Documentation:        http://docs.4suite.com/4DOM/Sax2.py.html
#
"""
Components for reading XML files from a SAX2 producer.
WWW: http://4suite.com/4DOM         e-mail: support@4suite.com

Copyright (c) 2000 Fourthought Inc, USA.   All Rights Reserved.
See  http://4suite.com/COPYRIGHT  for license and copyright information
"""

import sys, string, cStringIO
from xml.sax import saxlib, saxexts
import Sax2Lib
from xml.dom import Entity, DocumentType, Document
from xml.dom.Node import Node
from xml.dom import implementation
from xml.dom.ext import SplitQName
from xml.dom import XML_NAMESPACE, XMLNS_NAMESPACE
from xml.dom import Element
from xml.dom import Attr

class XmlDomGenerator(saxlib.HandlerBase,
                      Sax2Lib.LexicalHandler,
                      Sax2Lib.DTDDeclHandler,
                      Sax2Lib.NamespaceHandler):
    def __init__(self, doc=None, keepAllWs=0):
        self.__ownerDoc = None
        self.__rootNode = None
        #Set up the stack which keeps track of the nesting of DOM nodes.
        self.__nodeStack = []
        if doc:
            self.__ownerDoc = doc
            #Create a docfrag to hold all the generated nodes.
            self.__rootNode = self.__ownerDoc.createDocumentFragment()
            self.__nodeStack.append(self.__rootNode)
        self.__dt = None
        self.__xmlDecl = None
        self.__orphanedNodes = []
        self.__namespaces = {'xml': XML_NAMESPACE}
        self.__namespaceStack = []
        self.__keepAllWs = keepAllWs
        self.__currText = ''

    def __initRootNode(self, docElementUri, docElementName):
        if not self.__dt:
            self.__dt = implementation.createDocumentType(docElementName,'','')
        self.__ownerDoc = implementation.createDocument(docElementUri, docElementName, self.__dt)
        if self.__xmlDecl:
            decl_data = 'version="%s"' % (
                    self.__xmlDecl['version']
                    )
            if self.__xmlDecl['encoding']:
                decl_data = decl_data + ' encoding="%s"'%(
                    self.__xmlDecl['encoding']
                    )
            if self.__xmlDecl['standalone']:
                decl_data = decl_data + ' standalone="%s"'%(
                    self.__xmlDecl['standalone']
                    )
            xml_decl_node = self.__ownerDoc.createProcessingInstruction(
                'xml',
                decl_data
                )
            self.__ownerDoc.insertBefore(xml_decl_node, self.__ownerDoc.docType)
        before_doctype = 1
        for o_node in self.__orphanedNodes:
            if o_node[0] == 'pi':
                pi = self.__ownerDoc.createProcessingInstruction(
                    o_node[1],
                    o_node[2]
                    )
                if before_doctype:
                    self.__ownerDoc.insertBefore(pi, self.__dt)
                else:
                    self.__ownerDoc.appendChild(pi)
            elif o_node[0] == 'comment':
                comment = self.__ownerDoc.createComment(o_node[1])
                if before_doctype:
                    self.__ownerDoc.insertBefore(comment, self.__dt)
                else:
                    self.__ownerDoc.appendChild(comment)
            elif o_node[0] == 'doctype':
                before_doctype = 0
        self.__rootNode = self.__ownerDoc
        self.__nodeStack.append(self.__rootNode)
        return

    def __completeTextNode(self):
        #Note some parsers don;t report ignorable white space properly
        if self.__currText and len(self.__nodeStack) and self.__nodeStack[-1].nodeType != Node.DOCUMENT_NODE:
            new_text = self.__ownerDoc.createTextNode(self.__currText)
            self.__nodeStack[-1].appendChild(new_text)
        self.__currText = ''
        return

    def getRootNode(self):
        self.__completeTextNode()
        return self.__rootNode

    #Overridden DocumentHandler methods
    def processingInstruction (self, target, data):
        if self.__rootNode:
            self.__completeTextNode()
            pi = self.__ownerDoc.createProcessingInstruction(target, data)
            self.__nodeStack[-1].appendChild(pi)
        else:
            self.__orphanedNodes.append(('pi', target, data))
        return

    def startElement(self, name, attribs):
        self.__completeTextNode()
        old_nss = {}
        del_nss = []
        for curr_attrib_key, value in attribs.items():
            (prefix, local) = SplitQName(curr_attrib_key)
            if local == 'xmlns':
                if self.__namespaces.has_key(prefix):
                    old_nss[prefix] = self.__namespaces[prefix]
                else:
                    del_nss.append(prefix)
                if (prefix or value):
                    self.__namespaces[prefix] = attribs[curr_attrib_key]
                else:
                    del self.__namespaces[prefix]

        self.__namespaceStack.append((old_nss, del_nss))
        (prefix, local) = SplitQName(name)
        nameSpace = self.__namespaces.get(prefix, '')

        if self.__ownerDoc:
            new_element = self.__ownerDoc.createElementNS(nameSpace, (prefix and prefix + ':' +  local) or local)
        else:
            self.__initRootNode(nameSpace, name)
            new_element = self.__ownerDoc.documentElement

        for curr_attrib_key,curr_attrib_value in attribs.items():
            (prefix, local) = SplitQName(curr_attrib_key)
            if local == 'xmlns':
                namespace = XMLNS_NAMESPACE
                attr = self.__ownerDoc.createAttributeNS(namespace,
                                                         local + ':' + prefix)
            else:
                namespace = prefix and self.__namespaces.get(prefix, '') or ''
                attr = self.__ownerDoc.createAttributeNS(namespace,
                                                         (prefix and prefix + ':' + local) or local)
            attr.value = curr_attrib_value
            new_element.setAttributeNodeNS(attr)
        self.__nodeStack.append(new_element)
        return

    def endElement(self, name):
        self.__completeTextNode()
        new_element = self.__nodeStack[-1]
        del self.__nodeStack[-1]
        old_nss, del_nss = self.__namespaceStack[-1]
        del self.__namespaceStack[-1]
        self.__namespaces.update(old_nss)
        for prefix in del_nss:
            del self.__namespaces[prefix]
        if new_element != self.__ownerDoc.documentElement:
            self.__nodeStack[-1].appendChild(new_element)
        return

    def ignorableWhitespace(self, ch, start, length):
        """
        If 'keepAllWs' permits, add ignorable white-space as a text node.
        A Document node cannot contain text nodes directly.
        If the white-space occurs outside the root element, there is no place
        for it in the DOM and it must be discarded.
        """
        if self.__keepAllWs and self.__nodeStack[-1].nodeType !=  Node.DOCUMENT_NODE:
            self.__currText = self.__currText + ch[start:start+length]
        return

    def characters(self, ch, start, length):
        self.__currText = self.__currText + ch[start:start+length]
        return

    #Overridden LexicalHandler methods
    def xmlDecl(self, version, encoding, standalone):
        self.__xmlDecl = {'version': version, 'encoding': encoding, 'standalone': standalone}
        return

    def startDTD(self, doctype, publicID, systemID):
        if not self.__rootNode:
            self.__dt = implementation.createDocumentType(doctype, publicID, systemID)
            self.__orphanedNodes.append(('doctype'))
        else:
            raise 'Illegal DocType declaration'
        return

    def comment(self, text):
        if self.__rootNode:
            self.__completeTextNode()
            new_comment = self.__ownerDoc.createComment(text)
            self.__nodeStack[-1].appendChild(new_comment)
        else:
            self.__orphanedNodes.append(('comment', text))
        return

    def startCDATA(self):
        self.__completeTextNode()
        return

    def endCDATA(self):
        #NOTE: this doesn't handle the error where endCDATA is called
        #Without corresponding startCDATA.  Is this a problem?
        if self.__currText:
            new_text = self.__ownerDoc.createCDATASection(self.__currText)
            self.__nodeStack[-1].appendChild(new_text)
            self.__currText = ''
        return

    #Overridden DTDHandler methods
    def notationDecl (self, name, publicId, systemId):
        new_notation = self.__ownerDoc.getFactory().createNotation(self.__ownerDoc,  publicId, systemId, name)
        self.__ownerDoc.getDocumentType().getNotations().setNamedItem(new_notation)
        return

    def unparsedEntityDecl (self, publicId, systemId, notationName):
        new_notation = self.__ownerDoc.getFactory().createEntity(self.__ownerDoc,  publicId, systemId, notationName)
        self.__ownerDoc.getDocumentType().getEntities().setNamedItem(new_notation)
        return

    #Overridden ErrorHandler methods
    #FIXME: How do we handle warnings?

    def error(self, exception):
        raise exception

    def fatalError(self, exception):
        raise exception


def FromXmlStream(stream,
                  ownerDocument=None,
                  validate=0,
                  keepAllWs=0,
                  catName=None,
                  saxHandlerClass=XmlDomGenerator):
    #Create an XML DOM from SAX events
    parser = (validate and saxexts.XMLValParserFactory.make_parser()) or  saxexts.XMLParserFactory.make_parser()
    if catName:
        #set up the catalog, if there is one
        from xml.parsers.xmlproc import catalog
        cat_handler = catalog.SAX_catalog(catName, catalog.CatParserFactory())
        parser.setEntityResolver(cat_handler)
    handler = saxHandlerClass(ownerDocument, keepAllWs)
    parser.setDocumentHandler(handler)
    parser.setDTDHandler(handler)
    parser.setErrorHandler(handler)
    parser.parseFile(stream)
    return handler.getRootNode()

def FromXml(text,
            ownerDocument=None,
            validate=0,
            keepAllWs=0,
            catName=None,
            saxHandlerClass=XmlDomGenerator):
    fp = cStringIO.StringIO(text)
    rv = FromXmlStream(fp, ownerDocument, validate, keepAllWs, catName, saxHandlerClass)
    return rv

def FromXmlFile(fileName,
                ownerDocument=None,
                validate=0,
                keepAllWs=0,
                catName=None,
                saxHandlerClass=XmlDomGenerator):
    fp = open(fileName, 'r')
    rv = FromXmlStream(fp, ownerDocument, validate, keepAllWs, catName, saxHandlerClass)
    fp.close()
    return rv

def FromXmlUrl(url,
               ownerDocument=None,
               validate=0,
               keepAllWs=0,
               catName=None,
               saxHandlerClass=XmlDomGenerator):
    import urllib
    fp = urllib.urlopen(url)
    rv = FromXmlStream(fp, ownerDocument, validate, keepAllWs, catName, saxHandlerClass)
    fp.close()
    return rv

