"""Test possible errors in the setup script -- make sure they are
   caught and reported sensible."""

# created 2000/03/02, Greg Ward

__revision__ = "$Id$"

import sys, os, string
from shutil import rmtree
from popen2 import popen3
from distutils.util import mkpath


# The errors we're testing for are:
#   - bad module name (no corresponding .py file)
#   - bad package name (no corresponding dir or  __init__.py)
#   - bad package/module (conflict between .py and directory)
#   - bad package dir
#   - bad extension list format (not a list of tuples, or bad tuple)
#   - bad library list (ditto)
#   - bad source file in extension/library list
#   - bad include/library dir in extension/library list
#   - missing meta-data (warn only)
#   - bad extension command class
#
# Each expected error is described below as a tuple:
#   (description of expected error (printed to stdout),
#    text of setup script (except "from distutils.core import setup"),
#    list of files to create in the source tree (all empty),
#    distutils command that provokes the error,
#    regex to match expected error message)
#
# Each test will be run by writing its setup script into a temporary
# directory, chdir'ing into that directory, and running the setup script
# with the error-provoking commands.  (No options are needed because these
# are errors in the setup script, not on the command-line.)  The script is
# run with popen3(), so we can examine its stdout and stderr.

sample1 = (
    "bad module name (missing .py file)",
    "setup (modules = ['foo', 'bar'])",
    ['foo.py', 'ding/dong.py'],
    "build_py",
    "bad module name 'bar': could not find 'bar.py'")

sample2 = (
    "bad package name (missing dir)",
    "setup (packages = ['bar'])",
    ['foo.py', 'ding/dong.py'],
    "build_py",
    "bad package name 'bar': directory 'bar' not found")

sample3 = (
    "bad package name (missing __init__.py)",
    "setup (packages = ['bar'])",
    ['foo.py', 'bar/ding.py', 'bar/dong.py'],
    "build_py",
    "bad package name 'bar': 'bar/__init__.py' not found")



TMP_DIR = "tmp"


def write_setup_script (dir, contents):
    f = open (os.path.join (dir, "setup.py"), 'w')
    f.write ("from distutils.core import setup\n")
    f.write (contents)
    f.close ()

def create_files (dir, files):
    for file in files:
        mkpath (os.path.dirname (file))
        open (file, 'w').close()


def run_setup_script (dir, command):
    os.chdir (dir)
    (stdout, stdin, stderr) = popen3 ("%s setup.py %s" %
                                      (sys.executable, command))
    stdin.close ()
    out = string.join (stdout.readlines(), '')
    err = string.join (stderr.readlines(), '')
    stdout.close ()
    stderr.close ()
    os.chdir (os.pardir)
    return (out, err)


def try_setup (setup):
    (desc, script, files, command, error) = setup
    os.mkdir (TMP_DIR)
    write_setup_script (TMP_DIR, script)
    create_files (TMP_DIR, files)
    (stdout, stderr) = run_setup_script (TMP_DIR, command)
    print "stdout: >>>" + stdout + "<<<"
    print "stderr: >>>" + stderr + "<<<"
    rmtree (TMP_DIR)


if os.path.isdir (TMP_DIR):
    rmtree (TMP_DIR)

try_setup (sample1)
try_setup (sample2)
try_setup (sample3)

