"""Utility functions for writing test code:

      dump -
         dumps (almost) any Python object in a fairly human-readable
         form; especially handy for instances (actually taken from
         the 'dumper' module)

      try_it -
         eval a bit of code, checking carefully that the expected
         exception was raised (or that no exception was raised, if
         one wasn't expected)

      test -
         print a one-line summary of whether a test succeded
"""

# created 1999/02/05, Greg Ward

# XXX This is essentially the same as my mems.test.util module -- but I
# need a copy for Distutils.  This is ugly for both mainteance and legal
# reasons.  :-(

__revision__ = "$Id"

import sys
import dumper


# dump just comes from my dumper module
dump = dumper.dump


def caller (level=1):
    """Return the execution frame for some function up the call stack,
       by default (level=1) the immediate caller.  level=0 gives your
       own execution frame (that of the function that called 'caller');
       level=2 gives that of your caller's caller, and so on.  Raises
       RuntimeErro If level is greater than the depth of the call stack."""

    try:
        raise RuntimeError
    except RuntimeError:
        tb = sys.exc_info()[2]

    # Now 'tb' is the traceback containing the frame for *this* function;
    # at the very least, we want our *caller's* frame, so...
    frame = tb.tb_frame.f_back
    if frame is None:
        raise RuntimeError, "'caller' has no caller!"

    # Now follow 'next' pointers until we've reached the desired level
    while level > 0:
        frame = frame.f_back
        if frame is None:
            raise RuntimeError, 'call stack not deep enough'
        level = level - 1

    return frame


def extract_caller (frame=None):
    """Extracts and returns a (filename, lineno, function_name,
       code_line) tuple from an execution frame.  If 'frame' is not
       supplied or None, gets the execution frame of your caller and
       uses it."""

    import linecache
    
    # No execution frame given? then get that of our caller's caller
    if frame is None:
        frame = caller (2)

    filename = frame.f_code.co_filename
    lineno = frame.f_lineno
    function_name = frame.f_code.co_name
    code_line = linecache.getline (filename, lineno)

    return (filename, lineno, function_name, code_line)


def try_it (code, exc, descr=None, globals=None, locals=None):

    """Attempt to evaluate 'code', expecting it to raise exception 'exc'
       (which may be None if you don't expect an exception).  If
       'globals' and 'locals' are supplied, they represent the
       environment in which to evaluate 'code'; otherwise, the global
       and local namespaces of try_it's caller will be used.

       Prints a string starting with "ok: " if the code executed as
       expected (either raising the expected exception, or raising no
       exception if 'exc' was None); prints a string starting with "not
       ok: " if the expectations are violated.  If 'descr' supplied,
       prints it after the "ok"/"not ok" message, otherwise generates a
       message from 'code', the expected behaviour, the expected
       exception, and the observed behaviour."""
       
    frame = caller ()
    if globals is None: globals = frame.f_globals
    if locals is None: locals = frame.f_locals
    expected_exc = exc and exc.__name__ or "nothing"
    show_traceback = 0

    try:
        eval (code, globals, locals)
        if exc:
            ok = 0
            msg = "expected '%s' to raise %s" % (code, expected_exc)
        else:
            ok = 1
            msg = "'%s' evaluated successfully" % code
            
    except:
        (actual_exc, exc_value) = sys.exc_info()[0:2]
        if actual_exc is exc:
            ok = 1
            msg = 'as expected, "%s" raised %s ("%s")' % \
                  (code, expected_exc, exc_value)
        else:
            ok = 0
            msg = 'expected "%s" to raise %s, but raised %s ("%s")' % \
                  (code, expected_exc, actual_exc.__name__, exc_value)
            show_traceback = sys.exc_info ()
            #raise

    print (ok and "ok" or "not ok") + ": " + (descr or msg)
    if show_traceback:
        from traceback import print_exception
        apply (print_exception, show_traceback)
        
    # XXX would be useful to print stacktrace for unexpected exception!
    

def test (ok, descr):
    """Print a one-line summary of whether a test succeeded: "ok" or
       "not ok" (depending on whether 'ok' is true or false), followed
       by 'descr' (with a bit of intervening colon and whitespace
       to make it look pretty).

       Might someday also put in a number after the "ok"/"not ok" for
       better tracking of tests at runtime."""

    print (ok and "ok" or "not ok") + ": " + descr

# end test
