import java.awt.*;
import java.awt.event.*;
import java.sql.*;
import java.util.*;
import java.io.*;

public class SqlConsole extends Frame
{
    private static final String VERSION = "1.1b";

    private static final int FONT_STYLE = Font.PLAIN;
    private static final int FONT_SIZE = 14;
    private static final String PRINT_FONT_NAME = "Monospaced";
    private static final float FONT_PRINT_SCALE_FACTOR = (float)0.5; 
    private static final int FONT_PRINT_INSET_X = 30;
    private static final int FONT_PRINT_INSET_Y = 30; 
    private static final String FONT_WIDTH_MEASUREMENT_STRING =
        "  ABCDabcd01231234 pqrsPQRS 012  "; 

    private TextArea commandString;
    private TextArea resultString;

    public SqlConsole (DBConnection conn,   // null indicates no connection
		       String jdbc_driver,  // name of the JDBC driver, used only when the connection (conn) is null
		       String db_name,
		       boolean exit_on_close,  // when the user closes, the program exits.
		       boolean close_on_logout) // when user logs out, the window closes (and exits if exit_on_close == true 
	throws Exception
    {
	super ("SQL Console : not logged in");
	dbName = db_name;
	exitOnClose = exit_on_close;
	closeOnLogout = close_on_logout;

	if (conn != null) // already connected
	    {
		setTitle ("SQL Console : user \"" + 
			  conn.getUser() +
			  "\" host \"" + conn.getHost() + "\""); 
		dbConn = conn;
	    }
	else
	    {
		dbConn = new DBConnection (jdbc_driver); // load the class
	    }

	GridBagLayout gbl = new GridBagLayout();
	GridBagConstraints gbc = new GridBagConstraints();
	setLayout (gbl);

	MenuBar mbar = new MenuBar();
	setMenuBar (mbar);

	Menu file_menu = new Menu (" File ");
	mbar.add (file_menu);

	MenuItem logout_mi = new MenuItem (" logout ");
	logout_mi.addActionListener (new LogoutListener());
	file_menu.add (logout_mi);

	MenuItem save_mi = new MenuItem (" Save ... ");
	save_mi.addActionListener (new SaveListener());
	file_menu.add (save_mi);

	MenuItem print_mi = new MenuItem (" Print ... ");
	print_mi.addActionListener (new PrintListener());
	file_menu.add (print_mi);

	file_menu.addSeparator();
	
	MenuItem exit_mi = new MenuItem (" Exit ");
	exit_mi.addActionListener (new ExitListener());
	file_menu.add (exit_mi);

	Menu edit_menu = new Menu (" Edit ");
	mbar.add (edit_menu);

	MenuItem clear_mi = new MenuItem (" Clear ");
	clear_mi.addActionListener (new ResetListener());
	edit_menu.add (clear_mi);

	edit_menu.addSeparator();

	MenuItem select_mi = new MenuItem (" Select All ");
	select_mi.addActionListener (new SelectListener());
	edit_menu.add (select_mi);

	MenuItem copy_mi = new MenuItem (" Copy ");
	copy_mi.addActionListener (new CopyListener());
	edit_menu.add (copy_mi);

	MenuItem paste_mi = new MenuItem (" Paste ");
	paste_mi.addActionListener (new PasteListener());
	edit_menu.add (paste_mi);

	Panel panel1 = new Panel();
	panel1.setLayout (gbl);

	addToContainer (panel1,
			gbl,
			gbc,
			new Label ("SQL>"),
			0,
			0,
			1,
			1,
			GridBagConstraints.BOTH,
			GridBagConstraints.NORTHWEST,
			0, 0);

	commandString = new TextArea (5, 80);
	commandString.setFont (new Font ("Monospaced",
			      Font.PLAIN,
			      14));
	commandString.setEditable (false);

	addToContainer (panel1,
			gbl,
			gbc,
			commandString,
			0,
			1,
			1,
			1,
			GridBagConstraints.BOTH,
			GridBagConstraints.NORTHWEST,
			100, 0);
	commandString.addKeyListener (new SpecialKeyListener());

	Button submit_button = new Button (" Submit ");
	submit_button.addActionListener (new SubmitListener());
	addToContainer (panel1,
			gbl,
			gbc,
			submit_button,
			1,
			1,
			1,
			1,
			GridBagConstraints.NONE,
			GridBagConstraints.NORTHWEST,
			0, 0);

	addToContainer (panel1,
			gbl,
			gbc,
			new Label ("Result"),
			0,
			2,
			1,
			1,
			GridBagConstraints.NONE,
			GridBagConstraints.NORTHWEST,
			0, 0);

	resultString = new TextArea (8, 80);
	resultString.setFont (new Font ("Monospaced",
			      Font.PLAIN,
			      14));
	resultString.setEditable (false);
	addToContainer (panel1,
			gbl,
			gbc,
			resultString,
			0,
			3,
			1,
			1,
			GridBagConstraints.BOTH,
			GridBagConstraints.NORTHWEST,
			100, 100);

	// add panel1 to the dialog box
	addToContainer (this,
			gbl,
			gbc,
			panel1,
			0,
			0,
			1,
			1,
			GridBagConstraints.BOTH,
			GridBagConstraints.NORTHWEST,
			100, 100,
			new Insets (5, 5, 0, 5));

	Panel panel2 = new Panel();
	panel2.setLayout (gbl);

	Button logout_button = new Button (" logout ");
	logout_button.addActionListener (new LogoutListener());
	addToContainer (panel2,
			gbl,
			gbc,
			logout_button,
			0,
			0,
			1,
			1,
			GridBagConstraints.NONE,
			GridBagConstraints.CENTER,
			0, 0);

	Button reset_button = new Button (" Clear ");
	reset_button.addActionListener (new ResetListener());
	addToContainer (panel2,
			gbl,
			gbc,
			reset_button,
			1,
			0,
			1,
			1,
			GridBagConstraints.NONE,
			GridBagConstraints.CENTER,
			0, 0);

	// add panel2 to the dialog box
	addToContainer (this,
			gbl,
			gbc,
			panel2,
			0,
			1,
			1,
			1,
			GridBagConstraints.BOTH,
			GridBagConstraints.NORTHWEST,
			0, 0, new Insets (5, 5, 5, 5));	

	addWindowListener (new WindowCloseListener());

	pack();
	show();

	if (conn == null) // if no connection was specified
	    {
		while (login(db_name) == false) ;
	    }
    }

    // This constructor is for the case when the user has not logged on to the database.
    // A logon dialog is displayed
    public SqlConsole (String jdbc_driver, String db_name, boolean exit_on_close, boolean close_on_logout) throws Exception
    {
	this (null, jdbc_driver, db_name, exit_on_close, close_on_logout);
    }

    // This constructor is for the case where the user has already logged on to the database and close_on_logout == true.
    public SqlConsole (DBConnection conn, boolean exit_on_close) throws Exception
    {
	this (conn, "", "", exit_on_close, true);
    }    

    private boolean login (String db_name)
    {
	LoginDialog ld = new LoginDialog (this);
	String host = ld.getHost();
	if (host.length() <= 0) host = new String ("localhost");
	String db = ld.getDb();
	if (db.length() <= 0) db = new String ("test");

	try
	    {
		dbConn.login ("jdbc:" + db_name,
			      host,
			      ld.getUser(),
			      ld.getPassword(),
			      db);
		
		connection = dbConn.getConnection();
		setTitle ("SQL Console : user \"" 
			  + dbConn.getUser()
			  + "\" host \"" + dbConn.getHost() + "\"");
		commandString.setEditable (true);
		return true;
	    }
	catch (SQLException ex1)
	    {
		new YesNoDialog (this, "Error !",
				 ex1.getMessage(),
				 " Dismiss ", null);
		return false;
	    }
    }

    private void close ()
    {
	try
	    {
		if (connection != null) dbConn.dispose();
	    }
	catch (SQLException ex)
	    {
		new YesNoDialog (SqlConsole.this, "Error ! ", 
				 "SQLException : " + ex.getMessage(),
				 " Dismiss ", null);
	    }
	
	SqlConsole.this.hide();
	SqlConsole.this.dispose();
	
	if (exitOnClose == true)
	    {
		System.exit(0);
	    }	
    }

    private void logout ()
    {
	try
	    {
		dbConn.dispose();
		connection = null;
		clear();
		setTitle("SQL Console : not logged in");
		commandString.setEditable (false);

		if (closeOnLogout == true)
		    {
			hide();
			dispose();
			
			if (exitOnClose == true)
			    {
				System.exit (0);
			    }
		    }
		else
		    {
			while (login(dbName) == false) ;
		    }
	    }
	catch (SQLException ex)
	    {
		new YesNoDialog (SqlConsole.this, "Error ! ", 
				 "SQLException : " + ex.getMessage(),
				 " Dismiss ", null);		    
	    }
    }

    private void clear ()
    {
	resultString.setText ("");
	commandString.setText ("");
    }

    private void processQuery ()
    {
	String command = commandString.getText();

	StringTokenizer strtok = new StringTokenizer (command, ";");
	int num_tokens = strtok.countTokens();

	commandString.setText ("");
	for (int k = 0; k < num_tokens; k++)
	    {
		String token = strtok.nextToken().trim();

		if (token.length() > 0)
		    {
			resultString.append ("SQL> " + token + '\n');
			try
			    {
				Statement s = connection.createStatement();
				boolean result_available = s.execute (token);
			    
				if (result_available == true)
				    {
					ResultSet r = s.getResultSet();
					ResultSetMetaData m = r.getMetaData();
					    
					// print the result
					int cols = m.getColumnCount();
					int width = 0;
					String buffer = "";
					StringBuffer bar = new StringBuffer();
					int i;
					int row_count = 0;
					    
					// top bar
					for (i = 0; i < cols; i++)
					    {
						width += m.getColumnDisplaySize(i+1);
					    }
					width += 1 + cols;
					    
					for (i = 0; i < width; i++)
					    {
						bar.append ('-');
					    }
					bar.append ('\n');
					buffer += bar + "|";
					    
					// now print the columns
					for (i = 0; i < cols; i++)
					    {
						StringBuffer filler = new StringBuffer();
						String label = m.getColumnLabel(i+1);
						int size = m.getColumnDisplaySize (i+1);
						    
						int x;
						if (label.length() > size)
						    {
							label = label.substring (0, size);
						    }
						    
						if (label.length() < size)
						    {
							x = size - label.length();
							int j;
							for (j = 0; j < x; j++)
							    {
								filler.append(' ');
							    }
							    
							label = label + filler;
						    }
						    
						buffer = buffer + label + "|";
					    }
					    
					// lower bar
					buffer = buffer + '\n' + bar;
					    
					// now, add the rows
					while (r.next() == true)
					    {
						row_count++;
						buffer += "|";
						    
						    
						// format each column of the row
						for (i = 0; i < cols; i++)
						    {
							StringBuffer filler = new StringBuffer();
							Object value = r.getObject (i+1);
							    
							int size = m.getColumnDisplaySize (i+1);
							String str;
							    
							if (value == null)
							    {
								str = "";
							    }
							else
							    {
								str = value.toString();
							    }
							    
							if (str.length() > size)
							    {
								str = str.substring (0, size);
							    }
							    
							if (str.length() < size)
							    {
								int x = size - str.length();
								int j;
								for (j = 0; j < x; j++)
								    {
									filler.append (' ');
								    }
								str = str + filler;
							    }
							    
							str = str.replace ('\n', ' ');
							    
							    
							buffer = buffer + str + "|";
						    }
						buffer = buffer + '\n';
					    }
					buffer = buffer + bar + '\n' + "Row count = " + row_count + "\n\n";
					resultString.append  (buffer);
					    
					r.close();
				    }
				else
				    {
					resultString.append ("Number of affected rows = " + 
							     s.getUpdateCount() + "\n\n");
				    }
				    
				s.close();
			    }
			catch (SQLException ex)
			    {
				resultString.append  (ex.getMessage() + "\n\n");
				new YesNoDialog (this, "Error ! ", 
						 "SQLException : " + ex.getMessage(),
						 " Dismiss ", null);
				continue;
			    }
		    }
	    }
    }

    private void printOutput ()
    {
	String text = resultString.getText();
	if (text.length() == 0) return;

	Toolkit tk = getToolkit();
	PrintJob pj = tk.getPrintJob (SqlConsole.this, "Print SQL Output",
				      new Properties());
	if (pj != null)
	    {
		Graphics g = pj.getGraphics();

		Font font = new Font(PRINT_FONT_NAME,
				     FONT_STYLE,
				     (int)((float)FONT_SIZE * FONT_PRINT_SCALE_FACTOR));
		FontMetrics fm = getFontMetrics (font);
		int lineheight = fm.getHeight();
		int cwidth =  fm.stringWidth (FONT_WIDTH_MEASUREMENT_STRING) /
		    FONT_WIDTH_MEASUREMENT_STRING.length();
 
		Dimension dim = pj.getPageDimension();
		int pagewidth = dim.width;
		int pageheight = dim.height;
		int num_char_line = (pagewidth - (2 * FONT_PRINT_INSET_X)) / cwidth;
 
		g.setFont (font); 
		g.setColor (Color.black); 

		int line = 0;

		// tokenize the text into lines
		StringTokenizer strtok = new StringTokenizer (text, "\n");
		int num_tokens = strtok.countTokens();
		for (int i = 0; i < num_tokens; i++)
		    {
			String line_text = strtok.nextToken();
			int len = line_text.length();
			int num_lines = len / num_char_line;
			if ((len % num_char_line) > 0) num_lines++;

			int offset = 0;
			for (int j = 0; j < num_lines; j++)
			    {
				String submsg;
				try
				    {
					submsg = line_text.substring (offset,
								      offset+num_char_line);
				    }
				catch (StringIndexOutOfBoundsException ex)
				    {
					submsg = line_text.substring (offset);
				    }
				    
				offset+= num_char_line;
				    
				int ycoord = FONT_PRINT_INSET_Y + ((line+1)*lineheight);
				if (ycoord >= pageheight - FONT_PRINT_INSET_Y) // next page
				    {
					g.dispose();
					g = pj.getGraphics();
					g.setFont (font);
					    
					ycoord = FONT_PRINT_INSET_Y + lineheight;
					line = 0;
				    }
				    
				g.drawString (submsg, FONT_PRINT_INSET_X, ycoord);
				line++;
			    } // for j
		    } // for i
		g.dispose();
		pj.end();
	    }   
    }

    private void saveOutput ()
    {
	FileDialog fd = new FileDialog (SqlConsole.this, "Save SQL output to a file",
					FileDialog.SAVE);
	fd.show();
	String file_name = fd.getFile();
	if (file_name == null)
	    {
		// cancel selected
		return;
	    }

	File file = new File (fd.getDirectory(), file_name);
	String path_name = file.getAbsolutePath();
	try
	    {
		// save to this file
		FileOutputStream fos = new FileOutputStream (path_name, true);
		OutputStreamWriter osw = new OutputStreamWriter (fos);
		    
		String output = new String ("Output generated at " 
					    +  (new java.util.Date()) 
					    + " by user : "
					    + dbConn.getUser()
					    + " logged on to host : "
					    + dbConn.getHost()
					    + '\n'
					    + '\n'
					    + resultString.getText()
					    + '\n'
					    + '\n');
		osw.write (output, 0, output.length());
		osw.close();
	    }
	catch (IOException ex1)
	    {
		new YesNoDialog (SqlConsole.this, "Error ! ", 
				 "IO Exception : " + ex1.getMessage(),
				 " Dismiss ", null);
		return;
	    }
    }
    
    private void addToContainer (Container cont,
				 GridBagLayout gbl,
				 GridBagConstraints gbc,
				 Component comp,
				 int gridx,
				 int gridy,
				 int gridwidth,
				 int gridheight,
				 int fill,
				 int anchor,
				 int weightx,
				 int weighty,
				 Insets insets)
    {
	gbc.gridx = gridx;
	gbc.gridy = gridy;
	gbc.gridwidth = gridwidth;
	gbc.gridheight = gridheight;
	gbc.fill = fill;
	gbc.anchor = anchor;
	gbc.weightx = weightx;
	gbc.weighty = weighty;

	if (insets != null)
	    {
		gbc.insets = insets;
	    }

	gbl.setConstraints (comp, gbc);
	cont.add (comp);
    }

    private void addToContainer (Container cont,
				 GridBagLayout gbl,
				 GridBagConstraints gbc,
				 Component comp,
				 int gridx,
				 int gridy,
				 int gridwidth,
				 int gridheight,
				 int fill,
				 int anchor,
				 int weightx,
				 int weighty)
    {
	addToContainer (cont, gbl, gbc, comp, gridx, gridy, gridwidth, gridheight,
			fill, anchor, weightx, weighty, null);
    }



    class LogoutListener implements ActionListener
    {
	public void actionPerformed (ActionEvent e)
	{
	    logout();
	}
    };

    class WindowCloseListener extends WindowAdapter
    {
	public void windowClosing (WindowEvent e)
	{
	    close();
	}
    };

    class ExitListener implements ActionListener
    {
	public void actionPerformed (ActionEvent e)
	{
	    close();
	}	
    };

    class ResetListener implements ActionListener
    {
	public void actionPerformed (ActionEvent e)
	{
	    clear();
	}
    };

    class SelectListener implements ActionListener
    {
	public void actionPerformed (ActionEvent e)
	{
	    resultString.selectAll();
	}
    };

    class CopyListener implements ActionListener
    {
	public void actionPerformed (ActionEvent e)
	{
	    String text = commandString.getSelectedText();
	    if (text.length() == 0) // no text selected
		{
		    text = resultString.getSelectedText();
		    if (text.length() == 0) return;
		    else //unselect the area
			{
			    int cp = resultString.getCaretPosition();
			    resultString.select (0, 0);
			    resultString.setCaretPosition(cp);
			}
		}
	    else // unselect the area
		{
		    int cp = commandString.getCaretPosition();
		    commandString.select (0, 0);
		    commandString.setCaretPosition(cp);
		}

	    TextClipBoard.Instance().copyToClipBoard (text);
	}
    };

    class PasteListener implements ActionListener
    {
	public void actionPerformed (ActionEvent e)
	{
	    String text = TextClipBoard.Instance().getClipBoardContent ();
	    if (text.length() > 0)
		{
		    commandString.insert (text, commandString.getCaretPosition());
		}
	}
    };

    class SaveListener implements ActionListener
    {
	public void actionPerformed (ActionEvent e)
	{
	    saveOutput();
	}
    };

    class PrintListener implements ActionListener
    {
	public void actionPerformed (ActionEvent e)
	{
	    printOutput();
	}
    };

   class SubmitListener implements ActionListener
    {
	public void actionPerformed (ActionEvent e)
	{
	    processQuery();
	}
    };

    class SpecialKeyListener extends KeyAdapter
    {
	public void keyPressed (KeyEvent e)
	{
	    int key = e.getKeyCode();
	    Component parent = e.getComponent();

	    if (key == KeyEvent.VK_END)
		{
		    processQuery();
		}
	}
    };


    private Connection connection = null;
    private DBConnection dbConn = null;
    private boolean exitOnClose;
    private boolean closeOnLogout;
    private String dbName;

    public static void main (String[] args)
    {
	String jdbc_driver = new String ("org.gjt.mm.mysql.Driver");
	String db_name = new String ("mysql");

	for (int i = 0; i < args.length; i++)
	    {
		if (args[i].equals ("-d") == true)
		    {
			if (i == args.length - 1) // if this is the last argument
			    {
				System.out.println ("No driver specified");
				System.exit (1);
			    }

			i++;
			jdbc_driver = new String (args[i]);
			continue;
		    }
		else if (args[i].equals ("-u") == true)
		    {
			if (i == args.length - 1) // if this is the last argument
			    {
				System.out.println ("No database type specified");
				System.exit (1);
			    }

			i++;
			db_name = new String (args[i]);
			continue;
		    }
		else if (args[i].equals ("-v") == true)
		    {
			System.out.println ("SQL Console Version " + VERSION);
			System.exit (0);
		    }
		
		System.out.println ("Unknown option " + args[i]);
		System.exit (1);
	    }

	try
	    {
		SqlConsole  console = new SqlConsole(jdbc_driver, db_name, true, false);
	    }
	catch (Exception ex) 
	    {
		System.out.println ("Exception : " + ex.getMessage());
		System.exit (1);
	    }
    }
}









