#!/usr/bin/perl
#
# MRTG topN report generator
#
# copyright 2001 Derek Chen-Becker <dbecker@cpicorp.com>
#
# permission is hereby granted to modify/use this software
# as long as the original author(s) are credited.
#
# There is no warranty with this software, either written or
# implied.
#
# (neither are any claims made that I actually know Perl...)
#
# This script takes a directory where MRTG logs can be found  and picks 
# the top N logs in terms of most recent counter value and generates a 
# file called topN.html in the directory that the logs are in.

use Getopt::Std;

my $bgcolor = "White";
my $adminName = "Derek Chen-Becker";
my $adminMail = "dbecker\@cpicorp.com";
my $dataDivisor = 128; #divide the counter value (128 = kb/s)
my $dataLabel = "kb/s";

sub usage
{
    print <<"USAGEPRINT";
    
Generate topN reports from MRTG logs. Output to "topN.html" in the same
directory as the logs.

Usage: $0 [-m] [-v <ag|in|out>] [-c <count>] <log file directory>
 -m      :Use maximum values instead of averages (default = average)
 -v      :Which traffic to use for topN determination:
         ag      :Aggregate (in+out) (default)
         in      :Inbound traffic only
         out     :Outbound traffic only
 -c      :top <count> interface report is generated (default 5)
    
USAGEPRINT
    exit;
}

my %opts;

getopts("mv:c:",\%opts) or &usage;

if($#ARGV != 0)
{
    print "No log directory specified!\n";
    &usage;
}

if($opts{"m"})
{
    #print "Max values.\n";
    $inIndex = 3;
    $outIndex = 4;
} else {
    #print "Average values.\n";
    $inIndex = 1;
    $outIndex = 2;
}

$maxInterfaces = $opts{"c"} || 5;

#print "Parsing files from \"$ARGV[0]\"...\n";

@logfiles = glob("$ARGV[0]/*.log");

my %interfaceTraffic;
my %interfaceName;

for (@logfiles)
{
    #read in the second line
    $filepath = $_;
    open LOG, "< $filepath" or die "Cannot open $_ for read!\n";
    $line = <LOG>;
    $line = <LOG>;
    @vals = split ' ',$line;
    
    #change logfile into html file
    @fileparts = split(/\./,$filepath);
    $fileparts[$#fileparts] = "html";
    $webpage = join ".", @fileparts;
    
    #get filename without extension
    $#fileparts--;
    $newfilecomp = join ".", @fileparts;
    @fileparts = split(/[\/\\]/,$newfilecomp);
    $filename = @fileparts[$#fileparts];
    #print "name = $filename\n";
    
    $_ = $opts{"v"} || "ag";
    {
        /^ag/ && do {
            #print "Aggregate\n";
            $traffic = $vals[$inIndex] + $vals[$outIndex];
            last;
        };
        /^in/ && do {
            #print "Inbound\n";
            $traffic = $vals[$inIndex];
            last;
        };
        /^out/ && do {
            #print "Outbound\n";
            $traffic = $vals[$outIndex];
            last;
        };

        print "Invalid type: $_\n";
        exit;
    }
    
    #print "$webpage: $traffic\n";
    $interfaceTraffic{"$traffic"} = $webpage;
    $interfaceName{"$traffic"} = $filename;
}

#print "output to $ARGV[0]/topN.html\n";

#$outputDir = $opts{"o"} || ".";
$outputDir = $ARGV[0];

open HTML, "> $outputDir/topN.html" or die "Cannot open output file!\n";

$nowtime = localtime;

print HTML <<"HTMLOUT";
<html>
<title>Top $maxInterfaces talkers</title>
<body bgcolor="$bgcolor">

<h2>Top $maxInterfaces talkers</h2>
<h4>Generated $nowtime</h4>

<ol>
HTMLOUT

for (sort { $b <=> $a } keys %interfaceTraffic)
{
    if($maxInterfaces > 0)
    {
        #print "$interfaceTraffic{$_}: $_\n";
        $dataVal = sprintf "%.2f", $_ / $dataDivisor;
        print HTML "  <li><a href=\"$interfaceName{$_}.html\">$interfaceName{$_} ($dataVal $dataLabel)</a>\n";
        $maxInterfaces--;
    }
}

print HTML <<"TRAILER";
</ol>

Questions or comments please contact <a href="mailto:$adminMail">$adminName</a>

<hr>
<it>Copyright &copy; 2001, 
<a href="mailto:dbecker\@cpicorp.com">Derek Chen-Becker</a></it>
</body>
</html>
TRAILER

exit;
