/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

/**
 * includes
 */
#include "config.h"

/* I'm not sure wether or not these needs to be tested for */
#include <time.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>

/* pcap header(s) */
#ifdef HAVE_PCAP_H
	#include <pcap.h>
	#include <pcap-namedb.h>
#elif HAVE_PCAP_PCAP_H
	#include <pcap/pcap.h>
	#include <pcap/pcap-namedb.h>
#else
	#error pcap headers are missing, it seems that we havent compiled on your platform yet. Pleaes fix this and submit a patch to torgeir@trenger.ro!
#endif

/* errno.h */
#ifdef HAVE_ERRNO_H
	#include <errno.h>
#else
	#error errno.h missing, it seems that we havent compiled on your platform yet. Please fix this and submit a patch to torgeir@trenger.ro!
#endif

/* stdlib.h */
#ifdef HAVE_STDLIB_H
	#include <stdlib.h>
#else
	#error stdlib.h missing, it seems that we havent compiled on your platform  yet. Please fix this and submit a patch to torgeir@trenger.ro!
#endif

/* unistd.h */
#ifdef HAVE_UNISTD_H
	#include <unistd.h>
#else
	#error unistd.h missing, it seems that we havent compiled on your platform yet. Please fix this and submit a patch to torgeir@trenger.ro!
#endif

/* string.h */
#ifdef HAVE_STRING_H
	#include <string.h>
#else
	#error string.h missing, it seems that we havent compiled on your platform yet. Please fix this and submit a patch to torgeir@trenger.ro!
#endif

/* prototypes */
int main(int, char **);
void libpcap_listen();
void ProcessPacket(char *user, struct pcap_pkthdr * pkthdr, u_char * pkt);
int axtoi(char *hexStg);
void printhelp(void);
int addmacfile(char *input);

/* for easy reading the MAC-addy's */
struct EtherHdr
{
	unsigned char ether_dst[6];
	unsigned char ether_src[6];
	unsigned short ether_type;
};

struct macentry {
	unsigned char addr[5];
	unsigned long inbytes;
	unsigned long outbytes;
	int revert;
	unsigned long nextwrite;
	char *logfile;
};

#define MAXHOSTS 512
static struct macentry maclist[255][MAXHOSTS];



/* variables */
unsigned long waitsecs = 0;
char *netdev = NULL, *user = NULL, *jail = NULL, *myname = NULL;
int minutes = 5, promisc = 1, ready = 0;
extern char *optarg;    /* for getopt */
extern int optind;      /* for getopt */
struct passwd *user_pw = NULL;
char myver[] = "$Id: macwatch.c,v 1.8 2002/05/17 22:03:30 torgeir Exp $";


/*
 * name: main
 * arguments: int argc,**argv
 * purpose: start my application 
 */
int main (int argc,char **argv) {
	char ch,buf[256];
	int i,x;
	FILE *fh;

	/* re-set to make sure */
	for (i=0;i<255;i++) {
		for(x=0;x<MAXHOSTS;x++) {
			memset((char *)maclist[i][x].addr,0,5);
			maclist[i][x].logfile = NULL;
			maclist[i][x].inbytes = 0;
			maclist[i][x].outbytes = 0;
			maclist[i][x].nextwrite = 0;
		}
	}
	myname = argv[0];

	/* check what user want's to do */
	while((ch = getopt(argc, argv, "hpg:i:u:U:C:f:a:")) != -1) {
		switch (ch) {
			case 'a': /* add mac,file */
				if (addmacfile(optarg))
					return 1;
				break;
			case 'f': /* read mac and logfiles from file */ 
				fh = fopen(optarg,"r");
				if (!fh) {
					printf("Sorry - cannot read file %s: %s\n",optarg,strerror(errno));
					return (1);
				}
				while (!feof(fh)) {
					memset((char *)buf,0,256);
					if (!fgets(buf,254,fh))
						continue;
					
					// silently ignore lines that are less than 5 bytes
					if (strlen(buf) <= 5)
						continue;
					
					// minimum 19 bytes
					if (strlen(buf) <= 19) {
						fprintf(stderr,"Invalid line: %s\n",buf);
						continue;
					}
					addmacfile(buf);
				}
				fclose(fh);
				break;
			case 'u': /* update interval, in minutes */
				minutes = atoi(optarg);
				if (minutes <= 0) {
					printf("invalid update interval, minutes ONLY please!\n");
					return 1;
				}
				break;
			case 'U': /* other user to run as */
				user = malloc(strlen(optarg)+1);
				strcpy(user,optarg);
				break;
			case 'C': /* path to my jail */
				jail = malloc(strlen(optarg)+1);
				strcpy(jail,optarg);
				break;
			case 'i': /* interface */
				if (strlen(argv[1]) > 5) {
					printf("Sorry, devicename is too long!\n");
					return 1;
				}
				netdev = malloc(strlen(optarg)+1);
				strcpy(netdev,optarg);
				break;
			case 'p': /* non promisc mode */
				promisc = 0;
				break;
			
			case 'h':
				printhelp();
				return 1;
		}	
	} /* end, getopt */
	
	/* check if any mac-addy's have been added */
	if (!ready) {
		fprintf(stderr,"You must specify one or more address(es) to monitor!\n");
		printhelp();
		return 1;
	}
	
	/* check if we are supposed to run as another user */
	if (user != NULL) {
		user_pw = getpwnam(user);
		if (user_pw == NULL) {
			printf("getpwnam() failed while fetching user %s: %s\n",user,strerror(errno));
			return 1;
		}
	}
	
	/* check if we're supposed to be put in a jail */
	if (jail != NULL) {
		if (chroot(jail)) {
			printf("Could not chroot() to %s: %s\n",jail,strerror(errno));
			return 1;
		}
	}
	
	libpcap_listen();
	
	/* not reached */
 	return (0);
} /* end main() */

/*
 * just print the usage help
 */
void printhelp(void) {
	printf("%s, v%s\n",PACKAGE,VERSION);
	printf("Usage: %s\n",myname);
	printf("Optional parametres: \n");
	printf("\t-a\t add \"mac,file[,revert]\". i.e. 00:0A:1B:AA:03:3D,/var/log/gw\n");
	printf("\t-f\t read file of \"mac,file[,revert]\" entries\n");
	printf("\t-i\t interface. i.e. eth0\n");
	printf("\t-u\t update interval, in minutes. (how often to re-write the logfile)\n");
	printf("\t-p\t turn off promiscous sniffing\n");
	printf("\t-U\t user to run as, this is for the paranoid penguins\n");
	printf("\t-C\t path we should chroot to, also for the paranoid penguins\n");
	printf("\t\t  remember that the filename(s) MUST be relative to the jail!\n");
	printf("\n");
	printf("Please send bug-reports and comments to torgeir@trenger.ro\n\n");
	return;
} /* end, printmode() */

/* add mac,file to the monitor-list..  (mac,file[,revert]) */
int addmacfile (char *input) {
	char *mac, *logfile, *tmp, a[3];
	unsigned char gateway[6];
	int i,x,revert = 0;
	
	if (!input) {
		fprintf(stderr,"addmacfile(): no input given!\n");
		return 1;
	}
	mac = input;
	logfile = strstr(mac,",");
	if (!logfile) {
		fprintf(stderr,"invalid input: %s\n",input);
		return 1;
	}
	*logfile++ = '\0';
	if (*(logfile+strlen(logfile)-1) == '\n')
		*(logfile+strlen(logfile)-1) = '\0';

	tmp = strstr(logfile,",");
	if (tmp) {
		*tmp++ = '\0';
		
		// skip whitespace and tab's, if any
		while ((*tmp != '\0') && ((*tmp == ' ') || (*tmp == '\t')))
			tmp++;
		
		if ((*tmp == 'r') || (*tmp == 'R'))
			revert = 1;
	}
	
	if (strlen(mac) != 17) {
		printf("Wrong size for gateway_MAC!\n");
		return 1;
	}
	
	/* sanity check, bytes 2,5,8,11 and 14 should be ':' it if's a valid MAC */
	if (	(mac[2] != ':') \
		|| (mac[5] != ':') \
		|| (mac[8] != ':') \
		|| (mac[11] != ':') \
		|| (mac[14] != ':') \
	 ) {
		printf("invalid mac! remember to type the 0's and use : to separate each field\n");
		printhelp();
		return 1;
	}
	
	/* fix MAC-addy */
	a[0] = mac[0];
	a[1] = mac[1];
	gateway[0] = axtoi(a);
	a[0] = mac[3];
	a[1] = mac[4];
	gateway[1] = axtoi(a);
	a[0] = mac[6];
	a[1] = mac[7];
	gateway[2] = axtoi(a);
	a[0] = mac[9];
	a[1] = mac[10];
	gateway[3] = axtoi(a);
	a[0] = mac[12];
	a[1] = mac[13];
	gateway[4] = axtoi(a);
	a[0] = mac[15];
	a[1] = mac[16];
	gateway[5] = axtoi(a);
	
	// add to the list, start by finding the last entry used in this 'bucket'
	x = (unsigned int) gateway[5];
	for(i=0; maclist[x][i].logfile; i++) { ; }
	if (i>=MAXHOSTS) {
		fprintf(stderr,"Sorry, our limit has been reached!\n");
		fprintf(stderr," edit macwatch.c, and increase MAXHOSTS if needed!\n");
		return 1;
	}
	
	// okey, add the entry!
	maclist[x][i].logfile = strdup(logfile);
	maclist[x][i].addr[0] =  gateway[0];
	maclist[x][i].addr[1] =  gateway[1];
	maclist[x][i].addr[2] =  gateway[2];
	maclist[x][i].addr[3] =  gateway[3];
	maclist[x][i].addr[4] =  gateway[4];
	maclist[x][i].revert = revert;
	ready = 1;
	
	#ifdef DEBUG
	printf("Added MAC(%02X:%02X:%02X:%02X:%02X:%02X) FILE(%s)\n",
			maclist[x][i].addr[0],
			maclist[x][i].addr[1],
			maclist[x][i].addr[2],
			maclist[x][i].addr[3],
			maclist[x][i].addr[4],
			x,
			maclist[x][i].logfile);
	#endif
	
	return 0;	
} /* end, addmacfile() */


/**
 * handling of errors, for now we only print and die
 **/
void printerr (char *errmsg) /* take care of errors */
{
	fprintf(stderr,"An error occured! (%s) Exiting..\n", errmsg);
#ifdef HAVE_STRERROR
	fprintf(stderr,"strerror tells us: %s\n",strerror(errno));
#endif
	exit (-1);
} /* end, printerr() */

/*
 * name: libpcap_listen
 * arguments: (none)
 * purpose: init the libpcap interface and run the listener
 */
void libpcap_listen()
{
	char *pcap_cmd = NULL;
	char errorbuffer[PCAP_ERRBUF_SIZE];
	bpf_u_int32 localnet, netmask;		/* net addr holders */
	struct bpf_program fcode;		/* Finite state machine holder */
	pcap_t *pds;				/* packet descriptor for interface */
	int snaplen = 50;			/* snapshot-length, only want header-stuff */
	int datalink,i,x;
	FILE *tmpfile = NULL;
	
	/* autodetect interface if user didn't specify any! */
	if (netdev == NULL) {
		netdev = pcap_lookupdev(errorbuffer);
		if (netdev == NULL) {
			printf("pcap_lookupdev() failed: %s\n",errorbuffer);
			return;
		}
	}

	printf("initializing interface: %s.. \n",netdev);

	if(pcap_lookupnet(netdev, &localnet, &netmask, errorbuffer) < 0) {
		printerr("pcap_lookupnet didn't get it, setting defaults..");
		localnet = 0;
		netmask = 0;
	}

	pds = pcap_open_live(netdev,snaplen,promisc,0,errorbuffer);
	if(pds == NULL) {
		printerr("pcap_open_live: could not open device!");
	}
	
	i = pcap_snapshot(pds);
	if (snaplen < i) {
		printf("snaplen raised from default!\n");
		snaplen = i;
	}

	if(pcap_compile(pds, &fcode, pcap_cmd, 1, netmask) < 0) {
		printerr("pcap_compile: compilation failed, sorry\n");
	}
	
	if(pcap_setfilter(pds, &fcode) < 0) {
		printerr("pcap_setfilter failed\n");
	}

	datalink = pcap_datalink(pds);
	if(datalink < 0) {
		printerr("pcap_datalink failed\n");
	}

	/* okey, let's change to the user - if any */
	if (user_pw != NULL) {
		if (setgid(user_pw->pw_gid)) {
			printf("setgid(%d) failed: %s\n",(int)user_pw->pw_gid,strerror(errno));
			exit(1);
		}
		if (setuid(user_pw->pw_uid)) {
			printf("setuid(%d) failed: %s\n",(int)user_pw->pw_uid,strerror(errno));
			exit(1);
		}
	}

	/* now check that we actually have log-access..
	 * 
	 * if file doesen't exist - try to create it
	 *
	 * go through all entries..
	 */
	for(i=0;i<=255;i++) {
		
		for(x=0;maclist[i][x].logfile && (x < MAXHOSTS);x++) {
			
			if (access(maclist[i][x].logfile,F_OK)) {
				umask(077);
				tmpfile = fopen(maclist[i][x].logfile,"a");
				if (tmpfile == NULL) {
					printf("Could not write to logfile %s: %s\n",maclist[i][x].logfile,strerror(errno));
					exit(1);
				}
			} else { /* file exists - check for W_OK */
				if (access(maclist[i][x].logfile,W_OK)) {
					printf("Sorry - cannot write to logfile %s: %s\n",maclist[i][x].logfile,strerror(errno));
					exit(1);
				}
			}
		}
	}

	
	// some stuff we calculate in front to ceep CPU usage to a minimum..
	waitsecs = 60 * minutes;
	umask(077);

	/* daemonize */
#ifndef DEBUG 
	printf("ok, going into background\n");
	fclose(stdin);
	fclose(stdout);
	fclose(stderr);
	if (fork() != 0)
		exit(0);
#endif // ndef DEBUG
	
	if(pcap_loop(pds, -1, (pcap_handler) ProcessPacket, NULL) < 0) {
		printerr("pcap_loop died, exiting..");
	}
 
} /* end, libpcap_listen */

void ProcessPacket(char *user, struct pcap_pkthdr * pkthdr, u_char * pkt)
{
	FILE *fil = NULL;
	struct EtherHdr *head;
	int i,x;
	
	head = (struct EtherHdr *)pkt;
	
	/* check FROM addy */
	x = head->ether_src[5];
	for (i=0;(i < MAXHOSTS) && (maclist[x][i].logfile); i++) {
		if (maclist[x][i].addr[0] == head->ether_src[0]) {
			if (maclist[x][i].addr[1] == head->ether_src[1]) 
			if (maclist[x][i].addr[2] == head->ether_src[2]) 
			if (maclist[x][i].addr[3] == head->ether_src[3]) 
			if (maclist[x][i].addr[4] == head->ether_src[4]) {
				maclist[x][i].inbytes += (unsigned long)pkthdr->len;
				
				/* should we write logfile for this entry yet? */
				if (time(NULL) >= (maclist[x][i].nextwrite)) {
					fil = fopen(maclist[x][i].logfile,"w");
					maclist[x][i].nextwrite = time(NULL) + waitsecs;
					
					/* if user wants revert-mode then just write numbers the other way around */
					if (maclist[x][i].revert)
						fprintf(fil,"%lu\n%lu\n",maclist[x][i].outbytes,maclist[x][i].inbytes);
					else
						fprintf(fil,"%lu\n%lu\n",maclist[x][i].inbytes,maclist[x][i].outbytes);
			
					fclose(fil);
				}
				#ifdef DEBUG
				printf("+inbytes (%lu) for (%02X:%02X:%02X:%02X:%02X:%02X)\n",
						maclist[x][i].inbytes,
						maclist[x][i].addr[0],
						maclist[x][i].addr[1],
						maclist[x][i].addr[2],
						maclist[x][i].addr[3],
						maclist[x][i].addr[4],
						x
						);
				#endif
			}
			
		} 
	}
			
	/* check DESC addy */
	x = head->ether_dst[5];
	for (i=0;(i < MAXHOSTS) && (maclist[x][i].logfile); i++) {
		if (maclist[x][i].addr[0] == head->ether_dst[0]) {
			if (maclist[x][i].addr[1] == head->ether_dst[1]) 
			if (maclist[x][i].addr[2] == head->ether_dst[2]) 
			if (maclist[x][i].addr[3] == head->ether_dst[3]) 
			if (maclist[x][i].addr[4] == head->ether_dst[4]) {
				maclist[x][i].outbytes += (unsigned long)pkthdr->len;
				
				/* should we write logfile for this entry yet? */
				if (time(NULL) >= (maclist[x][i].nextwrite)) {
					fil = fopen(maclist[x][i].logfile,"w");
					maclist[x][i].nextwrite = time(NULL) + waitsecs;
					
					/* if user wants revert-mode then just write numbers the other way around */
					if (maclist[x][i].revert)
						fprintf(fil,"%lu\n%lu\n",maclist[x][i].outbytes,maclist[x][i].inbytes);
					else
						fprintf(fil,"%lu\n%lu\n",maclist[x][i].inbytes,maclist[x][i].outbytes);
			
					fclose(fil);
				}
				#ifdef DEBUG
				printf("+outbytes (%lu) for (%02X:%02X:%02X:%02X:%02X:%02X)\n",
						maclist[x][i].inbytes,
						maclist[x][i].addr[0],
						maclist[x][i].addr[1],
						maclist[x][i].addr[2],
						maclist[x][i].addr[3],
						maclist[x][i].addr[4],
						x
						);
				#endif
			}
			
		} 
	}
	
	return;
} /* end, ProcessPacket */

/* got this one from web, think it was borland's site */
int axtoi(char *hexStg) {
	int n = 0;         // position in string
	int m = 0;         // position in digit[] to shift
	int count;         // loop index
	int intValue = 0;  // integer value of hex string
	int digit[5];      // hold values to convert
	while (n < 4) {
		if (hexStg[n]=='\0')
			break;

		if (hexStg[n] > 0x29 && hexStg[n] < 0x40 ) //if 0 to 9
			digit[n] = hexStg[n] & 0x0f;            //convert to int
		else if (hexStg[n] >='a' && hexStg[n] <= 'f') //if a to f
			digit[n] = (hexStg[n] & 0x0f) + 9;      //convert to int
		else if (hexStg[n] >='A' && hexStg[n] <= 'F') //if A to F
			digit[n] = (hexStg[n] & 0x0f) + 9;      //convert to int
		else
			break;

		n++;
	}
	count = n;
	m = n - 1;
	n = 0;
	
	while(n < count) {
		// digit[n] is value of hex digit at position n
		// (m << 2) is the number of positions to shift
		// OR the bits into return value
		intValue = intValue | (digit[n] << (m << 2));
		m--;   // adjust the position to set
		n++;   // next digit to process
	}
	return (intValue);
}
