#include "nn.h"

#define MUT 2.0
#define DELTA 20.0
#define LAMBDA  0.6
#define LAMBDA_ 0.005
#define ALPHA 0.05
#define ALPHA_ 0.0001

nn::nn()
{
    n_header=0;    // no header by default
    separator=' '; // space as default separator
    n_elements=0;
    data_available=false;
    y=err=(double*)0;
    d=x=a=o=net=(double**)0;
    yt=ot=(double*)0;
    t_lines=n_lines=0;
    is_trained=false;
    is_shuffle=true;
    for(unsigned int i=0; i<MAXINPUT; i++){
	coder[i]=9999;
	defaults[i]=0;
	inputs[i]=0;
    }
}

nn::~nn()  // release memory
{
    if(is_trained==true)  mem_free();
    delete [] min;
    delete [] max;
    delete [] mean;
    delete [] var;
    delete [] corr;
    delete [] data;
    delete [] yt;
    delete [] ot;
}

void nn::set_n_hidden(unsigned int h)
{
    if(h<1 || h>MAXHIDDEN){
	fprintf(stderr,"number of hidden=%d must be in [1..30]",h);
	return;
    }
    n_hidden=h;
}

void nn::set_n_header(unsigned int h=0)
{
    n_header=h;
}

void nn::set_separator(char c=' ')
{
    separator=c;
}

void nn::set_y_output(int i)
{
	y_output=i;
}

void nn::set_coder(int i, int j)
{
    if(i<0 || i>MAXINPUT) return;
    coder[i]=j;
}

unsigned int nn::get_n_lines()
{
    return n_lines;
}

unsigned int nn::get_t_lines()
{
    return t_lines;
}

unsigned int nn::get_n_elements()
{
    return n_elements;
}

unsigned int nn::get_n_inputs()
{
    return n_inputs;
}

unsigned int nn::get_coder(int i)
{
    if(i<0 || (unsigned int) i>n_inputs){
	fprintf(stderr,"error in coder: i=%d  too big\n",i); 
	return 0;
    }
    return coder[i];
}

void nn::set_shuffle(bool flag)
{
    is_shuffle=flag;
}

void nn::set_norm(bool flag)
{
    is_norm=flag;
}

double nn::get_min(unsigned int i)
{
    if(i>n_elements){
	fprintf(stderr,"error get_min:%d selector out of range=%d\n"
		,i,n_elements);
	return -9999.0;
    }
    return min[i];
}

double nn::get_max(unsigned int i)
{
    if(i>n_elements){
	fprintf(stderr,"error get_max:%d selector out of range=%d\n"
		,i,n_elements);
	return -9999.0;
    }
    return max[i];
}

double nn::get_mean(unsigned int i)
{
    if(i>n_elements){
	fprintf(stderr,"error get_mean:%d selector out of range\n",i);
	return -9999.0;
    }
    return mean[i];
}

double nn::get_var(unsigned int i)
{
    if(i>n_elements){
	fprintf(stderr,"error get_var:%d selector out of range\n",i);
	return -9999.0;
    }
    return var[i];
}

double nn::get_corr(unsigned int i)
{
    if(i>n_elements){
	fprintf(stderr,"error get_corr:%d selector out of range\n",i);
	return -9999.0;
    }
    return corr[i];
}

int nn::get_y_output()
{
    return y_output;
}

double nn::get_output(unsigned int i)
{
    if(n_lines<=1 || y==(double*)0 || i>=n_lines) return -9999;
    return y[i];
}

double nn::get_network(unsigned int i)
{
   if(n_lines<=1 || o==(double**)0 || i>=n_lines) return -9999;
   return o[i][n_hidden];
}

double nn::get_t_network(unsigned int i)
{
    if(t_lines<=1 || ot==(double*)0 || i>=t_lines) return -9999;
    return ot[i];
}

double nn::get_t_output(unsigned int i)
{
    if(t_lines<=1 || y==(double*)0 || i>=t_lines) return -9999;
    return yt[i];
}


bool nn::read_data(const QString &f)
{
    QFile file(f);
    QString line;
    QString sep;
    unsigned int i = 0;
    fprintf(stderr,"n_header=%d y_output=%d\n",n_header,y_output);
    if ( file.open( IO_ReadOnly ) ) {
        QTextStream stream( &file );
	if(separator=="," || separator==";")
	    sep=separator;
	else
	    sep= QString("\\d")+separator;
	QRegExp mark( sep );
	while ( !stream.atEnd() ) {
            line = stream.readLine(); // line of text excluding '\n'
	    if(i++>n_header){
		n_elements=line.contains(mark)+1;
	    }
        }
        file.close();
    }
    else return false;
    n_lines=i-n_header;
    fprintf(stderr,"n_lines=%d n_elements=%d\n",n_lines,n_elements);
    // allocate memory
    min=new double[n_elements];
    max=new double[n_elements];
    mean=new double[n_elements];
    var=new double[n_elements];
    corr=new double[n_elements];
    data=(double**) new (double*)[n_lines];
    for(i=0; i<n_lines;i++)
	data[i]=new double[n_elements];
    if(!min || !max || !mean || !var || !corr || !data){
	fprintf(stderr,"(error nn::read_data): no space on device\n");
	return false;
    }
    for(i=0; i<n_elements; i++){
	min[i]=FLT_MAX;
	max[i]=-FLT_MAX;
	mean[i]=var[i]=corr[i]=0.0;
    }
    double x_;
    unsigned int j=0;
    unsigned int k=0;
    char c;
    if ( file.open( IO_ReadOnly ) ) {
        QTextStream stream( &file );
	while ( !stream.atEnd() ) {
	    if(k++<n_header){stream.readLine();}
	    else
	    { 
		for(i=0; i<n_elements; i++){
		    stream >> x_;
		    if(min[i]>x_) min[i]=x_;
		    if(max[i]<x_) max[i]=x_;
		    mean[i]+=x_;
		    data[j][i]=x_;
		    if(i<n_elements-1 && (separator=="," || separator==";"))
			stream >> c;
		    
		}
		j++;
	    }
        }
	for(i=0; i<n_elements; i++){
	    mean[i]/=n_lines;
	}
	if(y_output<0 || y_output>=(int) n_elements)y_output=n_elements-1;
	for(i=0; i<n_elements; i++){
	    for(j=0; j<n_lines; j++){
		var[i]+=(data[j][i]-mean[i])*(data[j][i]-mean[i]);
		corr[i]+=(data[j][i]-mean[i])*
		    (data[j][y_output]-mean[y_output]);
	    }
	    var[i]/=n_lines;
	    corr[i]/=n_lines;
	    corr[i]/=sqrt(var[i]);
  	}	
	for(i=0; i<n_elements; i++){
	    corr[i]/=sqrt(var[y_output]);
	}
	for(i=0; i<n_elements; i++){
	    var[i]=sqrt(var[i]);
	}
    }
    if(thresh<0.001)thresh=0.001;
    file.close();
    data_available=true;
    // generate and fill a random shuffle field
    shuffle_f=new int[n_lines];
    int *test_sh=new int[n_lines];
    if(shuffle_f==NULL || test_sh==NULL){
	fprintf(stderr,"error in read_data: no space on device\n");
	exit(1);
    }
    for(unsigned int i=0; i<n_lines; i++)
	test_sh[i]=0;
    int pointer=rand()%n_lines;
    for(unsigned int i=0; i<n_lines; i++){
	while(test_sh[pointer]!=0) pointer=rand()%n_lines;
	shuffle_f[i]=pointer;
	test_sh[pointer]=1;
    }
    delete [] test_sh;
    return true;
}

bool nn::gen_test_data(QString& in)
{
    QFile fin(in);
    QString o1=in+QString(".train");
    QString o2=in+QString(".test");
    QFile fo1(o1);
    QFile fo2(o2);
    QString line;
    QString sep;
    unsigned int i = 1;
    // save test and train data to disk
    fo1.open( IO_WriteOnly );
    fo2.open( IO_WriteOnly );
    QTextStream stream_o1( &fo1 );
    QTextStream stream_o2( &fo2 );
    time_t now = time(NULL);
    srand(now);
    fprintf(stderr,"n_header=%d y_output=%d\n",n_header,y_output);
    if ( fin.open( IO_ReadOnly ) ) {
        QTextStream stream( &fin );
	if(separator=="," || separator==";")
	    sep=separator;
	else
	    sep= QString("\\d")+separator;
	QRegExp mark( sep );
	n_lines=t_lines=0;
	while ( !stream.atEnd() ) {
            line = stream.readLine(); // line of text excluding '\n'
	    if(i++>n_header){
		n_elements=line.contains(mark)+1;
		if(((double)rand()/RAND_MAX)<0.8){
		    stream_o1 << line << endl;
		    n_lines++;
		}
		else{
		    stream_o2 << line << endl;
		    t_lines++;
		}
	    }
	}
    }
    else return false;
    fin.close();
    fo1.flush();
    fo2.flush();
    fo1.close();
    fo2.close();
    fprintf(stderr,"n_lines=%d t_lines=%d n_elements=%d\n"
	    ,n_lines,t_lines,n_elements);
    // allocate memory
    min=new double[n_elements];
    max=new double[n_elements];
    mean=new double[n_elements];
    var=new double[n_elements];
    corr=new double[n_elements];
    data=(double**) new (double*)[n_lines];
    for(i=0; i<n_lines;i++)
	data[i]=new double[n_elements];
    if(!min || !max || !mean || !var || !corr || !data){
	fprintf(stderr,"(error nn::gen_test_data): no space on device\n");
	return false;
    }
    for(i=0; i<n_elements; i++){
	min[i]=FLT_MAX;
	max[i]=-FLT_MAX;
	mean[i]=var[i]=corr[i]=0.0;
    }
    double x_;
    unsigned int j=0;
    char c;
    QFile fi1(o1); 
    QFile fi2(o2);
    if ( fi1.open( IO_ReadOnly ) ) {
        QTextStream stream( &fi1 );
	while ( !stream.atEnd() ) {
	    for(i=0; i<n_elements; i++){
		stream >> x_;
		if(min[i]>x_) min[i]=x_;
		if(max[i]<x_) max[i]=x_;
		mean[i]+=x_;
		data[j][i]=x_;
		if(separator=="," || separator==";")
		    stream >> c;
	    }
	    j++;
	}
	for(i=0; i<n_elements; i++){
	    mean[i]/=n_lines;
	}
	if(y_output<0 || y_output>=(int) n_elements)y_output=n_elements-1;
	for(i=0; i<n_elements; i++){
	    for(j=0; j<n_lines; j++){
		var[i]+=(data[j][i]-mean[i])*(data[j][i]-mean[i]);
		corr[i]+=(data[j][i]-mean[i])*
		    (data[j][y_output]-mean[y_output]);
	    }
	    var[i]/=n_lines;
	    corr[i]/=n_lines;
	    corr[i]/=sqrt(var[i]);
  	}	
	for(i=0; i<n_elements; i++){
	    corr[i]/=sqrt(var[y_output]);
	}
	for(i=0; i<n_elements; i++){
	    var[i]=sqrt(var[i]);
	}
    }
    if(thresh<0.001)thresh=0.001;
    fi1.close();
    tdata=(double**) new (double*)[t_lines];
    for(i=0; i<t_lines;i++)
	tdata[i]=new double[n_elements];
    if(!tdata){
	fprintf(stderr,"(error nn::gen_test_data): no space on device\n");
	return false;
    }
    j=0;
    if ( fi2.open( IO_ReadOnly ) ) {
        QTextStream stream( &fi2 );
	while ( !stream.atEnd() ) {
	    for(i=0; i<n_elements; i++){
		stream >> x_;
		tdata[j][i]=x_;
		if(i<n_elements-1 && (separator=="," || separator==";"))
		    stream >> c;
	    }
	    j++;
	}
	fi2.close();
    }
    else return false;
    yt=new double[t_lines];
    if(yt==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    ot=new (double)[t_lines];
    if(ot==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    data_available=true;
    // generate and fill a random shuffle field
    shuffle_f=new int[n_lines];
    int *test_sh=new int[n_lines];
    if(shuffle_f==NULL || test_sh==NULL){
	fprintf(stderr,"error in read_data: no space on device\n");
	exit(1);
    }
    for(unsigned int i=0; i<n_lines; i++)
	test_sh[i]=0;
    int pointer=rand()%n_lines;
    for(unsigned int i=0; i<n_lines; i++){
	while(test_sh[pointer]!=0) pointer=rand()%n_lines;
	shuffle_f[i]=pointer;
	test_sh[pointer]=1;
    }
    delete [] test_sh;
    return true;
}

bool nn::data_avail()
{
    return data_available;
}

void nn::set_max_error(double err)
{
    thresh=0.02*n_lines*err;
    if(thresh<0.001)thresh=0.001;
}

void nn::set_n_hidden(int h)
{
    n_hidden=h;
}

void nn::set_n_inputs(int n)
{
    n_inputs=n;
}

void nn::reset()
{
    if(w!=0){
	time_t now = time(NULL);
	srand(now);
	for(unsigned int i=0; i<n_weights;i++){
	    w[i]= 1-2*(double)rand()/RAND_MAX;
	}
    }	
    fprintf(stderr,"\n");
    if(alpha>0.000001*ALPHA)alpha=ALPHA;
    lambda=LAMBDA;
    err_=0.5*n_lines;
    err__=0.1;
    steps=0;
    mut=MUT;
    delta=DELTA;
}

int nn::train_lm()
{
    if(n_inputs >=30 || n_hidden>=30) return 1;
    if(n_elements==0) return 1;
    n_weights=n_inputs*n_hidden+2*n_hidden+1;
    n_outputs=(n_inputs+1)*n_hidden; // startpoint for weights to output
    // set weight factors
    time_t now = time(NULL);
    srand(now);
    for(unsigned int i=0; i<n_weights;i++){
        w[i]= 1-2*(double)rand()/RAND_MAX;
    }
    // allocate mem 
    err=new double[n_lines];
    if(err==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    y=new double[n_lines];
    if(y==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    o=new (double*)[n_lines];
    if(o==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    for(unsigned int i=0; i<n_lines; i++){
        o[i]=new double[n_hidden+1];
        if(o[i]==0){
            fprintf(stderr,"(error train_lm): could not allocate memory\n");
	    return -1;
        }
    }
    a=new (double*)[n_lines];
    if(a==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    for(unsigned int i=0; i<n_lines; i++){
        a[i]=new double[n_hidden+1];
        if(a[i]==0){
            fprintf(stderr,"(error train_lm): could not allocate memory\n");
	    return -1;
        }
    }
    net=new (double*)[n_lines];
    if(net==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    for(unsigned int i=0; i<n_lines; i++){
        net[i]=new double[n_hidden+1];
        if(net[i]==0){
            fprintf(stderr,"(error train_lm): could not allocate memory\n");
	    return -1;
        }
    }
    x=new (double*)[n_lines];
    if(x==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    for(unsigned int i=0; i<n_lines; i++){
        x[i]=new double[n_inputs];
        if(x[i]==0){
            fprintf(stderr,"(error train_lm): could not allocate memory\n");
	    return -1;
        }
    }

    err_=0.5*n_lines;
    err__=0.1;
    steps=0;
    mut=MUT;
    delta=DELTA;
    // construct x and y 
    for(unsigned int i=0; i<n_lines; i++){
	y[i]=2.0*(data[i][y_output]-min[y_output])/
		(max[y_output]-min[y_output])-1.0;
	for(unsigned int j=0; j<n_inputs;j++){
	    if(coder[j]==9999) return -1;
	    x[i][j]=2.0*(data[i][coder[j]]-min[coder[j]])/
		(max[coder[j]]-min[coder[j]])-1.0;
	}
    }
    is_trained=true;
    return 0;
}

double sigmoid(double x)
{
    return(2.0/(1.0+exp(-x))-1.0);
}

double activ(double x)
{
    return(1+exp(-x));
}


double nn::step_lm(int iter)
{
    int n=0;
    unsigned int i,j,k,l;
    unsigned int shuffle=0;
    while(err_>thresh && n<iter){
        n++;
        /* determine the errors */
        err_=0.0;
        // set all derivates to zero
        for(l=0;l<n_weights;l++) derr[l]=0.0;
        for(i=0; i<n_lines; i++){
	    if(is_shuffle==true)
		shuffle=shuffle_f[i];
	    else
		shuffle=i;
            for(k=0; k<n_hidden;k++){
                net[shuffle][k]=0.0;
                for(j=0;j<n_inputs;j++){
                    net[shuffle][k]+=w[j+k*(n_inputs+1)]*x[shuffle][j];
                }
                net[shuffle][k]+=w[k*(n_inputs+1)+n_inputs];
                a[shuffle][k]=activ(net[shuffle][k]);
                o[shuffle][k]=sigmoid(net[shuffle][k]);
            }
            net[shuffle][n_hidden]=0.0;
            for(k=0; k<n_hidden;k++){
                net[shuffle][n_hidden]+=w[k+(n_inputs+1)*n_hidden]
		    *o[shuffle][k];
            }
            net[shuffle][n_hidden]+=w[n_weights-1];
            o[shuffle][n_hidden]=oo=sigmoid(net[shuffle][n_hidden]);
            a[shuffle][n_hidden]=activ(net[shuffle][n_hidden]);
            err[shuffle]=(y[shuffle]-oo)*(y[shuffle]-oo);
            err_ += err[shuffle];
            // determine the derr/dwi
            for(k=0;k<n_hidden;k++){
                factor=exp(-net[shuffle][n_hidden]-net[shuffle][k])/
                    (a[shuffle][n_hidden]*a[shuffle][n_hidden]*
                     a[shuffle][k]*a[shuffle][k]);
                for(l=0;l<n_inputs;l++){
                    derr[l+k*(n_inputs+1)]+=
                        x[shuffle][l]*(o[shuffle][n_hidden]-y[shuffle])*factor;
                }
                derr[(k+1)*(n_inputs+1)-1]+=
                    (o[shuffle][n_hidden]-y[shuffle])*factor;
                derr[n_outputs+k]+=((o[shuffle][n_hidden]-y[shuffle])*
                                    o[shuffle][k]
				    *exp(-net[shuffle][n_hidden]))/
                    (a[shuffle][n_hidden]*a[shuffle][n_hidden]);
            }
            derr[n_weights-1]+=(o[shuffle][n_hidden]-y[shuffle])/
                (a[shuffle][n_hidden]*a[shuffle][n_hidden]);
        }
	        for(k=0;k<n_hidden;k++){
            for(l=0;l<n_inputs; l++){
                derr[l+k*(n_inputs+1)]*=8.0*w[k+(n_inputs+1)*n_hidden];
            }
            derr[(k+1)*(n_inputs+1)-1]*=8.0*w[k+(n_inputs+1)*n_hidden];
            derr[n_outputs+k]*=4.0;
        }
        derr[n_weights-1]*=4.0;
        // update weights
        scalar=0.0;
        for(l=0;l<n_weights;l++)
            scalar+=derr[l]*derr[l];
        scalar+=1/mut;
        for(k=0;k<n_weights;k++){
            val=0.0;
            for(l=0;l<n_weights;l++){
                if(l==k)  // diagonal elemets
                    val+=(1-derr[l]*derr[l]/scalar)*derr[k];
                else
                    val-=derr[l]*derr[l]*derr[k]/scalar;
            }
            val*=err_;
            w[k]-=val/delta;
        }
        if(err_<err__ && mut>0.5) mut-=0.01;
        if(err_>=err__ ) mut+=1.0;
        if(mut>MUT) mut=MUT;

        if(err__>=err_ && delta>1.00)delta-=0.01;
        if(err__<err_) delta+=0.1;
        if(delta>DELTA) delta=DELTA;
        err__=err_;
    }
//     double error=0.0;
//     for(i=0; i<n_lines; i++){
// 	error+=(o[i][n_hidden]-y[i])*(o[i][n_hidden]-y[i]);
// 	printf("%4d: %f %f %f %f %f %f %f\n"
// 	       ,i,x[i][0],x[i][1],y[i],o[i][n_hidden]
// 	       ,(o[i][n_hidden]-y[i]),error,err_);
//     }

    return sqrt(err_)/n_lines;
}

void nn::mem_free()
{
    // give memory free
    if(a!=0 && o!=0 && net!=0 && x!=0){
	for(unsigned int i=0; i<n_lines; i++){
	    delete [] a[i];
	    delete [] o[i];
	    delete [] net[i];
	    delete [] x[i];
	}
	delete [] a;
	delete [] o;
	delete [] net;
	delete [] x;
	a=o=net=x=(double**)0;
    }
    if(d!=0 && o!=0 && net!=0 && x!=0){
	for(unsigned int i=0; i<n_lines; i++){
	    delete [] d[i];
	    delete [] o[i];
	    delete [] net[i];
	    delete [] x[i];
	}
	delete [] d;
	delete [] o;
	delete [] net;
	delete [] x;
	d=x=o=net=(double**)0;
    }
}

int nn::train_bp()
{
    if(n_elements==0) return 1;
    if(n_inputs >=30 || n_hidden>=30) return 1;
    
    n_weights=n_inputs*n_hidden+2*n_hidden+1;
    n_outputs=(n_inputs+1)*n_hidden; // startpoint for weights to output
    // set weight factors
    time_t now = time(NULL);
    srand(now);
    for(unsigned int i=0; i<n_weights;i++){
        w[i]= 1-2*(double)rand()/RAND_MAX;
    }
    // allocate mem 
    if(err==0) err=new double[n_lines];
    if(err==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    if(y==0) y=new double[n_lines];
    if(y==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    o=new (double*)[n_lines];
    if(o==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    for(unsigned int i=0; i<n_lines; i++){
        o[i]=new double[n_hidden+1];
        if(o[i]==0){
            fprintf(stderr,"(error train_lm): could not allocate memory\n");
	    return -1;
        }
    }
    d=new (double*)[n_lines];
    if(d==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    for(unsigned int i=0; i<n_lines; i++){
        d[i]=new double[n_hidden+1];
        if(d[i]==0){
            fprintf(stderr,"(error train_lm): could not allocate memory\n");
	    return -1;
        }
    }
    net=new (double*)[n_lines];
    if(net==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    for(unsigned int i=0; i<n_lines; i++){
        net[i]=new double[n_hidden+1];
        if(net[i]==0){
            fprintf(stderr,"(error train_lm): could not allocate memory\n");
	    return -1;
        }
    }
    x=new (double*)[n_lines];
    if(x==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    for(unsigned int i=0; i<n_lines; i++){
        x[i]=new double[n_inputs];
        if(x[i]==0){
            fprintf(stderr,"(error train_lm): could not allocate memory\n");
	    return -1;
        }
    }

    err_=0.5*n_lines;
    err__=0.1;
    steps=0;
    mut=MUT;
    delta=DELTA;
    // construct x and y 
    for(unsigned int i=0; i<n_lines; i++){
	y[i]=2.0*(data[i][y_output]-min[y_output])/
		(max[y_output]-min[y_output])-1.0;
	for(unsigned int j=0; j<n_inputs;j++){
	    if(coder[j]==9999) return -1;
	    x[i][j]=2.0*(data[i][coder[j]]-min[coder[j]])/
		(max[coder[j]]-min[coder[j]])-1.0;
	}
    }
    err__=err_=n_lines;;
    steps=0;
    alpha=0.0;
    lambda=LAMBDA;
    if(thresh<0.001) thresh=0.001;
    is_trained=true;
    return 0;
}

int nn::train_bpm()
{
    train_bp();
    alpha=ALPHA;
    return 0;
}

double nn::step_bp(int iter)
{
    int n=0;
    unsigned int i,j,k,l;
    unsigned int shuffle=0;
    while(err_>thresh && n<iter){
        n++;
	/* determine the errors */
        err_=0.0;
        // set all derivates to zero
	for(i=0; i<n_lines; i++){
	    if(is_shuffle==true)
		shuffle=shuffle_f[i];
	    else
		shuffle=i;
            for(k=0; k<n_hidden;k++){
                net[shuffle][k]=0.0;
                for(j=0;j<n_inputs;j++){
                    net[shuffle][k]+=w[j+k*(n_inputs+1)]*x[shuffle][j];
                }
                net[shuffle][k]+=w[k*(n_inputs+1)+n_inputs];
                o[shuffle][k]=sigmoid(net[shuffle][k]);
            }
            net[shuffle][n_hidden]=0.0;
            for(k=0; k<n_hidden;k++){
                net[shuffle][n_hidden]+=w[k+(n_inputs+1)*n_hidden]*o[shuffle][k];
            }
            net[shuffle][n_hidden]+=w[n_weights-1];
            o[shuffle][n_hidden]=sigmoid(net[shuffle][n_hidden]);
            // error of output
            double m=(1.0-o[shuffle][n_hidden]*o[shuffle][n_hidden]);
            d[shuffle][n_hidden]=0.5*m*(y[shuffle]-o[shuffle][n_hidden]);
            // error of hidden
	    
            for(k=0; k<n_hidden;k++){
                d[shuffle][k]=0.5*(1.0-o[shuffle][k]*o[shuffle][k])
                    *w[k+(n_inputs+1)*n_hidden]*d[shuffle][n_hidden];
            }
            // momentum
            for(k=0;k<n_weights;k++){
                w[k]+=alpha*dw[k];
            }
            // dw for hidden nodes with input
	    for(k=0;k<n_hidden;k++){
		for(l=0;l<n_inputs;l++){
		    // update dw
                    dw[k*(n_inputs+1)+l]=lambda*d[shuffle][k]*x[shuffle][l];
                }
            }
	    // dw for hidden nodes bias
            for(k=0;k<n_hidden;k++){
                dw[(k+1)*n_inputs]=lambda*d[shuffle][k];
            }
            // dw for outputs nodes
            for(k=0;k<n_hidden;k++){
                dw[k+(n_outputs)]=lambda*d[shuffle][n_hidden]*o[shuffle][k];
            }
            dw[n_weights-1]=lambda*d[shuffle][n_hidden];
            for(k=0;k<n_weights;k++)
                w[k]+=dw[k];
	      
	    err_+=(o[shuffle][n_hidden]-y[shuffle])
		*(o[shuffle][n_hidden]-y[shuffle]);
	    // fprintf(stderr,"%d %lf\n",n,err_);
        }
	// fprintf(stderr,"%4d: %f %f %f\n",n,err_,lambda,alpha);
	
	if(err_>err__){
	    lambda/=(1.05);
	    if(lambda<LAMBDA_) lambda=LAMBDA_;
	    alpha-=0.002;
	    if(alpha<ALPHA_) alpha=ALPHA_;
	}
	else{
	    lambda+=0.01;
	    alpha+=0.002;
	    if(lambda>LAMBDA) lambda=LAMBDA;
	    if(alpha>ALPHA) alpha=ALPHA;
	}
	err__=err_;
    }
    // if(err_>n_lines) reset();
    return sqrt(err_)/n_lines;
}

bool nn::save_network(FILE *fp)
{
    fprintf(fp,"number_of_inputs:\t%d\n",n_inputs);
    fprintf(fp,"number_of_hidden:\t%d\n",n_hidden);
    for(unsigned int i=0;i<n_inputs;i++)
	fprintf(fp,"%lf %lf\n",min[coder[i]],max[coder[i]]);
    fprintf(fp,"%lf %lf\n",min[y_output],max[y_output]);
    for(unsigned int i=0;i<n_weights;i++)
	fprintf(fp,"%lf ",w[i]);
    fprintf(fp,"\n");
    return true;
}

bool nn::recalc_network()
{
    if(y==0) y=new double[n_lines];
    if(y==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    o=new (double*)[n_lines];
    if(o==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    for(unsigned int i=0; i<n_lines; i++){
        o[i]=new double[n_hidden+1];
        if(o[i]==0){
            fprintf(stderr,"(error train_lm): could not allocate memory\n");
	    return -1;
        }
    }
    net=new (double*)[n_lines];
    if(net==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    for(unsigned int i=0; i<n_lines; i++){
        net[i]=new double[n_hidden+1];
        if(net[i]==0){
            fprintf(stderr,"(error train_lm): could not allocate memory\n");
	    return -1;
        }
    }
    x=new (double*)[n_lines];
    if(x==0){
        fprintf(stderr,"(error train_lm): could not allocate memory\n");
        return -1;
    }
    for(unsigned int i=0; i<n_lines; i++){
        x[i]=new double[n_inputs];
        if(x[i]==0){
            fprintf(stderr,"(error train_lm): could not allocate memory\n");
	    return -1;
        }
    }
    for(unsigned int i=0; i<n_lines; i++){
	y[i]=2.0*(data[i][y_output]-min[y_output])/
	    (max[y_output]-min[y_output])-1.0;
	for(unsigned int j=0; j<n_inputs;j++){
	    if(coder[j]==9999) return false;
	    x[i][j]=2.0*(data[i][coder[j]]-min[coder[j]])/
		(max[coder[j]]-min[coder[j]])-1.0;
	}
    }

    for(unsigned int i=0; i<n_lines; i++){
	for(unsigned int k=0; k<n_hidden;k++){
	    net[i][k]=0.0;
	    for(unsigned int j=0;j<n_inputs;j++){
		net[i][k]+=w[j+k*(n_inputs+1)]*x[i][j];
	    }
	    net[i][k]+=w[k*(n_inputs+1)+n_inputs];
	    o[i][k]=sigmoid(net[i][k]);
	}
	net[i][n_hidden]=0.0;
	for(unsigned int k=0; k<n_hidden;k++){
	    net[i][n_hidden]+=w[k+(n_inputs+1)*n_hidden]*o[i][k];
	}
	net[i][n_hidden]+=w[n_weights-1];
	o[i][n_hidden]=sigmoid(net[i][n_hidden]);
   }
    return true;
}

bool nn::load_network(FILE *fp)
{
    if(data_available==false){
	fprintf(stderr,"(error load_network): read data first\n");
	return false;
    }
    char buf[255];
    fscanf(fp,"%s %d\n",buf,&n_inputs);
    fprintf(stderr,"number_of_inputs:\t%d\n",n_inputs);
    fscanf(fp,"%s %d\n",buf,&n_hidden);
    fprintf(stderr,"number_of_hidden:\t%d\n",n_hidden);
    n_weights=n_inputs*n_hidden+2*n_hidden+1;
    n_outputs=(n_inputs+1)*n_hidden; // startpoint for weights to output
    fprintf(stderr,"n_weigths: %d n_outputs: %d\n",
	    n_weights,n_outputs);
    // set weight factors
    mem_free();
    for(unsigned int i=0;i<n_inputs;i++){
	fscanf(fp,"%lf %lf\n",&min[coder[i]],&max[coder[i]]);
	fprintf(stderr,"%lf %lf\n",min[coder[i]],max[coder[i]]);
    }
    fscanf(fp,"%lf %lf\n",&min[y_output],&max[y_output]);
    fprintf(stderr,"%lf %lf\n",min[y_output],max[y_output]);
    for(unsigned int i=0;i<n_weights;i++){
	fscanf(fp,"%lf ",&w[i]);
	fprintf(stderr,"%lf ",w[i]);
    }
    fprintf(stderr,"\n");
    recalc_network();
    return true;
}

double nn::t_calc()
{
    double merr=0.0;
    xt=new (double*)[t_lines];
    if(xt==0){
        fprintf(stderr,"(error t_calc): could not allocate memory\n");
        return -1;
    }
    for(unsigned int i=0; i<t_lines; i++){
        xt[i]=new double[n_inputs];
        if(xt[i]==0){
            fprintf(stderr,"(error t_calc): could not allocate memory\n");
	    return -1;
        }
    }
    if(yt==0) yt=new double[t_lines];
    if(yt==0){
	fprintf(stderr,"(error t_calc): could not allocate memory\n");
	return false;
    }
    for(unsigned int i=0; i<t_lines; i++){
	yt[i]=2.0*(tdata[i][y_output]-min[y_output])/
	    (max[y_output]-min[y_output])-1.0;
	for(unsigned int j=0; j<n_inputs;j++){
	    if(coder[j]==9999) return 0.0;
	    xt[i][j]=2.0*(tdata[i][coder[j]]-min[coder[j]])/
		(max[coder[j]]-min[coder[j]])-1.0;
	}
    }
    for(unsigned int i=0; i<t_lines; i++){
	for(unsigned int k=0; k<n_hidden;k++){
	    net[i][k]=0.0;
	    for(unsigned int j=0;j<n_inputs;j++){
		net[i][k]+=w[j+k*(n_inputs+1)]*xt[i][j];
	    }
	    net[i][k]+=w[k*(n_inputs+1)+n_inputs];
	    o[i][k]=sigmoid(net[i][k]);
	}
	net[i][n_hidden]=0.0;
	for(unsigned int k=0; k<n_hidden;k++){
	    net[i][n_hidden]+=w[k+(n_inputs+1)*n_hidden]*o[i][k];
	}
	net[i][n_hidden]+=w[n_weights-1];
	o[i][n_hidden]=sigmoid(net[i][n_hidden]);
	ot[i]=o[i][n_hidden];
	merr+=(ot[i]-yt[i])*(ot[i]-yt[i]);
    }
    return sqrt(merr)/t_lines;
}

bool nn::set_default(unsigned int i, double val)
{
    if(i>n_inputs) return false;
    defaults[i]=val;
    return true;
}

bool nn::set_input(unsigned int i, unsigned int val)
{
    if(i>n_inputs) return false;
    inputs[i]=val;
    return true;
}

double nn::get_3d(double x, double y)
{
    // set defaults
    int flag=0;
    int count=0;
    for(unsigned int i=0; i<n_inputs; i++){
	if(flag==0 && inputs[i]==1){
	    defaults[i]=x;
	    flag=1;
	    continue;
	}
	if(flag==1 && inputs[i]==1){
	    defaults[i]=y;
	    flag++;
	}
    }
    for(unsigned int i=0; i<n_inputs; i++){
	if(inputs[i]==1 && count++<2 && is_norm==true)
	    defaults[i]=2.0*(defaults[i]-min[coder[i]])
		/(max[coder[i]]-min[coder[i]])-1.0;
    }
      // calc network
    for(unsigned int k=0; k<n_hidden;k++){
	net[0][k]=0.0;
	for(unsigned int j=0;j<n_inputs;j++){
	    net[0][k]+=w[j+k*(n_inputs+1)]*defaults[j];
	}
	net[0][k]+=w[k*(n_inputs+1)+n_inputs];
	o[0][k]=sigmoid(net[0][k]);
    }
    net[0][n_hidden]=0.0;
    for(unsigned int k=0; k<n_hidden;k++){
	net[0][n_hidden]+=w[k+(n_inputs+1)*n_hidden]*o[0][k];
    }
    net[0][n_hidden]+=w[n_weights-1];
    if(is_norm==true)
	return 0.5*(sigmoid(net[0][n_hidden])+1)*
	    (max[y_output]-min[y_output])+min[y_output];
    else
	return sigmoid(net[0][n_hidden]);
}

double nn::get_2d(double x)
{
    // set defaults
    int count=0;
    for(unsigned int i=0; i<n_inputs; i++){
	if(inputs[i]==1){
	    defaults[i]=x;
	}
    }
    for(unsigned int i=0; i<n_inputs; i++){
	if(inputs[i]==1 && count++<1)
	    defaults[i]=2.0*(defaults[i]-min[coder[i]])
		/(max[coder[i]]-min[coder[i]])-1.0;
    }
      // calc network
    for(unsigned int k=0; k<n_hidden;k++){
	net[0][k]=0.0;
	for(unsigned int j=0;j<n_inputs;j++){
	    net[0][k]+=w[j+k*(n_inputs+1)]*defaults[j];
	}
	net[0][k]+=w[k*(n_inputs+1)+n_inputs];
	o[0][k]=sigmoid(net[0][k]);
    }
    net[0][n_hidden]=0.0;
    for(unsigned int k=0; k<n_hidden;k++){
	net[0][n_hidden]+=w[k+(n_inputs+1)*n_hidden]*o[0][k];
    }
    net[0][n_hidden]+=w[n_weights-1];
    if(is_norm==true)
	return 0.5*(sigmoid(net[0][n_hidden])+1)*
	    (max[y_output]-min[y_output])+min[y_output];
    else
	return sigmoid(net[0][n_hidden]);
}

