#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <math.h>
#include <limits.h>

#include "wavfuncs.h"

/**

wavnorm - a 16 bit stereo .wav file normaliser

duncan@zog.net.au 19/7/1999

$Id: wavnorm.c,v 1.5 1999/07/28 13:18:06 duncan Exp $

**/



/** Needed to compile on RH 4.X for some reason **/

#ifndef MAP_FAILED
#define MAP_FAILED -1
#endif

int wavfd;

/** command line options - stored as globals **/
int verbose = 0;             /* -v */ 
int destructive = 0;         /* -d */
double threshold = -1.0;     /* -t */
double ratio = 1.0;          /* -r */


void usage(char *progname);
void normalise_wav_file(char *filename);

int main(int argc,char *argv[])
{
   int c;

   while ((c = getopt(argc, argv, "vdt:r:")) != EOF) 
   {
      switch(c) 
      {
         case 'v':
            verbose = 1;
            printf("%s version %s duncan@zog.net.au 1999\n",argv[0],VERSION);
            break;
         case 'd':
            destructive = 1;
            break;
         case 't':
            threshold = atof(optarg);
            if (verbose)
               printf("Using threshold of ratio of %f\n",threshold);
            break;
         case 'r':
            ratio = atof(optarg);
            break;
         default:
            usage(argv[0]);
            exit(1);
         }
   }

   if ((optind + 1) > argc)
   {
      usage(argv[0]);
      exit(1);
   }

   for (c = optind; c < argc; c++)
   {
      normalise_wav_file(argv[c]);
   }

   return(0);
}


void normalise_wav_file(char *filename)
{      
   char *inwavbuf;
   short *current;
   short *audio;
   short *audioend;
   WAVE_HEADER *wav_info;
   int wavfd;
   struct stat input_fstat;
   size_t increment;
   int loudest = 0;
   double loudestd;
   double total;
   double loudr;

   printf("processing %s\n",filename);

   wavfd = open(filename,O_RDWR,0600);
   if (wavfd == -1)
   {
      printf("Error: open() %s\n",strerror(errno));
      return;
   }

   if (fstat(wavfd,&input_fstat) != 0)
   {
      printf("Error: fstat() %s\n",strerror(errno));
      if (close(wavfd) != 0)
      {
         printf("Error: close() %s\n",strerror(errno));
         exit(1);
      }
      return;
   }

   if (input_fstat.st_size < sizeof(WAVE_HEADER))
   {
      printf("Error: %s not large enough to hold a .wav file header!\n",
         filename);
      return;
   }

   inwavbuf = mmap(NULL,input_fstat.st_size,
      PROT_READ | PROT_WRITE,MAP_SHARED,wavfd,0);
   if (inwavbuf == MAP_FAILED)
   {
      printf("Error: mmap() %s\n",strerror(errno));
      return;
   }

   audio = (short *)validate_wav_header(inwavbuf,verbose);

   if (audio == NULL)
   {
      printf("Error: %s not in suitable .wav format for this program\n",
         filename);
      return;
   }

   wav_info = (WAVE_HEADER *)inwavbuf;

   increment = wav_info->wBitsPerSample / 8;

   if (wav_info->nDataBytes != input_fstat.st_size - sizeof(WAVE_HEADER))
   {
      printf("   Warning: file length does not match length in header.\n");

      if (destructive)
      {
         printf("   --> fixing length to corrrect value\n");
         wav_info->nDataBytes = input_fstat.st_size - sizeof(WAVE_HEADER);
      }
   }

   current = audio;

   if (verbose)
      printf("   finding highest level...");

   audioend = (short *)((char *)audio + wav_info->nDataBytes);

   while (current < audioend)
   {
      if (abs(*current) > loudest)
      {
         if (verbose)
         {
            printf(".");
            fflush(stdout); /* so the dot appears quickly! */
            total = (double )abs(*current);
         }
         loudest = abs(*current);
         
      }
      current++;
   }


   loudestd = fabs((double )loudest);
   loudr =  SHRT_MAX / loudestd * ratio;
   if (verbose)
   {
      printf("\n");
      printf("    scaling to ratio %f of max\n",ratio);
      printf("    normalise ratio is %f\n",loudr);
   }

   if (destructive)
   {
      if (loudr > threshold)
      {
         if (verbose)
            printf("    normalising with ratio %f now...\n",loudr);

         /** we go backwards through the audio now, as presumably the
             last bytes we read are more likely to be buffered, so
             hopefully this is a bit faster? **/
  
         while (current > audio)
         {
            *current = (int )((double )*current * loudr);
            current--;
         }
      }
      else
      {
         if (verbose)
            printf("   threshold not hit, file not altered\n");
      }
   }

   munmap(inwavbuf,input_fstat.st_size);

   if (close(wavfd) != 0)
   {
      printf("Error: close() %s\n",strerror(errno));
      exit(1);
   }
      
}

void usage(char *progname)
{
   printf("%s: version %s\n",progname,VERSION);
   printf("%s [-t threshold] [-v] [-d] [-r] file.wav [file2.wav ...]\n",
      progname);
   printf("   -v verbose, and version info\n");
   printf("   -d destructive (actually alter the .wav!)\n");
   printf("   -t <threshold> threshhold ratio under which dont alter\n");
   printf("   -r <ratio>     ratio of max possible to scale (default 1.0)\n");
}
