/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "kalc.h"
#include "mem.h"


/*************************
 * Version 2.0.0 support *
 *************************/
typedef struct {
  int type;
  union {
    Complex cmp;
    double real;
    char *str;
  } value;
} Object200;

typedef struct {
  unsigned angleMode:     1;
  unsigned coordinate:    2;
  unsigned numberFormat:  2;
  unsigned decimalPlaces: 4;
  int nLines;
  char prompt[11];
} Options200;


int readStatus200(FILE *fp, int loadStack, int loadMem)
{
  /*
   * This function reads the status from a file of version 2.0.0
   *
   * Returns 1 on success, 0 on failure.
   */
  Options200 tmpOptions;
  int nStack;
  register int i;
  Object el;

  /* Get Options */
  fread(&tmpOptions, sizeof(tmpOptions), 1, fp);

  /* Convert options */
  userOptions.angleMode = tmpOptions.angleMode;
  userOptions.coordinate = tmpOptions.coordinate;
  userOptions.numberFormat = tmpOptions.numberFormat;
  userOptions.decimalPlaces = tmpOptions.decimalPlaces;
  userOptions.nLines = tmpOptions.nLines;
  strncpy(userOptions.prompt, tmpOptions.prompt, MAXPROMPTSIZE);

  /* Get number of stack elements */
  fread(&nStack, sizeof(int), 1, fp);

  /* Get each element and insert it in the stack if desired
     NOTE: We must load all objects because the memory is stored in
     sequence. */
  for (i = 0; i < nStack; i++) {
    el = loadObject200(fp);
    if (loadStack)
      insertObject(el);
  }

  /* Load memory contents */
  if (loadMem)
    loadMemory200(fp);

  return 1;
}

  
Object loadObject200(FILE *fp)
{
  /*
   * This option reads an object from the file whose pointer if fp,
   * returning the object. The caller must assure there is an object to
   * be read, otherwise unpredictable results may happen.
   *
   * For information on how the objects are stored, see the saveObject
   * function.
   *
   * NOTE: Read object in version 2.0.0 format.
   */

  Object200 obj;
  Object newObj;

  fread(&obj, sizeof(Object200), 1, fp);

  if (type(obj) == TYPE_STR || type(obj) == TYPE_ID) {
    char *string;
    size_t length;

    fread(&length, sizeof(length), 1, fp);
    string = (char *) calloc(length + 1, 1);
    if (!string)
      doError("Fatal", ERR_NOTENOUGHMEMORY);

    fread(string, 1, length, fp);
    obj.value.str = string;
  }

  newObj.type = obj.type;
  switch (newObj.type) {
  case TYPE_REAL:
    newObj.value.real = obj.value.real;
    break;

  case TYPE_CMP:
    newObj.value.cmp = obj.value.cmp;
    break;

  case TYPE_STR:
  case TYPE_ID:
    newObj.value.str = obj.value.str;
    break;
  }

  return newObj;
}


void loadMemory200(FILE *fp)
{
  /*
   * This functions load the memory saved in a file, stored in the format
   * specified in the saveMemory function.
   *
   * NOTE: Read object in version 2.0.0 format.
   */

  int length;
  char *name;
  Object *obj;

  while (1) {
    fread(&length, sizeof(length), 1, fp);
    if (length == -1)
      break;

    name = (char *) calloc(length + 1, 1);
    fread(name, 1, length, fp);

    obj = (Object *) malloc(sizeof(Object));
    *obj = loadObject200(fp);

    storeObject(name, obj);
  }
}
