/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <math.h>
#include <setjmp.h>

#include "cmp.h"
#include "kalc.h"


void f_rTOc(void)
{
  /*
   * This function calls the _f_rTOc function through the wrapper.
   */

  run2_1_Function(_f_rTOc, "r>c");
}


void f_reTOc(void)
{
  /*
   * This function calls the _f_reTOc function through the wrapper.
   */

  run1_1_Function(_f_reTOc, "re>c");
}


void f_imTOc(void)
{
  /*
   * This function calls the _f_imTOc function through the wrapper.
   */

  run1_1_Function(_f_imTOc, "im>c");
}


void f_cTOr(void)
{
  /*
   * This function calls the _f_cTOr function through the wrapper.
   */

  run1_2_Function(_f_cTOr, "c>r");
}


void f_re(void)
{
  /*
   * This function calls the _f_cTOre function through the wrapper.
   */

  run1_1_Function(_f_re, "re");
}


void f_im(void)
{
  /*
   * This function calls the _f_cTOim function through the wrapper.
   */

  run1_1_Function(_f_im, "im");
}


void f_conj(void)
{
  /*
   * This function calls the _f_conj function through the wrapper.
   */

  run1_1_Function(_f_conj, "conj");
}


void f_arg(void)
{
  /*
   * This function calls the _f_arg function through the wrapper.
   */

  run1_1_Function(_f_arg, "arg");
}


Object _f_rTOc(Object n, Object p, int *err)
{
  /*
   * This function returns a complex number whose real part is n and
   * whose imaginary part is p.
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.type = TYPE_CMP;
    n.value.cmp.re = n.value.real;
    n.value.cmp.im = p.value.real;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_reTOc(Object n, int *err)
{
  /*
   * This function returns a complex number whose real part is n and
   * whose imaginary part is 0.
   */
  
  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    __f_reTOc(&n);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_imTOc(Object n, int *err)
{
  /*
   * This function returns a complex number whose real part is zero
   * and whose imaginary part is n.
   */
  
  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.type = TYPE_CMP;
    n.value.cmp.im = n.value.real;
    n.value.cmp.re = 0;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_cTOr(Object n, Object *r, int *err)
{
  /*
   * This function returns the real part of n and stores its imaginary
   * part in *r.
   */

  if (type(n) == TYPE_CMP) {
    *err = ERR_NOERR;
    n.type = TYPE_REAL;
    n.value.real = n.value.cmp.re;

    r->type = TYPE_REAL;
    r->value.real = n.value.cmp.im;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_re(Object n, int *err)
{
  /*
   * This function returns the real part of n.
   */
  
  if (type(n) == TYPE_CMP) {
    *err = ERR_NOERR;
    n.type = TYPE_REAL;
    n.value.real = n.value.cmp.re;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_im(Object n, int *err)
{
  /*
   * This function returns the imaginary part of n.
   */
  
  if (type(n) == TYPE_CMP) {
    *err = ERR_NOERR;
    n.type = TYPE_REAL;
    n.value.real = n.value.cmp.im;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_conj(Object n, int *err)
{
  /*
   * This function returns the conjugate of its argument.
   * The conjugate of a real number is defined as itself.
   */
  
  switch (type(n)) {
  case TYPE_CMP:
    n.value.cmp = cmp_conj(n.value.cmp);

  case TYPE_REAL:	/* Do nothing for real numbers */
    *err = ERR_NOERR;
    break;

  default:
    *err = ERR_BADARGUMENTTYPE;
    break;
  }

  return n;
}


Object _f_arg(Object n, int *err)
{
  /*
   * This function returns the argument of a complex number.
   */

  switch (type(n)) {
  case TYPE_REAL:
    __f_reTOc(&n);

  case TYPE_CMP:
    *err = ERR_NOERR;
    n.type = TYPE_REAL;
    n.value.real = cmp_arg(n.value.cmp);
    if (userOptions.angleMode == ANGLE_DEG)
      n.value.real *= 180/PI;    
    break;
  }

  return n;
}


void __f_reTOc(Object *n)
{
  /*
   * This function converts the given object to a complex number,
   * assuming it is a real number.
   */
  
  n->type = TYPE_CMP;
  n->value.cmp.re = n->value.real;
  n->value.cmp.im = 0;
}
