/* Platform independent parts of DCALC - Reverse Polish Notation (RPN)
 * calculator.

    Contact info:
    bhepple@bit.net.au
    http://www.bit.net.au/~bhepple/dcalc

    Copyright (C) 1999 Bob Hepple

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; see the file COPYING.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.

 */

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h> /* Needed for Linux because atof() is not declared in math.h! */

#include "dcalc.h"

long    x, y, z, t, l,  /* Integer stack                             */
    	reg[NUMREGS];   /* Integer registers                         */
			/*                                           */
double  xf,yf,zf,tf,lf, /* Floating point stack                      */
    	regf[NUMREGS],  /* Floating point registers                  */
    	pi,             /* The usual pi.                             */
    	d_to_r,         /* Degrees to radians factor                 */
    	r_to_d;         /* Radians to degrees factor                 */
			/*                                           */
BOOLEAN dot_legal,      /* True if dot may be typed                  */
    	e_legal,        /* True if E (exponent operator) may be typed*/
    	entering,       /* True if we are still entering into X      */
    	lift_needed,    /* True if the stack needs lifting before    */
    			/* more characters are put into X            */
    	invert,         /* Invert flag - causes the next operation   */
    			/* to be inversed eg arcsin() instead of sin */
    	degree,         /* True if numbers are in degrees; else      */
    			/* radians                                   */
    	was_initialised = 0;
			/* To make sure program is only initialised once */
static BOOLEAN
    	last_was_fin_key;/* True if last operation was financial      */

char
    	inbuf[45],      /* Where typed characters are put            */
    	*inptr,         /* Pointer into inbuf                        */
    	outbuf[45],     /* General buffer used to format output      */
    	zero_string[] = "0",
    	div_by_zero[] = "Divide by 0!";

int
    	base,           /* base = 1 for ASCII, <0 for float, 8,      */
    			/* 10 or 16                                  */
    	fk_set,         /* The current set of function key labels    */
    	max_digits;     /* Maximum number of digits displayable      */

typedef unsigned char byte;

/* FUNCTION KEY DEFINITIONS */

struct funckey func[NUMFK][8] = {
    "Float  ", FLOAT,
    "Engin. ", ENG,
    "Binary ", BIN,
    "Octal  ", OCT,
    "Dec    ", DEC,
    "Hex    ", HEX,
    "Ascii  ", ASCIIM,
    "HELP   ", HELP,
    
    "Sin    ", SIN,
    "Cos    ", COS,
    "Tan    ", TAN,
    "Log E  ", LOGE,
    "Log 10 ", LOG10,
    "       ", ASCIIZ,
    "Inverse", INV,
    "HELP   ", HELP,
    
    "Square ", SQR,
    "S Root ", ROOT,
    "Cube   ", CUBE,
    "C root ", CROOT,
    "FRC    ", FRC,
    "INT    ", INT,
    "       ", ASCIIZ,
    "HELP   ", HELP,
    
    "Sum    ", SUM,
    "Mean   ", MEAN,
    "S Dev  ", S_DEV,
    "N      ", NSTAT,
    "Clear  ", SUM0,
    "Sum-   ", SUMR,
    "       ", ASCIIZ,
    "HELP   ", HELP,
    
    "N      ", NFIN,
    "I      ", INTST,
    "P Val  ", PVAL,
    "PMT    ", PMT,
    "F Val  ", FVAL,
    "       ", ASCIIZ,
    "Inverse", INV,
    "HELP   ", HELP,
};

void fmt_bin(str, x)
    char	*str;
    long	x;
{
    char 	inverted[50],
    reverted[50],
    *s, *t;
    BOOLEAN negative;
    
    if (x < 0) {
	negative = TRUE;
	x &= 0x7FFFFFFF;
    } else
	negative = FALSE;
    s = inverted;
    while (x != 0) {
	*(s++) = '0' + (char) (x % 2);
	x /= 2;
    }
    *s = ASCIIZ;
    if (negative) {
	while (strlen(inverted) < 31) {
	    *(s++) = '0';
	    *s = ASCIIZ;
	}
	*(s++) = '1';
    }
    if (s == inverted) /* x==0 */
	strcpy(str, zero_string);
    else {
	t = reverted;
	while (s != inverted)
	    *(t++) = *(--s);
	*t = ASCIIZ;
	sprintf(str, "%32.32s", reverted);
    }
}

void fmt_base(s, x, xf)
    char 	*s;
    long 	x;
    double 	xf;
{
    byte 	format[20];
    byte 	d, *p;
    union 	frig {
	long j;
	byte c[4];
    } frigger;
    int 	i,
    leading_zero;
    
    switch (base) {
    case 1:
	frigger.j = x;
	for (i = 0; i < 4; i++) {
#if REVERSE_BYTES
	    format[3 - i] = frigger.c[i];
#else
	    format[i] = frigger.c[i];
#endif
	}
	format[4] = 0;
#if PCSPECIFIC
	p = format;
	for (i = 0; i < 4; i++, p++)
	    if (*p == 0)
		*s++ = ' ';
	    else
		*s++ = *p;
	*s = 0;
#else
	/* Convert to canonical form */
	p = format;
	leading_zero = 1;
	for (i = 0; i < 4; i++, p++) {
	    if ((i < 3) && (*p == 0) && leading_zero)
		continue;
	    leading_zero = 0;
	    if (((*p > 126) && (*p < 158)) || (*p < 32)) {
		*s++ = '^';
		*s++ = *p + '@';
	    } else
		*s++ = *p;
	}
	*s = 0;
#endif
	break;
	
    case 2:
	fmt_bin(s, x);
	break;
    case 8:
	if (x == 0L)
	    strcpy(s, zero_string);
	else
	    sprintf(s, "%32.lo", x);
	break;
    case 10:
	if (x == 0L)
	    strcpy(s, zero_string);
	else
	    sprintf(s, "%32.ld", x);
	break;
    case 16:
	if (x == 0L)
	    strcpy(s, zero_string);
	else
	    sprintf(s, "%32.lx", x);
	break;
    default:
	if ((base < -9) || (fabs(xf) > 1.0E25)) {
	    i = -base;
	    if (i > 9)
		i -= 10;
	    sprintf(format, "%%32.%d", i);
	    strcat(format, "lg");
	    sprintf(s, format, xf);
	} else {
	    sprintf(format, "%%32.%d", -base);
	    strcat(format, "lf");
	    sprintf(s, format, xf);
	}
	break;
    }
}

void prep_for_output(string)
    char	*string;
    /* Put suitable separators in string and pad to WIDTH characters */
    /* Assume we are receiving a right justified number of any base  */
{
    char 	*s,
    *d,
    sep,
    *ft;
    int 	len,
    interval,
    count,
    negative;
    
    /* Make string WIDTH characters long by padding to left with SPACE */
    len = strlen(string);
    if (len < WIDTH) {
	d = string + WIDTH;
	s = string + len;
	while (s >= string)
	    *d-- = *s--;
	while (d >= string)
	    *d-- = SPACE;
    }
    
    switch (base) {
    case 1: /* ASCIIM */
	return; /* Nasty to return here - its easy to miss it when reading! */
    case 2: /* BINARY */
	sep = SPACE;
	interval = 8;
	break;
    case 8: /* OCTAL */
	sep = SPACE;
	interval = 3;
	break;
    case 10: /* DECIMAL */
	sep = COMMA;
	interval = 3;
	break;
    case 16: /* HEX */
	sep = SPACE;
	interval = 2;
	break;
    default: /* FLOAT or ENG */
	sep = COMMA;
	interval = 3;
	break;
    }
    
    /* Find first non-blank ... */
    s = string;
    while (*s && isspace(*s)) s++;
    if (*s == '-') {
	negative = TRUE;
	*s++ = SPACE;
    } else
	negative = FALSE;
    
    ft = NULL;
    
    if ((base < 0) && ((ft = strchr(s, '.')) ||
		       (ft = strchr(s, 'E'))))
	len = ft - s;
    else
	len = strlen(s);
    
    if (len > interval) {
	/* '(len - 1) / interval' is the number of separators to be added */
	d = s - (len - 1) / interval;
	
	if (ft == NULL)
	    ft = s + strlen(s);
	
	count = len % interval;
	if (count == 0)
	    count = interval;
	
	while (s < ft) {
	    *d++ = *s++;
	    if ((--count == 0) && (s < ft)) {
		*d++ = sep;
		count = interval;
	    }
	}
    }
    
    if (negative) {
	s = string;
	while (*s && isspace(*s)) s++;
	if (s > string)
	    *--s = '-';
    }
}

void print_base(x, xf)
    long	x;
    double	xf;
{
    fmt_base(outbuf, x, xf);
    prep_for_output(outbuf);
    print_string(outbuf);
}

long asc2int(s)
    char	*s;
{
    union 	frig {
	long j;
	byte c[4];
    } frigger;
    int 	i;
    
    for (i = 0; i < 4; i++)
#if REVERSE_BYTES
	frigger.c[3 - i] = *s++;
#else
    frigger.c[i] = *s++;
#endif
    return(frigger.j);
}

long bin2int(s)
    char 	*s;
{
    char 	*t;
    long 	accum;
    
    accum = 0;
    t = s;
    while (*t)
	accum = 2 * accum + (*(t++) - '0');
    return(accum);
}

void stop_entering()
{
    long 	temp;
    double 	tempf;
    
    if (entering) {
	entering = FALSE;
	switch (base) {
	case 1:
	    temp = asc2int(inbuf);
	    tempf = temp;
	    break;
	case 2:
	    temp = bin2int(inbuf);
	    tempf = temp;
	    break;
	case 8:
	    sscanf(inbuf, "%lo", &temp);
	    tempf = temp;
	    break;
	case 10:
	    sscanf(inbuf, "%ld", &temp);
	    tempf = temp;
	    break;
	case 16:
	    sscanf(inbuf, "%lx", &temp);
	    tempf = temp;
	    break;
	default:
	    tempf = atof(inbuf);
	    if (fabs(tempf) <= 0x7fffffff)
		temp = tempf;
	    else
		temp = 0;
	    break;
	}
	x = temp;
	xf = tempf;
    }
}

void push(inx, inxf)
    long	inx;
    double	inxf;
{
    stop_entering();
    t = z;        tf = zf;
    z = y;        zf = yf;
    y = x;        yf = xf;
    x = inx;      xf = inxf;
    lift_needed = TRUE;
}

void pop(outx, outxf)
    long	*outx;
    double	*outxf;
{
    stop_entering();
    *outx = x;    *outxf = xf;
    x = y;        xf = yf;
    y = z;        yf = zf;
    z = t;        zf = tf;
}

void dispnums()
{
    xwin_select();
    gotoxy(LEFTSIDE, STARTX_Y);
    if (entering) {
	strcpy(outbuf, inbuf);
	prep_for_output(outbuf);
	print_string(outbuf);
    } else
	print_base(x, xf);
    
    if (reg_select()) {
	gotoxy(LEFTSIDE, STARTT_Y);
	print_string("T");
	gotoxy(START_X, STARTT_Y);
	print_base(t, tf);
	
	gotoxy(LEFTSIDE, STARTZ_Y);
	print_string("Z");
	gotoxy(START_X, STARTZ_Y);
	print_base(z, zf);
	
	gotoxy(LEFTSIDE, STARTY_Y);
	print_string("Y");
	gotoxy(START_X, STARTY_Y);
	print_base(y, yf);
	
	gotoxy(LEFTSIDE, STARTL_Y);
	print_string("L");
	gotoxy(START_X, STARTL_Y);
	print_base(l, lf);
    }
}

void echo_char(c)
    byte	c;
    /* echo 'c' back to the screen and log in the input buffer */
{
    long 	temp;
    double 	tempf;
    byte 	*p;
    
    if ((c == BACKSPACE) && (base != 1)) {
	if (entering && (inptr > inbuf)) {
	    inptr--;
	    if ((*inptr == '.') && (base < 0))
		dot_legal = TRUE;
	    if ((*inptr == 'e') && (base < 0))
		e_legal = TRUE;
	    *inptr = ASCIIZ;
	} else
	    put_a_char(BELL);

    } else /* Neither backspace or ASCII mode */ {
	if (!entering) {
	    if (lift_needed) {
		temp = x;
		tempf = xf;
		push(temp, tempf);
		dispnums();
	    }
	    lift_needed = TRUE;
	    entering = TRUE;
	    dot_legal = TRUE;
	    e_legal = TRUE;
	    for (inptr = inbuf; inptr < inbuf + 4; inptr++)
		*inptr = ASCIIZ;
	    inptr = inbuf;
	}
	
	if (inptr < inbuf + max_digits)
	    if (base == 1) {
		for (p = inbuf; p < inbuf + 3; p++)
		    *p = *(p+1);
		inbuf[3] = c;
		inptr++;
	    } else {
		*inptr++ = c;
		*inptr = ASCIIZ;
	    }
	else {
	    put_a_char(BELL);
	    msg("Too many digits!");
	}
    }
    
    xwin_select();
    if (base != 1)
	strcpy(outbuf, inbuf);
    else {
	x = asc2int(inbuf);
	xf = x;
	fmt_base(outbuf, x, xf);
    }
    prep_for_output(outbuf);
    gotoxy(LEFTSIDE, STARTX_Y);
    print_string(outbuf);
}

void dispreg(i)
    int	i;
{
    if (reg_select()) {
	gotoxy(LEFTSIDE, i);
	outbuf[0] = '0' + i;
	outbuf[1] = '\0';
	print_string(outbuf);
	gotoxy(START_X, i);
	print_base(reg[i], regf[i]);
    }
}

void dispregs()
{
    int i;
    
    for (i = 0; i < NUMREGS; i++)
	dispreg(i);
}

void prinbase()
{
    xwin_select();
    gotoxy(BASE_X, BASE_Y);
    switch (base) {
    case 1:
	print_string("Ascii   ");
	break;
	
    case 2:
	print_string("Binary  ");
	break;
	
    case 8:
	print_string("Octal   ");
	break;
	
    case 10:
	print_string("Decimal ");
	break;
	
    case 16:
	print_string("Hex     ");
	break;
	
    default:
	if (base < -9) {
	    sprintf(outbuf, "Engin. %d", -(base + 10));
	    print_string(outbuf);
	} else {
	    sprintf(outbuf, "Float  %d", -base);
	    print_string(outbuf);
	}
	break;
    }
}

void print_inv()
{
    xwin_select();
    gotoxy(INV_X, INV_Y);
    if (invert)
	print_string("Inv   ");
    else
	print_string("      ");
}

void print_deg()
{
    xwin_select();
    gotoxy(TMODE_X, TMODE_Y);
    if (degree)
	print_string("Degrees");
    else
	print_string("Radians");
}

void display()
{
    int 	i;
    
    prinbase();
    print_deg();
    print_inv();
    dispnums();
    dispregs();
    invert = FALSE;
    print_inv();
    print_fk(fk_set);
}

int get_int(s)
    char	*s;
{
    COMMAND	c, d;
    char	buf[80];
    int	t, retval;
    
    msg(s);
    c = get_a_char(&d);
    if (isdigit(c))
	retval = c - '0';
    else {
	put_a_char(BELL);
	retval = -1;
    }
    clear_msg();
    return(retval);
}

void save_x()
{
    stop_entering();
    l = x;
    lf = xf;
}

long round(f)
    double	f;
{
    BOOLEAN	negative;
    long 	ret;
    
    negative = FALSE;
    if (f < 0.0) {
	negative = TRUE;
	f = -f;
    }
    ret = floor(f + 0.5);
    if (negative)
	ret = -ret;
    return(ret);
}

void change_base(c)
    COMMAND c;
{
    long	tempx;
    
    switch (c) {
    case ASCIIM:
	if (base != 1) {
	    stop_entering();
	    base = 1;
	    msg("Only function keys work in ASCII mode");
	    max_digits = 4;
	}
	break;
	
    case FLOAT:
	if (((tempx = get_int("Enter number of decimal places")) >= 0) &&
	    (tempx != -base)) {
	    stop_entering();
	    base = -tempx;
	    max_digits = 20;
	}
	break;
	
    case ENG:
	if (((tempx = get_int("Enter number of decimal places")) >= 0) && (tempx != -(10 + base))) {
	    stop_entering();
	    base = -(10 + tempx);
	    max_digits = 20;
	}
	break;
	
    case BIN:
	if (base != 2) {
	    stop_entering();
	    max_digits = 32;
	    base = 2;
	}
	break;
	
    case OCT:
	if (base != 8) {
	    stop_entering();
	    base = 8;
	    max_digits = 11;
	}
	break;
	
    case DEC:
	if (base != 10) {
	    stop_entering();
	    base = 10;
	    max_digits = 10;
	}
	break;
	
    case HEX:
	if (base != 16) {
	    stop_entering();
	    base = 16;
	    max_digits = 8;
	}
	break;
    default:
	put_a_char(BELL);
	break;
    }
    dispnums();
    dispregs();
    prinbase();
}

void arith(c)
    COMMAND c;
{
    long	tempx, tempy;
    double	tempxf, tempyf;
    char 	*c_ptr, c_buf[10];
    int	i;
    
    if (c == ',') { /* CHS */
	if (entering && (base < 0) && !e_legal) /* then CHS of exponent */ {
	    c_ptr = inbuf;
	    while (*c_ptr != 'E')
		c_ptr++;
	    c_ptr++;
	    strcpy(c_buf, c_ptr);
	    *c_ptr = ASCIIZ;
	    (*c_buf == '-') ? strcat(inbuf, "+"): strcat(inbuf, "-");
	    if ((*c_buf == '-') || (*c_buf == '+'))
		strcat(inbuf, &c_buf[1]);
	    else
		strcat(inbuf, c_buf);
	    inptr = &inbuf[strlen(inbuf)];
	} else {
	    pop(&tempx, &tempxf);
	    push(-tempx, -tempxf);
	}
	dispnums();
	return;
    }
    
    if (c == '@') { /* Pop up registers window */
	pop_up_reg();
	return;
    }
    
    save_x();
    pop(&tempx, &tempxf);
    
    if (c == '~') { /* operations on X only */
	tempx = ~tempx;
	tempxf = tempx;
    } else if (c == RECI) {
	if (((base > 0) && (tempx == 0)) || ((base < 0) && (tempxf == 0.0)))
	    msg(div_by_zero);
	else {
	    tempxf = 1.0 / tempxf;
	    tempx = round(tempxf);
	}
    } else { /* X & Y operations */
	pop(&tempy, &tempyf);
	switch (c) {
	case '+':
	    tempx = tempx + tempy;
	    tempxf = tempxf + tempyf;
	    break;
	case '-':
	    tempx = tempy - tempx;
	    tempxf = tempyf - tempxf;
	    break;
	case '*':
	    tempx = tempx * tempy;
	    tempxf = tempxf * tempyf;
	    break;
	case '/':
	    if (((base > 0) && (tempx == 0)) || ((base < 0) && (tempxf == 0.0))) {
		push(tempy, tempyf);
		msg(div_by_zero);
	    } else {
		if (tempx != 0)
		    tempx = tempy / tempx;
		else
		    tempx = 0;
		tempxf = tempyf / tempxf;
	    }
	    break;
	case '%':
	    tempx = tempx * tempy / 100;
	    tempxf = tempxf * tempyf / 100.0;
	    break;
	case '^':
	    tempx ^= tempy;
	    tempxf = tempx;
	    break;
	case '|':
	    tempx |= tempy;
	    tempxf = tempx;
	    break;
	case '&':
	    tempx &= tempy;
	    tempxf = tempx;
	    break;
	case '<':
	    tempy <<= tempx;
	    tempyf = tempy;
	    tempx = tempy;
	    tempxf = tempyf;
	    break;
	case '>':
	    tempy >>= tempx;
	    tempyf = tempy;
	    tempx = tempy;
	    tempxf = tempyf;
	    break;
	case '#':
	    tempy %= tempx;
	    tempyf = fmod(tempyf, tempxf);
	    tempx = tempy;
	    tempxf = tempyf;
	    break;
	case 'Y':
	    /* if x < 0 then make sure y is integer ... */
	    if (tempxf < 0.0) {
		tempx = 0;
		if (tempyf < 0.0) {
		    tempx = 1;
		    tempyf = -tempyf;
		}
		tempy = floor(tempyf + 0.5);
		tempyf = tempy;
		if (tempx)
		    tempyf = -tempyf;
	    }
	    
	    tempxf = pow(tempyf, tempxf);
	    tempx = tempxf;
	    break;
	    
	default:
	    put_a_char(BELL);
	    break;
	}
    }
    push(tempx, tempxf);
    dispnums();
}

void trig(c)
    COMMAND c;
{
    long	tempx, tempy;
    double	tempxf, tempyf;
    BOOLEAN	negative_root;
    
    /* TRIG FUNCTIONS ... */
    save_x();
    pop(&tempx, &tempxf);
    
    switch(c) {
    case SIN:
	if (invert) {
	    tempxf = asin(tempxf);
	    if (degree)
		tempxf *= r_to_d;
	    invert = FALSE;
	    print_inv();
	} else {
	    if (degree)
		tempxf = sin(d_to_r * tempxf);
	    else
		tempxf = sin(tempxf);
	}
	tempx = tempxf;
	break;
	
    case COS:
	if (invert) {
	    tempxf = acos(tempxf);
	    if (degree)
		tempxf *= r_to_d;
	    invert = FALSE;
	    print_inv();
	} else {
	    if (degree)
		tempxf = cos(d_to_r * tempxf);
	    else
		tempxf = cos(tempxf);
	}
	tempx = tempxf;
	break;
	
    case TAN:
	if (invert) {
	    tempxf = atan(tempxf);
	    if (degree)
		tempxf *= r_to_d;
	    invert = FALSE;
	    print_inv();
	} else {
	    if (degree)
		tempxf = tan(d_to_r * tempxf);
	    else
		tempxf = tan(tempxf);
	}
	tempx = tempxf;
	break;

    case ETOX:
	if (invert) {
	    tempxf = log(tempxf);
	    tempx = tempxf;
	    invert = FALSE;
	    print_inv();
	} else {
	    tempxf = exp(tempxf);
	    tempx = tempxf;
	}
	break;
	
    case LOGE:
	if (invert) {
	    tempxf = exp(tempxf);
	    tempx = tempxf;
	    invert = FALSE;
	    print_inv();
	} else {
	    tempxf = log(tempxf);
	    tempx = tempxf;
	}
	break;
	
    case LOG10:
	if (invert) {
	    tempxf = pow(10.0, tempxf);
	    tempx = tempxf;
	    invert = FALSE;
	    print_inv();
	} else {
	    tempxf = log10(tempxf);
	    tempx = tempxf;
	}
	break;
	
	/* Square root keys ... */
	
    case SQR:
	tempx *= tempx;
	tempxf *= tempxf;
	break;
	
    case ROOT:
	tempyf = tempx;
	tempx = round(sqrt(tempyf));
	tempxf = sqrt(tempxf);
	break;
	
    case CUBE:
	tempx = tempx * tempx * tempx;
	tempxf = tempxf * tempxf * tempxf;
	break;
	
    case CROOT:
	if (tempxf < 0.0) {
	    tempx = -tempx;
	    tempxf = -tempxf;
	    negative_root = TRUE;
	} else
	    negative_root = FALSE;
	
	/* Integer part ... */
	tempyf = tempx;
	tempx = round(pow(tempyf, 1.0 / 3.0));
	/* Floating point part ... */
	tempxf = pow(tempxf, 1.0 / 3.0);
	
	if (negative_root) {
	    tempx = -tempx;
	    tempxf = -tempxf;
	}
	break;
	
    case FRC:
	tempy = 0;
	if (tempxf < 0.0) {
	    tempy = 1;
	    tempxf = -tempxf;
	}
	tempxf = tempxf - floor(tempxf);
	if (tempy)
	    tempxf = -tempxf;
	tempx = 0;
	break;
	
    case INT:
	tempy = 0;
	if (tempxf < 0.0) {
	    tempy = 1;
	    tempxf = -tempxf;
	}
	tempxf = (double) floor(tempxf);
	if (tempy)
	    tempxf = -tempxf;
	break;
    default:
	put_a_char(BELL);
	break;
    }
    push(tempx, tempxf);
    dispnums();
}

void summation(c)
    COMMAND c;
{
    long	tempx, tempy;
    double	tempxf, tempyf;
    
    switch(c) {
    case SUM:
	save_x();
	pop(&tempx, &tempxf);
	reg[7]++;
	regf[7]++;
	reg[8] += tempx;
	regf[8] += tempxf;
	reg[9] += tempx * tempx;
	regf[9] += tempxf * tempxf;
	push(reg[7], regf[7]);
	dispreg(7);
	dispreg(8);
	dispreg(9);
	dispnums();
	msg("X accumulated");
	break;
	
    case SUMR:
	if (reg[7] > 0) {
	    save_x();
	    pop(&tempx, &tempxf);
	    reg[7]--;
	    regf[7]--;
	    reg[8] -= tempx;
	    regf[8] -= tempxf;
	    reg[9] -= tempx * tempx;
	    regf[9] -= tempxf * tempxf;
	    dispreg(7);
	    dispreg(8);
	    dispreg(9);
	    dispnums();
	    msg("X decremented");
	}
	break;
	
    case SUM0:
	reg[7] = 0;
	regf[7] = 0.0;
	reg[8] = 0;
	regf[8] = 0.0;
	reg[9] = 0;
	regf[9] = 0.0;
	dispreg(7);
	dispreg(8);
	dispreg(9);
	dispnums();
	msg("Accumulator registers cleared");
	break;
	
    case NSTAT:
	push(reg[7], regf[7]);
	dispnums();
	break;
	
    case MEAN:
	if (regf[7] == 0.0)
	    msg(div_by_zero);
	else {
	    tempxf = regf[8] / regf[7];
	    tempx = tempxf;
	    push(tempx, tempxf);
	    dispnums();
	}
	break;
	
    case S_DEV:
	/* s.dev = sqrt( n * sum(x^2) - (sum(x))^2 ) / n */
	if (regf[7] == 0.0)
	    msg(div_by_zero);
	else {
	    tempxf = sqrt( regf[7] * regf[9] - regf[8] * regf[8]) / regf[7];
	    tempx = tempxf;
	    push(tempx, tempxf);
	    dispnums();
	}
	break;
    default:
	put_a_char(BELL);
	break;
    }
}

void financial(c)
    COMMAND c;
{
    /*
     * Financial function register allocation ...
     *
     * 5 n
     * 6 interest
     * 7 present value
     * 8 payment
     * 9 future value
     *
     * Formulae are :
     *
     * 0 = PVAL + (1 + I.S) PMT (1 - (1+I)**-N ) / I + FVAL * (1+I)**-N
     *
     * where:
     *
     * N    = number of periods
     * PVAL = present value
     * I    = interest rate (as a fraction - user sees a percent)
     * PMT  = payment
     * FVAL = future value
     * S    = payment mode factor. 0 for payment at end of period, 1 for start.
     *        This version only uses S=0.
     */
    
#define NUM_REG		5
#define INT_REG		6
#define PVAL_REG	7
#define PMT_REG		8
#define FVAL_REG	9
    
    long	tempx, tempy;
    double	tempxf, tempyf;
    int 	i, count;
    BOOLEAN	this_was_fin_key;
    double interest, n, pval, pmt, fval, guess, epsilon;
    
    this_was_fin_key = FALSE;
    switch (c) {
    case PVAL:
	if (invert) {
	    push(reg[PVAL_REG], regf[PVAL_REG]);
	    invert = FALSE;
	    print_inv();
	    dispnums();
	    msg("Present value recalled");
	} else if (!last_was_fin_key) {
	    save_x();
	    pop(&tempx, &tempxf);
	    regf[PVAL_REG] = tempxf;
	    reg[PVAL_REG] = round(tempxf);
	    dispreg(PVAL_REG);
	    dispnums();
	    this_was_fin_key = TRUE;
	    msg("Present value entered");
	} else {
	    n = regf[NUM_REG];
	    interest = regf[INT_REG] / 100.0;
	    if (interest == 0.0) {
		msg(div_by_zero);
		break;
	    }
	    pmt = regf[PMT_REG];
	    fval = regf[FVAL_REG];
	    tempyf = pow(1.0 + interest, -n);
	    tempxf = pmt * (1.0 - tempyf) / interest
		+ fval * tempyf;
	    tempxf = -tempxf;
	    push(round(tempxf), tempxf);
	    regf[PVAL_REG] = tempxf;
	    reg[PVAL_REG] = round(tempxf);
	    dispreg(PVAL_REG);
	    dispnums();
	    msg("Present value computed");
	}
	break;
	
    case FVAL:
	if (invert) {
	    push(reg[FVAL_REG], regf[FVAL_REG]);
	    invert = FALSE;
	    print_inv();
	    dispnums();
	    msg("Future value recalled");
	} else if (!last_was_fin_key) {
	    save_x();
	    pop(&tempx, &tempxf);
	    regf[FVAL_REG] = tempxf;
	    reg[FVAL_REG] = round(tempxf);
	    dispreg(FVAL_REG);
	    dispnums();
	    this_was_fin_key = TRUE;
	    msg("Final value entered");
	} else {
	    n = regf[NUM_REG];
	    interest = regf[INT_REG] / 100.0;
	    pval = regf[PVAL_REG];
	    pmt = regf[PMT_REG];
	    tempyf = pow(1.0 + interest, n);
	    if ((tempyf == 0.0) || (interest == 0.0)) {
		msg(div_by_zero);
		break;
	    }
	    tempxf = (pval + pmt * (1.0 - 1.0 / tempyf) / interest) * tempyf;
	    tempxf = -tempxf;
	    push(round(tempxf), tempxf);
	    regf[FVAL_REG] = tempxf;
	    reg[FVAL_REG] = round(tempxf);
	    dispreg(FVAL_REG);
	    dispnums();
	    msg("Final value calculated");
	}
	break;
	
    case PMT:
	if (invert) {
	    push(reg[PMT_REG], regf[PMT_REG]);
	    invert = FALSE;
	    print_inv();
	    dispnums();
	    msg("Payment recalled");
	} else if (!last_was_fin_key) {
	    save_x();
	    pop(&tempx, &tempxf);
	    regf[PMT_REG] = tempxf;
	    reg[PMT_REG] = round(tempxf);
	    dispreg(PMT_REG);
	    dispnums();
	    this_was_fin_key = TRUE;
	    msg("Payment entered");
	} else {
	    n = regf[NUM_REG];
	    interest = regf[INT_REG] / 100.0;
	    pval = regf[PVAL_REG];
	    fval = regf[FVAL_REG];
	    tempyf = pow(1.0 + interest, -n);
	    if (tempyf == 1.0) {
		msg(div_by_zero);
		break;
	    }
	    tempxf = (pval + fval * tempyf) * interest / (1.0 - tempyf);
	    tempxf = -tempxf;
	    push(round(tempxf), tempxf);
	    regf[PMT_REG] = tempxf;
	    reg[PMT_REG] = round(tempxf);
	    dispreg(PMT_REG);
	    dispnums();
	    msg("Payment calculated");
	}
	break;
	
    case INTST:
	if (invert) {
	    push(reg[INT_REG], regf[INT_REG]);
	    invert = FALSE;
	    print_inv();
	    dispnums();
	    msg("Interest recalled");
	} else if (!last_was_fin_key) {
	    save_x();
	    pop(&tempx, &tempxf);
	    regf[INT_REG] = tempxf;
	    reg[INT_REG] = round(tempxf);
	    dispreg(INT_REG);
	    dispnums();
	    this_was_fin_key = TRUE;
	    msg("Interest rate (%) entered");
	} else {
	    n = regf[NUM_REG];
	    pval = regf[PVAL_REG];
	    pmt = regf[PMT_REG];
	    fval = regf[FVAL_REG];
	    /* use Newton-Raphson iteration ...
	       if we are seeking roots of f(x)=0 then
	       x_n+1 = x_n - f(x_n) / f'(x_n)
	       */
	    count = 0;
	    epsilon = 1000.0;
	    guess = 0.01; /* == 12% per annum */
	    while ((count < 10000) && (fabs(epsilon) > 0.0001)) {
		tempyf = pow(1.0 + guess, -n);
		tempxf = n * (pmt + pval * guess);
		if ((guess == -1.0) || (tempxf == 0.0)) {
		    count = 32000;
		    break;
		}
		epsilon = ( (pmt + pval * guess) - (pmt - fval * guess) * tempyf) /
		    (tempxf / (1.0 + guess) + pval + fval * tempyf);
		guess -= epsilon;
		count++;
	    }
	    tempxf = guess * 100.0; /* Convert to percent */
	    if (count < 10000) {
		push(round(tempxf), tempxf);
		regf[INT_REG] = tempxf;
		reg[INT_REG] = round(tempxf);
		dispreg(INT_REG);
		dispnums();
		msg("Interest rate (%) computed");
	    } else
		msg("Calculation does not converge!\007");
	}
	break;
	
    case NFIN:
	if (invert) {
	    push(reg[NUM_REG], regf[NUM_REG]);
	    invert = FALSE;
	    print_inv();
	    dispnums();
	    msg("Number of payments recalled");
	} else if (!last_was_fin_key) {
	    save_x();
	    pop(&tempx, &tempxf);
	    regf[NUM_REG] = tempxf;
	    reg[NUM_REG] = round(tempxf);
	    dispreg(NUM_REG);
	    dispnums();
	    this_was_fin_key = TRUE;
	    msg("Number of payments entered");
	} else {
	    interest = regf[INT_REG] / 100.0;
	    pval = regf[PVAL_REG];
	    pmt = regf[PMT_REG];
	    fval = regf[FVAL_REG];
	    tempxf = pmt + pval * interest;
	    if ((tempxf == 0.0) || (interest == 0.0)) {
		msg(div_by_zero);
		break;
	    }
	    tempxf = log((pmt - fval * interest) / tempxf)
		/ log(1.0 + interest);
	    push(round(tempxf), tempxf);
	    regf[NUM_REG] = tempxf;
	    reg[NUM_REG] = round(tempxf);
	    dispreg(NUM_REG);
	    dispnums();
	    msg("Number of payments computed");
	}
	break;
	
    default:
	put_a_char(BELL);
	break;
    }
    last_was_fin_key = FALSE;
    if (this_was_fin_key)
	last_was_fin_key = TRUE;
}

void process_digit(c)
    COMMAND c;
{
    switch (c) {
    case BACKSPACE:
    case '0':
    case '1':
	echo_char(c);
	break;
	
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
	if (base == 2)
	    put_a_char(BELL);
	else
	    echo_char(c);
	break;
	
    case '8':
    case '9':
	if ((base == 2) || (base == 8))
	    put_a_char(BELL);
	else
	    echo_char(c);
	break;
	
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'F':
	if (base != 16)
	    put_a_char(BELL);
	else
	    echo_char(c);
	break;
	
    case 'E':
	if (base == 16)
	    echo_char(c);
	else if (base > 0)
	    put_a_char(BELL);
	else {
	    if (e_legal) {
		echo_char(c);
		e_legal = FALSE;
	    }
	    else
		put_a_char(BELL);
	}
	break;
	
    case '.':
	if (base > 0) /* base == 1 (ASCIIM) never reaches here! */
	    put_a_char(BELL);
	else if (entering && !dot_legal)
	    put_a_char(BELL);
	else
	    echo_char(c);
	dot_legal = FALSE;
	break;
	
    default:
	put_a_char(BELL);
	break;
    }
}

void process(c)
    COMMAND	*c;
{
    double	tempxf, tempyf;
    long	tempx, tempy;
    COMMAND d;
    int	i, j;
    extern 	int is_resident;
    
    d = *c;
    
    /* ASCII MODE - almost any character can be echo'ed */
    if ((*c < MIN_COMMAND) && (base == 1)) {
	echo_char(*c); /* An ordinary character was typed */
	*c = 0;
	return;
    }
    
    if (d == INV) {
	invert = !invert;
	print_inv();
	return;
    }    
    
    if ((d == PVAL) ||
	(d == FVAL) ||
	(d == PMT) ||
	(d == INTST) ||
	(d == NFIN)) {            
	financial(*c);
	return;
    }
    
    last_was_fin_key = FALSE;
    
    if (((d >= '0') && (d <= '9')) ||
	((d >= 'A') && (d <= 'F')) ||
	(d == BACKSPACE) ||
	(d == '.')) {
	process_digit(d);
	return;
    }
    
    if (((d >= '!') && (d <= '/')) ||
	((d >= ':') && (d <= '@')) ||
	((d >= '[') && (d <= '`')) ||
	((d >= '{') && (d <= '~')) ||
	(d == 'Y')		   ||
	(d == RECI)) {
	arith(d);
	return;
    }
    
    switch (*c) {
    case FLOAT:
    case ASCIIM:
    case ENG:
    case BIN:
    case OCT:
    case DEC:
    case HEX:
	change_base(*c);
	break;
	
    case NEXTKEYS:
	fk_set++;
	if (fk_set == NUMFK)
	    fk_set = 0;
	invert = FALSE;
	print_inv();
	print_fk(fk_set);
	break;
	
    case PREVKEYS:
	fk_set--;
	if (fk_set < 0)
	    fk_set = NUMFK - 1;
	invert = FALSE;
	print_inv();
	print_fk(fk_set);
	break;
	
    case ROLLDOWN:
	pop(&tempx, &tempxf);
	t = tempx;
	tf = tempxf;
	dispnums();
	break;
	
    case CLX: /* Clear X */
	pop(&l, &lf);
	tempx = 0;
	tempxf = 0.0;
	push(tempx, tempxf);
	lift_needed = FALSE;
	dispnums();
	break;
	
    case 'M':
	degree = !degree;
	print_deg();
	break;
	
    case 'P':
	tempxf = pi;
	tempx = tempxf;
	push(tempx, tempxf);
	dispnums();
	break;
	
    case SIN:
    case COS:
    case TAN:
    case LOGE:
    case LOG10:
    case SQR:
    case ROOT:
    case CUBE:
    case CROOT:
    case FRC:
    case INT:
    case ETOX:
	trig(*c);
	break;
	
    case SUMR:
    case SUM0:
    case NSTAT:
    case MEAN:
    case S_DEV:
    case SUM:
	summation(*c);
	break;
	
    case ENTER:
	pop(&tempx, &tempxf);
	push(tempx, tempxf);
	push(tempx, tempxf);
	lift_needed = FALSE;
	dispnums();
	break;
	
    case 'L': /* Last X */
	push(l, lf);
	dispnums();
	break;
	
    case QUIT:
	if (is_resident)
	    *c = QUIT;
	else {
	    msg("Really quit? (Y or N)");
	    *c = get_a_char(&d);
	    if (toupper(*c) == 'Y')
		*c = QUIT;
	    else
		*c = '.';
	    clear_msg();
	}
	break;
	
    case 'R':
	i = get_int("Recall from register [0-9]");
	if ((i >= 0) && (i < NUMREGS)) {
	    tempx = reg[i];
	    tempxf = regf[i];
	    push(tempx, tempxf);
	    dispnums();
	}
	break;
	
    case 'S':
	stop_entering();
	msg("Store in register [0-9] (or [+-*/] reg.)");
	i = get_a_char(&j);
	clear_msg();
	j = i;
	if ((j=='+') || (j=='-') || (j=='*') || (j=='/'))
	    i = get_int("Store in register [0-9]");
	else {
	    j = 0;
	    if (isdigit(i))
		i -= '0';
	    else {
		put_a_char(BELL);
		break;
	    }
	}
	if ((i >= 0) && (i < NUMREGS)) {
	    switch (j) {
	    case 0:
		reg[i] = x;
		regf[i] = xf;
		break;
	    case '+':
		reg[i] += x;
		regf[i] += xf;
		break;
	    case '-':
		reg[i] -= x;
		regf[i] -= xf;
		break;
	    case '*':
		reg[i] *= x;
		regf[i] *= xf;
		break;
	    case '/':
		reg[i] /= x;
		regf[i] /= xf;
		break;
	    }
	    dispreg(i);
	}
	break;
	
    case 'X':
	pop(&tempx, &tempxf);
	pop(&tempy, &tempyf);
	push(tempx, tempxf);
	push(tempy, tempyf);
	dispnums();
	break;
	
    case '?':
    case 'H':
    case HELP:
	pop_up_help();
	break;
	
    default:
	put_a_char(BELL);
	break;
    }
}

void initialise()
{
    int	i;
    
    if (!was_initialised) {
	was_initialised = 1;
	os_init();
	
	x = y = z = t = l = 0;
	xf = yf = zf = tf = lf = 0.0;
	base = -2;
	dot_legal = TRUE;
	e_legal = TRUE;
	entering = FALSE;
	lift_needed = TRUE;
	invert = FALSE;
	degree = TRUE;
	last_was_fin_key = FALSE;
	inptr = inbuf;
	*inptr = ASCIIZ;
	fk_set = 0;
	max_digits = 20;
	pi = 4.0 * atan(1.0);
	d_to_r = pi / 180.0;
	r_to_d = 180.0 / pi;
	
	for (i = 0; i < NUMREGS; i++) {
	    reg[i] = 0;
	    regf[i] = 0.0;
	}
    }
}

void terminate()
{
    os_term();
}


