/* sched.c - planifica una funcin para ser llamada en
 * cada interrupcin del reloj */


/* Copyright (C) 1998 por Ori Pomerantz */


/* Los ficheros de cabeceras necesarios */

/* Estndar en los mdulos del ncleo */
#include <linux/kernel.h>   /* Estamos haciendo trabajo del ncleo */
#include <linux/module.h>   /* Especficamente, un mdulo */

/* Distribuido con CONFIG_MODVERSIONS */
#if CONFIG_MODVERSIONS==1
#define MODVERSIONS
#include <linux/modversions.h>
#endif        

/* Necesario porque usamos el sistema de ficheros proc */
#include <linux/proc_fs.h>

/* Planificamos tareas aqu */
#include <linux/tqueue.h>

/* Tambin necesitamos la habilidad para ponernos a dormir
 * y despertanos ms tarde */
#include <linux/sched.h>

/* En 2.2.3 /usr/include/linux/version.h se incluye una
 * macro para esto, pero 2.0.35 no lo hace - por lo tanto
 * lo aado aqu si es necesario. */
#ifndef KERNEL_VERSION
#define KERNEL_VERSION(a,b,c) ((a)*65536+(b)*256+(c))
#endif



/* El nmero de veces que la interrupcin del reloj
 * ha sido llamada */
static int TimerIntrpt = 0;


/* Esto lo usa cleanup, para evitar que el mdulo
 * sea descargado mientras intrpt_routine est
 * an en la cola de tareas */ 
static struct wait_queue *WaitQ = NULL;

static void intrpt_routine(void *);


/* La estructura de cola de tareas para esta tarea, de tqueue.h */
static struct tq_struct Task = {
  NULL,   /* Prximo elemento en la lista - queue_task har 
           * esto por nosotros */
  0,      /* Una bandera significando que todava no hemos 
	   * insertado en la cola de tareas */
  intrpt_routine, /* La funcin a ejecutar */
  NULL    /* El parmetro void* para esta funcin */
};



/* Esta funcin ser llamada en cada interrupcin de reloj.
 * Ntese que el puntero *void - funciones de la tarea
 * puede ser usado para ms de un propsito, obteniendo
 * cada vez un parmetro diferente. */

static void intrpt_routine(void *irrelevant)
{
  /* Incrementa el contador */
  TimerIntrpt++;

  /* Si cleanup nos quiere matar */
  if (WaitQ != NULL) 
    wake_up(&WaitQ);   /* Ahora cleanup_module puede retornar */
  else
    /* Nos vuelve a poner en la cola de tareas */
    queue_task(&Task, &tq_timer);  
}




/* Pone datos en el fichero del sistema de ficheros proc. */
int procfile_read(char *buffer, 
                  char **buffer_location, off_t offset, 
                  int buffer_length, int zero)
{
  int len;  /* Nmero de bytes usados actualmente */

  /* Esto es esttico por lo tanto permanecer en memoria 
   * cuando deje esta funcin */
  static char my_buffer[80];  

  static int count = 1;

  /* Damos toda nuestra informacin de una vez, por lo
   * tanto si alguien nos pregunta si tenemos ms
   * informacin la respuesta debera de ser no.
   */
  if (offset > 0)
    return 0;

  /* Rellena el buffer y obtiene su longitud */
  len = sprintf(my_buffer, 
                "Timer fue llamado %d veces\n", 
                TimerIntrpt);
  count++;

  /* Dice a la funcin que nos ha llamado dnde
   * est el buffer */
  *buffer_location = my_buffer;

  /* Retorna la longitud */
  return len;
}


struct proc_dir_entry Our_Proc_File = 
  {
    0, /* Nmero de inodo - ignralo, ser rellenado por 
        * proc_register_dynamic */
    5, /* Longitud del nombre del fichero */
    "sched", /* El nombre del fichero */
    S_IFREG | S_IRUGO, 
    /* Modo del fichero - este es un fichero normal que puede
     * ser leido por su dueo, su grupo, y por todo el mundo */
    1,  /* Nmero de enlaces (directorios donde  
         * el fichero es referenciado) */
    0, 0,  /* El uid y gid para el fichero - se lo damos 
            * a root */
    80, /* El tamao del fichero indicado por ls. */
    NULL, /* funciones que pueden ser realizadas en el 
           * inodo (enlace, borrado, etc.) - no 
           * soportamos ninguna. */
    procfile_read, 
    /* La funcin read para este fichero, la funcin llamada
     * cuando alguien intenta leer algo de l. */
    NULL 
    /* Podemos tener aqu una funcin para rellenar
     * el inodo del fichero, para permitirnos jugar con
     * los permisos, dueo, etc. */
  }; 


/* Inicializa el mdulo - registra el fichero proc */
int init_module()
{
  /* Pone la tarea en la cola de tareas tq_timer, por lo
   * tanto ser ejecutado en la siguiente interrupcin del reloj */
  queue_task(&Task, &tq_timer);

  /* Tiene xito si proc_register_dynamic tiene xito.
   * falla en otro caso */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,2,0)
  return proc_register(&proc_root, &Our_Proc_File);
#else
  return proc_register_dynamic(&proc_root, &Our_Proc_File);
#endif
}


/* Limpieza */
void cleanup_module()
{
  /* libera nuestro fichero /proc */
  proc_unregister(&proc_root, Our_Proc_File.low_ino);
  
  /* Duerme hasta que intrpt_routine es llamada por ltima
   * vez. Esto es necesario, porque en otro caso, desasignaremos
   * la memoria manteniendo intrpt_routine y Task mientras
   * tq_timer an las referencia. Destacar que no permitimos
   * seales que nos interrumpan.
   *
   * Como WaitQ no es ahora NULL, esto dice automticamente
   * a la rutina de interrupcin su momento de muerte. */
 sleep_on(&WaitQ);
}  






















