/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999	Eduardo M Kalinowski (ekalin@iname.com)
     
     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>

#include "cmp.h"
#include "kalc.h"

int lastArgN = 0;
Object *lastArgs[LASTARGS];



void run1_1_Function(Object (*f)(Object n, int *err), char *name)
{
  /*
   * This function runs the f function (whose name is given), which
   * takes one argument and returns one result. It checks that there
   * is at least one object in the stack.
   */

  int errCode;
  Object result;
  
  if (enoughArgs(1)) {
    saveLastArgs(1);
    result = (*f)(**tos, &errCode);  
  } else
    errCode = ERR_TOOFEWARGUMENTS;
  
  /* If we got a "Bad Argument Type Error", strip tags and try again */
  if (errCode == ERR_BADARGUMENTTYPE) {
    dtag(tos);
    result = (*f)(**tos, &errCode);
  }

  if (errCode)
    doError(name, errCode);
  else {
    /*
    insertObject(result);
    _f_swap();
    _f_drop();
    */
    _f_drop();
    insertObject(result);
  }
}


void run2_1_Function(Object (*f)(Object n, Object p, int *err), char *name)
{
  /*
   * This function runs the f function (whose address is given), which
   * takes two arguments and returns one result. It checks that there
   * is at least two object in the stack.
   */

  int errCode;
  Object result;

  if (enoughArgs(2)) {
    saveLastArgs(2);
    result = (*f)(**(tos - 1), **tos, &errCode);
  } else
    errCode = ERR_TOOFEWARGUMENTS;

  /* If we got a "Bad Argument Type Error", strip tags and try again */
  if (errCode == ERR_BADARGUMENTTYPE) {
    dtag(tos - 1);
    dtag(tos);
    result = (*f)(**(tos - 1), **tos, &errCode);
  }  

  if (errCode)
    doError(name, errCode);
  else {
    /*
    insertObject(result);
    _f_rolld(3, NULL);
    _f_drop();
    _f_drop();
    */
    _f_drop();
    _f_drop();
    insertObject(result);
  }
}


void run3_1_Function(Object (*f)(Object n, Object p, Object q, int *err),
		     char *name)
{
  /*
   * This function runs the f function (whose name is given), which
   * takes three arguments and returns one result. It checks that there
   * is at least one object in the stack.
   */

  int errCode;
  Object result;

  if (enoughArgs(3)) {
    saveLastArgs(3);
    result = (*f)(**(tos - 2), **(tos - 1), **tos, &errCode);
  } else
    errCode = ERR_TOOFEWARGUMENTS;

  /* If we got a "Bad Argument Type Error", strip tags and try again */
  if (errCode == ERR_BADARGUMENTTYPE) {
    dtag(tos - 2);
    dtag(tos - 1);
    dtag(tos);
    result = (*f)(**(tos - 2), **(tos - 1), **tos, &errCode);
  }

  if (errCode)
    doError(name, errCode);
  else {
    /*
    insertObject(result);
    _f_rolld(4, NULL);
    _f_drop();
    _f_drop();
    _f_drop();
    */
    _f_drop();
    _f_drop();
    _f_drop();
    insertObject(result);
  }
}
  
 

void run1_2_Function(Object (*f)(Object n, Object *r, int *err), char *name)
{
  /*
   * This function runs the f function (whose name is given), which
   * takes one argument and returns no results. It checks that there
   * is at least one object in the stack.
   */

  int errCode;
  Object result1, result2;

  if (enoughArgs(1)) {
    saveLastArgs(1);
    result1 = (*f)(**tos, &result2, &errCode);
  } else
    errCode = ERR_TOOFEWARGUMENTS;

  /* If we got a "Bad Argument Type Error", strip tags and try again */
  if (errCode == ERR_BADARGUMENTTYPE) {
    dtag(tos);
    result1 = (*f)(**tos, &result2, &errCode);
  }

  if (errCode)
    doError(name, errCode);
  else {
    _f_drop();
    insertObject(result1);
    insertObject(result2);
  }
}


void run1_0_Function(int (*f)(Object n), char *name)
{
  /*
   * This function runs the f function (whose name is given), which
   * takes one argument and returns nothing. It checks that there
   * is at least one object in the stack.
   */

  int errCode;
  
  if (enoughArgs(1)) {
    saveLastArgs(1);
    errCode = (*f)(**tos);
  } else
    errCode = ERR_TOOFEWARGUMENTS;

  /* If we got a "Bad Argument Type Error", strip tags and try again */
  if (errCode == ERR_BADARGUMENTTYPE) {
    dtag(tos);
    errCode = (*f)(**tos);
  }

  if (errCode)
    doError(name, errCode);
  else
    _f_drop();
}


#if 0
void run2_0_Function(int (*f)(Object n, Object p), char *name)
{
  /*
   * This function runs the f function (whose address is given). That
   * function does not return any result, although it is free to do what it
   * wants with the stack.
   */

  Object *obj1, *obj2;
  int errCode;

  if (enoughArgs(2)) {
    saveLastArgs(2);
    obj1 = *tos;
    obj2 = *(tos - 1);
    --tos, --tos;
    errCode = (*f)(*obj2, *obj1);
  } else
    errCode = ERR_TOOFEWARGUMENTS;

  /* If we get a "Bad Argument Type" error, strip tags and try again. */
  if (errCode == ERR_BADARGUMENTTYPE) {
    dtag(&obj2);
    dtag(&obj1);
    errCode = (*f)(*obj2, *obj1);
  }
 
  if (errCode)
    doError(name, errCode);
  else {
    freeObjectSpecials(*obj2);
    freeObjectSpecials(*obj1);
    free(obj2);
    free(obj1);
  }
}
#endif


void saveLastArgs(int n) 
{
  /*
   * This function saves n objects from the stack into the lastArg
   * area. Older objects are cleared.
   */

  register int i;
  
  if (n > LASTARGS)
    return;

  freeLastArgs();

  lastArgN = n;
  for (i = 0; i < n; i++)
    lastArgs[i] = objdup(*(tos - i));

  dirty = 1;
}


void freeLastArgs(void) 
{
  /*
   * This functions erases all arguments saved in lastArg area.
   */

  register int i;
  
  for (i = 0; i < lastArgN; i++) {
    freeObjectSpecials(*lastArgs[i]);
    free(lastArgs[i]);
  }
}


void storeLastArgs(FILE *fp) 
{
  /*
   * This function stores the saved last arguments to the file pointed
   * to by fp.
   * The format of the file stored is:
   * - An int representing the number of saved objects
   * - The objects itself, stored by saveObject()
   */

  register int i;

  fwrite(&lastArgN, sizeof(lastArgN), 1, fp);
  for (i = 0; i < lastArgN; i++)
    saveObject(lastArgs[i], fp);
}


void loadLastArgs(FILE *fp) 
{
  /*
   * This function reads the last arguments saved from the file pointed
   * to by fp. For the specification of how the objects are stored, see
   * the storeLastArgs() function.
   */

  register int i;
  Object el;

  fread(&lastArgN, sizeof(lastArgN), 1, fp);
  for (i = 0; i < lastArgN; i++) {
    el = loadObject(fp);
    lastArgs[i] = objdup(&el);
  }
}  
  

void doError(char *name, int errCode)
{
  /*
   * This function displays the error message for the specific error code.
   *
   * If errCode == ERR_CUSTOM, then name (which is normally prefixed before
   * the error message) is displayed as the error message.
   *
   * If errCode == ERR_NOTENOUGHMEMORY (a fatal error), the finalize()
   * function is called to save the stack and memory (or try, at least).
   */

  char *msg;

  switch (errCode) {
  case ERR_BADARGUMENTTYPE:
    msg = "Bad Argument Type";
    break;

  case ERR_BADARGUMENTVALUE:
    msg = "Bad Argument Value";
    break;
    
  case ERR_COULDNOTCHDIR:
    msg = "Couldn't Change Directory";
    break;

  case ERR_COULDNOTOPENFILE:
    msg = "Couldn't Open File";
    break;

  case ERR_INVALIDDATE:
    msg = "Invalid Date";
    break;

  case ERR_INVALIDVERSION:
    msg = "Object File Is Of Different Version";
    break;

  case ERR_MAXLOCALENVS:
    msg = "Maximum Number Of Local Environments Reached";
    break;

  case ERR_MISSINGOBJECT:
    msg = "Missing Object";
    break;

  case ERR_NONREALRESULT:
    msg = "Non-Real Result";
    break;

  case ERR_NOTEMPENVS:
    msg = "No Temporary Environments";
    break;

  case ERR_NOTENOUGHMEMORY:
    msg = "Not Enough Memory";
    break;

  case ERR_NOTINPROGRAM:
    msg = "Not Allowed Outside Program";
    break;

  case ERR_RECURSIONTOODEEP:
    msg = "Recursion Too Deep";
    break;

  case ERR_STACKOVERFLOW:
    msg = "Stack Overflow";
    break;

  case ERR_SYNTAXERROR:
    msg = "Syntax Error";
    break;

  case ERR_TOOFEWARGUMENTS:
    msg = "Too Few Arguments";
    break;
    
  case ERR_UNDEFINEDNAME:
    msg = "Undefined Name";
    break;

  case ERR_UNFINISHEDCOMP:
    msg = "Unfinished Composite Object";
    break;

  case ERR_UNSTARTEDPROG:
    msg = ">> Found Without Corresponding <<";
    break;

  case ERR_CUSTOM:
    msg = name;
    name = "";
    break;

  default:
    msg = "Unknown Error";
    break;
  }

  if (errCode != ERR_STACKOVERFLOW)
    printf("%s Error: %s\n", name, msg);
  else
    printf("Warning: %s\n", msg);

  if (errCode == ERR_NOTENOUGHMEMORY) {
    printf("Trying to save stack and memory...\n");
    finalize();
  }

  nFlow = -1;
  progNesting = 0;
  while (currMemLevel > 0)
    clearMemoryLevel(currMemLevel--);
  
  longjmp(jmpError, errCode);
}


/***********************************************
 * Table of functions, and searching functions *
 ***********************************************/
static FunctionList fList[] = {
  { "!", f_gamma },
  { "!=", f_neq },
  { "#", f_neq },
  { "$>id", f_strTOid },
  { "%", f_pctOf },
  { "%ch", f_pctCh },
  { "%t", f_pctT },
  { "*", f_multiply },
  { "+", f_add },
  { "-", f_subtract },
  { "->", f_arrow },
  { "-inf", f_minusInf },
  { "/", f_divide },
  { "<", f_lt },
  { "<<", startProgram },
  { "<=", f_lte },
  { "==", f_eq },
  { ">", f_gt },
  { ">=", f_gte },
  { ">>", endProgram },
  { ">hms", f_TOhms },
  { ">str", f_TOstr },
  { ">tag", f_TOtag },
  { "^", f_pow },
  { "_clvar", _doClvar },
  { "abnd", f_abnd },
  { "abort", exitAbort },
  { "abs", f_abs },
  { "acos", f_acos },
  { "acosh", f_acosh },
  { "acot", f_acot },
  { "acoth", f_acoth },
  { "acsc", f_acsc },
  { "acsch", f_acsch },
  { "alog", f_alog },
  { "and", f_and },
  { "arg", f_arg },
  { "asec", f_asec },
  { "asech", f_asech },
  { "asin", f_asin },
  { "asinh", f_asinh },
  { "asr", f_asr },
  { "atan", f_atan },
  { "atan2", f_atan2 },
  { "atanh", f_atanh },
  { "b>r", f_bTOr },
  { "bin", f_bin },
  { "c>r", f_cTOr },
  { "case", f_case },
  { "cd", doCd },
  { "ceil", f_ceil },
  { "chr", f_chr },
  { "chs", f_neg },
  { "cis", f_cis },
  { "clear", f_clear },
  { "clusr", doClvar },
  { "clvar", doClvar },
  { "coldStart", f_coldStart },
  { "comb", f_comb },
  { "conj", f_conj },
  { "cos", f_cos },
  { "cosh", f_cosh },
  { "cot", f_cot },
  { "coth", f_coth },
  { "csc", f_csc },
  { "csch", f_csch },
  { "cylin", f_cylin },
  { "d>r", f_dTOr },
  { "date", f_date },
  { "date+", f_dateAdd },
  { "ddays", f_ddays },
  { "dec", f_dec },
  { "deg", f_deg },
  { "depth", f_depth },
  { "diskRcl", diskRcl },
  { "diskSto", diskSto },
  { "do", f_do },
  { "dobind", f_dobind },
  { "dow", f_dow },
  { "dowstr", f_dowstr },
  { "drop", f_drop },
  { "drop2", f_drop2 },
  { "dropn", f_dropn },
  { "dtag", f_dtag },
  { "dup", f_dup },
  { "dup2", f_dup2 },
  { "dupdup", f_dupdup },
  { "dupn", f_dupn },
  { "else", f_else },
  { "end", f_end },
  { "eng", f_eng },
  { "eval", f_eval },
  { "exp", f_exp },
  { "expm1", f_expm1 },
  { "fix", f_fix },
  { "floor", f_floor },
  { "for", f_for },
  { "fp", f_fp },
  { "gcd", f_gcd },
  { "gd", f_gd },
  { "hav", f_hav },
  { "head", f_head },
  { "help", doHelp },
  { "hex", f_hex },
  { "hms+", f_hmsAdd },
  { "hms-", f_hmsSub },
  { "hms>", f_hmsTO },
  { "id>$", f_idTOstr },
  { "id>str", f_idTOstr },
  { "if", f_if },
  { "ift", f_ift },
  { "ifte", f_ifte },
  { "im", f_im },
  { "im>c", f_imTOc },
  { "inf", f_inf },
  { "inv", f_inv },
  { "invgd", f_invgd },
  { "ip", f_ip },
  { "keep", f_keep },
  { "lastarg", f_lastarg },
  { "lcm", f_lcm },
  { "lgamma", f_lgamma },
  { "ln", f_ln },
  { "lnp1", f_lnp1 },
  { "log", f_log },
  { "mant", f_mant },
  { "max", f_max },
  { "min", f_min },
  { "mod", f_mod },
  { "nan", f_nan },
  { "ndupn", f_ndupn },
  { "neg", f_neg },
  { "next", f_next },
  { "nip", f_nip },
  { "not", f_not },
  { "num", f_num },
  { "oct", f_oct },
  { "open", f_open },
  { "or", f_or },
  { "over", f_over },
  { "perm", f_perm },
  { "pi", f_pi },
  { "pick", f_pick },
  { "pick3", f_pick3 },
  { "polar", f_cylin },
  { "pos", f_pos },
  { "psign", f_psign },
  { "purge", doPurge },
  { "pwd", doPwd },
  { "q", finalize },
  { "quit", finalize },
  { "r>b", f_rTOb },
  { "r>c", f_rTOc },
  { "r>d", f_rTOd },
  { "rad", f_rad },
  { "rand", f_rand },
  { "rcl", doRcl },
  { "rcws", f_rcws },
  { "rdz", f_rdz },
  { "re", f_re },
  { "re>c", f_reTOc },
  { "rect", f_rect },
  { "repeat", f_repeat },
  { "repl", f_repl },
  { "rl", f_rl },
  { "rlb", f_rlb },
  { "rnd", f_rnd },
  { "roll", f_roll },
  { "rolld", f_rolld },
  { "rot", f_rot },
  { "rr", f_rr },
  { "rrb", f_rrb },
  { "save", f_save },
  { "saveAs", f_saveAs },
  { "sci", f_sci },
  { "sec", f_sec },
  { "sech", f_sech },
  { "set", doSet },
  { "shell", doShell },
  { "show", doShow },
  { "sign", f_sign },
  { "sin", f_sin },
  { "sinh", f_sinh },
  { "size", f_size },
  { "sl", f_sl },
  { "slb", f_slb },
  { "sq", f_sq },
  { "sqrt", f_sqrt },
  { "sr", f_sr },
  { "srb", f_srb },
  { "start", f_start },
  { "std", f_std },
  { "step", f_step },
  { "sto", doSto },
  { "str>", f_strTO },
  { "str>id", f_strTOid },
  { "stws", f_stws },
  { "sub", f_sub },
  { "swap", f_swap },
  { "tail", f_tail },
  { "tan", f_tan },
  { "tanh", f_tanh },
  { "then", f_then },
  { "time", f_time },
  { "tstr", f_tstr },
  { "type", f_type },
  { "unpick", f_unpick },
  { "unroll", f_rolld },
  { "unrot", f_unrot },
  { "until", f_until },
  { "vars", doVars },
  { "vers", f_vers },
  { "vtype", f_vtype },
  { "while", f_while },
  { "xor", f_xor },
  { "xpon", f_xpon },
  { "xroot", f_xroot },
};


int fnc_comp(const void *f1, const void *f2);

/*
 * The weird thing below is a function that takes a pointer to
 * character as argument, and returns a pointer to a function that
 * takes no arguments and returns nothing.
 */
void (*getFuncAddress(char *name))(void)
{
  /*
   * This function searches the function table for the function given
   * (with a binary search). It returns the address of the function
   * if successful, or NULL if the function was not found.
   */

  FunctionList target, *result;

  target.name = name;
  result = bsearch(&target,
		   fList,
		   sizeof(fList) / sizeof(FunctionList),
		   sizeof(FunctionList),
		   fnc_comp);
  if (result == NULL) return NULL;
  return result->address;
  
}

int fnc_comp(const void *f1, const void *f2) 
{
  return strcasecmp(((FunctionList *)f1)->name, ((FunctionList *)f2)->name);
}


char *getFuncName(void (*addr)(void)) 
{
  /*
   * This function returns the name of the function whose address is
   * given as argument, or NULL if not found. The pointer points to static
   * memory, and you should not change it.
   * Because the table of functions is sorted by function names, it is
   * necessary to do a linear search for the address. It is not worth
   * making a copy of the function table and sort that copy by address,
   * because that would take much memory, and because searching a
   * function's name given its address is much less often done than the
   * inverse operation.
   */
  register int i;
  int count = sizeof(fList) / sizeof(FunctionList);

  for (i = 0; i < count; i++)
    if (fList[i].address == addr)
      return fList[i].name;

  return NULL;
}
