/* The program memory in the HP-67 calculator. */

#ifndef PROGMEM_H
#define PROGMEM_H

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "calcfloat.h"

#include "hp67.h"
#include "datatypes.H"

extern run_mode runmode;

class CalcProgMem;

class ReturnAddr {
public:
  CalcProgMem *retaddr;
  unsigned int retstepnum;
  ReturnAddr *nextframe;
};



class KeyInfo {
public:
  char const *keyname;     /* The name of the key */
  int can_have_arg;  /* May have an argument (but not required to have one) */
  int immediate;     /* Can be used in immediate mode */
  int program;       /* Can be used in program mode */
};


class KeyType {

public:
  KeyInfo keycap;

private:

  int (*new_parse_arg)(char *rest, char **arg, char **comment);
  void (*new_print_line)(char *buffer, int blen, 
			 char const *arg, char const *comment);
  sp_acts (*new_action)(CalcProgMem *prog);

  static int keycount;
  static KeyType **sortedkeys;

public:

  KeyType(char const keyname[], 
	  int hasanarg,
	  int immediate,  /* Can be used in immediate mode */
	  int program,    /* Can be used in program mode */
	  int (* const fn_parse_arg)(char *rest, char **arg, char **comment),
	  void (*const fn_print_line)(char *buffer, int blen, 
				      char const *arg, char const *comment),
	  sp_acts (*const fn_action)(CalcProgMem *prog));

  ~KeyType(void) { free((void *) keycap.keyname);
		   if (--keycount == 0) free(sortedkeys); }
  int parse_arg(char *rest, char **arg, char **comment) const;

  void print_line(char *buffer, int blen, char const *arg,
		  char const *comment) const;
  
  sp_acts action(CalcProgMem *prog) const;

private: 
  static int compare_keys(const void *k1, const void *k2) {
    return strcmp((*((KeyType **)k1))->keycap.keyname, (*((KeyType **)k2))->keycap.keyname);
  }

  static int find_key(const void *key, const void *memb) {
    return strcmp((char *) key, (*((KeyType **)memb))->keycap.keyname);
  }

public: 
  inline static void keys_complete(KeyType *keylist[]);
  inline static sp_acts parse_command(char *inputline, CalcProgMem &target);

  static int numkeys(void) { return keycount; }

};



class ProgramSpace;

class CalcProgMem {

friend class KeyType;
friend class ProgramSpace;

public:
  int validcell;
  CalcFloat *value;
  char *argument;  /* Might be a label, might be a memory name */
  ProgramSpace *myspace;
  
private:
  KeyType *function;
  char *comment;   /* Part of the same malloc()ed block as argument */

  CalcProgMem *gototarget;   /* The target of a goto or gosub after label
				resolution */
  int gotolinenum;    /* if gototarget holds anything useful, gotolinenum is the number */
  
  CalcProgMem *prevmem;
  CalcProgMem *nextmem;

public:
  inline CalcProgMem(void);
  inline CalcProgMem(CalcProgMem const &from);
  inline CalcProgMem(char const *commandstr);

  inline CalcProgMem &operator= (CalcProgMem const &from);

  inline sp_acts do_step(void);
  void run(void);
  int perform_goto(void);
  void printcontents(char *buffer, int bufflen) const 
    {
      if (function == NULL) {
	value->printvalue(MAXDIGS, SCI, buffer, bufflen, NULL, NULL);
	if (argument != NULL && *comment != 0) {
	  strcat(buffer, " #");
	  strncat(buffer, comment, bufflen - strlen(buffer) - 1);
	}
      }
      else 
	function->print_line(buffer, bufflen, argument, comment);
    }

  CalcProgMem const *get_next_mem(void) const { return nextmem; }
  CalcProgMem const *get_prev_mem(void) const { return prevmem; }

private:
  inline void set_arg_comm(char const *arg, char const *comm);
  void copycontents(CalcProgMem *dest, CalcProgMem const &src) const;
};



class ProgramSpace {

friend class CalcProgMem;
friend sp_acts hp_run(CalcProgMem *);

private:
  ReturnAddr *frame0;           /* Next return address */
  CalcProgMem *firstmem;        /* Memory location 000 */
  CalcProgMem *lastmem;         /* Last memory location used */
  CalcProgMem *currentcell;     /* Insert after this cell. If NULL, 
				 * insert before cell 1 */
  unsigned int currentnum;

  unsigned int numcells;        /* Number of real, valid cells */
  CalcProgMem barrierlow;       /* An invalid cell to act as a barrier
				 * preventing wrap-around during execution */
  CalcProgMem barrierhigh;

  bool must_invalidate_gotos;

public:
  ProgramSpace(void) {
    frame0 = NULL;
    firstmem = &barrierlow;
    lastmem = &barrierhigh;
    currentcell = firstmem;
    currentnum = 0;
    numcells = 0;
    barrierhigh.nextmem = barrierlow.prevmem = NULL;
    barrierhigh.prevmem = &barrierlow;
    barrierlow.nextmem = &barrierhigh;
    must_invalidate_gotos = FALSE;
  }


  ~ProgramSpace(void) {
    erase_program();
  }

  inline void insert_cell(CalcProgMem &source);
  inline void delete_current_cell(void);
  CalcProgMem const *findlabel(char const *label, int *indirect);
  CalcProgMem const *findlinenum(int linenum);
  inline CalcProgMem const *get_currentcell_ptr(int *num) const {
    if (num != NULL) *num = currentnum;
    return currentcell; 
  }
  void push_retaddr(void);
  int pop_retaddr(void);
  void goto_stepnumber(unsigned int step);
  void erase_program(void);
  void writeoutput(FILE *ostream);

  sp_acts take_a_step(void) {
    CalcProgMem *oldpos = currentcell;
    ++(*this);
    return oldpos->do_step();
  }
  
  inline ProgramSpace &operator++(void);
  inline ProgramSpace &operator--(void);
  inline ProgramSpace &operator+=(int n);
  inline ProgramSpace &operator-=(int n);

private:
  void clear_return_stack(void);
  void invalidate_goto_ptrs(void);
};



/* Now, the KeyType functions. */



inline sp_acts KeyType::parse_command(char *inputline, CalcProgMem &target) {
  char *keyname, *arg, *comm, *remainder;
  char *cptr;
  KeyType **ans;
  
  // First, check to see if this is a valid number for the current arithmetic type
  if (target.value->parsevalue(inputline, &comm) == PARSE_OK) {
    /* A number */
    if ((cptr = strchr(comm, COMMENT)) != NULL) {
      target.set_arg_comm("", cptr + 1);
    } else {
      target.set_arg_comm("", "");
    }
    target.function = NULL;
    target.validcell = 1;
    return NONE;
  }
  
  
  // If we make it here, it's not an arithmetic value. A command, perhaps?
  
  keyname = inputline + strspn(inputline, WHITESPACE);
  cptr = keyname + strcspn(keyname, WHITESPACE);
  if (*cptr == 0)  /* No arg/comment stuff */
    *(cptr + 1) = 0;  /* Make room for an empty comment/arg */

  *cptr = 0;
  remainder = cptr + 1;    
  
  ans = (KeyType **) bsearch((const void *)keyname, sortedkeys, keycount,
			     sizeof(*sortedkeys),
			     (int (*)(const void *, const void *))find_key);

  if (ans == NULL) {
    target.validcell = 0;
    return NONE;
  } else {
    if ((runmode == IMMED &&
	 !(*ans)->keycap.immediate) ||
	(runmode == ENTER_PROG &&
	 !(*ans)->keycap.program)) {         // Command not valid at this time
      target.validcell = 0;
      return NONE;
    }

    target.function = *ans;
    switch ((*ans)->parse_arg(remainder, &arg, &comm)) {
    case 0:
      target.validcell = 1;
      target.set_arg_comm(arg, comm);
      return NONE;
      break;
    case 1:
      target.validcell = 1;
      target.set_arg_comm(arg, comm);
      return EXECUTE_NOW;
      break;
    default:
      target.validcell = 0;
      return NONE;
    }
  }
}


inline void KeyType::keys_complete(KeyType *keylist[]) {
  int i;
  
  sortedkeys = (KeyType **) malloc(keycount * sizeof(KeyType *));
  CHKMEM(sortedkeys);
  for (i = 0; i < keycount; i++)
    sortedkeys[i] = keylist[i];
  
  qsort((void *)sortedkeys, keycount, sizeof(*sortedkeys), 
	(int(*)(const void *, const void *)) compare_keys);
}





/************************************************
 CalcProgMem functions
*************************************************/




inline CalcProgMem::CalcProgMem(void)
{
  validcell = 0;
  comment = argument = NULL;
  value = new CalcFloat;
  gototarget = prevmem = nextmem = NULL;
}


inline CalcProgMem::CalcProgMem(CalcProgMem const &from)
{
  value = NULL;
  argument = NULL;
  copycontents(this, from);
}


inline CalcProgMem &CalcProgMem::operator= (CalcProgMem const &from)
{
  if (this != &from) {
    if (argument != NULL) free(argument);
    if (value != NULL) delete value;
    copycontents(this, from);
  }
  return *this;
}


sp_acts push_scalar(CalcProgMem *);

inline sp_acts CalcProgMem::do_step(void)
{
  if (!validcell) return ERROR;
  if (function == NULL) {
    return push_scalar(this);
  } else {
    return function->action(this);
  }
}


inline void ProgramSpace::insert_cell(CalcProgMem &source)
{
  CalcProgMem *holdn;

  if (must_invalidate_gotos)
    invalidate_goto_ptrs();

  clear_return_stack();

  if (currentcell == &barrierhigh) {   // Can't insert after barrier.
    --(*this);                      // Backup one and recurse in
    insert_cell(source);
    return;
  } else {
    holdn = new CalcProgMem(source);
    holdn->nextmem = currentcell->nextmem;
    holdn->prevmem = currentcell;
    currentcell->nextmem = holdn;
    if (holdn->nextmem != NULL)
      holdn->nextmem->prevmem = holdn;

    holdn->myspace = this;
    currentcell = holdn;
  }

  numcells++;
  currentnum++;
}


inline void CalcProgMem::set_arg_comm(char const *arg, char const *comm)
{
  if (argument != NULL) {
    free(argument);
    argument = comment = NULL;
  }
  if (arg == NULL) {
    argument = comment = NULL;
    return;
  }
  argument = (char *) malloc(strlen(arg) + strlen(comm) + 2);
  CHKMEM(argument);
  strcpy(argument, arg);
  comment = argument + strlen(arg) + 1;
  strcpy(comment, comm);
}



// The increment and decrement operators will move the program counter
// within the program space.
inline ProgramSpace &ProgramSpace::operator++(void)
{
  if ((currentcell = currentcell->nextmem) != NULL) {
    currentnum++;
  } else {
    currentcell = firstmem;
    currentnum = 0;
  }
  return *this;
}


inline ProgramSpace &ProgramSpace::operator--(void)
{
  if ((currentcell = currentcell->prevmem) != NULL) {
    currentnum--;
  } else {
    currentcell = lastmem;
    currentnum = numcells + 1;
  }
  return *this;
}


inline ProgramSpace &ProgramSpace::operator+=(int n)
{
  if (n < 0)
    return this->operator-=(-n);

  while (n--)
    this->operator++();

  return *this;
}


inline ProgramSpace &ProgramSpace::operator-=(int n)
{
  if (n < 0)
    return this->operator+=(-n);

  while (n--)
    this->operator--();

  return *this;
}


inline void ProgramSpace::delete_current_cell(void)
{
  CalcProgMem *holdcell = currentcell;
  
  if (must_invalidate_gotos)
    invalidate_goto_ptrs();

  clear_return_stack();

  if (numcells == 0 ||
      currentcell == firstmem ||
      currentcell == lastmem) return;   // No current cell, nothing to do

  currentcell = currentcell->prevmem;
  currentnum--;

  if (currentcell != NULL)
    currentcell->nextmem = holdcell->nextmem;

  if (holdcell->nextmem != NULL)
    holdcell->nextmem->prevmem = currentcell;

  delete holdcell;
  numcells--;
}

#endif  /* !PROGMEM_H */
